"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.KnowledgeBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const custom_resource_provider_1 = require("./custom-resource-provider");
const utils_1 = require("../../common/helpers/utils");
const opensearch_vectorindex_1 = require("../opensearch-vectorindex");
const opensearchserverless_1 = require("../opensearchserverless");
/**
 * Deploys a Bedrock Knowledge Base and configures a backend vector store.
 *
 * At the moment, only OpenSearch Serverless is supported as a vector store.
 * This construct creates the collection and index.
 */
class KnowledgeBase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * TagManager facilitates a common implementation of tagging for Constructs
         */
        this.cdkTagManager = new cdk.TagManager(cdk.TagType.MAP, 'Custom::Bedrock-KnowledgeBase');
        const embeddingsModel = props.embeddingsModel;
        this.instruction = props.instruction;
        const indexName = props.indexName ?? 'bedrock-knowledge-base-default-index';
        const vectorField = props.vectorField ?? 'bedrock-knowledge-base-default-vector';
        validateModel(embeddingsModel);
        this.name = (0, utils_1.generatePhysicalNameV2)(this, 'KB', { maxLength: 32 });
        const roleName = (0, utils_1.generatePhysicalNameV2)(this, 'AmazonBedrockExecutionRoleForKnowledgeBase', { maxLength: 64 });
        this.role = new iam.Role(this, 'Role', {
            roleName: roleName,
            assumedBy: new iam.ServicePrincipal('bedrock.amazonaws.com'),
        });
        this.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            principals: [new iam.ServicePrincipal('bedrock.amazonaws.com')],
            conditions: {
                StringEquals: {
                    'aws:SourceAccount': cdk.Stack.of(this).account,
                },
                ArnLike: {
                    'aws:SourceArn': cdk.Stack.of(this).formatArn({
                        service: 'bedrock',
                        resource: 'knowledge-base',
                        resourceName: '*',
                        arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                    }),
                },
            },
        }));
        this.role.addToPolicy(new iam.PolicyStatement({
            actions: ['bedrock:InvokeModel'],
            resources: [embeddingsModel.asArn(this)],
        }));
        if (props.vectorStore) {
            this.vectorStore = props.vectorStore;
        }
        else {
            this.vectorStore = new opensearchserverless_1.VectorCollection(this, 'KBVectors');
        }
        this.vectorStore.grantDataAccess(this.role);
        if (!props.vectorIndex) {
            this.vectorIndex = new opensearch_vectorindex_1.VectorIndex(this, 'KBIndex', {
                collection: this.vectorStore,
                indexName,
                vectorField,
                vectorDimensions: 1536,
                mappings: [
                    {
                        mappingField: 'AMAZON_BEDROCK_TEXT_CHUNK',
                        dataType: 'text',
                        filterable: true,
                    },
                    {
                        mappingField: 'AMAZON_BEDROCK_METADATA',
                        dataType: 'text',
                        filterable: false,
                    },
                ],
            });
            this.vectorIndex.node.addDependency(this.vectorStore);
        }
        else {
            this.vectorIndex = props.vectorIndex;
        }
        const crProvider = custom_resource_provider_1.BedrockCRProvider.getProvider(this);
        const knowledgeBase = new cdk.CustomResource(this, 'KB', {
            serviceToken: crProvider.serviceToken,
            resourceType: 'Custom::Bedrock-KnowledgeBase',
            properties: {
                knowledgeBaseConfiguration: {
                    type: 'VECTOR',
                    vectorKnowledgeBaseConfiguration: {
                        embeddingModelArn: embeddingsModel.asArn(this),
                    },
                },
                roleArn: this.role.roleArn,
                name: this.name,
                description: props.description,
                storageConfiguration: {
                    type: 'OPENSEARCH_SERVERLESS',
                    opensearchServerlessConfiguration: {
                        collectionArn: this.vectorStore.collectionArn,
                        vectorIndexName: indexName,
                        fieldMapping: {
                            vectorField: vectorField,
                            textField: 'AMAZON_BEDROCK_TEXT_CHUNK',
                            metadataField: 'AMAZON_BEDROCK_METADATA',
                        },
                    },
                },
                tags: this.cdkTagManager.renderedTags,
            },
        });
        const kbCRPolicy = new iam.Policy(this, 'KBCRPolicy', {
            roles: [crProvider.role],
            statements: [
                new iam.PolicyStatement({
                    actions: ['bedrock:CreateKnowledgeBase'],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    actions: [
                        'bedrock:UpdateKnowledgeBase',
                        'bedrock:DeleteKnowledgeBase',
                        'bedrock:TagResource',
                    ],
                    resources: [
                        cdk.Stack.of(this).formatArn({
                            service: 'bedrock',
                            resource: 'knowledge-base',
                            resourceName: '*',
                            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                        }),
                    ],
                }),
                new iam.PolicyStatement({
                    actions: ['iam:PassRole'],
                    resources: [this.role.roleArn],
                }),
            ],
        });
        knowledgeBase.node.addDependency(this.role);
        knowledgeBase.node.addDependency(kbCRPolicy);
        knowledgeBase.node.addDependency(this.vectorIndex);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(kbCRPolicy, [
            {
                id: 'AwsSolutions-IAM5',
                reason: "Bedrock CreateKnowledgeBase can't be restricted by resource.",
            },
        ], true);
        this.knowledgeBaseArn = knowledgeBase.getAttString('knowledgeBaseArn');
        this.knowledgeBaseId = knowledgeBase.getAttString('knowledgeBaseId');
    }
}
exports.KnowledgeBase = KnowledgeBase;
_a = JSII_RTTI_SYMBOL_1;
KnowledgeBase[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.bedrock.KnowledgeBase", version: "0.1.94" };
/**
 * Validate that Bedrock Knowledge Base can use the selected model.
 *
 * @internal This is an internal core function and should not be called directly.
 */
function validateModel(foundationModel) {
    if (!foundationModel.supportsKnowledgeBase) {
        throw new Error(`The model ${foundationModel} is not supported by Bedrock Knowledge Base.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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