/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as cdk from 'aws-cdk-lib';
import * as kms from 'aws-cdk-lib/aws-kms';
import * as s3 from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
import { KnowledgeBase } from './knowledge-base';
/**
 * Knowledge base can split your source data into chunks. A chunk refers to an
 * excerpt from a data source that is returned when the knowledge base that it
 * belongs to is queried. You have the following options for chunking your
 * data. If you opt for NONE, then you may want to pre-process your files by
 * splitting them up such that each file corresponds to a chunk.
 */
export declare enum ChunkingStrategy {
    /**
     * Amazon Bedrock splits your source data into chunks of the approximate size
     * that you set in the `fixedSizeChunkingConfiguration`.
     */
    FIXED_SIZE = "FIXED_SIZE",
    /**
     * `FIXED_SIZE` with the default chunk size of 300 tokens and 20% overlap.
     */
    DEFAULT = "DEFAULT",
    /**
     * Amazon Bedrock treats each file as one chunk. If you choose this option,
     * you may want to pre-process your documents by splitting them into separate
     * files.
     */
    NONE = "NONE"
}
/**
 * Properties for an S3 Data Source.
 */
export interface S3DataSourceProps {
    /**
     * The knowledge base that this data source belongs to.
     */
    readonly knowledgeBase: KnowledgeBase;
    /**
     * The name of the data source.
     */
    readonly dataSourceName: string;
    /**
     * The bucket that contains the data source.
     */
    readonly bucket: s3.IBucket;
    /**
     * The prefixes of the objects in the bucket that should be included in the data source.
     *
     * @default - All objects in the bucket.
     */
    readonly inclusionPrefixes?: string[];
    /**
     * The chunking strategy to use.
     *
     * @default ChunkingStrategy.DEFAULT
     */
    readonly chunkingStrategy?: ChunkingStrategy;
    /**
     * The maximum number of tokens to use in a chunk.
     *
     * @default 300
     */
    readonly maxTokens?: number;
    /**
     * The percentage of overlap to use in a chunk.
     *
     * @default 20
     */
    readonly overlapPercentage?: number;
    /**
     * The KMS key to use to encrypt the data source.
     *
     * @default Amazon Bedrock encrypts your data with a key that AWS owns and manages
     */
    readonly kmsKey?: kms.IKey;
}
/**
 * Sets up a data source to be added to a knowledge base.
 */
export declare class S3DataSource extends Construct {
    /**
     * The Data Source cfn resource.
     */
    readonly dataSource: cdk.CustomResource;
    /**
     * The unique identifier of the data source.
     */
    readonly dataSourceId: string;
    constructor(scope: Construct, id: string, props: S3DataSourceProps);
}
