"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3DataSource = exports.ChunkingStrategy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const custom_resource_provider_1 = require("./custom-resource-provider");
const CHUNKING_OVERLAP = 20;
const CHUNKING_MAX_TOKENS = 300;
/**
 * Knowledge base can split your source data into chunks. A chunk refers to an
 * excerpt from a data source that is returned when the knowledge base that it
 * belongs to is queried. You have the following options for chunking your
 * data. If you opt for NONE, then you may want to pre-process your files by
 * splitting them up such that each file corresponds to a chunk.
 */
var ChunkingStrategy;
(function (ChunkingStrategy) {
    /**
     * Amazon Bedrock splits your source data into chunks of the approximate size
     * that you set in the `fixedSizeChunkingConfiguration`.
     */
    ChunkingStrategy["FIXED_SIZE"] = "FIXED_SIZE";
    /**
     * `FIXED_SIZE` with the default chunk size of 300 tokens and 20% overlap.
     */
    ChunkingStrategy["DEFAULT"] = "DEFAULT";
    /**
     * Amazon Bedrock treats each file as one chunk. If you choose this option,
     * you may want to pre-process your documents by splitting them into separate
     * files.
     */
    ChunkingStrategy["NONE"] = "NONE";
})(ChunkingStrategy || (exports.ChunkingStrategy = ChunkingStrategy = {}));
/**
 * Sets up a data source to be added to a knowledge base.
 */
class S3DataSource extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const knowledgeBase = props.knowledgeBase;
        const dataSourceName = props.dataSourceName;
        const bucket = props.bucket;
        const inclusionPrefixes = props.inclusionPrefixes;
        const chunkingStrategy = props.chunkingStrategy ?? ChunkingStrategy.DEFAULT;
        const maxTokens = props.maxTokens ?? CHUNKING_MAX_TOKENS;
        const overlapPercentage = props.overlapPercentage ?? CHUNKING_OVERLAP;
        const kmsKey = props.kmsKey;
        bucket.grantRead(knowledgeBase.role);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(knowledgeBase.role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'The KB role needs read only access to all objects in the data source bucket.',
            },
        ], true);
        const crProvider = custom_resource_provider_1.BedrockCRProvider.getProvider(this);
        this.dataSource = new cdk.CustomResource(this, 'DataSource', {
            serviceToken: crProvider.serviceToken,
            resourceType: 'Custom::Bedrock-DataSource',
            properties: {
                knowledgeBaseId: knowledgeBase.knowledgeBaseId,
                name: dataSourceName,
                dataSourceConfiguration: {
                    type: 'S3',
                    s3Configuration: {
                        bucketArn: bucket.bucketArn,
                        inclusionPrefixes: inclusionPrefixes,
                    },
                },
                vectorIngestionConfiguration: vectorIngestionConfiguration(chunkingStrategy, maxTokens, overlapPercentage),
                serverSideEncryptionConfiguration: kmsKey ? {
                    kmsKeyArn: kmsKey.keyArn,
                } : undefined,
            },
        });
        this.dataSourceId = this.dataSource.getAtt('dataSourceId').toString();
        const dataSourceCRPolicy = new iam.Policy(this, 'DataSourceCRPolicy', {
            roles: [crProvider.role],
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'bedrock:CreateDataSource',
                        'bedrock:DeleteDataSource',
                        'bedrock:UpdateDataSource',
                    ],
                    resources: [
                        /** A weird race condition makes CreateDataSource fail due to permissions on the first attempt.
                         * A wildcard allows CFN to deploy this policy earlier and avoid the race. */
                        cdk.Stack.of(this).formatArn({
                            service: 'bedrock',
                            resource: 'knowledge-base',
                            resourceName: '*',
                            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                        }),
                        // knowledgeBase.knowledgeBaseArn,
                    ],
                }),
                new iam.PolicyStatement({
                    actions: ['s3:GetObject', 's3:ListBucket'],
                    resources: [bucket.bucketArn, `${bucket.bucketArn}/*`],
                }),
            ],
        });
        this.dataSource.node.addDependency(dataSourceCRPolicy);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(dataSourceCRPolicy, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'Bedrock CreateDataSource uses wildcards to access any object in the specified bucket. ' +
                    'A weird race condition makes CreateDataSource fail due to permissions on the first attempt. ' +
                    'Permitting CreateDataSource on all KnowledgeBases allows CFN to deploy this policy earlier and ' +
                    'avoid the race.',
            },
        ], true);
        // ToDo: scheduled ingestion jobs
    }
}
exports.S3DataSource = S3DataSource;
_a = JSII_RTTI_SYMBOL_1;
S3DataSource[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.bedrock.S3DataSource", version: "0.1.94" };
function vectorIngestionConfiguration(chunkingStrategy, maxTokens = CHUNKING_MAX_TOKENS, overlapPercentage = CHUNKING_OVERLAP) {
    if (chunkingStrategy === ChunkingStrategy.FIXED_SIZE) {
        if (maxTokens <= 20 || maxTokens >= 8000) {
            throw new Error(`maxTokens must be between 20 and 8000, got ${maxTokens}`);
        }
        if (overlapPercentage < 0 || overlapPercentage > 100) {
            throw new Error(`overlapPercentage must be between 0 and 100, got ${overlapPercentage}`);
        }
        return {
            chunkingConfiguration: {
                chunkingStrategy: chunkingStrategy,
                fixedSizeChunkingConfiguration: {
                    maxTokens,
                    overlapPercentage,
                },
            },
        };
    }
    else if (chunkingStrategy === ChunkingStrategy.NONE) {
        return {
            chunkingConfiguration: {
                chunkingStrategy,
            },
        };
    }
    else {
        return {};
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiczMtZGF0YS1zb3VyY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvY2RrLWxpYi9iZWRyb2NrL3MzLWRhdGEtc291cmNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7QUFFSCxtQ0FBbUM7QUFDbkMsMkNBQTJDO0FBRzNDLHFDQUEwQztBQUMxQywyQ0FBdUM7QUFFdkMseUVBQStEO0FBRy9ELE1BQU0sZ0JBQWdCLEdBQUcsRUFBRSxDQUFDO0FBQzVCLE1BQU0sbUJBQW1CLEdBQUcsR0FBRyxDQUFDO0FBRWhDOzs7Ozs7R0FNRztBQUNILElBQVksZ0JBZ0JYO0FBaEJELFdBQVksZ0JBQWdCO0lBQzFCOzs7T0FHRztJQUNILDZDQUF5QixDQUFBO0lBQ3pCOztPQUVHO0lBQ0gsdUNBQW1CLENBQUE7SUFDbkI7Ozs7T0FJRztJQUNILGlDQUFhLENBQUE7QUFDZixDQUFDLEVBaEJXLGdCQUFnQixnQ0FBaEIsZ0JBQWdCLFFBZ0IzQjtBQWtERDs7R0FFRztBQUNILE1BQWEsWUFBYSxTQUFRLHNCQUFTO0lBVXpDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBd0I7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBQzFDLE1BQU0sY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUM7UUFDNUMsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQztRQUM1QixNQUFNLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUNsRCxNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxnQkFBZ0IsQ0FBQyxPQUFPLENBQUM7UUFDNUUsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsSUFBSSxtQkFBbUIsQ0FBQztRQUN6RCxNQUFNLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxnQkFBZ0IsQ0FBQztRQUN0RSxNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDO1FBRzVCLE1BQU0sQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3JDLHlCQUFlLENBQUMsdUJBQXVCLENBQ3JDLGFBQWEsQ0FBQyxJQUFJLEVBQ2xCO1lBQ0U7Z0JBQ0UsRUFBRSxFQUFFLG1CQUFtQjtnQkFDdkIsTUFBTSxFQUFFLDhFQUE4RTthQUN2RjtTQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7UUFFRixNQUFNLFVBQVUsR0FBRyw0Q0FBaUIsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdkQsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtZQUMzRCxZQUFZLEVBQUUsVUFBVSxDQUFDLFlBQVk7WUFDckMsWUFBWSxFQUFFLDRCQUE0QjtZQUMxQyxVQUFVLEVBQUU7Z0JBQ1YsZUFBZSxFQUFFLGFBQWEsQ0FBQyxlQUFlO2dCQUM5QyxJQUFJLEVBQUUsY0FBYztnQkFDcEIsdUJBQXVCLEVBQUU7b0JBQ3ZCLElBQUksRUFBRSxJQUFJO29CQUNWLGVBQWUsRUFBRTt3QkFDZixTQUFTLEVBQUUsTUFBTSxDQUFDLFNBQVM7d0JBQzNCLGlCQUFpQixFQUFFLGlCQUFpQjtxQkFDckM7aUJBQ0Y7Z0JBQ0QsNEJBQTRCLEVBQUUsNEJBQTRCLENBQ3hELGdCQUFnQixFQUFFLFNBQVMsRUFBRSxpQkFBaUIsQ0FDL0M7Z0JBQ0QsaUNBQWlDLEVBQUUsTUFBTSxDQUFDLENBQUMsQ0FBQztvQkFDMUMsU0FBUyxFQUFFLE1BQU0sQ0FBQyxNQUFNO2lCQUN6QixDQUFDLENBQUMsQ0FBQyxTQUFTO2FBQ2Q7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBRXRFLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxvQkFBb0IsRUFBRTtZQUNwRSxLQUFLLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDO1lBQ3hCLFVBQVUsRUFBRTtnQkFDVixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLE9BQU8sRUFBRTt3QkFDUCwwQkFBMEI7d0JBQzFCLDBCQUEwQjt3QkFDMUIsMEJBQTBCO3FCQUMzQjtvQkFDRCxTQUFTLEVBQUU7d0JBQ1Q7cUdBQzZFO3dCQUM3RSxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7NEJBQzNCLE9BQU8sRUFBRSxTQUFTOzRCQUNsQixRQUFRLEVBQUUsZ0JBQWdCOzRCQUMxQixZQUFZLEVBQUUsR0FBRzs0QkFDakIsU0FBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsbUJBQW1CO3lCQUM3QyxDQUFDO3dCQUNGLGtDQUFrQztxQkFDbkM7aUJBQ0YsQ0FBQztnQkFDRixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLE9BQU8sRUFBRSxDQUFDLGNBQWMsRUFBRSxlQUFlLENBQUM7b0JBQzFDLFNBQVMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsR0FBRyxNQUFNLENBQUMsU0FBUyxJQUFJLENBQUM7aUJBQ3ZELENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBRXZELHlCQUFlLENBQUMsdUJBQXVCLENBQ3JDLGtCQUFrQixFQUNsQjtZQUNFO2dCQUNFLEVBQUUsRUFBRSxtQkFBbUI7Z0JBQ3ZCLE1BQU0sRUFBRSx3RkFBd0Y7b0JBQ3hGLDhGQUE4RjtvQkFDOUYsaUdBQWlHO29CQUNqRyxpQkFBaUI7YUFDMUI7U0FDRixFQUNELElBQUksQ0FDTCxDQUFDO1FBRUYsaUNBQWlDO0lBQ25DLENBQUM7O0FBeEdILG9DQXlHQzs7O0FBZ0JELFNBQVMsNEJBQTRCLENBQ25DLGdCQUFrQyxFQUNsQyxZQUFvQixtQkFBbUIsRUFDdkMsb0JBQTRCLGdCQUFnQjtJQUc1QyxJQUFJLGdCQUFnQixLQUFLLGdCQUFnQixDQUFDLFVBQVUsRUFBRSxDQUFDO1FBQ3JELElBQUksU0FBUyxJQUFJLEVBQUUsSUFBSSxTQUFTLElBQUksSUFBSSxFQUFFLENBQUM7WUFDekMsTUFBTSxJQUFJLEtBQUssQ0FBQyw4Q0FBOEMsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUM3RSxDQUFDO1FBRUQsSUFBSSxpQkFBaUIsR0FBRyxDQUFDLElBQUksaUJBQWlCLEdBQUcsR0FBRyxFQUFFLENBQUM7WUFDckQsTUFBTSxJQUFJLEtBQUssQ0FBQyxvREFBb0QsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDO1FBQzNGLENBQUM7UUFFRCxPQUFPO1lBQ0wscUJBQXFCLEVBQUU7Z0JBQ3JCLGdCQUFnQixFQUFFLGdCQUFnQjtnQkFDbEMsOEJBQThCLEVBQUU7b0JBQzlCLFNBQVM7b0JBQ1QsaUJBQWlCO2lCQUNsQjthQUNGO1NBQ0YsQ0FBQztJQUVKLENBQUM7U0FBTSxJQUFJLGdCQUFnQixLQUFLLGdCQUFnQixDQUFDLElBQUksRUFBRSxDQUFDO1FBRXRELE9BQU87WUFDTCxxQkFBcUIsRUFBRTtnQkFDckIsZ0JBQWdCO2FBQ2pCO1NBQ0YsQ0FBQztJQUVKLENBQUM7U0FBTSxDQUFDO1FBQ04sT0FBTyxFQUFFLENBQUM7SUFDWixDQUFDO0FBRUgsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCAqIGFzIGNkayBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBrbXMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWttcyc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMnO1xuaW1wb3J0IHsgTmFnU3VwcHJlc3Npb25zIH0gZnJvbSAnY2RrLW5hZyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuaW1wb3J0IHsgQmVkcm9ja0NSUHJvdmlkZXIgfSBmcm9tICcuL2N1c3RvbS1yZXNvdXJjZS1wcm92aWRlcic7XG5pbXBvcnQgeyBLbm93bGVkZ2VCYXNlIH0gZnJvbSAnLi9rbm93bGVkZ2UtYmFzZSc7XG5cbmNvbnN0IENIVU5LSU5HX09WRVJMQVAgPSAyMDtcbmNvbnN0IENIVU5LSU5HX01BWF9UT0tFTlMgPSAzMDA7XG5cbi8qKlxuICogS25vd2xlZGdlIGJhc2UgY2FuIHNwbGl0IHlvdXIgc291cmNlIGRhdGEgaW50byBjaHVua3MuIEEgY2h1bmsgcmVmZXJzIHRvIGFuXG4gKiBleGNlcnB0IGZyb20gYSBkYXRhIHNvdXJjZSB0aGF0IGlzIHJldHVybmVkIHdoZW4gdGhlIGtub3dsZWRnZSBiYXNlIHRoYXQgaXRcbiAqIGJlbG9uZ3MgdG8gaXMgcXVlcmllZC4gWW91IGhhdmUgdGhlIGZvbGxvd2luZyBvcHRpb25zIGZvciBjaHVua2luZyB5b3VyXG4gKiBkYXRhLiBJZiB5b3Ugb3B0IGZvciBOT05FLCB0aGVuIHlvdSBtYXkgd2FudCB0byBwcmUtcHJvY2VzcyB5b3VyIGZpbGVzIGJ5XG4gKiBzcGxpdHRpbmcgdGhlbSB1cCBzdWNoIHRoYXQgZWFjaCBmaWxlIGNvcnJlc3BvbmRzIHRvIGEgY2h1bmsuXG4gKi9cbmV4cG9ydCBlbnVtIENodW5raW5nU3RyYXRlZ3kge1xuICAvKipcbiAgICogQW1hem9uIEJlZHJvY2sgc3BsaXRzIHlvdXIgc291cmNlIGRhdGEgaW50byBjaHVua3Mgb2YgdGhlIGFwcHJveGltYXRlIHNpemVcbiAgICogdGhhdCB5b3Ugc2V0IGluIHRoZSBgZml4ZWRTaXplQ2h1bmtpbmdDb25maWd1cmF0aW9uYC5cbiAgICovXG4gIEZJWEVEX1NJWkUgPSAnRklYRURfU0laRScsXG4gIC8qKlxuICAgKiBgRklYRURfU0laRWAgd2l0aCB0aGUgZGVmYXVsdCBjaHVuayBzaXplIG9mIDMwMCB0b2tlbnMgYW5kIDIwJSBvdmVybGFwLlxuICAgKi9cbiAgREVGQVVMVCA9ICdERUZBVUxUJyxcbiAgLyoqXG4gICAqIEFtYXpvbiBCZWRyb2NrIHRyZWF0cyBlYWNoIGZpbGUgYXMgb25lIGNodW5rLiBJZiB5b3UgY2hvb3NlIHRoaXMgb3B0aW9uLFxuICAgKiB5b3UgbWF5IHdhbnQgdG8gcHJlLXByb2Nlc3MgeW91ciBkb2N1bWVudHMgYnkgc3BsaXR0aW5nIHRoZW0gaW50byBzZXBhcmF0ZVxuICAgKiBmaWxlcy5cbiAgICovXG4gIE5PTkUgPSAnTk9ORScsXG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYW4gUzMgRGF0YSBTb3VyY2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUzNEYXRhU291cmNlUHJvcHMge1xuICAvKipcbiAgICogVGhlIGtub3dsZWRnZSBiYXNlIHRoYXQgdGhpcyBkYXRhIHNvdXJjZSBiZWxvbmdzIHRvLlxuICAgKi9cbiAgcmVhZG9ubHkga25vd2xlZGdlQmFzZTogS25vd2xlZGdlQmFzZTtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBkYXRhIHNvdXJjZS5cbiAgICovXG4gIHJlYWRvbmx5IGRhdGFTb3VyY2VOYW1lOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgYnVja2V0IHRoYXQgY29udGFpbnMgdGhlIGRhdGEgc291cmNlLlxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0OiBzMy5JQnVja2V0O1xuICAvKipcbiAgICogVGhlIHByZWZpeGVzIG9mIHRoZSBvYmplY3RzIGluIHRoZSBidWNrZXQgdGhhdCBzaG91bGQgYmUgaW5jbHVkZWQgaW4gdGhlIGRhdGEgc291cmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFsbCBvYmplY3RzIGluIHRoZSBidWNrZXQuXG4gICAqL1xuICByZWFkb25seSBpbmNsdXNpb25QcmVmaXhlcz86IHN0cmluZ1tdO1xuICAvKipcbiAgICogVGhlIGNodW5raW5nIHN0cmF0ZWd5IHRvIHVzZS5cbiAgICpcbiAgICogQGRlZmF1bHQgQ2h1bmtpbmdTdHJhdGVneS5ERUZBVUxUXG4gICAqL1xuICByZWFkb25seSBjaHVua2luZ1N0cmF0ZWd5PzogQ2h1bmtpbmdTdHJhdGVneTtcbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIG51bWJlciBvZiB0b2tlbnMgdG8gdXNlIGluIGEgY2h1bmsuXG4gICAqXG4gICAqIEBkZWZhdWx0IDMwMFxuICAgKi9cbiAgcmVhZG9ubHkgbWF4VG9rZW5zPzogbnVtYmVyO1xuICAvKipcbiAgICogVGhlIHBlcmNlbnRhZ2Ugb2Ygb3ZlcmxhcCB0byB1c2UgaW4gYSBjaHVuay5cbiAgICpcbiAgICogQGRlZmF1bHQgMjBcbiAgICovXG4gIHJlYWRvbmx5IG92ZXJsYXBQZXJjZW50YWdlPzogbnVtYmVyO1xuICAvKipcbiAgICogVGhlIEtNUyBrZXkgdG8gdXNlIHRvIGVuY3J5cHQgdGhlIGRhdGEgc291cmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBBbWF6b24gQmVkcm9jayBlbmNyeXB0cyB5b3VyIGRhdGEgd2l0aCBhIGtleSB0aGF0IEFXUyBvd25zIGFuZCBtYW5hZ2VzXG4gICAqL1xuICByZWFkb25seSBrbXNLZXk/OiBrbXMuSUtleTtcbn1cblxuLyoqXG4gKiBTZXRzIHVwIGEgZGF0YSBzb3VyY2UgdG8gYmUgYWRkZWQgdG8gYSBrbm93bGVkZ2UgYmFzZS5cbiAqL1xuZXhwb3J0IGNsYXNzIFMzRGF0YVNvdXJjZSBleHRlbmRzIENvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBUaGUgRGF0YSBTb3VyY2UgY2ZuIHJlc291cmNlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRhdGFTb3VyY2U6IGNkay5DdXN0b21SZXNvdXJjZTtcbiAgLyoqXG4gICAqIFRoZSB1bmlxdWUgaWRlbnRpZmllciBvZiB0aGUgZGF0YSBzb3VyY2UuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGF0YVNvdXJjZUlkOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFMzRGF0YVNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICBjb25zdCBrbm93bGVkZ2VCYXNlID0gcHJvcHMua25vd2xlZGdlQmFzZTtcbiAgICBjb25zdCBkYXRhU291cmNlTmFtZSA9IHByb3BzLmRhdGFTb3VyY2VOYW1lO1xuICAgIGNvbnN0IGJ1Y2tldCA9IHByb3BzLmJ1Y2tldDtcbiAgICBjb25zdCBpbmNsdXNpb25QcmVmaXhlcyA9IHByb3BzLmluY2x1c2lvblByZWZpeGVzO1xuICAgIGNvbnN0IGNodW5raW5nU3RyYXRlZ3kgPSBwcm9wcy5jaHVua2luZ1N0cmF0ZWd5ID8/IENodW5raW5nU3RyYXRlZ3kuREVGQVVMVDtcbiAgICBjb25zdCBtYXhUb2tlbnMgPSBwcm9wcy5tYXhUb2tlbnMgPz8gQ0hVTktJTkdfTUFYX1RPS0VOUztcbiAgICBjb25zdCBvdmVybGFwUGVyY2VudGFnZSA9IHByb3BzLm92ZXJsYXBQZXJjZW50YWdlID8/IENIVU5LSU5HX09WRVJMQVA7XG4gICAgY29uc3Qga21zS2V5ID0gcHJvcHMua21zS2V5O1xuXG5cbiAgICBidWNrZXQuZ3JhbnRSZWFkKGtub3dsZWRnZUJhc2Uucm9sZSk7XG4gICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAga25vd2xlZGdlQmFzZS5yb2xlLFxuICAgICAgW1xuICAgICAgICB7XG4gICAgICAgICAgaWQ6ICdBd3NTb2x1dGlvbnMtSUFNNScsXG4gICAgICAgICAgcmVhc29uOiAnVGhlIEtCIHJvbGUgbmVlZHMgcmVhZCBvbmx5IGFjY2VzcyB0byBhbGwgb2JqZWN0cyBpbiB0aGUgZGF0YSBzb3VyY2UgYnVja2V0LicsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgdHJ1ZSxcbiAgICApO1xuXG4gICAgY29uc3QgY3JQcm92aWRlciA9IEJlZHJvY2tDUlByb3ZpZGVyLmdldFByb3ZpZGVyKHRoaXMpO1xuICAgIHRoaXMuZGF0YVNvdXJjZSA9IG5ldyBjZGsuQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0RhdGFTb3VyY2UnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IGNyUHJvdmlkZXIuc2VydmljZVRva2VuLFxuICAgICAgcmVzb3VyY2VUeXBlOiAnQ3VzdG9tOjpCZWRyb2NrLURhdGFTb3VyY2UnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBrbm93bGVkZ2VCYXNlSWQ6IGtub3dsZWRnZUJhc2Uua25vd2xlZGdlQmFzZUlkLFxuICAgICAgICBuYW1lOiBkYXRhU291cmNlTmFtZSxcbiAgICAgICAgZGF0YVNvdXJjZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICB0eXBlOiAnUzMnLFxuICAgICAgICAgIHMzQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgICAgYnVja2V0QXJuOiBidWNrZXQuYnVja2V0QXJuLFxuICAgICAgICAgICAgaW5jbHVzaW9uUHJlZml4ZXM6IGluY2x1c2lvblByZWZpeGVzLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHZlY3RvckluZ2VzdGlvbkNvbmZpZ3VyYXRpb246IHZlY3RvckluZ2VzdGlvbkNvbmZpZ3VyYXRpb24oXG4gICAgICAgICAgY2h1bmtpbmdTdHJhdGVneSwgbWF4VG9rZW5zLCBvdmVybGFwUGVyY2VudGFnZSxcbiAgICAgICAgKSxcbiAgICAgICAgc2VydmVyU2lkZUVuY3J5cHRpb25Db25maWd1cmF0aW9uOiBrbXNLZXkgPyB7XG4gICAgICAgICAga21zS2V5QXJuOiBrbXNLZXkua2V5QXJuLFxuICAgICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMuZGF0YVNvdXJjZUlkID0gdGhpcy5kYXRhU291cmNlLmdldEF0dCgnZGF0YVNvdXJjZUlkJykudG9TdHJpbmcoKTtcblxuICAgIGNvbnN0IGRhdGFTb3VyY2VDUlBvbGljeSA9IG5ldyBpYW0uUG9saWN5KHRoaXMsICdEYXRhU291cmNlQ1JQb2xpY3knLCB7XG4gICAgICByb2xlczogW2NyUHJvdmlkZXIucm9sZV0sXG4gICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAnYmVkcm9jazpDcmVhdGVEYXRhU291cmNlJyxcbiAgICAgICAgICAgICdiZWRyb2NrOkRlbGV0ZURhdGFTb3VyY2UnLFxuICAgICAgICAgICAgJ2JlZHJvY2s6VXBkYXRlRGF0YVNvdXJjZScsXG4gICAgICAgICAgXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgIC8qKiBBIHdlaXJkIHJhY2UgY29uZGl0aW9uIG1ha2VzIENyZWF0ZURhdGFTb3VyY2UgZmFpbCBkdWUgdG8gcGVybWlzc2lvbnMgb24gdGhlIGZpcnN0IGF0dGVtcHQuXG4gICAgICAgICAgICAgKiBBIHdpbGRjYXJkIGFsbG93cyBDRk4gdG8gZGVwbG95IHRoaXMgcG9saWN5IGVhcmxpZXIgYW5kIGF2b2lkIHRoZSByYWNlLiAqL1xuICAgICAgICAgICAgY2RrLlN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICAgICAgICAgIHNlcnZpY2U6ICdiZWRyb2NrJyxcbiAgICAgICAgICAgICAgcmVzb3VyY2U6ICdrbm93bGVkZ2UtYmFzZScsXG4gICAgICAgICAgICAgIHJlc291cmNlTmFtZTogJyonLFxuICAgICAgICAgICAgICBhcm5Gb3JtYXQ6IGNkay5Bcm5Gb3JtYXQuU0xBU0hfUkVTT1VSQ0VfTkFNRSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgLy8ga25vd2xlZGdlQmFzZS5rbm93bGVkZ2VCYXNlQXJuLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pLFxuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnLCAnczM6TGlzdEJ1Y2tldCddLFxuICAgICAgICAgIHJlc291cmNlczogW2J1Y2tldC5idWNrZXRBcm4sIGAke2J1Y2tldC5idWNrZXRBcm59LypgXSxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgdGhpcy5kYXRhU291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeShkYXRhU291cmNlQ1JQb2xpY3kpO1xuXG4gICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgZGF0YVNvdXJjZUNSUG9saWN5LFxuICAgICAgW1xuICAgICAgICB7XG4gICAgICAgICAgaWQ6ICdBd3NTb2x1dGlvbnMtSUFNNScsXG4gICAgICAgICAgcmVhc29uOiAnQmVkcm9jayBDcmVhdGVEYXRhU291cmNlIHVzZXMgd2lsZGNhcmRzIHRvIGFjY2VzcyBhbnkgb2JqZWN0IGluIHRoZSBzcGVjaWZpZWQgYnVja2V0LiAnICtcbiAgICAgICAgICAgICAgICAgICdBIHdlaXJkIHJhY2UgY29uZGl0aW9uIG1ha2VzIENyZWF0ZURhdGFTb3VyY2UgZmFpbCBkdWUgdG8gcGVybWlzc2lvbnMgb24gdGhlIGZpcnN0IGF0dGVtcHQuICcgK1xuICAgICAgICAgICAgICAgICAgJ1Blcm1pdHRpbmcgQ3JlYXRlRGF0YVNvdXJjZSBvbiBhbGwgS25vd2xlZGdlQmFzZXMgYWxsb3dzIENGTiB0byBkZXBsb3kgdGhpcyBwb2xpY3kgZWFybGllciBhbmQgJytcbiAgICAgICAgICAgICAgICAgICdhdm9pZCB0aGUgcmFjZS4nLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIHRydWUsXG4gICAgKTtcblxuICAgIC8vIFRvRG86IHNjaGVkdWxlZCBpbmdlc3Rpb24gam9ic1xuICB9XG59XG5cbmludGVyZmFjZSBGaXhlZFNpemVDaHVua2luZ0NvbmZpZyB7XG4gIG1heFRva2VuczogbnVtYmVyO1xuICBvdmVybGFwUGVyY2VudGFnZTogbnVtYmVyO1xufVxuXG5pbnRlcmZhY2UgQ2h1bmtpbmdDb25maWcge1xuICBjaHVua2luZ1N0cmF0ZWd5OiBDaHVua2luZ1N0cmF0ZWd5O1xuICBmaXhlZFNpemVDaHVua2luZ0NvbmZpZ3VyYXRpb24/OiBGaXhlZFNpemVDaHVua2luZ0NvbmZpZztcbn1cblxuaW50ZXJmYWNlIFZlY3RvckluZ2VzdGlvbkNvbmZpZyB7XG4gIGNodW5raW5nQ29uZmlndXJhdGlvbj86IENodW5raW5nQ29uZmlnO1xufVxuXG5mdW5jdGlvbiB2ZWN0b3JJbmdlc3Rpb25Db25maWd1cmF0aW9uKFxuICBjaHVua2luZ1N0cmF0ZWd5OiBDaHVua2luZ1N0cmF0ZWd5LFxuICBtYXhUb2tlbnM6IG51bWJlciA9IENIVU5LSU5HX01BWF9UT0tFTlMsXG4gIG92ZXJsYXBQZXJjZW50YWdlOiBudW1iZXIgPSBDSFVOS0lOR19PVkVSTEFQLFxuKTogVmVjdG9ySW5nZXN0aW9uQ29uZmlnIHtcblxuICBpZiAoY2h1bmtpbmdTdHJhdGVneSA9PT0gQ2h1bmtpbmdTdHJhdGVneS5GSVhFRF9TSVpFKSB7XG4gICAgaWYgKG1heFRva2VucyA8PSAyMCB8fCBtYXhUb2tlbnMgPj0gODAwMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBtYXhUb2tlbnMgbXVzdCBiZSBiZXR3ZWVuIDIwIGFuZCA4MDAwLCBnb3QgJHttYXhUb2tlbnN9YCk7XG4gICAgfVxuXG4gICAgaWYgKG92ZXJsYXBQZXJjZW50YWdlIDwgMCB8fCBvdmVybGFwUGVyY2VudGFnZSA+IDEwMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBvdmVybGFwUGVyY2VudGFnZSBtdXN0IGJlIGJldHdlZW4gMCBhbmQgMTAwLCBnb3QgJHtvdmVybGFwUGVyY2VudGFnZX1gKTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgY2h1bmtpbmdDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIGNodW5raW5nU3RyYXRlZ3k6IGNodW5raW5nU3RyYXRlZ3ksXG4gICAgICAgIGZpeGVkU2l6ZUNodW5raW5nQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIG1heFRva2VucyxcbiAgICAgICAgICBvdmVybGFwUGVyY2VudGFnZSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfTtcblxuICB9IGVsc2UgaWYgKGNodW5raW5nU3RyYXRlZ3kgPT09IENodW5raW5nU3RyYXRlZ3kuTk9ORSkge1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGNodW5raW5nQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBjaHVua2luZ1N0cmF0ZWd5LFxuICAgICAgfSxcbiAgICB9O1xuXG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuIHt9O1xuICB9XG5cbn0iXX0=