"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VectorCollection = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const oss = require("aws-cdk-lib/aws-opensearchserverless");
const constructs_1 = require("constructs");
const utils_1 = require("../../common/helpers/utils");
/**
 * Deploys an OpenSearch Serverless Collection to be used as a vector store.
 *
 * It includes all policies.
 */
class VectorCollection extends constructs_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        /**
         * An OpenSearch Access Policy document that will become `dataAccessPolicy`.
         * @private
         */
        this.dataAccessPolicyDocument = [];
        this.collectionName = (0, utils_1.generatePhysicalNameV2)(this, 'VectorStore', { maxLength: 32, lower: true });
        const encryptionPolicyName = (0, utils_1.generatePhysicalNameV2)(this, 'EncryptionPolicy', { maxLength: 32, lower: true });
        const encryptionPolicy = new oss.CfnSecurityPolicy(this, 'EncryptionPolicy', {
            name: encryptionPolicyName,
            type: 'encryption',
            policy: JSON.stringify({
                Rules: [
                    {
                        ResourceType: 'collection',
                        Resource: [`collection/${this.collectionName}`],
                    },
                ],
                AWSOwnedKey: true,
            }),
        });
        const networkPolicyName = (0, utils_1.generatePhysicalNameV2)(this, 'NetworkPolicy', { maxLength: 32, lower: true });
        const networkPolicy = new oss.CfnSecurityPolicy(this, 'NetworkPolicy', {
            name: networkPolicyName,
            type: 'network',
            policy: JSON.stringify([
                {
                    Rules: [
                        {
                            ResourceType: 'collection',
                            Resource: [`collection/${this.collectionName}`],
                        },
                        {
                            ResourceType: 'dashboard',
                            Resource: [`collection/${this.collectionName}`],
                        },
                    ],
                    AllowFromPublic: true,
                },
            ]),
        });
        const collection = new oss.CfnCollection(this, 'VectorCollection', {
            name: this.collectionName,
            type: 'VECTORSEARCH',
        });
        this.collectionArn = collection.attrArn;
        this.collectionId = collection.attrId;
        this.aossPolicy = new iam.ManagedPolicy(this, 'AOSSApiAccessAll', {
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: [
                        'aoss:APIAccessAll',
                    ],
                    resources: [collection.attrArn],
                }),
            ],
        });
        collection.addDependency(encryptionPolicy);
        collection.addDependency(networkPolicy);
        const isDataAccessPolicyNotEmpty = new cdk.CfnCondition(this, 'IsDataAccessPolicyNotEmpty', {
            expression: cdk.Fn.conditionNot(cdk.Fn.conditionEquals(0, cdk.Lazy.number({
                produce: () => this.dataAccessPolicyDocument.length,
            }))),
        });
        const dataAccessPolicyName = (0, utils_1.generatePhysicalNameV2)(this, 'DataAccessPolicy', { maxLength: 32, lower: true });
        this.dataAccessPolicy = new oss.CfnAccessPolicy(this, 'DataAccessPolicy', {
            name: dataAccessPolicyName,
            type: 'data',
            policy: cdk.Lazy.string({
                produce: () => JSON.stringify(this.dataAccessPolicyDocument),
            }),
        });
        this.dataAccessPolicy.cfnOptions.condition = isDataAccessPolicyNotEmpty;
    }
    /**
     * Grants the specified role access to data in the collection.
     * @param grantee The role to grant access to.
     */
    grantDataAccess(grantee) {
        this.dataAccessPolicyDocument.push({
            Rules: [
                {
                    Resource: [`collection/${this.collectionName}`],
                    Permission: [
                        'aoss:DescribeCollectionItems',
                        'aoss:CreateCollectionItems',
                        'aoss:UpdateCollectionItems',
                    ],
                    ResourceType: 'collection',
                },
                {
                    Resource: [`index/${this.collectionName}/*`],
                    Permission: [
                        'aoss:UpdateIndex',
                        'aoss:DescribeIndex',
                        'aoss:ReadDocument',
                        'aoss:WriteDocument',
                        'aoss:CreateIndex',
                    ],
                    ResourceType: 'index',
                },
            ],
            Principal: [
                grantee.roleArn,
            ],
            Description: '',
        });
        grantee.addManagedPolicy(this.aossPolicy);
    }
}
exports.VectorCollection = VectorCollection;
_a = JSII_RTTI_SYMBOL_1;
VectorCollection[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.opensearchserverless.VectorCollection", version: "0.1.94" };
//# sourceMappingURL=data:application/json;base64,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