"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildCustomResourceProvider = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const cr = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
function buildCustomResourceProvider(props) {
    const { providerName, codePath, handler, runtime, layers } = props;
    class CRProvider extends constructs_1.Construct {
        static getProvider(scope) {
            const stack = cdk.Stack.of(scope);
            const existing = stack.node.tryFindChild(providerName);
            if (existing) {
                return existing;
            }
            return new CRProvider(cdk.Stack.of(scope), providerName);
        }
        constructor(scope, id) {
            super(scope, id);
            this.role = new iam.Role(this, 'CRRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
                ],
            });
            const customResourceFunction = new lambda.Function(this, 'CustomResourcesFunction', {
                code: lambda.Code.fromDockerBuild(codePath),
                handler,
                runtime,
                layers,
                role: this.role,
                timeout: cdk.Duration.minutes(15),
                memorySize: 128,
                logRetention: logs.RetentionDays.ONE_WEEK,
                description: 'Custom Resource Provider',
            });
            const providerRole = new iam.Role(this, 'ProviderRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
                ],
            });
            this.provider = new cr.Provider(this, 'Provider', {
                onEventHandler: customResourceFunction,
                logRetention: logs.RetentionDays.ONE_WEEK,
                role: providerRole,
            });
            this.serviceToken = this.provider.serviceToken;
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(cdk.Stack.of(this), `${this.provider.node.path}/framework-onEvent/Resource`, [
                {
                    id: 'AwsSolutions-L1',
                    reason: 'Lambda runtime version is managed upstream by CDK.',
                },
            ]);
            for (let role of [this.role, providerRole]) {
                cdk_nag_1.NagSuppressions.addResourceSuppressions(role, [
                    {
                        id: 'AwsSolutions-IAM4',
                        reason: 'CDK CustomResource Lambda uses the AWSLambdaBasicExecutionRole AWS Managed Policy.',
                    },
                ]);
            }
            cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'CDK CustomResource Provider has a wildcard to invoke any version of the specific Custom Resource function.',
                    appliesTo: [{ regex: `/^Resource::<${id}${customResourceFunction.node.id}[A-Z0-9]+\\.Arn>:\\*$/g` }],
                },
            ], true);
        }
    }
    return CRProvider;
}
exports.buildCustomResourceProvider = buildCustomResourceProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3VzdG9tLXJlc291cmNlLXByb3ZpZGVyLWhlbHBlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9jb21tb24vaGVscGVycy9jdXN0b20tcmVzb3VyY2UtcHJvdmlkZXItaGVscGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBOzs7Ozs7Ozs7OztHQVdHO0FBQ0gsbUNBQW1DO0FBQ25DLDJDQUEyQztBQUMzQyxpREFBaUQ7QUFDakQsNkNBQTZDO0FBQzdDLG1EQUFtRDtBQUNuRCxxQ0FBMEM7QUFDMUMsMkNBQXVDO0FBa0R2QyxTQUFnQiwyQkFBMkIsQ0FBQyxLQUFzQjtJQUNoRSxNQUFNLEVBQUUsWUFBWSxFQUFFLFFBQVEsRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxHQUFHLEtBQUssQ0FBQztJQUVuRSxNQUFNLFVBQVcsU0FBUSxzQkFBUztRQUNoQyxNQUFNLENBQUMsV0FBVyxDQUFDLEtBQWdCO1lBQ2pDLE1BQU0sS0FBSyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ2xDLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFlBQVksQ0FBQyxDQUFDO1lBQ3ZELElBQUksUUFBUSxFQUFFLENBQUM7Z0JBQ2IsT0FBTyxRQUFzQixDQUFDO1lBQ2hDLENBQUM7WUFDRCxPQUFPLElBQUksVUFBVSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQzNELENBQUM7UUFNRCxZQUFZLEtBQWdCLEVBQUUsRUFBVTtZQUN0QyxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBRWpCLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxRQUFRLEVBQUU7Z0JBQ3ZDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztnQkFDM0QsZUFBZSxFQUFFO29CQUNmLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsMENBQTBDLENBQUM7aUJBQ3ZGO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsTUFBTSxzQkFBc0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLHlCQUF5QixFQUFFO2dCQUNsRixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsUUFBUSxDQUFDO2dCQUMzQyxPQUFPO2dCQUNQLE9BQU87Z0JBQ1AsTUFBTTtnQkFDTixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7Z0JBQ2YsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztnQkFDakMsVUFBVSxFQUFFLEdBQUc7Z0JBQ2YsWUFBWSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUTtnQkFDekMsV0FBVyxFQUFFLDBCQUEwQjthQUN4QyxDQUFDLENBQUM7WUFFSCxNQUFNLFlBQVksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtnQkFDdEQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO2dCQUMzRCxlQUFlLEVBQUU7b0JBQ2YsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQywwQ0FBMEMsQ0FBQztpQkFDdkY7YUFDRixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO2dCQUNoRCxjQUFjLEVBQUUsc0JBQXNCO2dCQUN0QyxZQUFZLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxRQUFRO2dCQUN6QyxJQUFJLEVBQUUsWUFBWTthQUNuQixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDO1lBRS9DLHlCQUFlLENBQUMsNkJBQTZCLENBQzNDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxFQUNsQixHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksNkJBQTZCLEVBQ3ZEO2dCQUNFO29CQUNFLEVBQUUsRUFBRSxpQkFBaUI7b0JBQ3JCLE1BQU0sRUFBRSxvREFBb0Q7aUJBQzdEO2FBQ0YsQ0FDRixDQUFDO1lBRUYsS0FBSyxJQUFJLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsWUFBWSxDQUFDLEVBQUUsQ0FBQztnQkFDM0MseUJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsSUFBSSxFQUNKO29CQUNFO3dCQUNFLEVBQUUsRUFBRSxtQkFBbUI7d0JBQ3ZCLE1BQU0sRUFBRSxvRkFBb0Y7cUJBQzdGO2lCQUNGLENBQ0YsQ0FBQztZQUNKLENBQUM7WUFFRCx5QkFBZSxDQUFDLHVCQUF1QixDQUNyQyxZQUFZLEVBQ1o7Z0JBQ0U7b0JBQ0UsRUFBRSxFQUFFLG1CQUFtQjtvQkFDdkIsTUFBTSxFQUFFLDRHQUE0RztvQkFDcEgsU0FBUyxFQUFFLENBQUMsRUFBRSxLQUFLLEVBQUUsZ0JBQWdCLEVBQUUsR0FBRyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsRUFBRSx5QkFBeUIsRUFBRSxDQUFDO2lCQUNyRzthQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7UUFDSixDQUFDO0tBQ0Y7SUFFRCxPQUFPLFVBQVUsQ0FBQztBQUNwQixDQUFDO0FBNUZELGtFQTRGQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ2F3cy1jZGstbGliL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyBjciBmcm9tICdhd3MtY2RrLWxpYi9jdXN0b20tcmVzb3VyY2VzJztcbmltcG9ydCB7IE5hZ1N1cHByZXNzaW9ucyB9IGZyb20gJ2Nkay1uYWcnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQ1JQcm92aWRlclByb3BzIHtcbiAgLyoqXG4gICAqIEEgZ2xvYmFsbHkgdW5pcXVlIG5hbWUgZm9yIHRoZSBDdXN0b20gUmVzb3VyY2UgcHJvdmlkZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZS5cbiAgICovXG4gIHJlYWRvbmx5IHByb3ZpZGVyTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBQYXRoIHRvIGN1c3RvbSByZXNvdXJjZSBwcm92aWRlciBMYW1iZGEgY29kZS4gVGhlIHBhY2thZ2Ugd2lsbCBiZSBidWlsdCB1c2luZyBsYW1iZGEuQ29kZS5mcm9tRG9ja2VyQnVpbGQoKS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lLlxuICAgKi9cbiAgcmVhZG9ubHkgY29kZVBhdGg6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIExhbWJkYSBoYW5kbGVyIGZ1bmN0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmUuXG4gICAqL1xuICByZWFkb25seSBoYW5kbGVyOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBydW50aW1lIG9mIHRoZSBMYW1iZGEgZnVuY3Rpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZS5cbiAgICovXG4gIHJlYWRvbmx5IHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lO1xuXG4gIC8qKlxuICAgKiBUaGUgbGlzdCBvZiBsYXllcnMgdG8gYXR0YWNoIHRvIHRoZSBMYW1iZGEgZnVuY3Rpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZS5cbiAgICovXG4gIHJlYWRvbmx5IGxheWVycz86IGxhbWJkYS5JTGF5ZXJWZXJzaW9uW107XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSUNSUHJvdmlkZXIge1xuICByb2xlOiBpYW0uUm9sZTtcbiAgcHJvdmlkZXI6IGNyLlByb3ZpZGVyO1xuICBzZXJ2aWNlVG9rZW46IHN0cmluZztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBJQ1JQcm92aWRlckNsYXNzIHtcbiAgZ2V0UHJvdmlkZXIoc2NvcGU6IENvbnN0cnVjdCk6IElDUlByb3ZpZGVyO1xufVxuXG5cbmV4cG9ydCBmdW5jdGlvbiBidWlsZEN1c3RvbVJlc291cmNlUHJvdmlkZXIocHJvcHM6IENSUHJvdmlkZXJQcm9wcyk6IElDUlByb3ZpZGVyQ2xhc3Mge1xuICBjb25zdCB7IHByb3ZpZGVyTmFtZSwgY29kZVBhdGgsIGhhbmRsZXIsIHJ1bnRpbWUsIGxheWVycyB9ID0gcHJvcHM7XG5cbiAgY2xhc3MgQ1JQcm92aWRlciBleHRlbmRzIENvbnN0cnVjdCB7XG4gICAgc3RhdGljIGdldFByb3ZpZGVyKHNjb3BlOiBDb25zdHJ1Y3QpOiBDUlByb3ZpZGVyIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gY2RrLlN0YWNrLm9mKHNjb3BlKTtcbiAgICAgIGNvbnN0IGV4aXN0aW5nID0gc3RhY2subm9kZS50cnlGaW5kQ2hpbGQocHJvdmlkZXJOYW1lKTtcbiAgICAgIGlmIChleGlzdGluZykge1xuICAgICAgICByZXR1cm4gZXhpc3RpbmcgYXMgQ1JQcm92aWRlcjtcbiAgICAgIH1cbiAgICAgIHJldHVybiBuZXcgQ1JQcm92aWRlcihjZGsuU3RhY2sub2Yoc2NvcGUpLCBwcm92aWRlck5hbWUpO1xuICAgIH1cblxuICAgIHB1YmxpYyByZWFkb25seSByb2xlOiBpYW0uUm9sZTtcbiAgICBwdWJsaWMgcmVhZG9ubHkgcHJvdmlkZXI6IGNyLlByb3ZpZGVyO1xuICAgIHB1YmxpYyByZWFkb25seSBzZXJ2aWNlVG9rZW46IHN0cmluZztcblxuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuU3RhY2ssIGlkOiBzdHJpbmcpIHtcbiAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICAgIHRoaXMucm9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnQ1JSb2xlJywge1xuICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlJyksXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgICAgY29uc3QgY3VzdG9tUmVzb3VyY2VGdW5jdGlvbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcywgJ0N1c3RvbVJlc291cmNlc0Z1bmN0aW9uJywge1xuICAgICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tRG9ja2VyQnVpbGQoY29kZVBhdGgpLFxuICAgICAgICBoYW5kbGVyLFxuICAgICAgICBydW50aW1lLFxuICAgICAgICBsYXllcnMsXG4gICAgICAgIHJvbGU6IHRoaXMucm9sZSxcbiAgICAgICAgdGltZW91dDogY2RrLkR1cmF0aW9uLm1pbnV0ZXMoMTUpLFxuICAgICAgICBtZW1vcnlTaXplOiAxMjgsXG4gICAgICAgIGxvZ1JldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9XRUVLLFxuICAgICAgICBkZXNjcmlwdGlvbjogJ0N1c3RvbSBSZXNvdXJjZSBQcm92aWRlcicsXG4gICAgICB9KTtcblxuICAgICAgY29uc3QgcHJvdmlkZXJSb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdQcm92aWRlclJvbGUnLCB7XG4gICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpLFxuICAgICAgICBtYW5hZ2VkUG9saWNpZXM6IFtcbiAgICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGUnKSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuXG4gICAgICB0aGlzLnByb3ZpZGVyID0gbmV3IGNyLlByb3ZpZGVyKHRoaXMsICdQcm92aWRlcicsIHtcbiAgICAgICAgb25FdmVudEhhbmRsZXI6IGN1c3RvbVJlc291cmNlRnVuY3Rpb24sXG4gICAgICAgIGxvZ1JldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9XRUVLLFxuICAgICAgICByb2xlOiBwcm92aWRlclJvbGUsXG4gICAgICB9KTtcblxuICAgICAgdGhpcy5zZXJ2aWNlVG9rZW4gPSB0aGlzLnByb3ZpZGVyLnNlcnZpY2VUb2tlbjtcblxuICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zQnlQYXRoKFxuICAgICAgICBjZGsuU3RhY2sub2YodGhpcyksXG4gICAgICAgIGAke3RoaXMucHJvdmlkZXIubm9kZS5wYXRofS9mcmFtZXdvcmstb25FdmVudC9SZXNvdXJjZWAsXG4gICAgICAgIFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBpZDogJ0F3c1NvbHV0aW9ucy1MMScsXG4gICAgICAgICAgICByZWFzb246ICdMYW1iZGEgcnVudGltZSB2ZXJzaW9uIGlzIG1hbmFnZWQgdXBzdHJlYW0gYnkgQ0RLLicsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICk7XG5cbiAgICAgIGZvciAobGV0IHJvbGUgb2YgW3RoaXMucm9sZSwgcHJvdmlkZXJSb2xlXSkge1xuICAgICAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICAgICAgcm9sZSxcbiAgICAgICAgICBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGlkOiAnQXdzU29sdXRpb25zLUlBTTQnLFxuICAgICAgICAgICAgICByZWFzb246ICdDREsgQ3VzdG9tUmVzb3VyY2UgTGFtYmRhIHVzZXMgdGhlIEFXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZSBBV1MgTWFuYWdlZCBQb2xpY3kuJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgKTtcbiAgICAgIH1cblxuICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgICBwcm92aWRlclJvbGUsXG4gICAgICAgIFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBpZDogJ0F3c1NvbHV0aW9ucy1JQU01JyxcbiAgICAgICAgICAgIHJlYXNvbjogJ0NESyBDdXN0b21SZXNvdXJjZSBQcm92aWRlciBoYXMgYSB3aWxkY2FyZCB0byBpbnZva2UgYW55IHZlcnNpb24gb2YgdGhlIHNwZWNpZmljIEN1c3RvbSBSZXNvdXJjZSBmdW5jdGlvbi4nLFxuICAgICAgICAgICAgYXBwbGllc1RvOiBbeyByZWdleDogYC9eUmVzb3VyY2U6Ojwke2lkfSR7Y3VzdG9tUmVzb3VyY2VGdW5jdGlvbi5ub2RlLmlkfVtBLVowLTldK1xcXFwuQXJuPjpcXFxcKiQvZ2AgfV0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgdHJ1ZSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIENSUHJvdmlkZXI7XG59Il19