"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Layer = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const child_process_1 = require("child_process");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const s3assets = require("aws-cdk-lib/aws-s3-assets");
const constructs_1 = require("constructs");
class Layer extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { runtime, architecture, path, additionalPackages, autoUpgrade, local } = props;
        const args = local ? [] : ['-t /asset-output/python'];
        if (additionalPackages) {
            args.push(...additionalPackages);
        }
        if (autoUpgrade) {
            args.push('--upgrade');
        }
        const layerAsset = new s3assets.Asset(this, 'LayerAsset', {
            path,
            bundling: local ? {
                // If local is true use the host to install the requirements
                image: runtime.bundlingImage,
                local: {
                    tryBundle(outputDir) {
                        (0, child_process_1.execSync)(`${local} -m venv venv`);
                        (0, child_process_1.execSync)('source venv/bin/activate');
                        (0, child_process_1.execSync)(`pip install -r ${path}/requirements.txt -t ${outputDir}/python ${args.join(' ')}`);
                        return true;
                    },
                },
            } : {
                // Default: Docker is used to install the requirements
                image: runtime.bundlingImage,
                platform: architecture.dockerPlatform,
                command: [
                    'bash',
                    '-c',
                    `pip install -r requirements.txt ${args.join(' ')}`,
                ],
                outputType: cdk.BundlingOutput.AUTO_DISCOVER,
                securityOpt: 'no-new-privileges:true',
                network: 'host',
            },
        });
        const layer = new lambda.LayerVersion(this, 'Layer', {
            code: lambda.Code.fromBucket(layerAsset.bucket, layerAsset.s3ObjectKey),
            compatibleRuntimes: [runtime],
            compatibleArchitectures: [architecture],
            ...props,
        });
        this.layer = layer;
    }
}
exports.Layer = Layer;
//# sourceMappingURL=data:application/json;base64,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