"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getlambdaSecuritygroup = exports.getPrivateSubnetIDs = exports.buildVpc = exports.CheckVpcProps = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const ec2 = require("aws-cdk-lib/aws-ec2");
function CheckVpcProps(propsObject) {
    let errorMessages = '';
    let errorFound = false;
    if ((propsObject.deployVpc || propsObject.vpcProps) && propsObject.existingVpc) {
        errorMessages += 'Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.CheckVpcProps = CheckVpcProps;
function buildVpc(scope, props) {
    if (props?.existingVpc) {
        return props?.existingVpc;
    }
    let defaultVpcProps = DefaultVpcProps();
    let cumulativeProps = defaultVpcProps;
    // Merge props provided by construct builder and by the end user
    // If user provided props are empty, the vpc will use only the builder provided props
    //cumulativeProps = consolidateProps(cumulativeProps, props?.userVpcProps, props?.constructVpcProps);
    const vpc = new ec2.Vpc(scope, 'Vpc', cumulativeProps);
    return vpc;
}
exports.buildVpc = buildVpc;
// get subnet id for passed vpc.
function getPrivateSubnetIDs(vpc) {
    return vpc.privateSubnets.map(subnet => subnet.subnetId);
}
exports.getPrivateSubnetIDs = getPrivateSubnetIDs;
// get lambda security group for passed VPC
function getlambdaSecuritygroup(scope, vpc) {
    let lambdaSecurityGroup = new ec2.SecurityGroup(scope, 'lambdaSecurityGroup', {
        vpc: vpc,
        allowAllOutbound: true,
        description: 'security group for lambda',
        securityGroupName: 'lambdaSecurityGroup',
    });
    return lambdaSecurityGroup;
}
exports.getlambdaSecuritygroup = getlambdaSecuritygroup;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Creates the default vpc props with public , private_with_egress and private_isolated subnet configuration.
 */
function DefaultVpcProps() {
    return {
        subnetConfiguration: [
            {
                name: 'public',
                subnetType: ec2.SubnetType.PUBLIC,
                cidrMask: 24,
            },
            {
                name: 'private',
                subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
                cidrMask: 24,
            },
            {
                name: 'isolated',
                subnetType: ec2.SubnetType.PRIVATE_ISOLATED,
                cidrMask: 24,
            },
        ],
        ipAddresses: ec2.IpAddresses.cidr('10.0.0.0/16'),
    };
}
//# sourceMappingURL=data:application/json;base64,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