"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContentGenerationAppSyncLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk_nag_1 = require("cdk-nag");
const base_class_1 = require("../../../common/base-class");
const lambda_builder_helper_1 = require("../../../common/helpers/lambda-builder-helper");
const s3_bucket_helper = require("../../../common/helpers/s3-bucket-helper");
const utils_1 = require("../../../common/helpers/utils");
const vpc_helper = require("../../../common/helpers/vpc-helper");
/**
 * @summary The ContentGenerationAppSyncLambda class.
 */
class ContentGenerationAppSyncLambda extends base_class_1.BaseClass {
    /**
     * @summary Constructs a new instance of the ContentGenerationAppSyncLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ContentGenerationAppSyncLambdaProps} props - user provided props for the construct.
     * @since 0.0.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            stage: props.stage,
            enableOperationalMetric: props.enableOperationalMetric,
            constructName: base_class_1.ConstructName.AWSCONTENTGENAPPSYNCLAMBDA,
            constructId: id,
            observability: props.observability,
        };
        this.updateEnvSuffix(baseProps);
        this.addObservabilityToConstruct(baseProps);
        vpc_helper.CheckVpcProps(props);
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingGeneratedAssetsBucketObj,
            bucketProps: props.generatedAssetsBucketProps,
        });
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
                securityGroupName: 'securityGroup' + this.stage,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'generateImageConstructLogGroup');
        const role = new iam.Role(this, 'generateImageConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + this.stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            versioned: true,
            lifecycleRules: [
                {
                    expiration: aws_cdk_lib_1.Duration.days(90),
                },
            ],
        });
        // Bucket containing the inputs assets (documents - text format) uploaded by the user
        let generatedAssetsBucket;
        if (!props.existingGeneratedAssetsBucketObj) {
            let tmpBucket;
            if (!props.generatedAssetsBucketProps) {
                tmpBucket = new s3.Bucket(this, 'generatedAssetsBucket' + this.stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'generated-asset-bucket' + this.stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [
                        {
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        },
                    ],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'generatedAssetsBucket' + this.stage, props.generatedAssetsBucketProps);
            }
            generatedAssetsBucket = tmpBucket;
            this.s3GenerateAssetsBucket = tmpBucket;
        }
        else {
            generatedAssetsBucket = props.existingGeneratedAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3GenerateAssetsBucketInterface = generatedAssetsBucket;
        // GraphQL API
        const generate_image_graphql_api = new appsync.GraphqlApi(this, 'generateImageGraphqlApi', {
            name: 'generateImageGraphqlApi' + this.stage,
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-contentgen-appsync-lambda/schema.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool: props.cognitoUserPool },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.IAM,
                    },
                ],
            },
            xrayEnabled: this.enablexray,
            logConfig: {
                fieldLogLevel: this.fieldLogLevel,
                retention: this.retention,
            },
        });
        this.graphqlApi = generate_image_graphql_api;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi
            ? generate_image_graphql_api.graphqlUrl
            : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi
            ? generate_image_graphql_api.apiId
            : props.existingMergedApi.attrApiId;
        const job_status_data_source = new appsync.NoneDataSource(this, 'NoneDataSourceQuestionAnswering', {
            api: this.graphqlApi,
            name: 'JobStatusDataSource',
        });
        job_status_data_source.createResolver('updateGenImageJobStatusResolver', {
            fieldName: 'updateGenerateImageStatus',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                          {
                              "version": "2017-02-28",
                              "payload": $util.toJson($context.args)
                          }
                          `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        if (!props.existingBusInterface) {
            this.generatedImageBus = new events.EventBus(this, 'generateImageEventBus' + this.stage, {
                eventBusName: 'generateImageEventBus' + this.stage,
            });
        }
        else {
            this.generatedImageBus = props.existingBusInterface;
        }
        // create httpdatasource with generate_image_graphql_api
        const event_bridge_datasource = this.graphqlApi.addEventBridgeDataSource('generateImageEventBridgeDataSource' + this.stage, this.generatedImageBus, {
            name: 'generateImageEventBridgeDataSource' + this.stage,
        });
        // Lambda function used to validate inputs in the step function
        const generate_image_function_role = new iam.Role(this, 'generate_image_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [
                                `arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['ec2:DescribeNetworkInterfaces'],
            resources: ['*'],
        }));
        // The lambda will genearte image and upload to s3
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:GetObject*', 's3:GetBucket*', 's3:List*'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3GenerateAssetsBucketInterface?.bucketName,
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' +
                    this.s3GenerateAssetsBucketInterface?.bucketName +
                    '/*',
            ],
        }));
        // Add Amazon Bedrock permissions to the IAM role for the Lambda function
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'bedrock:InvokeModel',
                'bedrock:InvokeModelWithResponseStream',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*',
            ],
        }));
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'comprehend:DetectEntities',
                'comprehend:DetectKeyPhrases',
                'comprehend:DetectPiiEntities',
                'comprehend:DetectSentiment',
                'comprehend:DetectSyntax',
            ],
            resources: ['*'],
        }));
        generate_image_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'rekognition:DetectModerationLabels',
                'rekognition:DetectText',
                'rekognition:DetectLabels',
                'rekognition:DetectFaces',
            ],
            resources: ['*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(generate_image_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const construct_docker_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-contentgen-appsync-lambda/src')),
            functionName: 'lambda_generate_image' + this.stage,
            description: 'Lambda function for generating image',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: generate_image_function_role,
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                OUTPUT_BUCKET: this.s3GenerateAssetsBucketInterface.bucketName,
            },
            ...(props.lambdaProvisionedConcurrency && {
                currentVersionOptions: {
                    provisionedConcurrentExecutions: props.lambdaProvisionedConcurrency,
                },
            }),
        };
        const generate_image_function = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'lambda_content_generation' + this.stage, construct_docker_lambda_props, props.customDockerLambdaProps);
        generate_image_function.currentVersion;
        const lambdaFunctions = [generate_image_function];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        // Add GraphQl permissions to the IAM role for the Lambda function
        generate_image_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['appsync:GraphQL'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' +
                    aws_cdk_lib_1.Aws.REGION +
                    ':' +
                    aws_cdk_lib_1.Aws.ACCOUNT_ID +
                    ':apis/' +
                    updateGraphQlApiId +
                    '/*',
            ],
        }));
        this.generatedImageBus.grantPutEventsTo(event_bridge_datasource.grantPrincipal);
        event_bridge_datasource.createResolver('GenerateImageResolver', {
            fieldName: 'generateImage',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                {
                  "version": "2018-05-29",
                  "operation": "PutEvents",
                  "events": [{
                      "source": "textToImage",
                      "detail": $util.toJson($context.arguments),
                      "detailType": "genAIdemo"
                  }]
                }

 
                        `),
            responseMappingTemplate: appsync.MappingTemplate.fromString(`
                        #if($ctx.error)
                            $util.error($ctx.error.message, $ctx.error.type, $ctx.result)
                        #end
                            $util.toJson($ctx.result)
                        `),
        });
        const rule = new events.Rule(this, 'textToImageRule' + this.stage, {
            description: 'Rule to trigger textToImage function',
            eventBus: this.generatedImageBus,
            eventPattern: {
                source: ['textToImage'],
            },
        });
        rule.addTarget(new targets.LambdaFunction(generate_image_function));
        this.cgLambdaFunction = generate_image_function;
    }
}
exports.ContentGenerationAppSyncLambda = ContentGenerationAppSyncLambda;
_a = JSII_RTTI_SYMBOL_1;
ContentGenerationAppSyncLambda[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.ContentGenerationAppSyncLambda", version: "0.1.94" };
//# sourceMappingURL=data:application/json;base64,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