"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomSageMakerEndpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const sagemaker = require("aws-cdk-lib/aws-sagemaker");
const sagemaker_endpoint_base_1 = require("./sagemaker-endpoint-base");
const base_class_1 = require("../../../common/base-class");
class CustomSageMakerEndpoint extends sagemaker_endpoint_base_1.SageMakerEndpointBase {
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            enableOperationalMetric: props.enableOperationalMetric,
            constructName: base_class_1.ConstructName.CUSTOMSAGEMAKERENDPOINT,
            constructId: id,
        };
        // No lambda function to use AWS SDK for service metric
        const lambdaFunctions = [];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        this.instanceType = props.instanceType;
        this.modelId = props.modelId;
        this.instanceCount = Math.max(1, props.instanceCount ?? 1);
        this.role = props.role ?? this.createSageMakerRole();
        this.grantPrincipal = this.role;
        this.modelDataUrl = props.modelDataUrl;
        this.startupHealthCheckTimeoutInSeconds = props.startupHealthCheckTimeoutInSeconds ?? 600;
        this.environment = props.environment;
        this.modelDataDownloadTimeoutInSeconds = props.modelDataDownloadTimeoutInSeconds ?? 600;
        const image = props.container.bind(this, this.grantPrincipal).imageName;
        const modelIdStr = this.modelId.split('/').join('-').split('.').join('-');
        const isArtifactCompressed = this.modelDataUrl.endsWith('.tar.gz');
        const model = new sagemaker.CfnModel(scope, `${modelIdStr}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            primaryContainer: isArtifactCompressed ? {
                image,
                mode: 'SingleModel',
                modelDataUrl: this.modelDataUrl,
                environment: {
                    SAGEMAKER_REGION: cdk.Aws.REGION,
                    ...this.environment,
                },
            } : {
                image,
                mode: 'SingleModel',
                modelDataSource: {
                    s3DataSource: {
                        compressionType: 'None',
                        s3DataType: 'S3Prefix',
                        s3Uri: this.modelDataUrl,
                    },
                },
                environment: {
                    SAGEMAKER_REGION: cdk.Aws.REGION,
                    ...this.environment,
                },
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.modelId,
                },
            ],
            vpcConfig: props.vpcConfig,
        });
        const endpointConfig = new sagemaker.CfnEndpointConfig(scope, `EndpointConfig-${id}`, {
            productionVariants: [
                {
                    instanceType: this.instanceType.toString(),
                    initialVariantWeight: 1,
                    initialInstanceCount: this.instanceCount,
                    variantName: 'AllTraffic',
                    volumeSizeInGb: props.volumeSizeInGb,
                    modelName: model.getAtt('ModelName').toString(),
                    containerStartupHealthCheckTimeoutInSeconds: this.startupHealthCheckTimeoutInSeconds,
                    modelDataDownloadTimeoutInSeconds: this.modelDataDownloadTimeoutInSeconds,
                },
            ],
        });
        endpointConfig.addDependency(model);
        const endpoint = new sagemaker.CfnEndpoint(scope, `${modelIdStr}-endpoint-${id}`, {
            endpointName: props.endpointName,
            endpointConfigName: endpointConfig.getAtt('EndpointConfigName').toString(),
            tags: [
                {
                    key: 'modelId',
                    value: this.modelId,
                },
            ],
        });
        endpoint.addDependency(endpointConfig);
        this.cfnModel = model;
        this.cfnEndpoint = endpoint;
        this.cfnEndpointConfig = endpointConfig;
        this.endpointArn = endpoint.ref;
    }
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['sagemaker:InvokeEndpoint'],
            resourceArns: [this.endpointArn],
        });
    }
}
exports.CustomSageMakerEndpoint = CustomSageMakerEndpoint;
_a = JSII_RTTI_SYMBOL_1;
CustomSageMakerEndpoint[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.CustomSageMakerEndpoint", version: "0.1.94" };
//# sourceMappingURL=data:application/json;base64,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