"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HuggingFaceSageMakerEndpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const sagemaker = require("aws-cdk-lib/aws-sagemaker");
const sagemaker_endpoint_base_1 = require("./sagemaker-endpoint-base");
const base_class_1 = require("../../../common/base-class");
/**
 * @summary The HuggingFaceSageMakerEndpoint class.
 */
class HuggingFaceSageMakerEndpoint extends sagemaker_endpoint_base_1.SageMakerEndpointBase {
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            enableOperationalMetric: props.enableOperationalMetric,
            constructName: base_class_1.ConstructName.HUGGINGFACESAGEMAKERENDPOINT,
            constructId: id,
        };
        // No lambda function to use AWS SDK for service metric
        const lambdaFunctions = [];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        this.modelId = props.modelId;
        this.instanceType = props.instanceType;
        this.instanceCount = Math.max(1, props.instanceCount ?? 1);
        this.role = props.role ?? this.createSageMakerRole();
        this.grantPrincipal = this.role;
        this.startupHealthCheckTimeoutInSeconds = props.startupHealthCheckTimeoutInSeconds ?? 600;
        this.environment = props.environment;
        const image = props.container.bind(this, this.grantPrincipal).imageName;
        const modelIdStr = this.modelId.split('/').join('-').split('.').join('-');
        const model = new sagemaker.CfnModel(scope, `${modelIdStr}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            primaryContainer: {
                image,
                mode: 'SingleModel',
                environment: {
                    SAGEMAKER_CONTAINER_LOG_LEVEL: '20',
                    SAGEMAKER_REGION: cdk.Aws.REGION,
                    HF_MODEL_ID: this.modelId,
                    ...this.environment,
                },
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.modelId,
                },
            ],
            vpcConfig: props.vpcConfig,
        });
        const endpointConfig = new sagemaker.CfnEndpointConfig(scope, `EndpointConfig-${id}`, {
            productionVariants: [
                {
                    instanceType: this.instanceType.toString(),
                    initialVariantWeight: 1,
                    initialInstanceCount: this.instanceCount,
                    variantName: 'AllTraffic',
                    modelName: model.getAtt('ModelName').toString(),
                    containerStartupHealthCheckTimeoutInSeconds: this.startupHealthCheckTimeoutInSeconds,
                },
            ],
        });
        endpointConfig.addDependency(model);
        const endpoint = new sagemaker.CfnEndpoint(scope, `${modelIdStr}-endpoint-${id}`, {
            endpointName: props.endpointName,
            endpointConfigName: endpointConfig.getAtt('EndpointConfigName').toString(),
            tags: [
                {
                    key: 'modelId',
                    value: this.modelId,
                },
            ],
        });
        endpoint.addDependency(endpointConfig);
        this.cfnModel = model;
        this.cfnEndpoint = endpoint;
        this.cfnEndpointConfig = endpointConfig;
        this.endpointArn = endpoint.ref;
    }
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['sagemaker:InvokeEndpoint'],
            resourceArns: [this.endpointArn],
        });
    }
}
exports.HuggingFaceSageMakerEndpoint = HuggingFaceSageMakerEndpoint;
_a = JSII_RTTI_SYMBOL_1;
HuggingFaceSageMakerEndpoint[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.HuggingFaceSageMakerEndpoint", version: "0.1.94" };
//# sourceMappingURL=data:application/json;base64,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