"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JumpStartSageMakerEndpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const sagemaker = require("aws-cdk-lib/aws-sagemaker");
const core_1 = require("aws-cdk-lib/core");
const jumpstart_constants_1 = require("./private/jumpstart-constants");
const sagemaker_endpoint_base_1 = require("./sagemaker-endpoint-base");
const construct_name_enum_1 = require("../../../common/base-class/construct-name-enum");
/**
 * @summary The JumpStartSageMakerEndpoint class.
 */
class JumpStartSageMakerEndpoint extends sagemaker_endpoint_base_1.SageMakerEndpointBase {
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            enableOperationalMetric: props.enableOperationalMetric,
            constructName: construct_name_enum_1.ConstructName.JUMPSTARTSAGEMAKERENDPOINT,
            constructId: id,
        };
        // No lambda function to use AWS SDK for service metric
        const lambdaFunctions = [];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        this.model = props.model;
        this.instanceType = props.instanceType;
        this.instanceCount = Math.max(1, props.instanceCount ?? 1);
        this.acceptEula = props.acceptEula ?? false;
        this.role = props.role ?? this.createSageMakerRole();
        this.grantPrincipal = this.role;
        this.startupHealthCheckTimeoutInSeconds = props.startupHealthCheckTimeoutInSeconds ?? 600;
        this.environment = props.environment;
        this.spec = this.model.bind();
        if (!this.acceptEula && this.spec.requiresEula) {
            throw new Error('The AcceptEula value must be explicitly defined as True in order to accept the EULA for the model ' + this.spec.modelId + '. You are responsible for reviewing and complying with any applicable license terms and making sure they are acceptable for your use case before downloading or using a model.');
        }
        this.region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(this.region)) {
            throw new Error('Region is unresolved. You should explicitly specify the region in the environment.');
        }
        const instanceType = this.verifyInstanceType();
        const instanseBaseType = instanceType.split('.')[1];
        let model;
        if (this.spec.modelPackageArns) {
            if (this.environment) {
                throw new Error('Environment variables are not supported for model packages.');
            }
            model = this.getModelFromPackage(scope, id, props.vpcConfig);
        }
        else {
            const environment = this.buildEnvironment(instanceType);
            model = this.getModelFromArtifact(scope, id, instanceType, instanseBaseType, environment, props.vpcConfig);
        }
        const endpointConfig = new sagemaker.CfnEndpointConfig(scope, `EndpointConfig-${id}`, {
            productionVariants: [
                {
                    instanceType,
                    initialVariantWeight: 1,
                    initialInstanceCount: this.instanceCount,
                    variantName: 'AllTraffic',
                    modelName: model.getAtt('ModelName').toString(),
                    containerStartupHealthCheckTimeoutInSeconds: this.startupHealthCheckTimeoutInSeconds,
                },
            ],
        });
        endpointConfig.addDependency(model);
        const endpoint = new sagemaker.CfnEndpoint(scope, `${this.spec.modelId}-endpoint-${id}`, {
            endpointConfigName: endpointConfig.getAtt('EndpointConfigName').toString(),
            endpointName: 'jumpstart-' + props.endpointName,
        });
        endpoint.addDependency(endpointConfig);
        this.cfnModel = model;
        this.cfnEndpoint = endpoint;
        this.cfnEndpointConfig = endpointConfig;
        this.endpointArn = endpoint.ref;
    }
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['sagemaker:InvokeEndpoint'],
            resourceArns: [this.endpointArn],
        });
    }
    verifyInstanceType() {
        let instanceType = this.spec.defaultInstanceType;
        if (this.instanceType) {
            instanceType = this.instanceType.toString();
        }
        const supportedInstanceTypes = this.spec.instanceTypes;
        if (!supportedInstanceTypes.includes(instanceType)) {
            throw new Error(`The instance type ${instanceType} is not supported. Default instance type: ${this.spec.defaultInstanceType}. Supported instance types: ${supportedInstanceTypes.join(', ')}.`);
        }
        return instanceType;
    }
    buildEnvironment(instanceType) {
        const configEnvironment = this.spec.instanceVariants?.find((v) => v.instanceType === instanceType)?.environment;
        const environment = {
            ...(this.spec.environment ?? {}),
            ...configEnvironment,
            ...this.environment,
        };
        return environment;
    }
    getModelFromArtifact(scope, id, instanceType, instanceBaseType, environment, vpcConfig) {
        const key = this.spec.prepackedArtifactKey ?? this.spec.artifactKey;
        const bucket = jumpstart_constants_1.JumpStartConstants.JUMPSTART_LAUNCHED_REGIONS[this.region]?.contentBucket;
        if (!bucket) {
            throw new Error(`JumpStart is not available in the region ${this.region}.`);
        }
        const modelArtifactUrl = `s3://${bucket}/${key}`;
        const isArtifactCompressed = modelArtifactUrl.endsWith('.tar.gz');
        const imageUriKey = this.spec.instanceVariants
            ?.find((v) => v.instanceType === instanceBaseType)
            ?.imageUri?.replace('$', '');
        if (!imageUriKey) {
            throw new Error(`The image uri is not available for instance type ${instanceType}.`);
        }
        const image = this.spec.instanceAliases?.find((v) => v.region === this.region)?.aliases[imageUriKey];
        if (!image) {
            throw new Error(`The image uri is not available for instance type ${instanceType} in region ${this.region}.`);
        }
        const model = new sagemaker.CfnModel(scope, `${this.spec.modelId}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            enableNetworkIsolation: true,
            primaryContainer: isArtifactCompressed ? {
                // True: Artifact is a tarball
                image,
                modelDataUrl: modelArtifactUrl,
                environment,
            } : {
                // False: Model is uncompressed
                image,
                modelDataSource: {
                    s3DataSource: {
                        compressionType: 'None',
                        s3DataType: 'S3Prefix',
                        s3Uri: modelArtifactUrl,
                        modelAccessConfig: {
                            acceptEula: this.acceptEula,
                        },
                    },
                },
                environment,
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.spec.modelId,
                },
                {
                    key: 'modelVersion',
                    value: this.spec.version,
                },
            ],
            vpcConfig: vpcConfig,
        });
        return model;
    }
    getModelFromPackage(scope, id, vpcConfig) {
        const modelPackageArns = this.spec.modelPackageArns || {};
        const supportedRegions = Object.keys(modelPackageArns);
        if (!supportedRegions.includes(this.region)) {
            throw new Error(`The model package is not available in the region ${this.region}. Supported regions: ${supportedRegions.join(', ')}.`);
        }
        const modelPackageArn = modelPackageArns[this.region];
        const model = new sagemaker.CfnModel(scope, `${this.spec.modelId}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            enableNetworkIsolation: true,
            primaryContainer: {
                modelPackageName: modelPackageArn,
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.spec.modelId,
                },
                {
                    key: 'modelVersion',
                    value: this.spec.version,
                },
            ],
            vpcConfig: vpcConfig,
        });
        return model;
    }
}
exports.JumpStartSageMakerEndpoint = JumpStartSageMakerEndpoint;
_a = JSII_RTTI_SYMBOL_1;
JumpStartSageMakerEndpoint[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.JumpStartSageMakerEndpoint", version: "0.1.94" };
//# sourceMappingURL=data:application/json;base64,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