"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QaAppsyncOpensearch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk_nag_1 = require("cdk-nag");
const base_class_1 = require("../../../common/base-class");
const lambda_builder_helper_1 = require("../../../common/helpers/lambda-builder-helper");
const opensearch_helper = require("../../../common/helpers/opensearch-helper");
const s3_bucket_helper = require("../../../common/helpers/s3-bucket-helper");
const utils_1 = require("../../../common/helpers/utils");
const vpc_helper = require("../../../common/helpers/vpc-helper");
/**
 * @summary The QaAppsyncOpensearch class.
 */
class QaAppsyncOpensearch extends base_class_1.BaseClass {
    /**
     * @summary Constructs a new instance of the RagAppsyncStepfnOpensearch class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {QaAppsyncOpensearchProps} props - user provided props for the construct.
     * @since 0.0.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            stage: props.stage,
            enableOperationalMetric: props.enableOperationalMetric,
            constructName: base_class_1.ConstructName.AWSQAAPPSYNCOPENSEARCH,
            constructId: id,
            observability: props.observability,
        };
        this.updateEnvSuffix(baseProps);
        this.addObservabilityToConstruct(baseProps);
        vpc_helper.CheckVpcProps(props);
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
                securityGroupName: 'securityGroup' + this.stage,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'qaConstructLogGroup');
        const role = new iam.Role(this, 'qaConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + this.stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            versioned: true,
            lifecycleRules: [
                {
                    expiration: aws_cdk_lib_1.Duration.days(90),
                },
            ],
        });
        // Bucket containing the inputs assets (documents - text format) uploaded by the user
        let inputAssetsBucket;
        if (!props.existingInputAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'inputAssetsQABucket' + this.stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'input-asset-qa-bucket' + this.stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [
                        {
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        },
                    ],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'InputAssetsQABucket' + this.stage, props.bucketInputsAssetsProps);
            }
            inputAssetsBucket = tmpBucket;
            this.s3InputAssetsBucket = tmpBucket;
        }
        else {
            inputAssetsBucket = props.existingInputAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3InputAssetsBucketInterface = inputAssetsBucket;
        // GraphQL API
        const question_answering_graphql_api = new appsync.GraphqlApi(this, 'questionAnsweringGraphqlApi', {
            name: 'questionAnsweringGraphqlApi' + this.stage,
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-qa-appsync-opensearch/schema.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool: props.cognitoUserPool },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.IAM,
                    },
                ],
            },
            xrayEnabled: this.enablexray,
            logConfig: {
                fieldLogLevel: this.fieldLogLevel,
                retention: this.retention,
            },
        });
        this.graphqlApi = question_answering_graphql_api;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi
            ? question_answering_graphql_api.graphqlUrl
            : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi
            ? question_answering_graphql_api.apiId
            : props.existingMergedApi.attrApiId;
        const job_status_data_source = new appsync.NoneDataSource(this, 'NoneDataSourceQuestionAnswering', {
            api: this.graphqlApi,
            name: 'JobStatusDataSource',
        });
        job_status_data_source.createResolver('updateQAJobStatusResolver', {
            fieldName: 'updateQAJobStatus',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                          {
                              "version": "2017-02-28",
                              "payload": $util.toJson($context.args)
                          }
                          `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        if (!props.existingBusInterface) {
            this.qaBus = new events.EventBus(this, 'questionAnsweringEventBus' + this.stage, {
                eventBusName: 'questionAnsweringEventBus' + this.stage,
            });
        }
        else {
            this.qaBus = props.existingBusInterface;
        }
        // create httpdatasource with question_answering_graphql_api
        const event_bridge_datasource = this.graphqlApi.addEventBridgeDataSource('questionAnsweringEventBridgeDataSource' + this.stage, this.qaBus, {
            name: 'questionAnsweringEventBridgeDataSource' + this.stage,
        });
        let SecretId = 'NONE';
        if (props.openSearchSecret) {
            SecretId = props.openSearchSecret.secretName;
        }
        // Lambda function used to validate inputs in the step function
        const question_answering_function_role = new iam.Role(this, 'question_answering_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [
                                `arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        // Minimum permissions for a Lambda functienvon to execute while accessing a resource within a VPC
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['ec2:DescribeNetworkInterfaces'],
            resources: ['*'],
        }));
        // The lambda will access the opensearch credentials
        if (props.openSearchSecret) {
            props.openSearchSecret.grantRead(question_answering_function_role);
        }
        // The lambda will pull processed files and create embeddings
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:GetObject*', 's3:GetBucket*', 's3:List*'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3InputAssetsBucketInterface?.bucketName,
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' +
                    this.s3InputAssetsBucketInterface?.bucketName +
                    '/*',
            ],
        }));
        if (props.existingOpensearchDomain) {
            question_answering_function_role.addToPolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['es:*'],
                resources: [
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':es:' +
                        aws_cdk_lib_1.Aws.REGION +
                        ':' +
                        aws_cdk_lib_1.Aws.ACCOUNT_ID +
                        ':domain/' +
                        props.existingOpensearchDomain.domainName +
                        '/*',
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':es:' +
                        aws_cdk_lib_1.Aws.REGION +
                        ':' +
                        aws_cdk_lib_1.Aws.ACCOUNT_ID +
                        ':domain/' +
                        props.existingOpensearchDomain.domainName,
                ],
            }));
        }
        if (props.existingOpensearchServerlessCollection) {
            question_answering_function_role.addToPolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['aoss:APIAccessAll'],
                resources: [
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':aoss:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':collection/' + props.existingOpensearchServerlessCollection.attrId,
                ],
            }));
        }
        // Add Amazon Bedrock permissions to the IAM role for the Lambda function
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'bedrock:InvokeModel',
                'bedrock:InvokeModelWithResponseStream',
                'bedrock:ListFoundationModels',
            ],
            // ListFoundationModels has no specific resource type
            resources: [
                '*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(question_answering_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const construct_docker_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-qa-appsync-opensearch/question_answering/src')),
            functionName: 'lambda_question_answering' + this.stage,
            description: 'Lambda function for question answering',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: question_answering_function_role,
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                INPUT_BUCKET: this.s3InputAssetsBucketInterface.bucketName,
                OPENSEARCH_API_NAME: opensearch_helper.getOpenSearchApiName(props),
                OPENSEARCH_DOMAIN_ENDPOINT: opensearch_helper.getOpenSearchEndpoint(props),
                OPENSEARCH_INDEX: props.openSearchIndexName,
                OPENSEARCH_SECRET_ID: SecretId,
            },
            ...(props.lambdaProvisionedConcurrency && {
                currentVersionOptions: {
                    provisionedConcurrentExecutions: props.lambdaProvisionedConcurrency,
                },
            }),
        };
        const question_answering_function = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'lambda_question_answering' + this.stage, construct_docker_lambda_props, props.customDockerLambdaProps);
        question_answering_function.currentVersion;
        const lambdaFunctions = [question_answering_function];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        // Add GraphQl permissions to the IAM role for the Lambda function
        question_answering_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['appsync:GraphQL'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' +
                    aws_cdk_lib_1.Aws.REGION +
                    ':' +
                    aws_cdk_lib_1.Aws.ACCOUNT_ID +
                    ':apis/' +
                    updateGraphQlApiId +
                    '/*',
            ],
        }));
        this.qaBus.grantPutEventsTo(event_bridge_datasource.grantPrincipal);
        event_bridge_datasource.createResolver('QuestionAnsweringResolver', {
            fieldName: 'postQuestion',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                        {
                            "version": "2018-05-29",
                            "operation": "PutEvents",
                            "events": [{
                                "source": "questionanswering",
                                "detail": $util.toJson($context.arguments),
                                "detailType": "Question answering"
                            }
                            ]
                        } 
                        `),
            responseMappingTemplate: appsync.MappingTemplate.fromString(`
                        #if($ctx.error)
                            $util.error($ctx.error.message, $ctx.error.type, $ctx.result)
                        #end
                            $util.toJson($ctx.result)
                        `),
        });
        const rule = new events.Rule(this, 'QuestionAnsweringRule' + this.stage, {
            description: 'Rule to trigger question answering function',
            eventBus: this.qaBus,
            eventPattern: {
                source: ['questionanswering'],
            },
        });
        rule.addTarget(new targets.LambdaFunction(question_answering_function));
        this.qaLambdaFunction = question_answering_function;
    }
}
exports.QaAppsyncOpensearch = QaAppsyncOpensearch;
_a = JSII_RTTI_SYMBOL_1;
QaAppsyncOpensearch[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.QaAppsyncOpensearch", version: "0.1.94" };
/**
 * Construct warning
 */
QaAppsyncOpensearch.CONSTRUCT_SCHEMA_UPDATE_WARNING = `
  Attention QaAppsyncOpensearch users, an update has been made to 
  the GraphQL schema.To ensure continued functionality, please review 
  and update your GraphQL mutations and subscriptions to align with 
  the new schema.This schema update enables enhanced capabilities 
  and optimizations,so adopting the changes is recommended. 
  Please refer to the construct documentation for details 
  on the schema changes and examples of updated GraphQL statements.
  Reach out to the support team if you need assistance 
  updating your integration codebase.  
  `;
//# sourceMappingURL=data:application/json;base64,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