"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RagAppsyncStepfnOpensearch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const stepfn = require("aws-cdk-lib/aws-stepfunctions");
const stepfn_task = require("aws-cdk-lib/aws-stepfunctions-tasks");
const cdk_nag_1 = require("cdk-nag");
const base_class_1 = require("../../../common/base-class");
const construct_name_enum_1 = require("../../../common/base-class/construct-name-enum");
const lambda_builder_helper_1 = require("../../../common/helpers/lambda-builder-helper");
const opensearch_helper = require("../../../common/helpers/opensearch-helper");
const s3_bucket_helper = require("../../../common/helpers/s3-bucket-helper");
const utils_1 = require("../../../common/helpers/utils");
const vpc_helper = require("../../../common/helpers/vpc-helper");
/**
   * @summary The RagAppsyncStepfnOpensearch class.
   */
class RagAppsyncStepfnOpensearch extends base_class_1.BaseClass {
    /**
       * @summary Constructs a new instance of the RagAppsyncStepfnOpensearch class.
       * @param {cdk.App} scope - represents the scope for all the resources.
       * @param {string} id - this is a a scope-unique id.
       * @param {RagAppsyncStepfnOpensearchProps} props - user provided props for the construct.
       * @since 0.0.0
       * @access public
       */
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            stage: props.stage,
            enableOperationalMetric: props.enableOperationalMetric,
            constructName: construct_name_enum_1.ConstructName.AWSRAGAPPSYNCSTEPFNOPENSEARCH,
            constructId: id,
            observability: props.observability,
        };
        this.updateEnvSuffix(baseProps);
        this.addObservabilityToConstruct(baseProps);
        vpc_helper.CheckVpcProps(props);
        opensearch_helper.CheckOpenSearchProps(props);
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingProcessedAssetsBucketObj,
            bucketProps: props.bucketProcessedAssetsProps,
        });
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
                securityGroupName: 'securityGroup' + this.stage,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'ingestionConstructLogGroup');
        const role = new iam.Role(this, 'ingestionConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + this.stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            versioned: true,
            lifecycleRules: [{
                    expiration: aws_cdk_lib_1.Duration.days(90),
                }],
        });
        // Bucket containing the inputs assets (documents - multiple modalities) uploaded by the user
        let inputAssetsBucket;
        if (!props.existingInputAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'inputAssetsBucket' + this.stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'input-assets-bucket' + this.stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [{
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        }],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'InputAssetsBucket' + this.stage, props.bucketInputsAssetsProps);
            }
            inputAssetsBucket = tmpBucket;
            this.s3InputAssetsBucket = tmpBucket;
        }
        else {
            inputAssetsBucket = props.existingInputAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3InputAssetsBucketInterface = inputAssetsBucket;
        // Bucket containing the processed assets (documents - text format) uploaded by the user
        let processedAssetsBucket;
        if (!props.existingProcessedAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'processedAssetsBucket' + this.stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'processed-assets-bucket' + this.stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [{
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        }],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'processedAssetsBucket' + this.stage, props.bucketProcessedAssetsProps);
            }
            processedAssetsBucket = tmpBucket;
            this.s3ProcessedAssetsBucket = tmpBucket;
        }
        else {
            processedAssetsBucket = props.existingProcessedAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3ProcessedAssetsBucketInterface = processedAssetsBucket;
        // GraphQL API
        const ingestion_graphql_api = new appsync.GraphqlApi(this, 'ingestionGraphqlApi', {
            name: 'ingestionGraphqlApi' + this.stage,
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-rag-appsync-stepfn-opensearch/schema.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool: props.cognitoUserPool },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.IAM,
                    },
                ],
            },
            xrayEnabled: this.enablexray,
            logConfig: {
                fieldLogLevel: this.fieldLogLevel,
                retention: this.retention,
            },
        });
        this.graphqlApi = ingestion_graphql_api;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi ? ingestion_graphql_api.graphqlUrl : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi ? ingestion_graphql_api.apiId : props.existingMergedApi.attrApiId;
        const job_status_data_source = new appsync.NoneDataSource(this, 'NoneDataSourceIngestion', {
            api: this.graphqlApi,
            name: 'JobStatusDataSource',
        });
        job_status_data_source.createResolver('updateIngestionJobStatusResolver', {
            fieldName: 'updateIngestionJobStatus',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                    {
                        "version": "2017-02-28",
                        "payload": $util.toJson($context.args)
                    }
                    `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        if (!props.existingBusInterface) {
            this.ingestionBus = new events.EventBus(this, 'ingestionEventBus' + this.stage, {
                eventBusName: 'ingestionEventBus' + this.stage,
            });
        }
        else {
            this.ingestionBus = props.existingBusInterface;
        }
        // create httpdatasource with ingestion_graphql_api
        const event_bridge_datasource = this.graphqlApi.addEventBridgeDataSource('ingestionEventBridgeDataSource' + this.stage, this.ingestionBus, {
            name: 'ingestionEventBridgeDataSource' + this.stage,
        });
        const construct_input_validation_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/input_validation/src')),
            functionName: 'ingestion_input_validation_docker' + this.stage,
            description: 'Lambda function for validating input files formats',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        };
        // Lambda function used to validate inputs in the step function
        const validate_input_function = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'lambda_function_validation_input' + this.stage, construct_input_validation_lambda_props, props.customInputValidationDockerLambdaProps);
        // Add GraphQl permissions to the IAM role for the Lambda function
        validate_input_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        // The lambda will pull documents from the input bucket, transform them, and upload
        // the artifacts to the processed bucket
        // we don't use grant read here since it has no effect in case of existing buckets provided by the user
        const s3_transformer_job_function_role = new iam.Role(this, 's3_transformer_job_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'rekognition:DetectModerationLabels',
            ],
            resources: ['*'],
        }));
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['bedrock:*'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*',
            ],
        }));
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
                's3:GetObject*',
                's3:GetBucket*',
                's3:List*',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3InputAssetsBucketInterface?.bucketName,
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3InputAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:PutObjectRetention',
                's3:List*',
                's3:GetBucket*',
                's3:Abort*',
                's3:DeleteObject*',
                's3:PutObjectLegalHold',
                's3:PutObjectTagging',
                's3:PutObjectVersionTagging',
                's3:PutObject',
                's3:GetObject*'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName,
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        // Add GraphQl permissions to the IAM role for the Lambda function
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(s3_transformer_job_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const construct__file_transformer_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/s3_file_transformer/src')),
            functionName: 's3_file_transformer_docker' + this.stage,
            description: 'Lambda function for converting files from their input format to text',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: s3_transformer_job_function_role,
            environment: {
                INPUT_BUCKET: this.s3InputAssetsBucketInterface.bucketName,
                OUTPUT_BUCKET: this.s3ProcessedAssetsBucketInterface.bucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        };
        const s3_transformer_job_function = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'lambda_function_s3_file_transformer' + this.stage, construct__file_transformer_lambda_props, props.customFileTransformerDockerLambdaProps);
        let SecretId = 'NONE';
        if (props.openSearchSecret) {
            SecretId = props.openSearchSecret.secretName;
        }
        const embeddings_job_function_role = new iam.Role(this, 'embeddings_job_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
                's3:GetObject*',
                's3:GetBucket*',
                's3:List*',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName,
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        if (props.existingOpensearchDomain) {
            embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['es:*'],
                resources: [
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':es:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':domain/' + props.existingOpensearchDomain.domainName + '/*',
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':es:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':domain/' + props.existingOpensearchDomain.domainName,
                ],
            }));
        }
        if (props.existingOpensearchServerlessCollection) {
            embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['aoss:APIAccessAll'],
                resources: [
                    'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':aoss:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':collection/' + props.existingOpensearchServerlessCollection.attrId,
                ],
            }));
        }
        // Add Amazon Bedrock permissions to the IAM role for the Lambda function
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['bedrock:*'],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(embeddings_job_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        // The lambda will access the opensearch credentials
        if (props.openSearchSecret) {
            props.openSearchSecret.grantRead(embeddings_job_function_role);
        }
        const construct_embeddings_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/embeddings_job/src')),
            functionName: 'embeddings_job_docker' + this.stage,
            description: 'Lambda function for creating documents chunks, embeddings and storing them in Amazon Opensearch',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: embeddings_job_function_role,
            environment: {
                OUTPUT_BUCKET: this.s3ProcessedAssetsBucketInterface.bucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                OPENSEARCH_INDEX: props.openSearchIndexName,
                OPENSEARCH_API_NAME: opensearch_helper.getOpenSearchApiName(props),
                OPENSEARCH_DOMAIN_ENDPOINT: opensearch_helper.getOpenSearchEndpoint(props),
                OPENSEARCH_SECRET_ID: SecretId,
            },
        };
        // Lambda function performing the embedding job
        const embeddings_job_function = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'lambda_function_embeddings_job' + this.stage, construct_embeddings_lambda_props, props.customEmbeddingsDockerLambdaProps);
        const lambdaFunctions = [embeddings_job_function, s3_transformer_job_function, validate_input_function];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        // Add GraphQl permissions to the IAM role for the Lambda function
        embeddings_job_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        // Step function definition
        const inputValidationTask = new stepfn_task.LambdaInvoke(this, 'Validate Ingestion Input', {
            lambdaFunction: validate_input_function,
            resultPath: '$.validation_result',
        });
        const fileTransformationTask = new stepfn_task.LambdaInvoke(this, 'Download and transform document to raw text', {
            lambdaFunction: s3_transformer_job_function,
            resultPath: '$.s3_transformer_result',
        });
        const embeddingsTask = new stepfn_task.LambdaInvoke(this, 'Generate embeddings from processed documents and store them', {
            lambdaFunction: embeddings_job_function,
            resultPath: '$',
        });
        const validate_input_choice = new stepfn.Choice(this, 'Is Valid Ingestion Parameters?', {
            outputPath: '$.validation_result.Payload.files',
        });
        const run_files_in_parallel = new stepfn.Map(this, 'Map State', {
            maxConcurrency: 100,
        }).itemProcessor(fileTransformationTask);
        const jobFailed = new stepfn.Fail(this, 'Job Failed', {
            cause: 'Validation job failed',
            error: 'DescribeJob returned FAILED',
        });
        const definition = inputValidationTask.next(validate_input_choice.when(stepfn.Condition.booleanEquals('$.validation_result.Payload.isValid', false), jobFailed).otherwise(run_files_in_parallel.next(embeddingsTask)));
        const maxLogGroupNameLength = 255;
        const logGroupPrefix = '/aws/vendedlogs/states/constructs/';
        const maxGeneratedNameLength = maxLogGroupNameLength - logGroupPrefix.length;
        const nameParts = [
            aws_cdk_lib_1.Stack.of(scope).stackName, // Name of the stack
            scope.node.id, // Construct ID
            'StateMachineLogRag', // Literal string for log group name portion
        ];
        const logGroupName = (0, utils_1.generatePhysicalName)(logGroupPrefix, nameParts, maxGeneratedNameLength);
        const ragLogGroup = new logs.LogGroup(this, 'ingestionStepFunctionLogGroup', {
            logGroupName: logGroupName,
        });
        const ingestion_step_function = new stepfn.StateMachine(this, 'IngestionStateMachine', {
            stateMachineName: 'IngestionStateMachine' + this.stage,
            definitionBody: stepfn.DefinitionBody.fromChainable(definition),
            timeout: aws_cdk_lib_1.Duration.minutes(30),
            logs: {
                destination: ragLogGroup,
                level: stepfn.LogLevel.ALL,
            },
            tracingEnabled: this.enablexray,
        });
        this.stateMachine = ingestion_step_function;
        this.ingestionBus.grantPutEventsTo(event_bridge_datasource.grantPrincipal);
        event_bridge_datasource.createResolver('ingestDocumentResolver', {
            fieldName: 'ingestDocuments',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                    {
                        "version": "2018-05-29",
                        "operation": "PutEvents",
                        "events": [{
                            "source": "ingestion",
                            "detail": $util.toJson($context.arguments),
                            "detailType": "genAIdemo"
                        }
                        ]
                    } 
                    `),
            responseMappingTemplate: appsync.MappingTemplate.fromString(`
                    #if($ctx.error)
                        $util.error($ctx.error.message, $ctx.error.type, $ctx.result)
                    #end
                        $util.toJson($ctx.result)
                    `),
        });
        const rule = new events.Rule(this, 'ingestionRule' + this.stage, {
            description: 'Rule to trigger ingestion function',
            eventBus: this.ingestionBus,
            eventPattern: {
                source: ['ingestion'],
            },
        });
        rule.addTarget(new targets.SfnStateMachine(this.stateMachine));
        this.embeddingsLambdaFunction = embeddings_job_function;
        this.fileTransformerLambdaFunction = s3_transformer_job_function;
        this.inputValidationLambdaFunction = validate_input_function;
    }
}
exports.RagAppsyncStepfnOpensearch = RagAppsyncStepfnOpensearch;
_a = JSII_RTTI_SYMBOL_1;
RagAppsyncStepfnOpensearch[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.RagAppsyncStepfnOpensearch", version: "0.1.94" };
//# sourceMappingURL=data:application/json;base64,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