"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SummarizationAppsyncStepfn = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const sqs = require("aws-cdk-lib/aws-sqs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const sfnTask = require("aws-cdk-lib/aws-stepfunctions-tasks");
const cdk_nag_1 = require("cdk-nag");
const base_class_1 = require("../../../common/base-class/base-class");
const construct_name_enum_1 = require("../../../common/base-class/construct-name-enum");
const eventBridge = require("../../../common/helpers/eventbridge-helper");
const lambda_builder_helper_1 = require("../../../common/helpers/lambda-builder-helper");
const redisHelper = require("../../../common/helpers/redis-helper");
const s3BucketHelper = require("../../../common/helpers/s3-bucket-helper");
const utils_1 = require("../../../common/helpers/utils");
const vpcHelper = require("../../../common/helpers/vpc-helper");
class SummarizationAppsyncStepfn extends base_class_1.BaseClass {
    /**
       * @summary Constructs a new instance of the SummarizationAppsyncStepfn class.
       * @param {Construct} scope - represents the scope for all the resources.
       * @param {string} id - this is a a scope-unique id.
       * @param {SummarizationAppsyncStepfnProps} props - user provided props for the construct.
       * @since 0.0.0
       * @access public
       */
    constructor(scope, id, props) {
        super(scope, id);
        const baseProps = {
            stage: props.stage,
            enableOperationalMetric: props.enableOperationalMetric,
            constructName: construct_name_enum_1.ConstructName.AWSSUMMARIZATIONAPPSYNCSTEPFN,
            constructId: id,
            observability: props.observability,
        };
        this.updateEnvSuffix(baseProps);
        this.addObservabilityToConstruct(baseProps);
        // vpc
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'summarizationConstructLogGroup');
        const vpcFlowLogrole = new iam.Role(this, 'summarizationConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, vpcFlowLogrole),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + this.stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            versioned: true,
            lifecycleRules: [{
                    expiration: aws_cdk_lib_1.Duration.days(90),
                }],
        });
        // bucket for input document
        s3BucketHelper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        if (props?.existingInputAssetsBucketObj) {
            this.inputAssetBucket = props.existingInputAssetsBucketObj;
        }
        else if (props?.bucketInputsAssetsProps) {
            this.inputAssetBucket = new s3.Bucket(this, 'inputAssetsSummaryBucket' + this.stage, props.bucketInputsAssetsProps);
        }
        else {
            const bucketName = 'input-assets-summary-bucket' + this.stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID;
            this.inputAssetBucket = new s3.Bucket(this, 'inputAssetsSummaryBucket' + this.stage, {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                encryption: s3.BucketEncryption.S3_MANAGED,
                bucketName: bucketName,
                serverAccessLogsBucket: serverAccessLogBucket,
                enforceSSL: true,
                versioned: true,
                lifecycleRules: [{
                        expiration: aws_cdk_lib_1.Duration.days(90),
                    }],
            });
        }
        // bucket for transformed document
        s3BucketHelper.CheckS3Props({
            existingBucketObj: props.existingProcessedAssetsBucketObj,
            bucketProps: props.bucketProcessedAssetsProps,
        });
        if (props?.existingProcessedAssetsBucketObj) {
            this.processedAssetBucket = props.existingProcessedAssetsBucketObj;
        }
        else if (props?.bucketProcessedAssetsProps) {
            this.processedAssetBucket = new s3.Bucket(this, 'processedAssetsSummaryBucket' + this.stage, props.bucketProcessedAssetsProps);
        }
        else {
            const bucketName = 'processed-assets-summary-bucket' + this.stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID;
            this.processedAssetBucket = new s3.Bucket(this, 'processedAssetsSummaryBucket' + this.stage, {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                encryption: s3.BucketEncryption.S3_MANAGED,
                bucketName: bucketName,
                serverAccessLogsBucket: serverAccessLogBucket,
                enforceSSL: true,
                versioned: true,
                lifecycleRules: [{
                        expiration: aws_cdk_lib_1.Duration.days(90),
                    }],
            });
        }
        // set up redis cluster
        redisHelper.CheckRedisClusterProps(props);
        // build redis cluster only when cfnCacheClusterProps is set
        if (props?.cfnCacheClusterProps) {
            const redisSecurityGroup = redisHelper.getRedisSecurityGroup(this, {
                existingVpc: this.vpc,
            });
            const redisProps = {
                existingVpc: this.vpc,
                cfnCacheClusterProps: props.cfnCacheClusterProps,
                subnetIds: vpcHelper.getPrivateSubnetIDs(this.vpc),
                inboundSecurityGroup: this.securityGroup,
                redisSecurityGroup: redisSecurityGroup,
                redisPort: 8686,
            };
            this.redisCluster = redisHelper.buildRedisCluster(this, redisProps);
            redisHelper.setInboundRules(redisSecurityGroup, this.securityGroup, redisProps.redisPort);
        }
        else {
            this.redisCluster = props?.existingRedisCulster;
        }
        const redisHost = this.redisCluster?.attrRedisEndpointAddress;
        const redisPort = this.redisCluster?.attrRedisEndpointPort;
        eventBridge.CheckEventBridgeProps(props);
        // Create event bridge
        this.eventBridgeBus = eventBridge.buildEventBus(this, {
            existingEventBusInterface: props.existingBusInterface,
            eventBusProps: props.eventBusProps,
        });
        // make it generic for other auth config as well
        const authorizationConfig = {
            defaultAuthorization: {
                authorizationType: appsync.AuthorizationType.USER_POOL,
                userPoolConfig: { userPool: props.cognitoUserPool },
            },
            additionalAuthorizationModes: [
                {
                    authorizationType: appsync.AuthorizationType.IAM,
                },
            ],
        };
        const apiName = props.summaryApiName || 'summaryApi';
        // graphql api for summary. client invoke this api with given schema and cognito user pool auth.
        const summarizationGraphqlApi = new appsync.GraphqlApi(this, 'summarizationGraphqlApi' + this.stage, {
            name: apiName + this.stage,
            logConfig: {
                fieldLogLevel: this.fieldLogLevel,
                retention: this.retention,
            },
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-summarization-appsync-stepfn/schema.graphql')),
            authorizationConfig: authorizationConfig,
            xrayEnabled: this.enablexray,
        });
        this.graphqlApi = summarizationGraphqlApi;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi ? summarizationGraphqlApi.graphqlUrl : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi ? summarizationGraphqlApi.apiId : props.existingMergedApi.attrApiId;
        // bucket
        const transformedAssetBucketName = this.processedAssetBucket.bucketName;
        const inputAssetBucketName = this.inputAssetBucket.bucketName;
        const isFileTransformationRequired = props?.isFileTransformationRequired || 'false';
        // role
        const inputvalidatorLambdaRole = new iam.Role(this, 'inputvalidatorLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                'appsync:GraphQL'],
            resources: ['arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + inputAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + transformedAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(inputvalidatorLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const construct_input_validation_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/input_validator')),
            functionName: 'summary_input_validator' + this.stage,
            description: 'Lambda function to validate input for summary api',
            vpc: this.vpc,
            tracing: this.lambdaTracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 1),
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            role: inputvalidatorLambdaRole,
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        };
        // Lambda function used to validate inputs in the step function
        const inputValidatorLambda = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'inputValidatorLambda' + this.stage, construct_input_validation_lambda_props, props.customInputValidationDockerLambdaProps);
        // role
        const documentReaderLambdaRole = new iam.Role(this, 'documentReaderLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:PutObject',
                'appsync:GraphQL'],
            resources: ['arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + inputAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + transformedAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(documentReaderLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used',
            },
        ], true);
        const construct_document_reader_lambda_props = {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/document_reader')),
            functionName: 'summary_document_reader' + this.stage,
            description: 'Lambda function to read the input transformed document',
            vpc: this.vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 1),
            tracing: this.lambdaTracing,
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            role: documentReaderLambdaRole,
            environment: {
                REDIS_HOST: redisHost,
                REDIS_PORT: redisPort,
                TRANSFORMED_ASSET_BUCKET: transformedAssetBucketName,
                INPUT_ASSET_BUCKET: inputAssetBucketName,
                IS_FILE_TRANSFORMED: isFileTransformationRequired,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        };
        // Lambda function used to read documents in the step function
        const documentReaderLambda = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'documentReaderLambda' + this.stage, construct_document_reader_lambda_props, props.customDocumentReaderDockerLambdaProps);
        const summaryChainType = props?.summaryChainType || 'stuff';
        // role
        const summaryGeneratorLambdaRole = new iam.Role(this, 'summaryGeneratorLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:PutObject',
                'appsync:GraphQL',
                'bedrock:InvokeModel',
                'bedrock:InvokeModelWithResponseStream'],
            resources: ['arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + inputAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':s3:::' + transformedAssetBucketName + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
                'arn:' + aws_cdk_lib_1.Aws.PARTITION + ':bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(summaryGeneratorLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const construct_generate_summary_lambda_props = {
            functionName: 'summary_generator' + this.stage,
            description: 'Lambda function to generate the summary',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/summary_generator')),
            vpc: this.vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(10),
            tracing: this.lambdaTracing,
            role: summaryGeneratorLambdaRole,
            environment: {
                REDIS_HOST: redisHost,
                REDIS_PORT: redisPort,
                ASSET_BUCKET_NAME: transformedAssetBucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                SUMMARY_LLM_CHAIN_TYPE: summaryChainType,
            },
        };
        // Lambda function used to generate the summary in the step function
        const generateSummarylambda = (0, lambda_builder_helper_1.buildDockerLambdaFunction)(this, 'generateSummarylambda' + this.stage, construct_generate_summary_lambda_props, props.customSummaryGeneratorDockerLambdaProps);
        this.inputAssetBucket?.grantRead(generateSummarylambda);
        this.processedAssetBucket?.grantReadWrite(generateSummarylambda);
        this.inputAssetBucket?.grantRead(documentReaderLambda);
        this.processedAssetBucket?.grantReadWrite(documentReaderLambda);
        const lambdaFunctions = [documentReaderLambda, generateSummarylambda, inputValidatorLambda];
        this.updateConstructUsageMetricCode(baseProps, scope, lambdaFunctions);
        // create datasource at appsync
        const SummaryStatusDataSource = new appsync.NoneDataSource(this, 'noneDataSource' + this.stage, {
            api: summarizationGraphqlApi,
            name: 'SummaryStatusDataSource',
        });
        SummaryStatusDataSource.createResolver('summaryResponseresolver' + this.stage, {
            typeName: 'Mutation',
            fieldName: 'updateSummaryJobStatus',
            requestMappingTemplate: appsync.MappingTemplate.fromString('{ "version": "2017-02-28", "payload": $util.toJson($context.args) }'),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        // step functions
        const inputValidationTask = new sfnTask.LambdaInvoke(this, 'Validate Input ', {
            lambdaFunction: inputValidatorLambda,
            resultPath: '$.validation_result',
        });
        const documentReaderTask = new sfnTask.LambdaInvoke(this, 'Read document and check summary in cache ', {
            lambdaFunction: documentReaderLambda,
            resultPath: '$.document_result',
        });
        const generateSummaryTask = new sfnTask.LambdaInvoke(this, 'Generate Summary with llm', {
            lambdaFunction: generateSummarylambda,
            resultPath: '$.summary_result',
        });
        const dlq = new sqs.Queue(this, 'dlq', {
            queueName: 'summarydlq' + this.stage,
            retentionPeriod: aws_cdk_lib_1.Duration.days(7),
            enforceSSL: true,
        });
        const jobFailed = new sfn.Fail(this, 'Failed', {
            comment: 'AWS summary Job failed',
        });
        const jobSuccess = new sfn.Succeed(this, 'succeeded', {
            comment: 'AWS summary Job succeeded',
        });
        // step function choice steps
        const validateInputChoice = new sfn.Choice(this, 'is Valid Parameters?', {
            outputPath: '$.validation_result.Payload.files',
        });
        const summaryfromCacheChoice = new sfn.Choice(this, 'is Summary in Cache?', {});
        const fileStatusForSummarization = new sfn.Choice(this, 'is file status ready for summarization?', {
            outputPath: '$.document_result.Payload',
        });
        // step function, run files in parallel
        const runFilesInparallel = new sfn.Map(this, 'Run Files in Parallel', {
            maxConcurrency: 100,
        }).itemProcessor(documentReaderTask.next(summaryfromCacheChoice
            .when(sfn.Condition.booleanEquals('$.document_result.Payload.is_summary_available', true), jobSuccess).otherwise(fileStatusForSummarization.when(sfn.Condition.stringMatches('$.document_result.Payload.status', 'Error'), jobSuccess).otherwise(generateSummaryTask.next(jobSuccess)))));
        const maxLogGroupNameLength = 255;
        const logGroupPrefix = '/aws/vendedlogs/states/constructs/';
        const maxGeneratedNameLength = maxLogGroupNameLength - logGroupPrefix.length;
        const nameParts = [
            aws_cdk_lib_1.Stack.of(scope).stackName, // Name of the stack
            scope.node.id, // Construct ID
            'StateMachineLogSummarization', // Literal string for log group name portion
        ];
        const logGroupName = (0, utils_1.generatePhysicalName)(logGroupPrefix, nameParts, maxGeneratedNameLength);
        const summarizationLogGroup = new logs.LogGroup(this, 'summarizationLogGroup', {
            logGroupName: logGroupName,
        });
        // step function definition
        const definition = inputValidationTask.next(validateInputChoice
            .when(sfn.Condition.booleanEquals('$.validation_result.Payload.isValid', false), jobFailed)
            .otherwise(runFilesInparallel));
        // step function
        const summarizationStepFunction = new sfn.StateMachine(this, 'summarizationStepFunction', {
            definitionBody: sfn.DefinitionBody.fromChainable(definition),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            logs: {
                destination: summarizationLogGroup,
                level: sfn.LogLevel.ALL,
            },
            tracingEnabled: this.enablexray,
        });
        this.stateMachine = summarizationStepFunction;
        // event bridge datasource for summarization api
        const eventBridgeDataSource = summarizationGraphqlApi.addEventBridgeDataSource('eventBridgeDataSource', this.eventBridgeBus);
        this.eventBridgeBus.grantPutEventsTo(eventBridgeDataSource.grantPrincipal);
        // add resolver on summary graphql schema
        eventBridgeDataSource.createResolver('generateSummary', {
            typeName: 'Mutation',
            fieldName: 'generateSummary',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
         {
          "version": "2018-05-29",
          "operation": "PutEvents",
          "events": [{
              "source": "summary",
              "detail": $util.toJson($context.arguments),
              "detailType": "genAIdemo"
          }
          ]
  } 
  `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('#if($ctx.error)$utilerror($ctx.error.message, $ctx.error.type, $ctx.result) #end $util.toJson($ctx.result)'),
        });
        new events.Rule(this, 'SummaryMutationRule', {
            description: 'Summary Mutation Rule',
            eventBus: this.eventBridgeBus,
            eventPattern: {
                source: ['summary'],
            },
            targets: [
                new targets.SfnStateMachine(summarizationStepFunction, {
                    deadLetterQueue: dlq,
                    retryAttempts: 1,
                }),
            ],
        });
        this.documentReaderLambdaFunction = documentReaderLambda;
        this.inputValidationLambdaFunction = inputValidatorLambda;
        this.summaryGeneratorLambdaFunction = generateSummarylambda;
    }
}
exports.SummarizationAppsyncStepfn = SummarizationAppsyncStepfn;
_a = JSII_RTTI_SYMBOL_1;
SummarizationAppsyncStepfn[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.SummarizationAppsyncStepfn", version: "0.1.94" };
//# sourceMappingURL=data:application/json;base64,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