##################################### TERMS OF USE ###########################################
# The following code is provided for demonstration purpose only, and should not be used      #
# without independent verification. Recorded Future makes no representations or warranties,  #
# express, implied, statutory, or otherwise, regarding any aspect of this code or of the     #
# information it may retrieve, and provides it both strictly “as-is” and without assuming    #
# responsibility for any information it may retrieve. Recorded Future shall not be liable    #
# for, and you assume all risk of using, the foregoing. By using this code, Customer         #
# represents that it is solely responsible for having all necessary licenses, permissions,   #
# rights, and/or consents to connect to third party APIs, and that it is solely responsible  #
# for having all necessary licenses, permissions, rights, and/or consents to any data        #
# accessed from any third party API.                                                         #
##############################################################################################

from functools import total_ordering
from typing import Annotated, Optional

from pydantic import AfterValidator, BeforeValidator, Field

from ..common_models import RFBaseModel
from ..helpers.helpers import Validators
from .models import DynamicInfo, Metadata, PEInfo, SampleInfo, StaticInfo


@total_ordering
class SandboxReport(RFBaseModel):
    """Validate data received from the `/v1/reports` endpoint.

    This class supports hashing, equality comparison, string representation, and total ordering
    of `SandboxReport` instances.

    Hashing:
        Returns a hash value based on the report `file`, `task`, `sample.completed`.

    Equality:
        Checks equality between two `SandboxReport` instances by `id_` when both are present;
        otherwise falls back to comparing (`file`, `task`, `sample.completed`).

    Greater-than Comparison:
        Defines a greater-than comparison between two `SandboxReport` instances based on the sample
        completed time, then created time, and finally the `file` name.

    String Representation:
        Returns a string representation of the `SandboxReport` instance including the `file`,
        `score`, `task`, and created timestamp.


        ```python
        >>> print(report)
        Sandbox Report of: sample.exe, Score: 85, Task: behavioural1, Submitted: 2024-05-21 10:42:30
        ```
    Ordering:
        The ordering of `SandboxReport` instances is determined primarily by the sample completed
        time. If two instances have the same completed time, the created time is used as a secondary
        criterion, followed by the report `file` name.
    """

    file: str
    id_: Optional[str] = None
    task: str
    dynamic: DynamicInfo
    metadata: Metadata
    pe: Optional[PEInfo] = None
    sample: SampleInfo
    static: StaticInfo

    def __hash__(self):
        return hash((self.file, self.task, self.sample.completed))

    def __eq__(self, other: 'SandboxReport'):
        return (self.file, self.task, self.sample.completed) == (
            other.file,
            other.task,
            other.sample.completed,
        )

    def __gt__(self, other: 'SandboxReport'):
        if self.sample.completed and other.sample.completed:
            return self.sample.completed > other.sample.completed

        if self.sample.created and other.sample.created:
            return self.sample.created > other.sample.created

        return self.file > other.file

    def __str__(self):
        return 'Sandbox Report of: {}, Score: {}, Task: {}, Submitted: {}'.format(
            self.file, self.sample.score or 'N/A', self.task, self.sample.created or 'N/A'
        )


def _split_time(value: str):
    return value.split('T')[0]


class MalwareReportIn(RFBaseModel):
    """Validate data sent to the `/v1/reports` endpoint."""

    query: str
    sha256: str
    start_date: Annotated[
        str, BeforeValidator(Validators.convert_relative_time), AfterValidator(_split_time)
    ]
    end_date: Annotated[
        str, BeforeValidator(Validators.convert_relative_time), AfterValidator(_split_time)
    ]
    my_enterprise: bool
    limit: int = Field(ge=1, le=10)
