##################################### TERMS OF USE ###########################################
# The following code is provided for demonstration purpose only, and should not be used      #
# without independent verification. Recorded Future makes no representations or warranties,  #
# express, implied, statutory, or otherwise, regarding any aspect of this code or of the     #
# information it may retrieve, and provides it both strictly “as-is” and without assuming    #
# responsibility for any information it may retrieve. Recorded Future shall not be liable    #
# for, and you assume all risk of using, the foregoing. By using this code, Customer         #
# represents that it is solely responsible for having all necessary licenses, permissions,   #
# rights, and/or consents to connect to third party APIs, and that it is solely responsible  #
# for having all necessary licenses, permissions, rights, and/or consents to any data        #
# accessed from any third party API.                                                         #
##############################################################################################

import logging
from typing import Annotated

from pydantic import validate_call
from typing_extensions import Doc

from ..constants import DEFAULT_LIMIT
from ..endpoints import EP_MALWARE_INTEL_REPORTS
from ..helpers import debug_call
from ..helpers.helpers import connection_exceptions
from ..rf_client import RFClient
from .errors import MalwareIntelReportError
from .malware_intel import MalwareReportIn, SandboxReport


class MalwareIntelMgr:
    """Manages requests for Recorded Future Malware Intelligence API."""

    def __init__(self, rf_token: str = None):
        """Initializes the `MalwareIntelMgr` object.

        Args:
            rf_token (str, optional): Recorded Future API token.
        """
        self.log = logging.getLogger(__name__)
        self.rf_client = RFClient(api_token=rf_token) if rf_token else RFClient()

    @debug_call
    @validate_call
    @connection_exceptions(ignore_status_code=[], exception_to_raise=MalwareIntelReportError)
    def reports(
        self,
        query: Annotated[str, Doc('The filtering query to perform.')],
        sha256: Annotated[str, Doc('The SHA256 to search.')],
        start_date: Annotated[
            str, Doc('The starting date, format YYYY-MM-DD or relative like -1d.')
        ],
        end_date: Annotated[str, Doc('The ending date, format YYYY-MM-DD or relative like -1d.')],
        my_enterprise: Annotated[
            bool, Doc('If the report has been submitted by your enterprise.')
        ] = False,
        limit: Annotated[int, Doc('Maximum number of reports.')] = DEFAULT_LIMIT,
    ) -> Annotated[list[SandboxReport], Doc('A list of reports that have been found.')]:
        """Search for the reports matching the specified parameters.

        Endpoint:
            `/malware-intelligence/v1/reports`

        Raises:
            ValidationError: If any supplied parameter is of incorrect type.
            MalwareIntelReportError: If API error occurs.
        """
        data = {
            'query': query,
            'sha256': sha256,
            'start_date': start_date,
            'end_date': end_date,
            'my_enterprise': my_enterprise,
            'limit': limit,
        }

        data = MalwareReportIn.model_validate(data).json()

        reports = self.rf_client.request('post', EP_MALWARE_INTEL_REPORTS, data=data).json()[
            'reports'
        ]

        return [SandboxReport.model_validate(r) for r in reports]
