##################################### TERMS OF USE ###########################################
# The following code is provided for demonstration purpose only, and should not be used      #
# without independent verification. Recorded Future makes no representations or warranties,  #
# express, implied, statutory, or otherwise, regarding any aspect of this code or of the     #
# information it may retrieve, and provides it both strictly “as-is” and without assuming    #
# responsibility for any information it may retrieve. Recorded Future shall not be liable    #
# for, and you assume all risk of using, the foregoing. By using this code, Customer         #
# represents that it is solely responsible for having all necessary licenses, permissions,   #
# rights, and/or consents to connect to third party APIs, and that it is solely responsible  #
# for having all necessary licenses, permissions, rights, and/or consents to any data        #
# accessed from any third party API.                                                         #
##############################################################################################

import itertools
from typing import TYPE_CHECKING, Union

from markdown_strings import bold, unordered_list

from ...analyst_notes import AnalystNote
from ...analyst_notes.markdown import _cleanup_insikt_note_text
from ...constants import TIMESTAMP_STR
from ...enrich.lookup import CVSS, CVSSV3, EnrichmentData
from ...markdown import MarkdownMaker
from ...markdown.markdown import divider, header, html_collapsible

if TYPE_CHECKING:
    from ...playbook_alerts.playbook_alerts import PBA_CyberVulnerability

DEFAULT_AI_INSIGHTS_MSG = 'Recorded Future AI Insights are unavailable.'


def _add_ai_insights(
    pba: 'PBA_CyberVulnerability', md_maker: MarkdownMaker, extra_content: list[EnrichmentData]
) -> None:
    result = []
    for entity in extra_content:
        if (
            entity.content.entity.name == pba.panel_status.entity_name
            and entity.content.ai_insights
        ):
            ai_insights = (
                entity.content.ai_insights.text
                or entity.content.ai_insights.comment
                or DEFAULT_AI_INSIGHTS_MSG
            )
            result.append(f'{ai_insights}')
            md_maker.add_section('Recorded Future AI Insights', result)


def _format_cvss(cvss: Union[CVSS, CVSSV3], title: str, html_tags: bool) -> str:
    """Extract data from enirched content. Parse timestamps and cvss score."""
    content = []
    if not cvss:
        return ''

    for key, val in cvss:
        dt_val = ''
        if key in ('modified', 'created', 'published', 'last_modified') and val:
            dt_val = val.strftime(TIMESTAMP_STR)

        if val is not None:
            k = key.replace('_', ' ') + ':'
            content.append(f'* {bold(k.title())} {dt_val or val}\n')

    if not content:
        return ''

    content = '\n\n' + ''.join(sorted(content)) + '\n'
    if html_tags:
        return f'\n{html_collapsible(title, content)}'
    return f'{bold(title)}\n{content}'


def _add_vulnerability_overview(
    pba: 'PBA_CyberVulnerability',
    md_maker: MarkdownMaker,
    html_tags: bool,
    extra_content: list[EnrichmentData],
) -> None:
    result = []

    risk_score = ''
    enriched_cve = None
    for entity in extra_content:
        if entity.content.entity.name == pba.panel_status.entity_name:
            enriched_cve = entity
            risk_score = entity.content.risk.score or 0

    result.append(f'{bold("Name:")} {pba.panel_status.entity_name}  ')
    if risk_score:
        result.append(f'{bold("Risk Score:")} {risk_score}  ')

    targets = (
        [target.name for target in pba.panel_status.targets] if pba.panel_status.targets else []
    )
    matches = f'Matches in Watch List ({len(targets)}):'
    matches = f'{bold(matches)} {", ".join(targets)}  '
    result.append(matches)

    result.append(f'{bold("Criticality:")} {pba.panel_status.entity_criticality}  ')
    result.append(f'{bold("Lifecycle:")} {pba.lifecycle_stage}  ')

    if enriched_cve:
        result.append(_format_cvss(enriched_cve.content.cvss, 'CVSS v2', html_tags))
        result.append(_format_cvss(enriched_cve.content.cvssv3, 'CVSS v3', html_tags))

    md_maker.add_section('Vulnerability Overview', result)


def _add_assessments(pba: 'PBA_CyberVulnerability', md_maker: MarkdownMaker) -> None:
    result = []
    if pba.panel_evidence_summary:
        risk_rules = ''.join(
            f'{header(risk_rule.rule, 5)}\n{risk_rule.description}\n\n'
            for risk_rule in pba.panel_evidence_summary.summary.risk_rules
        )
        result.append(risk_rules)
        if risk_rules:
            md_maker.add_section('Assessments', result)


def _add_insikt_notes(
    pba: 'PBA_CyberVulnerability', md_maker: MarkdownMaker, extra_content: list[AnalystNote]
) -> None:
    result = []
    for insikt_note in pba.panel_evidence_summary.insikt_notes:
        topic = (
            insikt_note.topic
            if isinstance(insikt_note.topic, str)
            else ', '.join(insikt_note.topic)
        )

        note_title = f'{bold("Title:")} {insikt_note.title}  '
        published_date = f'{bold("Published:")} {insikt_note.published.strftime(TIMESTAMP_STR)}  '
        topic = f'{bold("Topic:")} {topic}  '
        note_text = f'{_cleanup_insikt_note_text(insikt_note.fragment)}  '
        validation_urls = ''

        for entity in extra_content:
            if entity.id_ == insikt_note.id_:
                validation_urls = unordered_list(
                    [url.name for url in entity.attributes.validation_urls]
                )
                if validation_urls:
                    validation_urls = f'{bold("Validation URLs:")}\n{validation_urls}\n'
                note_text = f'{bold("Text:")} {_cleanup_insikt_note_text(entity.attributes.text)}\n'

        note_md = [note_title, published_date, topic, validation_urls, note_text, divider()]
        result.extend(note_md)

    if result:
        md_maker.add_section('Insikt Notes', result)


def _add_affected_products(pba: 'PBA_CyberVulnerability', md_maker: MarkdownMaker) -> None:
    if not pba.panel_evidence_summary.affected_products:
        return

    rows = []
    affected_products = {
        str(vendor): list(products)
        for vendor, products in itertools.groupby(
            sorted(product.name for product in pba.panel_evidence_summary.affected_products),
            key=lambda x: x.split()[0],
        )
    }

    affected_products = dict(sorted(affected_products.items(), key=lambda item: item[0]))

    for vendor in affected_products:
        rows.append(f'{header(vendor, 5)}\n')
        products = [f'* {p}' for p in sorted(affected_products[vendor])]

        rows.extend(products)

    total_products = sum(len(products) for products in affected_products.values())

    section_title = f'Affected Products ({total_products})'

    md_maker.add_section(section_title, rows)


def _cyber_vulnerability_markdown(
    pba: 'PBA_CyberVulnerability',
    md_maker: MarkdownMaker,
    html_tags: bool,
    extra_content: Union[list[EnrichmentData], list[AnalystNote]],
) -> str:
    if extra_content is None:
        extra_content = []
    analyst_notes = [_ for _ in extra_content if isinstance(_, AnalystNote)]
    enrichment = [_ for _ in extra_content if isinstance(_, EnrichmentData)]

    _add_ai_insights(pba, md_maker, enrichment)
    _add_vulnerability_overview(pba, md_maker, html_tags, enrichment)
    _add_assessments(pba, md_maker)
    _add_insikt_notes(pba, md_maker, analyst_notes)
    _add_affected_products(pba, md_maker)

    return md_maker.format_output()
