#!/usr/bin/env python3
"""
Markdown Formatter Demo

This example demonstrates the markdown display type with:
- Frontmatter support
- Various markdown elements
- Custom element creation
- Builder pattern usage

Run:
    uv run python examples/markdown_demo.py
"""

from dataclasses import dataclass
from replkit2 import App
from replkit2.textkit import markdown, MarkdownElement


# Custom markdown element
class Note(MarkdownElement):
    """Custom note element with colored background (in markdown)."""

    element_type = "note"

    def __init__(self, content: str, color: str = "blue"):
        self.content = content
        self.color = color

    @classmethod
    def from_dict(cls, data: dict) -> "Note":
        return cls(content=data.get("content", ""), color=data.get("color", "blue"))

    def render(self) -> str:
        return f"> **Note ({self.color}):** {self.content}"


@dataclass
class DemoState:
    """Simple state for the demo."""

    documents: list[dict] = None

    def __post_init__(self):
        if self.documents is None:
            self.documents = []


app = App("markdown_demo", DemoState)


@app.command(display="markdown")
def welcome(state):
    """Show welcome message with various markdown elements."""
    return {
        "frontmatter": {"title": "Welcome to ReplKit2 Markdown", "version": "1.0", "author": "ReplKit2 Team"},
        "elements": [
            {"type": "heading", "content": "Getting Started", "level": 1},
            {"type": "text", "content": "ReplKit2 now supports markdown display with a rich set of elements."},
            {"type": "heading", "content": "Features", "level": 2},
            {
                "type": "list",
                "items": [
                    "YAML frontmatter for metadata",
                    "Standard markdown elements",
                    "Custom element support",
                    "Builder pattern for easy construction",
                ],
            },
            {"type": "heading", "content": "Code Example", "level": 2},
            {
                "type": "code_block",
                "content": '@app.command(display="markdown")\ndef my_command(state):\n    return {"elements": [...]}',
                "language": "python",
            },
            {
                "type": "blockquote",
                "content": "The markdown formatter preserves the data-first philosophy of ReplKit2.",
            },
            {"type": "note", "content": "This is a custom element!", "color": "green"},
            {"type": "raw", "content": "\n---\n\n*Generated by ReplKit2*"},
        ],
    }


@app.command(display="markdown")
def guide(state):
    """Show guide using the builder pattern."""
    return (
        markdown()
        .frontmatter(title="ReplKit2 Markdown Guide", date="2025-01-28", tags=["documentation", "markdown"])
        .heading("Using the Builder")
        .text("The markdown builder provides a fluent interface for creating documents.")
        .heading("Basic Elements", level=2)
        .text("You can add various elements in sequence:")
        .list(
            [
                "Text paragraphs",
                "Headings with levels 1-6",
                "Code blocks with syntax highlighting",
                "Blockquotes for emphasis",
                "Lists (ordered and unordered)",
            ]
        )
        .heading("Advanced Usage", level=2)
        .code_block(
            """from replkit2.textkit import markdown

doc = (markdown()
    .frontmatter(title="My Doc")
    .heading("Chapter 1")
    .text("Content goes here...")
    .build())""",
            language="python",
        )
        .blockquote("Remember: The builder returns a dict with 'elements' and 'frontmatter' fields.")
        .heading("Custom Elements", level=2)
        .text("You can also use custom elements with the generic element() method:")
        .element("note", content="Custom elements auto-register!", color="yellow")
        .build()
    )


@app.command(display="markdown")
def api_docs(state):
    """Generate API documentation in markdown."""
    return {
        "frontmatter": {"title": "API Reference", "type": "reference", "auto_generated": True},
        "elements": [
            {"type": "heading", "content": "MarkdownElement API"},
            {"type": "text", "content": "Base class for all markdown elements."},
            {"type": "heading", "content": "Required Methods", "level": 2},
            {
                "type": "code_block",
                "content": """class MyElement(MarkdownElement):
    element_type = "my_element"  # Required
    
    @classmethod
    def from_dict(cls, data: dict) -> "MyElement":
        # Create instance from dict
        pass
    
    def render(self) -> str:
        # Return markdown string
        pass""",
                "language": "python",
            },
            {"type": "heading", "content": "Auto-Registration", "level": 2},
            {
                "type": "text",
                "content": "Elements automatically register when the class is defined. No manual registration needed!",
            },
            {
                "type": "list",
                "items": [
                    "Define element_type class variable",
                    "Implement from_dict() class method",
                    "Implement render() instance method",
                ],
                "ordered": True,
            },
            {"type": "note", "content": "See the Note class in this file for a complete example.", "color": "blue"},
        ],
    }


@app.command()
def add_doc(state, title: str, content: str):
    """Add a document to track."""
    doc = {"title": title, "content": content, "id": len(state.documents) + 1}
    state.documents.append(doc)
    return f"Added document #{doc['id']}: {title}"


@app.command(display="table", headers=["ID", "Title", "Length"])
def list_docs(state):
    """List all tracked documents."""
    return [{"ID": doc["id"], "Title": doc["title"], "Length": len(doc["content"])} for doc in state.documents]


if __name__ == "__main__":
    app.run(title="Markdown Formatter Demo")
