# PyDistLinalg/decompositions.py

import dask.array as da
from .core import _ensure_dask_array # Используем нашу внутреннюю функцию

def cholesky(A, chunks='auto'):
    """Разложение Холецкого для положительно определенной симметричной матрицы."""
    A_da = _ensure_dask_array(A, chunks)
    # dask.array.linalg.cholesky требует, чтобы A была симметричной и положительно определенной
    return da.linalg.cholesky(A_da)

def qr(A, chunks='auto'):
    """QR-разложение."""
    A_da = _ensure_dask_array(A, chunks)
    Q, R = da.linalg.qr(A_da)
    return Q, R

def svd(A, chunks='auto'):
    """Сингулярное разложение (SVD)."""
    A_da = _ensure_dask_array(A, chunks)
    U, s, V = da.linalg.svd(A_da)
    return U, s, V

def eigh(A, chunks='auto'):
    """Собственные значения и векторы для симметричной или эрмитовой матрицы."""
    A_da = _ensure_dask_array(A, chunks)
    # Возвращает собственные значения и собственные векторы (w, v)
    w, v = da.linalg.eigh(A_da)
    return w, v

# LU-разложение:
# dask.array.linalg не имеет прямой высокоуровневой функции для LU-разложения,
# так как оно сложнее для распределенной реализации общего случая без
# специальной блокировки и обмена данными.
# Для LU часто используются специализированные библиотеки или подходы.
# В Dask можно было бы реализовать LU, используя блокировку и вызовы numpy.linalg.lu
# на чанках, но это была бы более сложная и менее производительная реализация,
# чем специализированные библиотеки (например, ScaLAPACK).
# Для данного модуля, мы пока оставим его как концептуальную возможность.
def lu(A, chunks='auto'):
    """
    LU-разложение.
    Примечание: Прямая эффективная реализация LU-разложения для произвольных
    распределенных матриц в Dask является сложной и часто требует специализированных
    библиотек или ручной оркестровки, которая выходит за рамки простой
    обертки dask.array.linalg.
    """
    raise NotImplementedError("LU-разложение для распределенных массивов пока не реализовано в PyDistLinalg "
                              "из-за его сложности в распределенной среде.")