from __future__ import annotations

import re
import shutil
import subprocess
import tempfile
from dataclasses import dataclass
from pathlib import Path
from typing import Any


@dataclass
class PCIDevice:
    vendor: str
    """
    Vendor ID of the PCI device.
    """
    path: str
    """
    Path to the PCI device in sysfs.
    """
    address: str
    """
    Address of the PCI device.
    """
    class_: bytes
    """
    Class of the PCI device.
    """
    config: bytes
    """
    Device ID of the PCI device.
    """


def get_pci_devices(
    address: list[str] | str | None = None,
    vendor: list[str] | str | None = None,
) -> list[PCIDevice]:
    """
    Get PCI devices.

    Args:
        address: List of PCI addresses or a single address to filter by.
        vendor: List of vendor IDs or a single vendor ID to filter by.

    Returns:
        List of PCIDevice objects.

    """
    pci_devices = []
    sysfs_pci_path = Path("/sys/bus/pci/devices")
    if not sysfs_pci_path.exists():
        return pci_devices

    if address and isinstance(address, str):
        address = [address]
    if vendor and isinstance(vendor, str):
        vendor = [vendor]

    for dev_path in sysfs_pci_path.iterdir():
        dev_address = dev_path.name
        if address and dev_address not in address:
            continue

        dev_vendor_file = dev_path / "vendor"
        if not dev_vendor_file.exists():
            continue
        with dev_vendor_file.open("r") as vf:
            dev_vendor = vf.read().strip()
            if vendor and dev_vendor not in vendor:
                continue

        dev_class_file = dev_path / "class"
        dev_config_file = dev_path / "config"
        if not dev_class_file.exists() or not dev_config_file.exists():
            continue

        with dev_class_file.open("rb") as f:
            dev_class = f.read().strip()
        with dev_config_file.open("rb") as f:
            dev_config = f.read().strip()

        pci_devices.append(
            PCIDevice(
                vendor=dev_vendor,
                path=str(dev_path),
                address=dev_address,
                class_=dev_class,
                config=dev_config,
            ),
        )

    return pci_devices


@dataclass
class DeviceFile:
    path: str
    """
    Path to the device file.
    """
    number: int | None = None
    """
    Number of the device file.
    """


def get_device_files(pattern: str, directory: Path | str = "/dev") -> list[DeviceFile]:
    r"""
    Get device files with the given pattern.

    Args:
        pattern:
            Pattern of the device files to search for.
            Pattern must include a regex group for the number,
            e.g nvidia(?P<number>\d+).
        directory:
            Directory to search for device files,
            e.g /dev.

    Returns:
        List of DeviceFile objects.

    """
    if "(?P<number>" not in pattern:
        msg = "Pattern must include a regex group for the number, e.g nvidia(?P<number>\\d+)."
        raise ValueError(msg)

    if isinstance(directory, str):
        directory = Path(directory)

    device_files = []
    if not directory.exists():
        return device_files

    regex = re.compile(f"^{directory!s}/{pattern}$")
    for file_path in directory.iterdir():
        matched = regex.match(str(file_path))
        if not matched:
            continue
        file_number = matched.group("number")
        try:
            file_number = int(file_number)
        except ValueError:
            file_number = None
        device_files.append(
            DeviceFile(
                path=str(file_path),
                number=file_number,
            ),
        )

    # Sort by number in ascending order, None values at the end
    return sorted(
        device_files,
        key=lambda df: (df.number is None, df.number),
    )


def support_command(command: str) -> bool:
    """
    Determine whether a command is available.

    Args:
        command:
            The name of the command to check.

    Returns:
        True if the command is available, False otherwise.

    """
    return shutil.which(command) is not None


def execute_shell_command(command: str, cwd: str | None = None) -> str | None:
    """
    Execute a shell command and return its output.

    Args:
        command:
            The command to run.
        cwd:
            The working directory to run the command in, or None to use a temporary directory.

    Returns:
        The output of the command.

    Raises:
        If the command fails or returns a non-zero exit code.

    """
    if cwd is None:
        cwd = tempfile.gettempdir()

    command = command.strip()
    if not command:
        msg = "Command is empty"
        raise ValueError(msg)

    try:
        result = subprocess.run(  # noqa: S602
            command,
            capture_output=True,
            check=False,
            shell=True,
            text=True,
            cwd=cwd,
            encoding="utf-8",
        )
    except Exception as e:
        msg = f"Failed to run command '{command}'"
        raise RuntimeError(msg) from e
    else:
        if result.returncode != 0:
            msg = f"Unexpected result: {result}"
            raise RuntimeError(msg)

        return result.stdout


def execute_command(command: list[str], cwd: str | None = None) -> str | None:
    """
    Execute a command and return its output.

    Args:
        command:
            The command to run.
        cwd:
            The working directory to run the command in, or None to use a temporary directory.

    Returns:
        The output of the command.

    Raises:
        If the command fails or returns a non-zero exit code.

    """
    if cwd is None:
        cwd = tempfile.gettempdir()

    if not command:
        msg = "Command list is empty"
        raise ValueError(msg)

    try:
        result = subprocess.run(  # noqa: S603
            command,
            capture_output=True,
            check=False,
            text=True,
            cwd=cwd,
            encoding="utf-8",
        )
    except Exception as e:
        msg = f"Failed to run command '{command}'"
        raise RuntimeError(msg) from e
    else:
        if result.returncode != 0:
            msg = f"Unexpected result: {result}"
            raise RuntimeError(msg)

        return result.stdout


def safe_int(value: Any, default: int = 0) -> int:
    """
    Safely convert a value to int.

    Args:
        value:
            The value to convert.
        default:
            The default value to return if conversion fails.

    Returns:
        The converted int value, or 0 if conversion fails.

    """
    if value is None:
        return default
    if isinstance(value, int):
        return value
    try:
        return int(value)
    except (ValueError, TypeError):
        return default


def safe_float(value: Any, default: float = 0.0) -> float:
    """
    Safely convert a value to float.

    Args:
        value:
            The value to convert.
        default:
            The default value to return if conversion fails.

    Returns:
        The converted float value, or 0.0 if conversion fails.

    """
    if value is None:
        return default
    if isinstance(value, float):
        return value
    try:
        return float(value)
    except (ValueError, TypeError):
        return default


def safe_bool(value: Any, default: bool = False) -> bool:
    """
    Safely convert a value to bool.

    Args:
        value:
            The value to convert.
        default:
            The default value to return if conversion fails.

    Returns:
        The converted bool value, or False if conversion fails.

    """
    if value is None:
        return default
    if isinstance(value, bool):
        return value
    if isinstance(value, str):
        value_lower = value.strip().lower()
        if value_lower in ("true", "1", "yes", "on"):
            return True
        if value_lower in ("false", "0", "no", "off"):
            return False
    try:
        return bool(value)
    except (ValueError, TypeError):
        return default


def safe_str(value: Any, default: str = "") -> str:
    """
    Safely convert a value to str.

    Args:
        value:
            The value to convert.
        default:
            The default value to return if conversion fails.

    Returns:
        The converted str value, or an empty string if conversion fails.

    """
    if value is None:
        return default
    if isinstance(value, str):
        return value
    if isinstance(value, bytes):
        try:
            return value.decode("utf-8", errors="ignore")
        except (ValueError, TypeError):
            return default
    try:
        return str(value)
    except (ValueError, TypeError):
        return default
