"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IngressV1Beta1Backend = exports.IngressV1Beta1 = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
/**
 * Ingress is a collection of rules that allow inbound connections to reach the endpoints defined by a backend.
 *
 * An Ingress can be configured to give services
 * externally-reachable urls, load balance traffic, terminate SSL, offer name
 * based virtual hosting etc.
 *
 * @stability stable
 */
class IngressV1Beta1 extends base_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _c;
        super(scope, id);
        this._rulesPerHost = {};
        this.apiObject = new k8s.KubeIngressV1Beta1(this, 'Resource', {
            metadata: props.metadata,
            spec: {
                backend: cdk8s_1.Lazy.any({ produce: () => { var _c; return (_c = this._defaultBackend) === null || _c === void 0 ? void 0 : _c._toKube(); } }),
                rules: cdk8s_1.Lazy.any({ produce: () => this.synthRules() }),
            },
        });
        if (props.defaultBackend) {
            this.addDefaultBackend(props.defaultBackend);
        }
        this.addRules(...(_c = props.rules) !== null && _c !== void 0 ? _c : []);
    }
    /**
     * (deprecated) Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        if (!this._defaultBackend && Object.keys(this._rulesPerHost).length === 0) {
            return ['ingress with no rules or default backend'];
        }
        return [];
    }
    /**
     * Defines the default backend for this ingress.
     *
     * A default backend capable of
     * servicing requests that don't match any rule.
     *
     * @param backend The backend to use for requests that do not match any rule.
     * @stability stable
     */
    addDefaultBackend(backend) {
        this.addRules({ backend });
    }
    /**
     * Specify a default backend for a specific host name.
     *
     * This backend will be used as a catch-all for requests
     * targeted to this host name (the `Host` header matches this value).
     *
     * @param host The host name to match.
     * @param backend The backend to route to.
     * @stability stable
     */
    addHostDefaultBackend(host, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend });
    }
    /**
     * Adds an ingress rule applied to requests to a specific host and a specific HTTP path (the `Host` header matches this value).
     *
     * @param host The host name.
     * @param path The HTTP path.
     * @param backend The backend to route requests to.
     * @stability stable
     */
    addHostRule(host, path, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend, path });
    }
    /**
     * Adds an ingress rule applied to requests sent to a specific HTTP path.
     *
     * @param path The HTTP path.
     * @param backend The backend to route requests to.
     * @stability stable
     */
    addRule(path, backend) {
        this.addRules({ backend, path });
    }
    /**
     * Adds rules to this ingress.
     *
     * @param rules The rules to add.
     * @stability stable
     */
    addRules(...rules) {
        var _c, _d;
        for (const rule of rules) {
            // default backend is not really a rule
            if (!rule.host && !rule.path) {
                if (this._defaultBackend) {
                    throw new Error('a default backend is already defined for this ingress');
                }
                this._defaultBackend = rule.backend;
                continue;
            }
            const host = (_c = rule.host) !== null && _c !== void 0 ? _c : '';
            const backend = rule.backend;
            const path = rule.path;
            if (path && !path.startsWith('/')) {
                throw new Error(`ingress paths must begin with a "/": ${path}`);
            }
            const routes = this._rulesPerHost[host] = (_d = this._rulesPerHost[host]) !== null && _d !== void 0 ? _d : [];
            // check if we already have a rule for this host/path
            if (routes.find(r => r.path === path)) {
                throw new Error(`there is already an ingress rule for ${host}${path}`);
            }
            routes.push({ backend: backend._toKube(), path });
        }
    }
    synthRules() {
        const rules = new Array();
        for (const [host, paths] of Object.entries(this._rulesPerHost)) {
            rules.push({
                host: host ? host : undefined,
                http: { paths: paths.sort(sortByPath) },
            });
        }
        return rules.length > 0 ? rules : undefined;
    }
}
exports.IngressV1Beta1 = IngressV1Beta1;
_a = JSII_RTTI_SYMBOL_1;
IngressV1Beta1[_a] = { fqn: "cdk8s-plus-17.IngressV1Beta1", version: "1.0.0-beta.27" };
/**
 * The backend for an ingress path.
 *
 * @stability stable
 */
class IngressV1Beta1Backend {
    constructor(backend) {
        this.backend = backend;
    }
    /**
     * A Kubernetes `Service` to use as the backend for this path.
     *
     * @param service The service object.
     * @stability stable
     */
    static fromService(service, options = {}) {
        if (service.ports.length === 0) {
            throw new Error('service does not expose any ports');
        }
        let servicePort;
        if (service.ports.length === 1) {
            servicePort = service.ports[0].port;
        }
        else {
            if (options.port !== undefined) {
                const found = service.ports.find(p => p.port === options.port);
                if (found) {
                    servicePort = found.port;
                }
                else {
                    throw new Error(`service exposes ports ${service.ports.map(p => p.port).join(',')} but backend is defined to use port ${options.port}`);
                }
            }
            else {
                throw new Error(`unable to determine service port since service exposes multiple ports: ${service.ports.map(x => x.port).join(',')}`);
            }
        }
        if (options.port !== undefined && servicePort !== options.port) {
            throw new Error(`backend defines port ${options.port} but service exposes port ${servicePort}`);
        }
        return new IngressV1Beta1Backend({
            serviceName: service.name,
            servicePort,
        });
    }
    /**
     * @internal
     */
    _toKube() { return this.backend; }
}
exports.IngressV1Beta1Backend = IngressV1Beta1Backend;
_b = JSII_RTTI_SYMBOL_1;
IngressV1Beta1Backend[_b] = { fqn: "cdk8s-plus-17.IngressV1Beta1Backend", version: "1.0.0-beta.27" };
function sortByPath(lhs, rhs) {
    var _c, _d;
    const p1 = (_c = lhs.path) !== null && _c !== void 0 ? _c : '';
    const p2 = (_d = rhs.path) !== null && _d !== void 0 ? _d : '';
    return p1.localeCompare(p2);
}
//# sourceMappingURL=data:application/json;base64,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