# MAOS Backend API Examples\n\nThis directory contains comprehensive examples demonstrating how to use the Multi-Agent Orchestration System (MAOS) as a backend API for building scalable, distributed applications.\n\n## Overview\n\nThe MAOS system provides a powerful orchestration platform that enables:\n\n- **Distributed Task Execution**: Break down complex tasks into manageable subtasks\n- **Multi-Agent Coordination**: Deploy specialized agents for different task types\n- **Resource Management**: Dynamic allocation and optimization of computational resources\n- **Real-time Monitoring**: System health, performance metrics, and task progress tracking\n- **Claude Code Integration**: Seamless integration with Claude Code for AI-assisted development\n\n## Quick Start\n\n### 1. Installation\n\n```bash\n# Install dependencies\npip install -r requirements.txt\n\n# For development\npip install -r requirements-dev.txt\n```\n\n### 2. Running the API Server\n\n```bash\n# Start the MAOS backend API server\npython examples/backend_api_example.py\n\n# Or with custom settings\npython examples/backend_api_example.py --host 0.0.0.0 --port 8000 --debug\n```\n\n### 3. Testing the API\n\n```bash\n# Run client examples\npython examples/client_examples.py\n\n# Run specific example types\npython examples/client_examples.py --example tasks\npython examples/client_examples.py --example agents\npython examples/client_examples.py --example claude\n```\n\n## Examples Structure\n\n### Backend API Server (`backend_api_example.py`)\n\nComplete FastAPI-based backend server demonstrating:\n\n- **System Initialization**: Orchestrator setup with all components\n- **RESTful API Endpoints**: Full CRUD operations for tasks, agents, and resources\n- **Health Monitoring**: Basic and detailed health checks\n- **Error Handling**: Comprehensive error handling and logging\n- **Claude Integration**: Endpoints for Claude Code integration\n- **Production Patterns**: Graceful shutdown, signal handling, background cleanup\n\n**Key Features:**\n- Auto-scaling agent management\n- Resource allocation strategies\n- Task decomposition patterns\n- Real-time metrics and monitoring\n\n### Client Examples (`client_examples.py`)\n\nComprehensive client library and examples showing:\n\n#### Task Orchestration Patterns\n- **Simple Task Execution**: Single task submission and monitoring\n- **Parallel Workflows**: Multiple concurrent task execution\n- **Hierarchical Decomposition**: Complex task breakdown strategies\n- **Resource-Intensive Processing**: High-performance computing patterns\n\n#### Agent Management\n- **Specialized Agent Creation**: Task-specific agent deployment\n- **Health Monitoring**: System and component health tracking\n- **Performance Metrics**: Real-time performance monitoring\n\n#### Claude Code Integration\n- **Task Submission**: Submit tasks from Claude Code\n- **Status Monitoring**: Track Claude task execution\n- **Result Retrieval**: Get results and handle callbacks\n\n## Docker Deployment\n\n### Production Deployment\n\n```bash\n# Build and run with docker-compose\ncd examples/docker\ndocker-compose up -d\n\n# View logs\ndocker-compose logs -f maos-api\n\n# Scale services\ndocker-compose up -d --scale maos-api=3\n```\n\n### Development Environment\n\n```bash\n# Run development stack\ndocker-compose -f docker-compose.yml -f docker-compose.dev.yml up\n\n# Run tests in container\ndocker-compose --profile testing up test-runner\n\n# Code quality checks\ndocker-compose --profile quality up linting\n```\n\n## API Endpoints\n\n### Core Endpoints\n\n| Method | Endpoint | Description |\n|--------|----------|-------------|\n| GET | `/health` | Basic health check |\n| GET | `/health/detailed` | Detailed system health |\n| POST | `/api/v1/tasks` | Submit new task |\n| GET | `/api/v1/tasks/{id}` | Get task details |\n| DELETE | `/api/v1/tasks/{id}` | Cancel task |\n| POST | `/api/v1/agents` | Create agent |\n| GET | `/api/v1/system/status` | System status |\n| GET | `/api/v1/system/metrics` | System metrics |\n\n### Claude Integration Endpoints\n\n| Method | Endpoint | Description |\n|--------|----------|-------------|\n| POST | `/api/v1/claude/tasks` | Submit Claude task |\n| GET | `/api/v1/claude/tasks/{id}` | Get Claude task status |\n| GET | `/api/v1/claude/orchestrator/status` | Claude orchestrator status |\n\n## Configuration\n\n### Environment Variables\n\n```bash\n# Environment\nMAOS_ENV=production          # development, testing, staging, production\nLOG_LEVEL=INFO               # DEBUG, INFO, WARNING, ERROR\n\n# Redis Configuration\nREDIS_HOST=localhost\nREDIS_PORT=6379\nREDIS_PASSWORD=secret\n\n# Security\nSECURITY_REQUIRE_ENCRYPTION=true\nSECURITY_REQUIRE_AUTH=true\nSECURITY_RATE_LIMIT=1000\n\n# Performance\nMESSAGE_BUS_MAX_QUEUE_SIZE=10000\nMESSAGE_BUS_WORKER_COUNT=4\n```\n\n### API Configuration\n\n```python\nclass APIConfig:\n    host = \"0.0.0.0\"\n    port = 8000\n    max_agents = 100\n    max_concurrent_tasks = 1000\n    default_task_timeout = 300\n    max_task_timeout = 3600\n```\n\n## Task Patterns\n\n### 1. Simple Task Execution\n\n```python\n# Submit a data processing task\ntask_response = await client.submit_task(\n    name=\"Data Analysis\",\n    description=\"Analyze customer behavior patterns\",\n    priority=\"high\",\n    parameters={\n        \"dataset\": \"customers_2024\",\n        \"analysis_type\": \"behavioral\"\n    },\n    resource_requirements={\n        \"cpu_cores\": 2.0,\n        \"memory_mb\": 4096\n    },\n    timeout_seconds=600\n)\n```\n\n### 2. Parallel Processing\n\n```python\n# Submit multiple parallel tasks\ntasks = [\n    {\"name\": f\"Process Partition {i}\", \"parameters\": {\"partition\": i}}\n    for i in range(1, 4)\n]\n\nfor task in tasks:\n    await client.submit_task(**task, decomposition_strategy=\"parallel\")\n```\n\n### 3. Hierarchical Decomposition\n\n```python\n# Submit complex ML pipeline\npipeline_response = await client.submit_task(\n    name=\"ML Pipeline\",\n    description=\"Complete ML workflow\",\n    parameters={\n        \"model_type\": \"random_forest\",\n        \"dataset\": \"customer_churn\",\n        \"stages\": [\"preprocess\", \"train\", \"validate\", \"deploy\"]\n    },\n    decomposition_strategy=\"hierarchical\",\n    timeout_seconds=1800\n)\n```\n\n## Agent Types\n\n### Specialized Agents\n\n```python\n# Data Science Agent\ndata_agent = await client.create_agent(\n    agent_type=\"data_scientist\",\n    capabilities=[\"data_processing\", \"computation\"],\n    configuration={\n        \"python_version\": \"3.11\",\n        \"ml_libraries\": [\"sklearn\", \"pandas\", \"numpy\"]\n    },\n    max_concurrent_tasks=2\n)\n\n# API Integration Agent\napi_agent = await client.create_agent(\n    agent_type=\"api_handler\",\n    capabilities=[\"api_integration\", \"communication\"],\n    configuration={\n        \"protocols\": [\"REST\", \"GraphQL\"],\n        \"auth_methods\": [\"OAuth2\", \"JWT\"]\n    },\n    max_concurrent_tasks=5\n)\n```\n\n## Monitoring and Observability\n\n### Health Checks\n\n```python\n# Basic health check\nhealth = await client.health_check()\nprint(f\"Status: {health['status']}\")\n\n# Detailed health with components\ndetailed = await client.detailed_health_check()\nfor component, status in detailed['components'].items():\n    print(f\"{component}: {status['status']}\")\n```\n\n### System Metrics\n\n```python\n# Get comprehensive metrics\nmetrics = await client.get_system_metrics()\nprint(f\"Tasks completed: {metrics['orchestrator']['tasks_completed']}\")\nprint(f\"Agents active: {metrics['agent_manager']['active_agents']}\")\n```\n\n## Error Handling\n\nThe system provides comprehensive error handling:\n\n```python\ntry:\n    response = await client.submit_task(name=\"Test Task\")\nexcept aiohttp.ClientResponseError as e:\n    if e.status == 400:\n        print(f\"Bad request: {e.message}\")\n    elif e.status == 500:\n        print(f\"Server error: {e.message}\")\nexcept asyncio.TimeoutError:\n    print(\"Request timed out\")\n```\n\n## Best Practices\n\n### 1. Resource Management\n- Always specify appropriate resource requirements\n- Use decomposition strategies for complex tasks\n- Monitor resource utilization\n\n### 2. Error Handling\n- Implement proper retry logic\n- Handle timeout scenarios\n- Log errors with context\n\n### 3. Performance\n- Use appropriate task priorities\n- Batch similar operations\n- Monitor system metrics\n\n### 4. Security\n- Enable authentication in production\n- Use HTTPS for external access\n- Validate all input parameters\n\n## Troubleshooting\n\n### Common Issues\n\n1. **Task Not Starting**\n   - Check agent availability\n   - Verify resource requirements\n   - Review system health\n\n2. **High Memory Usage**\n   - Check for resource leaks\n   - Review concurrent task limits\n   - Monitor agent performance\n\n3. **Connection Issues**\n   - Verify Redis connectivity\n   - Check network configuration\n   - Review firewall settings\n\n### Debug Mode\n\n```bash\n# Run with debug logging\npython examples/backend_api_example.py --debug\n\n# Check logs\ntail -f maos_api.log\n```\n\n## Contributing\n\nSee the main project README for contribution guidelines.\n\n## License\n\nThis project is licensed under the MIT License.\n"