#!/usr/bin/env python3
"""Create a PDF or HTML document from a UDN songsheet."""

import argparse
import datetime
import shutil
import sys
from pathlib import Path

from udn_songbook import Song

# so we know which profiles are defined
from udn_songbook.config import load_settings

settings = load_settings()


def parse_cmdline(argv: list[str]) -> argparse.Namespace:
    """Process commandline options and arguments, provide help."""

    preamble = f"""
Generates a PDF or HTML songsheet from a UDN input file

Files are generated in your CWD (current working directory),
unless you specify a path to an output file.

Profiles can be used to define common output formats
(whether to display chords etc).

Profiles are defined in TOML files and merged.

Currently discovered config files:
  - {"\n  - ".join([str(f) for f in settings.settings_file])}
    """

    parser = argparse.ArgumentParser(
        description=preamble, formatter_class=argparse.RawTextHelpFormatter
    )
    parser.add_argument(
        "filename",
        type=Path,
        help="path to a filename in UDN format",
    )
    parser.add_argument(
        "-o",
        "--output",
        type=Path,
        help="name of output file (or directory) "
        "autogenerated from input filename if omitted. "
        "If a directory, will be created and the default filename used within it",
    )

    parser.add_argument(
        "-d",
        "--destdir",
        type=Path,
        help="output directory for generated files (will be created if missing)",
    )

    parser.add_argument(
        "-f",
        "--force",
        action="store_true",
        default=False,
        help="overwrite in-place rather than renaming existing output files",
    )

    parser.add_argument(
        "-v",
        "--verbose",
        action="store_true",
        default=False,
        help="Actually tell me what is being done",
    )

    parser.add_argument(
        "-s",
        "--style",
        type=Path,
        help="Path to custom stylesheet, or name of built-in one.",
    )

    outgrp = parser.add_argument_group(
        "Customised Output", "Options for changing default output"
    )

    outgrp.add_argument(
        "-p",
        "--profile",
        choices=settings.get("profile").keys(),
        default="default",
        help="Select a output profile (singer, etc).",
    )

    outgrp.add_argument(
        "--html",
        action="store_true",
        default=False,
        help="Create a songsheet in HTML format, rather than the default PDF",
    )

    outgrp.add_argument(
        "-t",
        "--transpose",
        metavar="SEMITONES",
        type=int,
        help="transpose song by this many semitones before rendering. Can be negative",
    )

    stylegrp = parser.add_argument_group(
        "Content Customisation",
        "Settings present in a chosen profile (see --profile) take precedence",
    )
    stylegrp.add_argument(
        "--chords",
        action="store_true",
        default=False,
        help="Include inline chord names",
    )
    stylegrp.add_argument(
        "--notes",
        action="store_true",
        default=False,
        help="Include inline performance notes ('slowly' etc)",
    )
    stylegrp.add_argument(
        "--diagrams", action="store_true", default=False, help="Include chord diagrams"
    )
    stylegrp.add_argument(
        "--singer-notes",
        action="store_true",
        default=False,
        help="Include inline singer notes (for duets etc)",
    )

    opts = parser.parse_args(argv)

    opts.filename = Path(opts.filename)

    if not opts.output:
        opts.output = opts.filename.with_suffix(".html" if opts.html else ".pdf")

    if opts.destdir:
        opts.output = opts.destdir / opts.output.name

    return opts


def main():
    """
    Main functionality
    """
    opts = parse_cmdline(sys.argv[1:])

    song = Song(opts.filename)

    profile = settings.get(f"profiles.{opts.profile}", {})

    if not opts.style:
        opts.style = Path(profile.get("stylesheet", "portrait.css"))

    # figure out stylesheets
    if opts.style and not any([
        opts.style.with_suffix(".css").exists(),
        (song.styles_dir / opts.style.with_suffix(".css")).exists(),
    ]):
        print(f"Cannot locate stylesheet {opts.style}")
        sys.exit(2)

    # we provided a stylesheet, does it exist?

    if opts.transpose:
        if opts.verbose:
            print(f"transposing song by {opts.transpose} semitones")
        song.transpose(opts.transpose)

    if opts.output.is_file() and not opts.force:
        print(f"output file {opts.output} already exists, backing it up")
        shutil.copyfile(
            opts.output,
            opts.output.with_suffix(
                datetime.datetime.now().strftime(f"{opts.output.suffix}.%s")
            ),
        )

    context = {
        "chords": opts.chords,
        "notes": opts.notes,
        "diagrams": False,  # future-proofing
        "singer_notes": opts.singer_notes,
        "credits": False,  # future-proofing
    }

    if opts.verbose:
        print(f"Rendering {opts.filename} as {opts.output}")

    if opts.html:
        with opts.output.open(mode="w") as dest:
            dest.write(song.html(standalone=True, chords=opts.chords, notes=opts.notes))
        with open(opts.output, "w") as dest:
            dest.write(
                song.html(
                    stylesheet=opts.style,
                    profile=opts.profile,
                    verbose=opts.verbose,
                    **context,
                )
            )

    else:
        song.pdf(
            destfile=opts.output,
            stylesheet=opts.style,
            profile=opts.profile,
            verbose=opts.verbose,
            **context,
        )


if __name__ == "__main__":
    main()
