# CoMed: A Comprehensive Framework for Drug Co-Medication Risk Analysis

[![Python 3.8+](https://img.shields.io/badge/python-3.8+-blue.svg)](https://www.python.org/downloads/)
[![PyPI version](https://badge.fury.io/py/comed.svg)](https://badge.fury.io/py/comed)
[![License: BSD-2-Clause](https://img.shields.io/badge/License-BSD--2--Clause-blue.svg)](https://opensource.org/licenses/BSD-2-Clause)

## ⚠️ **IMPORTANT DISCLAIMER**

> **🚨 CRITICAL NOTICE: This tool is designed for research and educational purposes only.**
> 
> **CoMed is NOT intended for direct clinical use and should NOT be used as the sole basis for clinical decision-making.**
> 
> - ✅ **Intended Use**: Research, education, and clinical decision support for healthcare professionals
> - ❌ **NOT for**: Direct patient care, automated clinical decisions, or replacing professional medical judgment
> - 🔬 **Target Users**: Clinical researchers, healthcare professionals, and medical students
> - ⚖️ **Responsibility**: Always consult qualified healthcare professionals for clinical decisions
> 
> **By using this software, you acknowledge that it is for research and educational purposes only.**

## 🎯 Overview

CoMed is a comprehensive framework for analyzing drug co-medication risks using advanced AI techniques including Retrieval-Augmented Generation (RAG), Chain-of-Thought (CoT) reasoning, and multi-agent collaboration. It automates the process of searching medical literature, analyzing drug interactions, and generating detailed risk assessment reports.

### Why CoMed?

- **Research Efficiency**: Automate literature review for drug interaction studies
- **Comprehensive Analysis**: Combine multiple AI approaches for thorough risk assessment
- **Reproducible Results**: Standardized methodology for consistent analysis
- **Scalable Processing**: Handle multiple drug combinations efficiently
- **Evidence-Based**: Ground analysis in peer-reviewed medical literature

## 🔧 Key Features

### 1. **Literature Retrieval & Analysis**
- Automated PubMed search with intelligent query construction
- Relevance filtering and ranking of medical literature
- Statistical analysis of retrieved papers
- Support for custom search parameters

### 2. **Advanced AI Reasoning**
- Chain-of-Thought reasoning for step-by-step analysis
- Multi-agent collaboration for comprehensive assessment
- Conflict resolution and consensus building
- Confidence-weighted decision making

### 3. **Comprehensive Risk Assessment**
- Multi-dimensional risk analysis (pharmacokinetic, pharmacodynamic, clinical)
- Evidence-based risk scoring
- Detailed interaction mechanism analysis
- Clinical recommendation generation

### 4. **Flexible Architecture**
- Modular design for easy customization
- Support for different LLM providers
- Configurable analysis parameters
- Extensible agent system

## 📦 Installation

### Prerequisites
- Python 3.8 or higher
- OpenAI API key (or compatible LLM API)

### Install from PyPI
```bash
pip install comed
```

### Install from Source
```bash
git clone https://github.com/studentiz/comed.git
cd comed
pip install -e .
```

## 🚀 Quick Start

### Basic Setup

```python
import os
import comed

# Configure your LLM API
os.environ["MODEL_NAME"] = "gpt-4o"  # or "gpt-3.5-turbo", "qwen2.5-32b-instruct"
os.environ["API_BASE"] = "https://api.openai.com/v1"
os.environ["API_KEY"] = "your-api-key-here"

# Initialize with drug list
drugs = ["warfarin", "aspirin", "ibuprofen"]
com = comed.CoMedData(drugs)

# Run complete analysis
report_path = com.run_full_analysis(retmax=30, verbose=True)
print(f"Report generated at: {report_path}")
```

### Step-by-Step Analysis

```python
# 1. Search medical literature
com.search(retmax=20, email="your.email@example.com")

# 2. Analyze drug associations
com.analyze_associations()

# 3. Assess risks
com.analyze_risks()

# 4. Generate report
com.generate_report("Drug_Interaction_Report.html")
```

## 📚 Usage Examples

### Example 1: Cardiovascular Drug Interactions

```python
import os
import comed

# Set up API credentials
os.environ["MODEL_NAME"] = "gpt-4o"
os.environ["API_BASE"] = "https://api.openai.com/v1"
os.environ["API_KEY"] = "your-api-key-here"

# Analyze cardiovascular drug combinations
cardiovascular_drugs = ["warfarin", "aspirin", "clopidogrel", "metoprolol"]
com = comed.CoMedData(cardiovascular_drugs)

# Run analysis with method chaining
com.search(retmax=50) \
   .analyze_associations() \
   .analyze_risks() \
   .generate_report("Cardiovascular_Interactions.html")
```

### Example 2: Diabetes Medication Analysis

```python
# Analyze diabetes medication interactions
diabetes_drugs = ["metformin", "insulin", "glipizide", "pioglitazone"]
com = comed.CoMedData(diabetes_drugs)

# Incremental analysis
com.search(retmax=30)
com.analyze_associations()

# Add more drugs and continue analysis
com.add_drugs(["sitagliptin", "canagliflozin"])
com.search(retmax=30)
com.analyze_risks()
com.generate_report("Diabetes_Medication_Analysis.html")
```

### Example 3: Multi-Agent Collaboration

```python
from comed import MultiAgentSystem

# Initialize multi-agent system
agent_system = MultiAgentSystem(
    model_name="gpt-4o",
    api_key="your-key",
    api_base="https://api.openai.com/v1"
)

# Analyze specific drug combination
drug1, drug2 = "warfarin", "aspirin"
abstract = "Literature abstract content..."

# Use consensus collaboration
result = agent_system.process_drug_combination(
    drug1, drug2, abstract, 
    collaboration_mode="consensus"
)

print(f"Risk Assessment: {result['risk_analysis']}")
print(f"Safety Recommendation: {result['safety_assessment']}")
print(f"Clinical Guidance: {result['clinical_recommendation']}")
```

### Example 4: Batch Processing

```python
# Process multiple drug combinations
drug_combinations = [
    ["warfarin", "aspirin"],
    ["metformin", "lisinopril"],
    ["atorvastatin", "amlodipine"]
]

for i, drugs in enumerate(drug_combinations):
    com = comed.CoMedData(drugs)
    com.search(retmax=20)
    com.analyze_associations()
    com.analyze_risks()
    com.generate_report(f"Combination_{i+1}_Report.html")
```

## 🎮 Demo Scripts

Run the included demo scripts to see CoMed in action:

```bash
# Basic functionality demo
python examples/basic_demo.py

# Quick start tutorial
python examples/quick_start.py

# Multi-agent system demo
python examples/simple_agent_test.py

# Load existing data and analyze
python examples/load_and_analyze.py
```

## 🔧 Advanced Configuration

### Environment Variables

```bash
export MODEL_NAME="gpt-4o"
export API_BASE="https://api.openai.com/v1"
export API_KEY="your-api-key"
export LOG_DIR="logs"
export OLD_OPENAI_API="No"  # Set to "Yes" for older OpenAI API format
```

### Custom LLM Configuration

```python
# Configure different LLM providers
com = comed.CoMedData(["drug1", "drug2"])
com.set_config({
    'model_name': 'qwen2.5-32b-instruct',
    'api_base': 'https://your-llm-api.com/v1',
    'api_key': 'your-api-key'
})
```

### Analysis Parameters

```python
# Customize search parameters
com.search(
    retmax=50,  # Number of papers to retrieve
    email="your.email@example.com",  # Required for PubMed
    date_range=("2020/01/01", "2024/12/31")  # Optional date filter
)

# Customize analysis depth
com.analyze_associations(
    confidence_threshold=0.7,  # Minimum confidence for associations
    include_mechanisms=True    # Include interaction mechanisms
)
```

## 🏗️ Architecture

### Core Components

```
CoMed Framework
├── RAG Module (rag.py)
│   ├── Literature retrieval from PubMed
│   ├── Relevance scoring and filtering
│   └── Statistical analysis
├── CoT Module (cot.py)
│   ├── Chain-of-thought reasoning
│   ├── Step-by-step analysis
│   └── Result interpretation
├── Multi-Agent Module (agents.py)
│   ├── RiskAnalysisAgent
│   ├── SafetyAgent
│   ├── ClinicalAgent
│   └── Collaboration protocols
└── Core Module (core.py)
    ├── Component integration
    ├── Configuration management
    └── Result aggregation
```

### Data Flow

```
Drug Combinations → Literature Search → Association Analysis → Risk Assessment → Report Generation
        ↓                    ↓                    ↓                    ↓
   Input Validation    PubMed Retrieval    CoT Reasoning    Multi-Agent Analysis
        ↓                    ↓                    ↓                    ↓
   Query Construction   Relevance Filtering   Evidence Analysis   Consensus Building
        ↓                    ↓                    ↓                    ↓
   Search Execution    Statistical Analysis   Risk Scoring      Final Report
```

## 📊 Performance & Evaluation

CoMed provides built-in performance monitoring and evaluation capabilities:

```python
# Monitor analysis performance
com = comed.CoMedData(["warfarin", "aspirin"])
com.search(retmax=20)

# Get performance statistics
stats = com.get_performance_stats()
print(f"Papers retrieved: {stats['papers_retrieved']}")
print(f"Analysis time: {stats['analysis_time']:.2f}s")
print(f"Success rate: {stats['success_rate']:.2%}")
```

## 📚 API Reference

### Core Classes

- `CoMedData`: Main analysis class
- `RAGSystem`: Literature retrieval system
- `CoTReasoner`: Chain-of-thought reasoning
- `MultiAgentSystem`: Multi-agent collaboration

### Key Methods

#### Analysis Methods
- `search(retmax, email, date_range)`: Search medical literature
- `analyze_associations(confidence_threshold)`: Analyze drug associations
- `analyze_risks(risk_dimensions)`: Assess interaction risks
- `generate_report(filename)`: Generate HTML report

#### Multi-Agent Methods
- `process_drug_combination(drug1, drug2, abstract, mode)`: Process with agents
- `get_agent_stats()`: Get agent performance statistics
- `set_collaboration_mode(mode)`: Set collaboration strategy

#### Configuration Methods
- `set_config(config_dict)`: Set system configuration
- `add_drugs(drug_list)`: Add drugs to analysis
- `save_data(filename)`: Save analysis data
- `load_data(filename)`: Load existing data

## 🛠️ Development

### Adding Custom Agents

```python
from comed.agents import Agent

class CustomAnalysisAgent(Agent):
    def __init__(self, model_name, api_key, api_base):
        super().__init__(
            name="CustomAnalysisAgent",
            description="Custom drug analysis",
            model_name=model_name,
            api_key=api_key,
            api_base=api_base
        )
    
    def _execute_task(self, input_data):
        # Implement custom analysis logic
        return {"custom_result": "Analysis result"}
```

### Extending Analysis Modules

```python
from comed.rag import RAGSystem

class CustomRAGSystem(RAGSystem):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
    
    def custom_search_method(self, query):
        # Implement custom search logic
        pass
```

## 🤝 Contributing

We welcome contributions in various forms:

1. **Code Contributions**: New features, bug fixes, performance optimizations
2. **Documentation**: Better examples, tutorials, API documentation
3. **Testing**: Unit tests, integration tests, benchmark tests
4. **Research**: New evaluation metrics, test scenarios, and performance studies

### Development Setup

```bash
# Clone the repository
git clone https://github.com/studentiz/comed.git
cd comed

# Install in development mode
pip install -e .

# Install development dependencies (optional)
pip install -r requirements-dev.txt

# Set up environment variables
export MODEL_NAME="gpt-4o"
export API_BASE="https://api.openai.com/v1"
export API_KEY="your-api-key-here"
```

## 📄 License

This project is licensed under the BSD-2-Clause License. See [LICENSE](LICENSE) file for details.

## 🙏 Acknowledgments

Thanks to the research community for valuable feedback and contributions. Special thanks to reviewers who helped improve the framework's modularity and evaluation capabilities.

## 📞 Contact

- **Project Homepage**: https://github.com/studentiz/comed
- **Issue Reports**: Please use [GitHub Issues](https://github.com/studentiz/comed/issues)
- **Email**: studentiz@live.com

---

**Note**: This framework is for research purposes only and should not be used for clinical decision-making. Any medical decisions should be made in consultation with qualified healthcare professionals.