"""
RIT API Client - A comprehensive wrapper for the Rotman Interactive Trader API.

This module provides a clean interface for interacting with the RIT REST API,
with built-in rate limiting, error handling, and DataFrame support.
"""

import time
from typing import Optional, Dict, List, Union, Literal
import requests
import pandas as pd
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry

from .config import Config
from .exceptions import (
    RateLimitException,
    AuthenticationException,
    EndpointNotAvailableException,
    OrderException,
    InvalidParameterException,
    RITException,
)


class RITClient:
    """
    Client for interacting with the RIT API.

    Usage:
        client = RITClient()
        case_info = client.get_case()
        securities_df = client.get_securities()
    """

    def __init__(
        self,
        api_key: Optional[str] = None,
        base_url: Optional[str] = None,
        max_retries: Optional[int] = None,
        timeout: Optional[int] = None,
    ):
        """
        Initialize the RIT API client.

        Args:
            api_key: RIT API key (defaults to Config.RIT_API_KEY)
            base_url: Base URL for API (defaults to Config.BASE_URL)
            max_retries: Maximum retry attempts (defaults to Config.MAX_RETRIES)
            timeout: Request timeout in seconds (defaults to Config.TIMEOUT)
        """
        self.api_key = api_key or Config.RIT_API_KEY
        self.base_url = (base_url or Config.BASE_URL).rstrip("/")
        self.max_retries = max_retries or Config.MAX_RETRIES
        self.timeout = timeout or Config.TIMEOUT

        if not self.api_key:
            Config.validate()

        # Setup session with retry logic
        self.session = requests.Session()
        retry_strategy = Retry(
            total=3,
            backoff_factor=0.3,
            status_forcelist=[500, 502, 503, 504],
        )
        adapter = HTTPAdapter(max_retries=retry_strategy)
        self.session.mount("http://", adapter)
        self.session.mount("https://", adapter)

        # Default headers
        self.headers = {"accept": "application/json", "X-API-Key": self.api_key}

    def _request(
        self,
        method: str,
        endpoint: str,
        params: Optional[Dict] = None,
        retry_on_rate_limit: bool = True,
    ) -> Union[Dict, List]:
        """
        Make an API request with error handling and rate limit retry.

        Args:
            method: HTTP method (GET, POST, DELETE)
            endpoint: API endpoint (without base URL)
            params: Query parameters
            retry_on_rate_limit: Whether to automatically retry on rate limit

        Returns:
            API response as dict or list

        Raises:
            Various RITException subclasses based on error type
        """
        url = f"{self.base_url}/{endpoint.lstrip('/')}"
        retries = 0

        while retries <= self.max_retries:
            try:
                response = self.session.request(
                    method=method,
                    url=url,
                    headers=self.headers,
                    params=params,
                    timeout=self.timeout,
                )

                # Handle rate limiting
                if response.status_code == 429:
                    if not retry_on_rate_limit or retries >= self.max_retries:
                        wait_time = float(response.headers.get("Retry-After", Config.RETRY_DELAY))
                        raise RateLimitException(wait_time)

                    wait_time = float(response.headers.get("Retry-After", Config.RETRY_DELAY))
                    time.sleep(wait_time)
                    retries += 1
                    continue

                # Handle authentication errors
                if response.status_code == 401:
                    raise AuthenticationException("Authentication failed. Check your API key.")

                # Handle bad requests
                if response.status_code == 400:
                    error_msg = response.text
                    try:
                        error_data = response.json()
                        error_msg = error_data.get("message", error_msg)
                    except:
                        pass
                    raise InvalidParameterException(f"Bad request: {error_msg}")

                # Handle not found / not available
                if response.status_code == 404:
                    raise EndpointNotAvailableException(
                        f"Endpoint '{endpoint}' not found or not available in this case."
                    )

                # Handle other errors
                if response.status_code >= 400:
                    raise RITException(
                        f"API request failed with status {response.status_code}: {response.text}"
                    )

                # Success
                if response.status_code == 200:
                    return response.json()

                # Unexpected success code
                return response.json()

            except requests.exceptions.Timeout:
                raise RITException(f"Request timeout after {self.timeout} seconds")
            except requests.exceptions.ConnectionError:
                raise RITException("Connection error. Is the RIT Client running and API enabled?")
            except (
                RateLimitException,
                AuthenticationException,
                EndpointNotAvailableException,
                InvalidParameterException,
                RITException,
            ):
                raise
            except Exception as e:
                raise RITException(f"Unexpected error: {str(e)}")

        raise RateLimitException(Config.RETRY_DELAY, "Max retries exceeded")

    # ========== Case Information ==========

    def get_case(self) -> Dict:
        """
        Get information about the current case.

        Returns:
            Dict with case info including name, period, tick, status, etc.
        """
        return self._request("GET", "/case")

    def get_case_status(self) -> str:
        """
        Get the current case status.

        Returns:
            Case status: 'ACTIVE', 'PAUSED', or 'STOPPED'
        """
        return self.get_case()["status"]

    def is_case_active(self) -> bool:
        """Check if the case is currently active."""
        return self.get_case_status() == "ACTIVE"

    # ========== Trader Information ==========

    def get_trader(self) -> Dict:
        """
        Get information about the currently signed in trader.

        Returns:
            Dict with trader_id, first_name, last_name, nlv
        """
        return self._request("GET", "/trader")

    def get_nlv(self) -> float:
        """Get the trader's current Net Liquid Value."""
        return self.get_trader()["nlv"]

    # ========== Trading Limits ==========

    def get_limits(self) -> pd.DataFrame:
        """
        Get trading limits for the current case.

        Returns:
            DataFrame with limit information (name, gross, net, limits, fines)
        """
        data = self._request("GET", "/limits")
        return pd.DataFrame(data)

    # ========== News ==========

    def get_news(self, after: Optional[int] = None, limit: int = 20) -> pd.DataFrame:
        """
        Get the most recent news.

        Args:
            after: Retrieve only news with news_id greater than this value
            limit: Maximum number of news items to retrieve

        Returns:
            DataFrame with news (news_id, period, tick, ticker, headline, body)
        """
        params = {"limit": limit}
        if after is not None:
            params["after"] = after

        data = self._request("GET", "/news", params=params)
        return pd.DataFrame(data)

    def get_latest_news(self) -> Optional[Dict]:
        """
        Get the single most recent news item.

        Returns:
            Dict with news item or None if no news
        """
        df = self.get_news(limit=1)
        return df.iloc[0].to_dict() if not df.empty else None

    # ========== Securities ==========

    def get_securities(self, ticker: Optional[str] = None) -> pd.DataFrame:
        """
        Get list of available securities and associated positions.

        Args:
            ticker: Optional ticker to filter by specific security

        Returns:
            DataFrame with security information and positions
        """
        params = {"ticker": ticker} if ticker else {}
        data = self._request("GET", "/securities", params=params)
        return pd.DataFrame(data)

    def get_security(self, ticker: str) -> Dict:
        """
        Get information for a specific security.

        Args:
            ticker: Security ticker (case-sensitive)

        Returns:
            Dict with security information
        """
        df = self.get_securities(ticker=ticker)
        if df.empty:
            raise InvalidParameterException(f"Security '{ticker}' not found")
        return df.iloc[0].to_dict()

    def get_order_book(self, ticker: str, limit: int = 20) -> Dict[str, pd.DataFrame]:
        """
        Get the order book for a security.

        Args:
            ticker: Security ticker (case-sensitive)
            limit: Maximum number of orders per side

        Returns:
            Dict with 'bid' and 'ask' DataFrames
        """
        params = {"ticker": ticker, "limit": limit}
        data = self._request("GET", "/securities/book", params=params)

        return {"bid": pd.DataFrame(data.get("bid", [])), "ask": pd.DataFrame(data.get("ask", []))}

    def get_security_history(
        self, ticker: str, period: Optional[int] = None, limit: int = 20
    ) -> pd.DataFrame:
        """
        Get OHLC history for a security.

        Args:
            ticker: Security ticker (case-sensitive)
            period: Period to retrieve data from (defaults to current)
            limit: Maximum number of records

        Returns:
            DataFrame with tick, open, high, low, close
        """
        params = {"ticker": ticker, "limit": limit}
        if period is not None:
            params["period"] = period

        data = self._request("GET", "/securities/history", params=params)
        return pd.DataFrame(data)

    def get_time_and_sales(
        self,
        ticker: str,
        after: Optional[int] = None,
        period: Optional[int] = None,
        limit: int = 20,
    ) -> pd.DataFrame:
        """
        Get time & sales history for a security.

        Args:
            ticker: Security ticker (case-sensitive)
            after: Retrieve only data with id greater than this value
            period: Period to retrieve data from (defaults to current)
            limit: Maximum number of records

        Returns:
            DataFrame with id, period, tick, price, quantity
        """
        params = {"ticker": ticker, "limit": limit}
        if after is not None:
            params["after"] = after
        if period is not None:
            params["period"] = period

        data = self._request("GET", "/securities/tas", params=params)
        return pd.DataFrame(data)

    # ========== Orders ==========

    def get_orders(
        self, status: Literal["OPEN", "TRANSACTED", "CANCELLED"] = "OPEN"
    ) -> pd.DataFrame:
        """
        Get a list of all orders.

        Args:
            status: Order status filter ('OPEN', 'TRANSACTED', 'CANCELLED')

        Returns:
            DataFrame with order information
        """
        params = {"status": status}
        data = self._request("GET", "/orders", params=params)
        return pd.DataFrame(data)

    def get_open_orders(self) -> pd.DataFrame:
        """Get all open orders."""
        return self.get_orders(status="OPEN")

    def submit_order(
        self,
        ticker: str,
        order_type: Literal["MARKET", "LIMIT"],
        quantity: float,
        action: Literal["BUY", "SELL"],
        price: Optional[float] = None,
        dry_run: bool = False,
    ) -> Dict:
        """
        Submit a new order.

        Args:
            ticker: Security ticker (case-sensitive)
            order_type: 'MARKET' or 'LIMIT'
            quantity: Order quantity
            action: 'BUY' or 'SELL'
            price: Limit price (required if order_type is 'LIMIT')
            dry_run: Only for MARKET orders - simulates execution without placing

        Returns:
            Dict with order information

        Raises:
            OrderException: If order submission fails
            EndpointNotAvailableException: If orders not allowed in this case
        """
        params = {"ticker": ticker, "type": order_type, "quantity": quantity, "action": action}

        if order_type == "LIMIT":
            if price is None:
                raise InvalidParameterException("Price is required for LIMIT orders")
            params["price"] = price
        elif price is not None:
            params["price"] = price

        if dry_run and order_type == "MARKET":
            params["dry_run"] = 1

        try:
            return self._request("POST", "/orders", params=params)
        except (InvalidParameterException, EndpointNotAvailableException) as e:
            raise OrderException(str(e))

    def get_order(self, order_id: int) -> Dict:
        """
        Get details of a specific order.

        Args:
            order_id: The order ID

        Returns:
            Dict with order information
        """
        return self._request("GET", f"/orders/{order_id}")

    def cancel_order(self, order_id: int) -> bool:
        """
        Cancel an open order.

        Args:
            order_id: The order ID to cancel

        Returns:
            True if successful
        """
        result = self._request("DELETE", f"/orders/{order_id}")
        return result.get("success", False)

    def cancel_all_orders(self) -> List[int]:
        """
        Cancel all open orders.

        Returns:
            List of cancelled order IDs
        """
        result = self._request("POST", "/commands/cancel", params={"all": 1})
        return result.get("cancelled_order_ids", [])

    def cancel_orders_by_ticker(self, ticker: str) -> List[int]:
        """
        Cancel all open orders for a specific ticker.

        Args:
            ticker: Security ticker

        Returns:
            List of cancelled order IDs
        """
        result = self._request("POST", "/commands/cancel", params={"ticker": ticker})
        return result.get("cancelled_order_ids", [])

    def cancel_orders_by_ids(self, order_ids: List[int]) -> List[int]:
        """
        Cancel specific orders by ID.

        Args:
            order_ids: List of order IDs to cancel

        Returns:
            List of actually cancelled order IDs
        """
        ids_str = ",".join(map(str, order_ids))
        result = self._request("POST", "/commands/cancel", params={"ids": ids_str})
        return result.get("cancelled_order_ids", [])

    # ========== Assets (Commodities Case) ==========

    def get_assets(self, ticker: Optional[str] = None) -> pd.DataFrame:
        """
        Get list of available assets.

        Args:
            ticker: Optional ticker to filter by specific asset

        Returns:
            DataFrame with asset information

        Note:
            This endpoint is primarily for commodities cases
        """
        params = {"ticker": ticker} if ticker else {}
        try:
            data = self._request("GET", "/assets", params=params)
            return pd.DataFrame(data)
        except EndpointNotAvailableException:
            raise EndpointNotAvailableException("Assets endpoint not available in this case type")

    def get_asset_history(
        self, ticker: Optional[str] = None, period: Optional[int] = None, limit: int = 20
    ) -> pd.DataFrame:
        """
        Get activity log for assets.

        Args:
            ticker: Optional ticker filter
            period: Period to retrieve data from
            limit: Maximum number of records

        Returns:
            DataFrame with asset activity history
        """
        params = {"limit": limit}
        if ticker:
            params["ticker"] = ticker
        if period is not None:
            params["period"] = period

        try:
            data = self._request("GET", "/assets/history", params=params)
            return pd.DataFrame(data)
        except EndpointNotAvailableException:
            raise EndpointNotAvailableException("Assets endpoint not available in this case type")

    def get_leases(self) -> pd.DataFrame:
        """
        Get list of all assets currently being leased or used.

        Returns:
            DataFrame with lease information
        """
        try:
            data = self._request("GET", "/leases")
            return pd.DataFrame(data)
        except EndpointNotAvailableException:
            raise EndpointNotAvailableException("Leases endpoint not available in this case type")

    def lease_asset(
        self,
        ticker: str,
        from_tickers: Optional[List[str]] = None,
        quantities: Optional[List[float]] = None,
    ) -> Dict:
        """
        Lease or use an asset.

        Args:
            ticker: Asset ticker to lease
            from_tickers: Source tickers (for assets like refineries)
            quantities: Source quantities (must match from_tickers length)

        Returns:
            Dict with lease information
        """
        params = {"ticker": ticker}

        if from_tickers and quantities:
            if len(from_tickers) != len(quantities):
                raise InvalidParameterException("from_tickers and quantities must have same length")

            for i, (from_tick, qty) in enumerate(zip(from_tickers, quantities), 1):
                params[f"from{i}"] = from_tick
                params[f"quantity{i}"] = qty

        try:
            return self._request("POST", "/leases", params=params)
        except EndpointNotAvailableException:
            raise EndpointNotAvailableException("Leases endpoint not available in this case type")

    def use_leased_asset(
        self, lease_id: int, from_tickers: List[str], quantities: List[float]
    ) -> Dict:
        """
        Use a leased asset.

        Args:
            lease_id: The lease ID
            from_tickers: Source tickers
            quantities: Source quantities

        Returns:
            Dict with updated lease information
        """
        if len(from_tickers) != len(quantities):
            raise InvalidParameterException("from_tickers and quantities must have same length")

        params = {}
        for i, (from_tick, qty) in enumerate(zip(from_tickers, quantities), 1):
            params[f"from{i}"] = from_tick
            params[f"quantity{i}"] = qty

        return self._request("POST", f"/leases/{lease_id}", params=params)

    def get_lease(self, lease_id: int) -> Dict:
        """
        Get details of a specific lease.

        Args:
            lease_id: The lease ID

        Returns:
            Dict with lease information
        """
        return self._request("GET", f"/leases/{lease_id}")

    def cancel_lease(self, lease_id: int) -> bool:
        """
        Cancel/unlease an asset.

        Args:
            lease_id: The lease ID

        Returns:
            True if successful
        """
        result = self._request("DELETE", f"/leases/{lease_id}")
        return result.get("success", False)

    # ========== Tenders ==========

    def get_tenders(self) -> pd.DataFrame:
        """
        Get list of all active tenders.

        Returns:
            DataFrame with tender information
        """
        try:
            data = self._request("GET", "/tenders")
            return pd.DataFrame(data)
        except EndpointNotAvailableException:
            raise EndpointNotAvailableException("Tenders endpoint not available in this case type")

    def accept_tender(self, tender_id: int, price: float) -> bool:
        """
        Accept a tender.

        Args:
            tender_id: The tender ID
            price: Bid price (must match tender price for fixed-bid tenders)

        Returns:
            True if successful
        """
        params = {"price": price}
        result = self._request("POST", f"/tenders/{tender_id}", params=params)
        return result.get("success", False)

    def decline_tender(self, tender_id: int) -> bool:
        """
        Decline a tender.

        Args:
            tender_id: The tender ID

        Returns:
            True if successful
        """
        result = self._request("DELETE", f"/tenders/{tender_id}")
        return result.get("success", False)

    # ========== Utility Methods ==========

    def get_market_snapshot(self) -> Dict[str, pd.DataFrame]:
        """
        Get a complete snapshot of the current market state.

        Returns:
            Dict with 'case', 'trader', 'securities', 'limits', 'news' data
        """
        return {
            "case": self.get_case(),
            "trader": self.get_trader(),
            "securities": self.get_securities(),
            "limits": self.get_limits(),
            "news": self.get_news(limit=5),
            "orders": self.get_open_orders(),
        }

    def __repr__(self) -> str:
        """String representation of the client."""
        return f"RITClient(base_url='{self.base_url}')"
