import unittest

import numpy as np
import pandas as pd
import patsy

from inmoose.deseq2 import DESeq, makeExampleDESeqDataSet
from inmoose.diffexp import DEResults, meta_de
from inmoose.edgepy import (
    DGEList,
    exactTest,
    glmLRT,
    glmQLFTest,
)
from inmoose.utils import Factor


class Test(unittest.TestCase):
    def test_meta_de(self):
        # test contrasts
        dds = makeExampleDESeqDataSet(n=200, m=12, seed=42)
        dds.obs["condition"] = Factor(np.repeat([1, 2, 3], 4))
        dds.obs["group"] = Factor(np.repeat([[1, 2]], 6, axis=0).flatten())
        dds.counts()[:, 0] = np.repeat([100, 200, 800], 4)

        dds.design = "~ group + condition"

        # DE with deseq
        dds = DESeq(dds)
        deseq_res = dds.results()

        # DE with edgepy
        d = DGEList(counts=dds.X.T, group=dds.obs["group"])
        d.design = patsy.dmatrix("~ group + condition", data=dds.obs)
        d = d.estimateGLMCommonDisp()

        # DE with edgepy (LRT)
        edgepy_lrt_res = glmLRT(d.glmFit())

        # DE with edgepy (QL FTest)
        edgepy_ql_res = glmQLFTest(d.glmQLFit())

        # DE with edgepy (exactTest)
        edgepy_et_res = exactTest(d)

        # meta-analysis
        des = [deseq_res, edgepy_lrt_res, edgepy_ql_res, edgepy_et_res]
        res = meta_de(des)

        # combined logFC and confidence intervals are controlled against metafor
        # adjusted p-values have no external control, and are tested for non-regression
        ref = pd.DataFrame(
            {
                "combined logFC (CI_L)": [
                    0.6660151964096146,
                    -2.6386163106710177,
                    -2.4294256191695074,
                    -0.6488512767501585,
                    -2.91706443340857,
                    -3.521554387539502,
                    -0.013715570856962955,
                    -2.44433271322974,
                    -3.0287062069792716,
                    -1.2632361321964904,
                    -0.4565657144319961,
                    -1.050526185309593,
                    -2.898567141180949,
                    -0.5981094405745551,
                    0.542876997367387,
                    -2.5842836601101418,
                    -1.4881728468777093,
                    0.2816630199971988,
                    -1.621914242625586,
                    -2.175531983553664,
                    -2.426621491203676,
                    -2.2862239530712416,
                    0.6061652115075944,
                    -1.7988866511533614,
                    -1.3911906679104926,
                    -1.6781046771044665,
                    -1.5607619003814168,
                    -2.7313354443639586,
                    -1.819587410152225,
                    -1.8461737799605058,
                    -2.396354475817337,
                    -4.150733072719027,
                    -0.866612921109145,
                    -1.6209230508004846,
                    -1.3932528679807576,
                    0.06535739301240662,
                    -0.8614695037171913,
                    -0.8949387070851511,
                    -0.1885094288572462,
                    -2.32066999356063,
                    -1.077924632221578,
                    0.37794895248328375,
                    -0.5536367454374285,
                    -0.21445770439079448,
                    -1.037823547101704,
                    -2.0371872963903646,
                    -2.4346661814814206,
                    -0.4321573992340394,
                    -0.2907332240528021,
                    -0.8539476086445773,
                    -1.234310858262552,
                    -2.7156517079926124,
                    -2.8582388336368183,
                    -1.2745838405371699,
                    -2.231898985996819,
                    -1.8279970283385685,
                    -4.480837497725425,
                    -0.8011169122478685,
                    -1.9799835053398835,
                    -1.0525039883843812,
                    -0.11662674646446902,
                    -2.544956075527203,
                    -1.064065093263388,
                    -0.8655309481531789,
                    -3.13762536071928,
                    -0.10540785386966345,
                    -2.518975904098701,
                    -2.0158475722803484,
                    -1.6728430608728435,
                    -0.5029904144572839,
                    -2.7112359453355257,
                    -1.608063992292449,
                    -0.12750004447413454,
                    -0.8225585773011765,
                    -1.772440064055683,
                    -0.9260166010678244,
                    -2.313239524972868,
                    -2.2068757891973796,
                    -1.786675849820731,
                    -1.0262766837558557,
                    -1.642001176385648,
                    0.2736409499254513,
                    -1.6755199304028023,
                    -2.4188282013360873,
                    -1.1577393065221067,
                    -2.243289671608783,
                    -1.5618172554069623,
                    -2.954474402652569,
                    -1.4590021786483582,
                    -2.371121434638752,
                    0.4108524092617336,
                    0.7039471918836333,
                    -1.773048451498628,
                    -0.3511337912563779,
                    -2.29667638756921,
                    -1.4953656452894795,
                    0.4109470746831836,
                    -1.620862282034234,
                    -4.385686353557798,
                    -1.264780589472257,
                    -1.0905046361675372,
                    -1.0403768171893486,
                    -1.9921809567227817,
                    -1.0215481046788315,
                    -1.6201315863839372,
                    -2.3497397485281812,
                    -0.9990128404438292,
                    -5.336487545760831,
                    -1.0623709246799449,
                    -0.4845713539981751,
                    -1.1092279063453114,
                    -3.022406150569415,
                    -0.7788181579284017,
                    -1.7145383153607154,
                    -1.3029315696858457,
                    -3.635458688461912,
                    -1.7449955806992654,
                    -0.37369875378310724,
                    -1.0063648668773697,
                    -3.121738602010506,
                    -0.9469631938814649,
                    -1.106923489229556,
                    -1.497550281217571,
                    -2.515290508312053,
                    -0.9689865181728943,
                    -0.04387428241650848,
                    -0.37155469916385075,
                    0.2678112004933926,
                    -2.385590578613881,
                    -0.9158592706903643,
                    -1.8301813463727143,
                    0.07460142020882277,
                    -1.8353686638038753,
                    -3.390945408814847,
                    0.02301767559765966,
                    -1.9324366910144262,
                    -0.678599506709188,
                    -2.309748259623439,
                    -1.2948255656204237,
                    0.35319753181102975,
                    -1.0127834932344837,
                    -2.176779495718015,
                    -1.5315051529282733,
                    0.6836811923317061,
                    -2.23977305266835,
                    -1.891197146518215,
                    -4.784144827741974,
                    -1.8037230037715726,
                    -1.9295223989664199,
                    -0.7340126641402587,
                    -1.8476750640373827,
                    -2.3842945946277867,
                    -1.2424865760658015,
                    -1.245280551728335,
                    -1.215932212899843,
                    -1.9632263729514479,
                    -1.801494956920078,
                    -0.2662585394490714,
                    -1.5599833295880043,
                    -4.881835644107957,
                    -3.7493825881515503,
                    -2.67304198399075,
                    -1.536839751850891,
                    -2.19184090843123,
                    -1.94974510590935,
                    -0.12555539343831423,
                    -2.7007627007566892,
                    -1.4724196092118091,
                    -1.1455256591450884,
                    -3.115445854154167,
                    -1.0084793313977904,
                    -0.3940975540315449,
                    0.2856517318181431,
                    -1.9711379193519085,
                    -3.7499696990142724,
                    -2.2126112079601503,
                    -0.8543495093303507,
                    -1.9283857091263117,
                    -2.264787656888103,
                    -3.7275147424298174,
                    -0.554008090864221,
                    -2.474562426499207,
                    -0.9685815584803881,
                    -0.989172560751711,
                    -0.7884366564329295,
                    -1.6045380562019598,
                    -0.7654273118624542,
                    -4.289165687959561,
                    -3.5234762679288174,
                    -1.3877715238168173,
                    -0.4356219067620921,
                    -0.4168421506416269,
                    0.1318434023527142,
                    -0.020915214579250163,
                    -0.9953178311101134,
                    -0.09157891037095012,
                    -0.43160817845695565,
                    -2.8193999937632803,
                    -0.9965176866672529,
                    -1.3939413540076844,
                ],
                "combined logFC (CI_R)": [
                    3.354872058019537,
                    -0.34054646318095894,
                    0.35083068561892405,
                    1.2758758057915247,
                    0.987419819649228,
                    -0.43883016404534936,
                    2.1355572315208398,
                    0.5080118956718125,
                    -0.534349148975036,
                    3.472246864111826,
                    1.536101548350488,
                    0.8515220599803147,
                    -0.8176581829870475,
                    1.447354013999258,
                    2.6303737680491617,
                    -0.036105150077430315,
                    0.49647965833900265,
                    3.0187769587400037,
                    0.38285920569357956,
                    -0.02243840495832705,
                    0.7756234225826169,
                    -0.03228481757257251,
                    2.56069239477196,
                    0.42803292672457405,
                    0.6664263638908428,
                    1.0336396693493475,
                    0.4703412064756911,
                    -0.7449809664746245,
                    1.2827268308386777,
                    0.22347520052144698,
                    -0.42579173099623957,
                    -1.0378149913948602,
                    1.474202085650837,
                    0.9832345898694661,
                    0.5979839574608838,
                    1.9537523108751609,
                    1.4094483690114512,
                    1.3045573581800696,
                    2.076718012172452,
                    -0.28070340978151154,
                    0.9447694039293841,
                    2.3437117766301405,
                    1.8183642690425976,
                    1.8861062700963296,
                    1.0774997747736272,
                    0.014536046442994888,
                    -0.3734987169876831,
                    1.6657293834968137,
                    1.7371529269569606,
                    1.6934459447290249,
                    0.7288005882108818,
                    -0.4582096934338016,
                    0.47222573634692555,
                    0.7525448340628036,
                    -0.05053394143011802,
                    0.666381149024404,
                    -0.3611533539173757,
                    1.1327677959957496,
                    1.9093349096061205,
                    1.6224956915201059,
                    2.8092931584641434,
                    -0.4675924133882219,
                    0.9556382394675933,
                    1.1537936002185782,
                    0.4719726272864868,
                    1.8940426428930035,
                    -0.370165159637702,
                    0.5148047144587685,
                    0.37474384949352624,
                    3.582528610121325,
                    1.266931510128373,
                    2.2158467226115492,
                    3.071101744852342,
                    1.1813346784457122,
                    0.18990362116929815,
                    1.1106002066088052,
                    -0.12768912119813525,
                    -0.1844407501803229,
                    0.25835875418293686,
                    1.1774869629070284,
                    0.3507702603130787,
                    2.7960199923829494,
                    0.6503959195118854,
                    -0.02699953769695007,
                    1.5711833206054435,
                    0.4520306257582095,
                    0.5672425711692818,
                    0.766116704264352,
                    0.5643657171094311,
                    -0.27795639060702126,
                    2.366747950705241,
                    2.77534421368124,
                    0.41134113503556835,
                    1.969588958548259,
                    1.2844090294384958,
                    0.9380549081602017,
                    3.274251586886894,
                    0.6590123200119615,
                    0.750549465611706,
                    2.018855699691608,
                    1.0562073319536494,
                    0.950802281270808,
                    0.5938590357378501,
                    1.0494322805270517,
                    1.5076695060610152,
                    0.9887948817674262,
                    4.264921321686238,
                    1.0328063734921642,
                    0.8621704647652939,
                    4.993500038553787,
                    0.9499110359553256,
                    -0.8997966342353689,
                    1.4795464281564392,
                    0.874904298760736,
                    0.7724717737068273,
                    0.8436676453709686,
                    0.38875016354958924,
                    1.6996550100286307,
                    1.0180844678588887,
                    -0.8432083259898191,
                    1.814332444242015,
                    0.89595219174541,
                    0.47768062281587376,
                    -0.6122504151106485,
                    0.9676127377450849,
                    1.9942096848086273,
                    2.5322997447126463,
                    2.6138264680652976,
                    0.06389822388153621,
                    1.3473614017305102,
                    0.2707537486533764,
                    4.160256742758929,
                    0.3225311846941912,
                    -0.916644886148936,
                    2.691811189135298,
                    0.5218401717051796,
                    1.2741589329453422,
                    0.12126997384548743,
                    0.5949804141792554,
                    2.316820003047362,
                    1.0110795928142386,
                    0.6586624521693996,
                    3.422985512826824,
                    2.7421890347062328,
                    1.5513174592326577,
                    0.25964239369840003,
                    0.20181520548352871,
                    0.9264018383524469,
                    0.015928832730718212,
                    1.28513806241418,
                    0.7355117760711144,
                    0.8107859106491883,
                    2.436102208526566,
                    1.0699877757552179,
                    0.8751364190659427,
                    0.05081803315723321,
                    0.22775593873046085,
                    1.6470249080421566,
                    1.8052433343747047,
                    -0.6862411763247649,
                    0.7657421312167374,
                    -0.48935736430145216,
                    0.3473665242861057,
                    0.9672738541238843,
                    0.37303455240566996,
                    2.348076008955688,
                    0.22513005048675083,
                    0.6067512405141617,
                    0.8955766331258332,
                    -0.6949808575425094,
                    1.313934433369968,
                    1.6405906148006886,
                    2.68193589852294,
                    0.7671934502323494,
                    2.818888058186232,
                    -0.18521862612875628,
                    1.0476709254412027,
                    0.5922832182532727,
                    -0.08490117418915877,
                    -0.7311606079226332,
                    1.3351236101816348,
                    -0.4398791608381776,
                    1.151171699680487,
                    1.743948784448906,
                    1.2591808745192232,
                    0.4065967359112883,
                    1.2607500621625511,
                    -1.182757686341431,
                    -0.9549093767155574,
                    0.5752653648553867,
                    1.4776432917947777,
                    1.8303526779984838,
                    2.45286628466279,
                    1.9298533345564415,
                    1.351808248409464,
                    2.2091228037508985,
                    1.5322246498088472,
                    -0.5191787229012983,
                    1.0055531958855577,
                    0.8993415317328557,
                ],
                "combined logFC": [
                    2.010443627214576,
                    -1.4895813869259882,
                    -1.0392974667752917,
                    0.31351226452068304,
                    -0.964822306879671,
                    -1.9801922757924255,
                    1.0609208303319384,
                    -0.9681604087789637,
                    -1.7815276779771538,
                    1.1045053659576676,
                    0.539767916959246,
                    -0.09950206266463923,
                    -1.8581126620839983,
                    0.4246222867123514,
                    1.5866253827082744,
                    -1.3101944050937862,
                    -0.4958465942693533,
                    1.6502199893686014,
                    -0.6195275184660032,
                    -1.0989851942559956,
                    -0.8254990343105295,
                    -1.159254385321907,
                    1.5834288031397772,
                    -0.6854268622143936,
                    -0.36238215200982493,
                    -0.32223250387755953,
                    -0.5452103469528629,
                    -1.7381582054192914,
                    -0.2684302896567735,
                    -0.8113492897195295,
                    -1.4110731034067885,
                    -2.5942740320569437,
                    0.303794582270846,
                    -0.3188442304655092,
                    -0.39763445525993685,
                    1.0095548519437838,
                    0.27398943264712994,
                    0.2048093255474592,
                    0.944104291657603,
                    -1.3006867016710706,
                    -0.06657761414609703,
                    1.3608303645567121,
                    0.6323637618025846,
                    0.8358242828527676,
                    0.019838113835961624,
                    -1.0113256249736848,
                    -1.4040824492345518,
                    0.6167859921313871,
                    0.7232098514520792,
                    0.41974916804222373,
                    -0.252755135025835,
                    -1.586930700713207,
                    -1.1930065486449464,
                    -0.26101950323718315,
                    -1.1412164637134685,
                    -0.5808079396570822,
                    -2.4209954258214,
                    0.1658254418739405,
                    -0.03532429786688163,
                    0.2849958515678623,
                    1.3463332059998372,
                    -1.5062742444577124,
                    -0.05421342689789732,
                    0.14413132603269962,
                    -1.3328263667163966,
                    0.89431739451167,
                    -1.4445705318682016,
                    -0.7505214289107899,
                    -0.6490496056896586,
                    1.5397690978320207,
                    -0.7221522176035764,
                    0.30389136515955006,
                    1.4718008501891038,
                    0.1793880505722678,
                    -0.7912682214431924,
                    0.0922918027704904,
                    -1.2204643230855017,
                    -1.1956582696888511,
                    -0.764158547818897,
                    0.07560513957558634,
                    -0.6456154580362846,
                    1.5348304711542002,
                    -0.5125620054454585,
                    -1.2229138695165187,
                    0.20672200704166846,
                    -0.8956295229252867,
                    -0.4972873421188403,
                    -1.0941788491941085,
                    -0.44731823076946353,
                    -1.3245389126228866,
                    1.3888001799834873,
                    1.7396457027824366,
                    -0.6808536582315298,
                    0.8092275836459406,
                    -0.5061336790653573,
                    -0.27865536856463896,
                    1.842599330785039,
                    -0.4809249810111363,
                    -1.817568443973046,
                    0.3770375551096756,
                    -0.017148652106944023,
                    -0.044787267959270266,
                    -0.6991609604924658,
                    0.01394208792411003,
                    -0.056231040161460966,
                    -0.6804724333803774,
                    1.6329542406212045,
                    -2.1518405861343335,
                    -0.10010022995732543,
                    2.2544643422778057,
                    -0.07965843519499288,
                    -1.961101392402392,
                    0.3503641351140188,
                    -0.41981700829998975,
                    -0.26522989798950924,
                    -1.3958955215454716,
                    -0.6781227085748381,
                    0.6629781281227617,
                    0.005859800490759473,
                    -1.9824734640001627,
                    0.43368462518027506,
                    -0.10548564874207311,
                    -0.5099348292008486,
                    -1.5637704617113508,
                    -0.0006868902139046985,
                    0.9751677011960594,
                    1.0803725227743979,
                    1.440818834279345,
                    -1.1608461773661722,
                    0.21575106552007295,
                    -0.779713798859669,
                    2.117429081483876,
                    -0.7564187395548421,
                    -2.1537951474818917,
                    1.357414432366479,
                    -0.7052982596546232,
                    0.29777971311807716,
                    -1.094239142888976,
                    -0.3499225757205842,
                    1.3350087674291957,
                    -0.0008519502101226178,
                    -0.7590585217743078,
                    0.9457401799492752,
                    1.7129351135189694,
                    -0.3442277967178461,
                    -0.8157773764099074,
                    -2.2911648111292227,
                    -0.43866058270956276,
                    -0.9567967831178508,
                    0.2755626991369606,
                    -0.5560816439831341,
                    -0.7867543419892992,
                    0.5968078162303823,
                    -0.08764638798655852,
                    -0.17039789691695012,
                    -0.9562041698971073,
                    -0.7868695090948086,
                    0.6903831842965426,
                    0.12263000239335031,
                    -2.7840384102163607,
                    -1.4918202284674065,
                    -1.5811996741461012,
                    -0.5947366137823926,
                    -0.6122835271536728,
                    -0.78835527675184,
                    1.111260307758687,
                    -1.2378163251349692,
                    -0.4328341843488237,
                    -0.12497451300962753,
                    -1.9052133558483382,
                    0.15272755098608884,
                    0.6232465303845719,
                    1.4837938151705414,
                    -0.6019722345597796,
                    -0.4655408204140199,
                    -1.1989149170444533,
                    0.09666070805542601,
                    -0.6680512454365195,
                    -1.1748444155386308,
                    -2.2293376751762253,
                    0.390557759658707,
                    -1.4572207936686923,
                    0.09129507060004953,
                    0.37738811184859755,
                    0.23537210904314693,
                    -0.5989706601453357,
                    0.2476613751500485,
                    -2.735961687150496,
                    -2.2391928223221873,
                    -0.4062530794807153,
                    0.5210106925163428,
                    0.7067552636784284,
                    1.292354843507752,
                    0.9544690599885957,
                    0.1782452086496753,
                    1.0587719466899743,
                    0.5503082356759458,
                    -1.6692893583322892,
                    0.00451775460915238,
                    -0.24729991113741429,
                ],
                "adjusted combined pval": [
                    0.002710867577467824,
                    0.5286922590236601,
                    0.3635268253933128,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.40841535987809763,
                    0.8552303143709908,
                    0.45397729003343607,
                    0.249091508520245,
                    0.2061819814543404,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.249091508520245,
                    0.9999997373138294,
                    0.40841535987809763,
                    0.7200961165080368,
                    0.9999997373138294,
                    0.5115953505542589,
                    0.9999997373138294,
                    0.845756654704669,
                    0.5676481671711865,
                    0.8395655047552425,
                    0.36422621419559664,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9239752719474906,
                    0.9999997373138294,
                    0.2321774685303417,
                    0.5115953505542589,
                    0.9239752719474906,
                    0.5070330046729763,
                    0.09174034333551025,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.7200961165080368,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.6847287526005329,
                    0.9999997373138294,
                    0.5442138166006407,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9658912215110148,
                    0.5892132685871182,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.8063836512383536,
                    0.9999997373138294,
                    0.3635268253933128,
                    0.878906678789298,
                    0.9999997373138294,
                    0.845756654704669,
                    0.9999997373138294,
                    0.02353676743785122,
                    0.9999997373138294,
                    0.4372483768805375,
                    0.6899654596418524,
                    0.9239752719474906,
                    0.5115953505542589,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.19158373821572414,
                    0.9999997373138294,
                    0.6681718367617152,
                    0.7139678179747181,
                    0.9999997373138294,
                    0.14361191965472642,
                    0.5626777439383062,
                    0.9999997373138294,
                    0.5115953505542589,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.8395655047552425,
                    0.6847287526005329,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.6847287526005329,
                    0.9999997373138294,
                    0.45397729003343607,
                    0.9999997373138294,
                    0.5115953505542589,
                    0.9999997373138294,
                    0.19266295410826276,
                    0.9999997373138294,
                    0.6847287526005329,
                    0.5274203426647385,
                    0.33573852446575575,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.5115953505542589,
                    0.9999997373138294,
                    0.01737015553722674,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.845756654704669,
                    0.5306621528598132,
                    0.13720519336046125,
                    0.014432955706249317,
                    0.9999997373138294,
                    0.249091508520245,
                    0.9999997373138294,
                    0.29234617948435815,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.18171356404052635,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.249091508520245,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.249091508520245,
                    0.9999997373138294,
                    0.9550708068178158,
                    0.45143711607921805,
                    0.6761159592036801,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.14764567543600687,
                    0.8655808208804949,
                    0.13720519336046125,
                    0.7952134754914685,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.4372483768805375,
                    0.9999997373138294,
                    0.5442138166006407,
                    0.9999997373138294,
                    0.7200961165080368,
                    0.9999997373138294,
                    0.4372483768805375,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.04693144675687698,
                    0.8395655047552425,
                    0.8395655047552425,
                    0.9999997373138294,
                    0.8395655047552425,
                    0.6210330306914035,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9681522788336918,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.7223703085727059,
                    0.02353676743785122,
                    0.19266295410826276,
                    0.4993454158594806,
                    0.9999997373138294,
                    0.878906678789298,
                    0.5115953505542589,
                    0.9999997373138294,
                    0.29234617948435815,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.19266295410826276,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.7200961165080368,
                    0.9999997373138294,
                    0.03371060943211578,
                    0.7200961165080368,
                    0.9999997373138294,
                    0.7200961165080368,
                    0.9550708068178158,
                    0.18171356404052635,
                    0.9999997373138294,
                    0.5306621528598132,
                    0.9999997373138294,
                    0.7200961165080368,
                    0.9999997373138294,
                    0.9850862188123334,
                    0.9999997373138294,
                    0.04693144675687698,
                    0.19266295410826276,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.8280732314616722,
                    0.9327511393735692,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.9999997373138294,
                    0.5115953505542589,
                    0.9999997373138294,
                    0.9999997373138294,
                ],
            },
            index=[f"gene{i}" for i in range(200)],
        )
        pd.testing.assert_frame_equal(res, ref, rtol=1e-4)

    def test_meta_de_nan(self):
        """check robustness of meta_de to NaN values"""
        idx = ["gene1", "gene2", "gene3"]
        res1 = DEResults(
            {
                "log2FoldChange": [1.0, np.nan, 3.0],
                "lfcSE": [0.5, np.nan, 0.5],
                "pvalue": [0.1, np.nan, 0.2],
            },
            index=idx,
        )
        res2 = DEResults(
            {
                "log2FoldChange": [2.0, 3.0, 4.0],
                "lfcSE": [0.5, 0.5, 0.5],
                "pvalue": [0.1, 0.15, 0.2],
            },
            index=idx,
        )
        res3 = DEResults(
            {
                "log2FoldChange": [3.0, np.nan, 5.0],
                "lfcSE": [0.5, np.nan, 0.5],
                "pvalue": [0.2, np.nan, 0.1],
            },
            index=idx,
        )
        meta1 = meta_de([res1, res2, res3])
        meta2 = meta_de([res1, res3])
        ref1 = pd.DataFrame(
            {
                "combined logFC (CI_L)": [
                    0.8684142716330685,
                    np.nan,
                    2.868414271633068,
                ],
                "combined logFC (CI_R)": [
                    3.1315857283669315,
                    np.nan,
                    5.131585728366931,
                ],
                "combined logFC": [2, np.nan, 4],
                "adjusted combined pval": [
                    0.13058835750961412,
                    0.15,
                    0.13058835750961412,
                ],
            },
            index=idx,
        )
        ref2 = pd.DataFrame(
            {
                "combined logFC (CI_L)": [
                    0.0400360524639638,
                    np.nan,
                    2.0400360524639636,
                ],
                "combined logFC (CI_R)": [
                    3.9599639475360364,
                    np.nan,
                    5.959963947536036,
                ],
                "combined logFC": [2, np.nan, 4],
                "adjusted combined pval": [
                    0.09824046010856295,
                    np.nan,
                    0.09824046010856295,
                ],
            },
            index=idx,
        )
        pd.testing.assert_frame_equal(meta1, ref1)
        pd.testing.assert_frame_equal(meta2, ref2)
