use crate::expression::compiler::config::CompilationConfig;
use crate::transform::TransformTrait;
use async_trait::async_trait;

use datafusion::arrow::array::RecordBatch;
use datafusion::prelude::DataFrame;
use datafusion_common::utils::SingleRowListArrayBuilder;
use datafusion_common::{DFSchema, ScalarValue};
use datafusion_expr::Expr;
use datafusion_functions_aggregate::expr_fn::{max, min};
use std::sync::Arc;
use vegafusion_common::column::unescaped_col;
use vegafusion_common::datatypes::to_numeric;
use vegafusion_common::error::{Result, ResultWithContext};
use vegafusion_core::proto::gen::transforms::Extent;
use vegafusion_core::task_graph::task_value::TaskValue;

#[async_trait]
impl TransformTrait for Extent {
    async fn eval(
        &self,
        sql_df: DataFrame,
        config: &CompilationConfig,
    ) -> Result<(DataFrame, Vec<TaskValue>)> {
        let output_values = if self.signal.is_some() {
            let (min_expr, max_expr) = min_max_exprs(self.field.as_str(), sql_df.schema())?;

            let extent_df = sql_df
                .clone()
                .aggregate(Vec::new(), vec![min_expr, max_expr])?;

            let logical_plan = extent_df.logical_plan().clone();
            let result_table = config.plan_executor.execute_plan(logical_plan).await?;
            let result_batch = result_table.to_record_batch()?;
            let extent_list = extract_extent_list(&result_batch)?;
            vec![extent_list]
        } else {
            Vec::new()
        };

        Ok((sql_df, output_values))
    }
}

fn min_max_exprs(field: &str, schema: &DFSchema) -> Result<(Expr, Expr)> {
    let field_col = unescaped_col(field);
    let min_expr = min(to_numeric(field_col.clone(), schema)?).alias("__min_val");
    let max_expr = max(to_numeric(field_col, schema)?).alias("__max_val");
    Ok((min_expr, max_expr))
}

fn extract_extent_list(batch: &RecordBatch) -> Result<TaskValue> {
    let min_val_array = batch
        .column_by_name("__min_val")
        .with_context(|| "No column named __min_val".to_string())?;
    let max_val_array = batch
        .column_by_name("__max_val")
        .with_context(|| "No column named __max_val".to_string())?;

    let min_val_scalar = ScalarValue::try_from_array(min_val_array, 0).unwrap();
    let max_val_scalar = ScalarValue::try_from_array(max_val_array, 0).unwrap();

    // Build two-element list of the extents
    let extent_list = TaskValue::Scalar(ScalarValue::List(Arc::new(
        SingleRowListArrayBuilder::new(ScalarValue::iter_to_array(vec![
            min_val_scalar,
            max_val_scalar,
        ])?)
        .with_nullable(true)
        .build_list_array(),
    )));
    Ok(extent_list)
}
