"""G-code generation functionality."""

import math
from pathlib import Path
from typing import List, TextIO

from microweldr.core.config import Config
from microweldr.core.models import WeldPath, WeldPoint


class GCodeGenerator:
    """Generator for G-code files from weld paths."""

    def __init__(self, config: Config) -> None:
        """Initialize G-code generator."""
        self.config = config

    def generate_file(
        self,
        weld_paths: List[WeldPath],
        output_path: str | Path,
        skip_bed_leveling: bool = False,
        margin_info: dict = None,
    ) -> None:
        """Generate G-code file from weld paths."""
        output_path = Path(output_path)

        with open(output_path, "w") as f:
            self._write_header(f, weld_paths)
            self._write_pre_calibration_heating(f)
            self._write_initialization(f, skip_bed_leveling)
            self._write_final_heating(f)
            self._write_user_pause(f, margin_info)
            self._write_welding_sequence(f, weld_paths)
            self._write_cooldown(f)

    def _write_header(self, f: TextIO, weld_paths: List[WeldPath]) -> None:
        """Write G-code header."""
        f.write("; Generated by MicroWeldr\n")
        f.write("; Prusa Core One Plastic Welding G-code\n")
        f.write(f"; Total paths: {len(weld_paths)}\n\n")

    def _write_pre_calibration_heating(self, f: TextIO) -> None:
        """Start bed heating before calibration for efficiency."""
        bed_temp = self.config.get("temperatures", "bed_temperature")
        use_chamber_heating = self.config.get(
            "temperatures", "use_chamber_heating", True
        )
        chamber_temp = self.config.get(
            "temperatures", "chamber_temperature", 35
        )  # Default 35°C

        f.write("; Start heating before calibration (efficient timing)\n")

        # Chamber heating (optional)
        if use_chamber_heating:
            f.write(f"; Heat chamber to {chamber_temp}°C (Core One)\n")
            f.write(f"M141 S{chamber_temp} ; Set chamber temperature\n")
            f.write(f"M191 S{chamber_temp} ; Wait for chamber temperature\n\n")
        else:
            f.write("; Chamber heating disabled (sensor not available)\n\n")

        # Start bed heating (don't wait yet - let it heat during calibration)
        f.write(f"; Start heating bed to {bed_temp}°C (heating during calibration)\n")
        f.write(f"M140 S{bed_temp} ; Set bed temperature (start heating)\n\n")

    def _write_initialization(self, f: TextIO, skip_bed_leveling: bool) -> None:
        """Write printer initialization commands."""
        layed_back_mode = self.config.get(
            "printer", "layed_back_mode", False
        )  # Default to upright mode (layed back is experimental)

        if layed_back_mode:
            # Print warning about layed back mode
            print(
                "⚠️  WARNING: Layed back mode is EXPERIMENTAL and may not work properly!"
            )
            print(
                "⚠️  Known issues: calibration conflicts, Z-axis problems, coordinate issues"
            )
            print(
                "⚠️  Recommendation: Set layed_back_mode = false for reliable operation"
            )
            print("⚠️  Continue at your own risk - manual intervention may be required")
            print()
            f.write("; Initialize printer (layed back mode - printer on its back!)\n")
            f.write("; IMPORTANT: Manually position print head before starting!\n")
            f.write("; Expected position: Rear right corner of bed\n")
            f.write(
                "; All positioning fully manual - no homing to avoid X/Y conflicts\n\n"
            )

            f.write("G90 ; Absolute positioning\n")
            f.write("M83 ; Relative extruder positioning\n")
            f.write("M84 S0 ; Disable stepper timeout for layed back operation\n")

            # Trust X/Y but will home Z during heating
            f.write(
                "G92 X0 Y0 ; Set current X/Y position as origin (manual positioning trusted)\n\n"
            )

            # Always skip bed leveling for layed back operation
            f.write("; Bed leveling disabled for layed back operation\n")
            f.write("; Manual bed preparation required (printer is relaxing)\n\n")
        else:
            # Standard operation mode
            f.write("; Initialize printer (standard operation mode)\n")
            f.write("G90 ; Absolute positioning\n")
            f.write("M83 ; Relative extruder positioning\n")
            f.write("G28 ; Home all axes\n\n")

            # Bed leveling (optional)
            if not skip_bed_leveling:
                f.write("; Bed leveling\n")
                f.write("G29 ; Auto bed leveling\n\n")
            else:
                f.write("; Bed leveling disabled\n\n")

    def _write_final_heating(self, f: TextIO) -> None:
        """Wait for bed temperature and heat nozzle."""
        bed_temp = self.config.get("temperatures", "bed_temperature")
        nozzle_temp = self.config.get("temperatures", "nozzle_temperature")
        layed_back_mode = self.config.get("printer", "layed_back_mode", False)

        # Z-axis calibration while bed heats up (layed back mode only)
        if layed_back_mode:
            f.write("; Z-axis calibration while bed heats up (efficient timing)\n")
            f.write(
                "; IMPORTANT: Manually position Z-axis at desired height before starting\n"
            )
            f.write(
                "; Skipping automatic Z-homing to avoid X/Y conflicts in layed back mode\n"
            )
            f.write(
                "G92 Z0 ; Set current Z position as zero reference (trust manual positioning)\n"
            )
            f.write(
                "G1 Z10 F150 ; Move Z to safe position slowly (no rush when layed back)\n\n"
            )

        # Now wait for bed temperature (should be ready or nearly ready)
        f.write(f"; Wait for bed to reach target temperature\n")
        f.write(f"M190 S{bed_temp} ; Wait for bed temperature\n\n")

        # Heat nozzle
        f.write(f"; Heat nozzle to {nozzle_temp}°C\n")
        f.write(f"M104 S{nozzle_temp} ; Set nozzle temperature\n")
        f.write(f"M109 S{nozzle_temp} ; Wait for nozzle temperature\n\n")

    def _write_user_pause(self, f: TextIO, margin_info: dict = None) -> None:
        """Write user pause for plastic sheet insertion."""
        f.write("; Pause for user to insert plastic sheets\n")

        # Create message with margin information if available
        if margin_info:
            message = f"Place film (margins F/B:{margin_info['front_back']}, L/R:{margin_info['left_right']})"
            # Truncate if too long for LCD
            if len(message) > 64:
                message = f"Place film (F/B:{margin_info['front_back']}, L/R:{margin_info['left_right']})"
        else:
            message = "Insert plastic sheets and press continue"

        f.write(f"M117 {message}\n")
        f.write("M0 ; Pause - Insert plastic sheets and press continue\n")
        f.write("M117 Starting welding sequence...\n\n")

    def _write_welding_sequence(self, f: TextIO, weld_paths: List[WeldPath]) -> None:
        """Write the main welding sequence."""
        move_height = self.config.get("movement", "move_height")
        z_speed = self.config.get("movement", "z_speed")
        travel_speed = self.config.get("movement", "travel_speed")

        f.write(f"G1 Z{move_height} F{z_speed} ; Move to safe height\n\n")

        current_nozzle_temp = self.config.get("temperatures", "nozzle_temperature")

        for path in weld_paths:
            f.write(f"; Processing path: {path.svg_id} (type: {path.weld_type})\n")

            if path.weld_type == "stop":
                # Handle stop points with custom messages
                message = path.pause_message or "Manual intervention required"
                # Clean message for LCD display (remove problematic characters)
                safe_message = (
                    message.replace('"', "'").replace(";", ",").replace("\n", " ")[:64]
                )  # Prusa LCD limit
                f.write(f"; User stop requested\n")
                f.write(f"M117 {safe_message}\n")
                f.write(f"M0 ; Pause for user action\n")
                f.write(f"M117 Continuing welding...\n\n")
                continue
            elif path.weld_type == "pipette":
                # Handle pipetting stops for microfluidic device filling
                message = path.pause_message or "Pipette filling required"
                # Clean message for LCD display (remove problematic characters)
                safe_message = (
                    message.replace('"', "'").replace(";", ",").replace("\n", " ")[:64]
                )  # Prusa LCD limit
                f.write(f"; Pipetting stop - fill pouch with pipette\n")
                f.write(f"M117 {safe_message}\n")
                f.write(f"M0 ; Pause for pipetting\n")
                f.write(f"M117 Continuing welding...\n\n")
                continue

            # Get settings for this weld type
            weld_config = self.config.get_section(f"{path.weld_type}_welds")

            # Check for custom temperature on this path
            target_temp = (
                path.custom_temp
                if path.custom_temp is not None
                else weld_config["weld_temperature"]
            )

            # Set temperature if different
            if target_temp != current_nozzle_temp:
                current_nozzle_temp = target_temp
                if path.custom_temp is not None:
                    f.write(
                        f"M104 S{current_nozzle_temp} ; Set custom temperature {current_nozzle_temp}°C\n"
                    )
                    f.write(
                        f"M109 S{current_nozzle_temp} ; Wait for custom temperature\n\n"
                    )
                else:
                    f.write(
                        f"M104 S{current_nozzle_temp} ; Set temperature for {path.weld_type} welds\n"
                    )
                    f.write(f"M109 S{current_nozzle_temp} ; Wait for temperature\n\n")

            # Process path with multi-pass welding
            self._write_multipass_welding(
                f, path, weld_config, move_height, travel_speed, z_speed
            )

            f.write("\n")

    def _write_multipass_welding(
        self,
        f: TextIO,
        path,
        weld_config: dict,
        move_height: float,
        travel_speed: int,
        z_speed: int,
    ) -> None:
        """Write multi-pass welding sequence for a path."""

        initial_spacing = weld_config["initial_dot_spacing"]
        final_spacing = weld_config["dot_spacing"]
        cooling_time = weld_config["cooling_time_between_passes"]

        # Calculate how many passes we need
        spacing_ratio = initial_spacing / final_spacing
        num_passes = max(1, int(math.log2(spacing_ratio)) + 1)

        f.write(
            f"; Multi-pass welding: {num_passes} passes from {initial_spacing}mm to {final_spacing}mm spacing\n"
        )

        # Generate all weld points for all passes
        all_passes_points = self._generate_multipass_points(
            path.points, initial_spacing, final_spacing, num_passes
        )

        # Execute each pass
        for pass_num, pass_points in enumerate(all_passes_points, 1):
            if not pass_points:
                continue

            f.write(f"; Pass {pass_num}/{num_passes}\n")

            for point in pass_points:
                # Move to position at safe height
                f.write(
                    f"G1 X{point.x:.3f} Y{point.y:.3f} Z{move_height} F{travel_speed}\n"
                )

                # Lower to weld height - use custom height if specified (path-level or point-level)
                weld_height = (
                    point.custom_height
                    if point.custom_height is not None
                    else (
                        path.custom_height
                        if path.custom_height is not None
                        else weld_config["weld_height"]
                    )
                )
                f.write(f"G1 Z{weld_height:.3f} F{z_speed}\n")

                # Dwell for welding - use custom dwell time if specified (path-level or point-level)
                dwell_time = (
                    point.custom_dwell
                    if point.custom_dwell is not None
                    else (
                        path.custom_dwell
                        if path.custom_dwell is not None
                        else weld_config["spot_dwell_time"]
                    )
                )
                dwell_ms = int(dwell_time * 1000)
                if point.custom_dwell is not None or path.custom_dwell is not None:
                    f.write(f"G4 P{dwell_ms} ; Custom dwell time {dwell_time}s\n")
                else:
                    f.write(f"G4 P{dwell_ms} ; Dwell for welding\n")

                # Raise to safe height
                f.write(f"G1 Z{move_height} F{z_speed}\n")

            # Cooling time between passes (except after the last pass)
            if pass_num < num_passes and cooling_time > 0:
                cooling_ms = int(cooling_time * 1000)
                f.write(f"G4 P{cooling_ms} ; Cooling time between passes\n")

            f.write(f"; End of pass {pass_num}\n")

    def _generate_multipass_points(
        self,
        original_points,
        initial_spacing: float,
        final_spacing: float,
        num_passes: int,
    ):
        """Generate points for each pass of multi-pass welding."""

        if num_passes == 1:
            return [original_points]

        # Create a continuous path from all points
        all_path_points = []
        for i in range(len(original_points) - 1):
            start = original_points[i]
            end = original_points[i + 1]

            # Calculate distance
            dx = end.x - start.x
            dy = end.y - start.y
            distance = math.sqrt(dx * dx + dy * dy)

            if distance == 0:
                continue

            # Generate points at final spacing along this segment
            num_points = max(1, int(distance / final_spacing))

            for j in range(num_points + 1):
                t = j / num_points if num_points > 0 else 0
                x = start.x + t * dx
                y = start.y + t * dy
                all_path_points.append((x, y, start.weld_type))

        # Now distribute these points across passes
        passes = [[] for _ in range(num_passes)]

        # First pass: every 2^(num_passes-1) point
        step = 2 ** (num_passes - 1)
        for i in range(0, len(all_path_points), step):
            x, y, weld_type = all_path_points[i]
            passes[0].append(WeldPoint(x, y, weld_type))

        # Subsequent passes: fill in between previous pass points
        for pass_num in range(1, num_passes):
            step = 2 ** (num_passes - 1 - pass_num)
            offset = step

            for i in range(offset, len(all_path_points), step * 2):
                if i < len(all_path_points):
                    x, y, weld_type = all_path_points[i]
                    passes[pass_num].append(WeldPoint(x, y, weld_type))

        return passes

    def _write_cooldown(self, f: TextIO) -> None:
        """Write cooldown and end sequence for Prusa Core One."""
        cooldown_temp = self.config.get("temperatures", "cooldown_temperature")
        use_chamber_heating = self.config.get(
            "temperatures", "use_chamber_heating", True
        )
        layed_back_mode = self.config.get("printer", "layed_back_mode", True)

        f.write("; Cool down (Core One)\n")
        f.write(f"M104 S{cooldown_temp} ; Cool nozzle\n")
        f.write(f"M140 S{cooldown_temp} ; Cool bed\n")

        if use_chamber_heating:
            f.write(f"M141 S0 ; Turn off chamber heating\n")

        # Skip homing in layed back mode to avoid conflicts
        if not layed_back_mode:
            f.write("G28 X Y ; Home X and Y\n")
        else:
            f.write("; Skipping X/Y homing in layed back mode (avoid conflicts)\n")

        f.write("M84 ; Disable steppers\n")
        f.write("; End of G-code\n")
