"""module to automatically wrap methods in autogenerated low-level code and re-expose them as high-level functions"""

import inspect
from typing import Optional
import warnings

from beartype import beartype
from box import Box

from deeporigin.auth import get_tokens
from deeporigin.config import get_value
from deeporigin.platform import Client
from deeporigin.platform.recording import RequestRecorder
from deeporigin.utils.constants import API_ENDPOINT
from deeporigin.utils.core import _get_method

warnings.filterwarnings(
    "ignore",
    module="pydantic._internal._fields",
)

# we're importing this here after the warnings are suppressed
# because importing this raises UserWarnings we don't want to show
import do_sdk_platform  # noqa: E402

# recorder instance used when a high-level Client has recording=True
_RECORDER = RequestRecorder()


@beartype
def _add_functions_to_module(
    module,
    api_name: str,
) -> list:
    """utility function to dynamically add functions to a module

    This function works by calling setattr on the module.

    Args:
        module (str): name of the module
        api_name (str): name of the API

    Returns:
        set of methods that were added
    """
    methods = _get_client_methods(
        _get_api_client(
            api_name=api_name,
            configure=False,
        )
    )

    sanitized_methods = []

    for method in methods:
        # clean up the name so that it's more readable
        # here we're removing the first word, which is the same as the api name
        sanitized_method_name = "_".join(method.split("_")[1:])

        sanitized_method_name = sanitized_method_name.replace(
            "_without_preload_content", ""
        ).lstrip("_")

        sanitized_methods.append(sanitized_method_name)

        # add this function as an attribute to this module
        # so that we can call it
        setattr(
            module,
            sanitized_method_name,
            _create_function(
                method_path=method,
                api_name=api_name,
            ),
        )

    return sanitized_methods


@beartype
def _get_api_client(
    *,
    api_name: str,
    configure: bool = True,
    token: Optional[str] = None,
    api_endpoint: Optional[str] = None,
):
    """return a configured client for the API we want to access

    Args:
        api_name (str): name of the API
        configure (bool): whether to configure the client with authentication

    Returns:
        configured client

    """

    if configure:
        if api_endpoint is None:
            env = get_value()["env"]
            api_endpoint = API_ENDPOINT[env]

        if token is None:
            token = get_tokens()["access"]

        configuration = do_sdk_platform.Configuration(
            host=api_endpoint,
            access_token=token,
        )

        client = do_sdk_platform.ApiClient(configuration=configuration)
    else:
        client = do_sdk_platform.ApiClient()

    api_class = getattr(do_sdk_platform, api_name)
    client = api_class(api_client=client)
    return client


@beartype
def _get_client_methods(client) -> set:
    """utility function to get methods from the client that return raw responses from the server"""
    methods = set(
        [
            attr
            for attr in dir(client)
            if callable(getattr(client, attr))
            and not attr.startswith("_")
            and "without_preload_content" in attr
        ]
    )

    return methods


def _create_function(
    *,
    method_path: str,
    api_name: str,
):
    """utility function the dynamically creates functions
    that wrap low-level functions in the DeepOrigin data API"""

    # we're constructing a client solely for the purposes
    # of inspecting its methods and extracting
    # function signatures. So we don't need any
    # authentication
    internal_client = _get_api_client(
        configure=False,
        api_name=api_name,
    )

    method = _get_method(internal_client, method_path)

    signature = inspect.signature(method)

    # we're not going to use this. deleting for clarity
    del internal_client

    def dynamic_function(
        *,
        client: Client = None,
        **kwargs,
    ):
        """dynamic function that wraps low-level functions in the DeepOrigin platform API"""

        if client is None:
            # no client was provided, so we'll create a new one
            # using config and api tokens cached on disk
            client = _get_api_client(
                api_name=api_name,
            )
            client_org_key = None
            recording_enabled = False
            recording_db_path = None
            client.is_mock = False

        else:
            if not isinstance(client, Client):
                raise ValueError(
                    "client must be a Client instance. Create a client instance from deeporigin.platform.Client"
                )
            # convert the DO Client to the low-level client
            client_org_key = client.org_key
            recording_enabled = getattr(client, "recording", False)
            recording_db_path = None

            if not client.is_mock:
                # convert the DO Client to the low-level client
                client = _get_api_client(
                    api_name=api_name,
                    token=client.token,
                    api_endpoint=client.api_endpoint,
                )
                client.is_mock = False

        method = _get_method(client, method_path)

        # Insert org_key if not present in kwargs, and
        # if it's required by the method
        if "org_key" in signature.parameters and (kwargs.get("org_key") is None):
            if client_org_key is None:
                kwargs["org_key"] = get_value()["org_key"]
            else:
                kwargs["org_key"] = client_org_key

        # call the low level API method
        _start_time = (
            inspect.currentframe()
        )  # dummy to avoid import time for time module here
        import time as _time  # local import to keep module import surface minimal

        t0 = _time.perf_counter_ns()
        response = method(**kwargs)

        if not client.is_mock:
            if 400 <= response.status < 600:
                content = response.read().decode("utf-8", errors="replace")

                raise ValueError(
                    f"HTTP request failed with status: {response.status} - {response.reason} - {content}"
                )

            if not isinstance(response, dict):
                response = response.json()

            # Prepare payload for recording before Box wrapping
            if isinstance(response, bool):
                payload_for_record = response
            elif isinstance(response, dict):
                if "data" in response.keys():
                    payload_for_record = response["data"]
                else:
                    payload_for_record = response
            elif isinstance(response, list):
                payload_for_record = response
            else:
                payload_for_record = None

            if recording_enabled:
                t1 = _time.perf_counter_ns()
                duration_ms = (t1 - t0) / 1_000_000
                try:
                    method_name = method_path
                    _RECORDER.record(
                        method=method_name,
                        kwargs=kwargs,
                        response=payload_for_record,
                        duration_ms=int(duration_ms),
                        db_path=recording_db_path,
                    )
                except Exception:
                    # Recording failures must not impact normal execution
                    print("Recording failed")
                    pass

        if isinstance(response, bool):
            return response

        elif isinstance(response, dict):
            if "data" in response.keys():
                response = response["data"]
                if isinstance(response, list):
                    response = [Box(item) for item in response]
                else:
                    response = Box(response)
            else:
                response = Box(response)

        elif isinstance(response, list):
            response = [Box(item) for item in response]
        else:
            raise NotImplementedError(f"Unexpected response type: {type(response)}")

        return response

    # attach the signature of the underlying method to the
    # function so that IDEs can display it properly
    dynamic_function.__signature__ = signature

    return dynamic_function
