# Copyright 2025 Luminary Cloud, Inc. All Rights Reserved.
from typing import Any, Iterator, Tuple
from google.protobuf.message import Message as _ProtoMessage


# Gives derived classes the ability to generate the code that would set the values of their members.
# Evidently this is most suited for dataclasses.
# Note that to allow recursion, the lines generated by this class DO NOT contain the name of the
# object being generated, e.g. they will be of the form " = Type()", ".field = val".
# Derived classes can use _to_code_helper to insert an adequate object name.
class CodeRepr:
    def _to_code(self, hide_defaults: bool = True, use_tmp_objs: bool = True) -> str:
        # Returns the code representation of a value with special cases for enums and classes
        # that also implement "_to_code" (thus triggering recursion), otherwise uses the default
        # conversion to string.
        def to_string(val: Any) -> str:
            if "<enum " in str(type(val)):
                str_val = val.__repr__()
                return str_val.split(": ")[0][1:]
            try:
                return val._to_code(hide_defaults, use_tmp_objs)
            except:
                return str(val)

        # Returns the code to set the value of a given field of the class.
        def generate(field: str, val: Any) -> str:
            str_val = to_string(val)
            code = ""
            if issubclass(type(val), _ProtoMessage):
                # Special case for protos.
                # TODO(pedro): This is not working 100%.
                lines = str_val.split("\n")
                short = len(lines) <= 4
                code += f".{field} = {type(val).__name__}("
                if not short:
                    # Default and set each field vs init fields in constructor.
                    code += ")\n"
                needs_warning = False
                for line in lines:
                    if ":" in line:
                        if short:
                            mod_line = line.replace(": ", "=")
                            code += f"{mod_line}, "
                        else:
                            mod_line = line.replace(":", " =")
                            code += f".{field}.{mod_line}\n"
                    elif line:
                        needs_warning = True
                if short:
                    code = code.strip(", ") + ")\n"
                if needs_warning:
                    code = f"# NOTE: Google protobuf types are not fully supported yet.\n{code}"

            elif str_val.endswith("\n"):
                # Special case for fields with their own "_to_code".
                for line in str_val.split("\n"):
                    if line.startswith("new_") or line.startswith("#"):
                        code += f"{line}\n"
                    elif line.startswith(".new_"):
                        code += f"{line[1:]}\n"
                    elif line:
                        code += f".{field}{line}\n"
            elif isinstance(val, str):
                code += f'.{field} = "{str_val}"\n'
            else:
                code += f".{field} = {str_val}\n"
            return code

        # Returns the code representation of map keys, with special cases for enums and strings.
        def map_key(key: Any) -> str:
            if isinstance(key, str):
                return f'"{key}"'
            if "<enum " in str(type(key)):
                str_key = key.__repr__()
                return str_key.split(": ")[0][1:]
            return str(key)

        # Helper to generate short names for temporary objects that are built and inserted into
        # containers.
        def short_name(field: str) -> str:
            if not "_" in field:
                return field
            return "".join([word[0] for word in field.split("_")])

        # Create the top level object (default constructed), we do some manipulation of module paths
        # (extracted from the type info) to make them consistent with the aliased imports in some
        # __init__ files.
        if "luminarycloud" in str(type(self)):
            path = (
                str(type(self))
                .split("'")[1]
                .replace("luminarycloud.params.simulation", "sim_params")
                .replace("luminarycloud.outputs.output_definitions", "outputs")
            )
            names = path.split(".")
            if names[-2].endswith("_"):
                path = path.replace(names[-2] + ".", "")
            code = f" = {path}()\n"
        else:
            code = f" = {type(self).__name__}()\n"

        default = type(self)()
        for field, val in self._iterate_fields_for_code_gen():
            # Skip default values.
            if hide_defaults and val == vars(default).get(field):
                continue
            field_short_name = short_name(field)
            if isinstance(val, dict):
                code += "# Clear the map and insert new items.\n"
                code += f".{field} = {{}}\n"
                # Sort dictionary items by key to ensure deterministic ordering
                for key, value in sorted(val.items(), key=lambda x: map_key(x[0])):
                    # Simplify the code for maps of built-in types by not creating a temporary
                    # instance.
                    if isinstance(value, (bool, int, float)):
                        code += f".{field}[{map_key(key)}] = {value}\n"
                    elif isinstance(value, str):
                        code += f'.{field}[{map_key(key)}] = "{value}"\n'
                    else:
                        code += f"# Insert a new {type(value).__name__}.\n"
                        if use_tmp_objs:
                            code += generate(f"new_{field_short_name}", value)
                            code += f".{field}[{map_key(key)}] = new_{field_short_name}\n"
                        else:
                            code += generate(f"{field}[{map_key(key)}]", value)
            elif isinstance(val, list):
                # Similar to maps, for classes with more complex representations create temporaries
                # and append, otherwise use the default list representation (which also covers
                # empty lists).
                if len(val) > 0 and to_string(val[0]).endswith("\n"):
                    code += f"# Clear the list of {type(val[0]).__name__} and append new items.\n"
                    code += f".{field} = []\n"
                    for i, value in enumerate(val):
                        code += f"# Append a new {type(value).__name__}.\n"
                        if use_tmp_objs:
                            code += generate(f"new_{field_short_name}", value)
                            code += f".{field}.append(new_{field_short_name})\n"
                        else:
                            code += f".{field}.append(None)\n"
                            code += generate(f"{field}[{i}]", value)
                else:
                    code += f".{field} = {to_string(val)}\n"
            else:
                code += generate(field, val)
        return code

    # All fields yielded here will be included in code generation. By default, we iterate over all
    # "public" instance variables (i.e. ones that do not begin with "_") and properties, but a class
    # can override this if it's needs to do something funny like conditionally hide fields.
    def _iterate_fields_for_code_gen(self) -> Iterator[Tuple[str, Any]]:
        # Yield all instance variables that are not "private"
        # Sort by field name to ensure deterministic ordering
        for item in sorted(vars(self).items()):
            if item[0].startswith("_"):
                continue
            yield item

        # Yield all properties
        for name in sorted(dir(self)):
            if isinstance(getattr(type(self), name, None), property):
                yield (name, getattr(self, name))

    # See the class description.
    def _to_code_helper(
        self, obj_name: str = "obj", hide_defaults: bool = True, use_tmp_objs: bool = True
    ) -> str:
        code = ""
        for line in self._to_code(hide_defaults, use_tmp_objs).split("\n"):
            if line:
                if line.startswith(".new_"):
                    code += f"{line[1:]}\n"
                elif line[0] == "." or line.startswith(" = "):
                    code += f"{obj_name}{line}\n"
                else:
                    # Handles comments and temporary objects.
                    code += f"{line}\n"
        return code
