# Copyright 2025 Luminary Cloud, Inc. All Rights Reserved.

from .._client import get_default_client
from .._proto.api.v0.luminarycloud.output_definition import output_definition_pb2 as outdefpb
from .._proto.frontend.output import output_pb2 as feoutputpb
from .._proto.output import output_pb2 as outputpb
from dataclasses import dataclass, field
from ..types import Vector3
from typing import cast, overload
from ..enum import (
    CalculationType,
    MomentConventionType,
    OutputDefinitionIncludes,
    QuantityType,
    ResidualType,
    SpaceAveragingType,
    Vector3Component,
    VolumeReductionType,
)
from .._helpers import CodeRepr


# CalculationType enum comes from the simulation protobuf, but that's just a copy-paste of the
# CalculationType enum in the frontend output protobuf. The OutputNode proto needs the frontend
# output proto type, so this helps us appease the type checker with a type cast.
def _proto_calc_type(calc_type: CalculationType) -> feoutputpb.CalculationType.ValueType:
    return cast(feoutputpb.CalculationType.ValueType, calc_type)


# TODO: make a simpler way to create Derived definitions, like from a string expression


@dataclass
class TrailingAverageConfig(CodeRepr):
    """Configuration for a trailing average calculation of an OutputDefinition."""

    averaging_iterations: int = 0
    "Number of trailing iterations to average over."


@dataclass
class ConvergenceMonitoringConfig(CodeRepr):
    """Configuration for a convergence monitoring calculation of an OutputDefinition."""

    averaging_iterations: int = 0
    "Number of trailing iterations to average over."
    iterations_to_consider: int = 0
    "Number of iterations to consider for convergence analysis."


@dataclass
class OutputDefinitionInclusions(CodeRepr):
    """Specifies which values should be calculated for an OutputDefinition."""

    base_value: bool = True
    "Whether to include the base value, i.e. the quantity itself."
    coefficient: bool = False
    "Whether to compute the quantity coefficient. Only valid for forces and moments, ignored for other quantity types."
    trailing_average: TrailingAverageConfig | None = None
    "If present, computes a trailing average over the quantity."
    convergence_monitoring: ConvergenceMonitoringConfig | None = None
    "If present, computes a convergence monitoring criterion for the quantity."

    def _apply_to_proto(self, proto: feoutputpb.OutputNode) -> None:
        if self.base_value:
            proto.include[OutputDefinitionIncludes.BASE] = True
        if self.coefficient:
            proto.include[OutputDefinitionIncludes.COEFFICIENT] = True
        if self.trailing_average is not None:
            proto.trail_avg_iters = self.trailing_average.averaging_iterations
            if self.base_value:
                proto.include[OutputDefinitionIncludes.TIME_AVERAGE] = True
            if self.coefficient:
                proto.include[OutputDefinitionIncludes.COEFFICIENT_TIME_AVERAGE] = True
        if self.convergence_monitoring is not None:
            proto.average_iters = self.convergence_monitoring.averaging_iterations
            proto.analysis_iters = self.convergence_monitoring.iterations_to_consider
            proto.include[OutputDefinitionIncludes.MAX_DEV] = True

    @classmethod
    def _from_proto(cls, proto: feoutputpb.OutputNode) -> "OutputDefinitionInclusions":
        instance = cls(
            base_value=proto.include.get(OutputDefinitionIncludes.BASE, False),
            coefficient=proto.include.get(OutputDefinitionIncludes.COEFFICIENT, False),
        )
        if proto.trail_avg_iters > 0 and (
            proto.include.get(OutputDefinitionIncludes.TIME_AVERAGE)
            or proto.include.get(OutputDefinitionIncludes.COEFFICIENT_TIME_AVERAGE)
        ):
            instance.trailing_average = TrailingAverageConfig(
                averaging_iterations=proto.trail_avg_iters,
            )
        if (
            proto.average_iters > 0
            and proto.analysis_iters > 0
            and proto.include.get(OutputDefinitionIncludes.MAX_DEV)
        ):
            instance.convergence_monitoring = ConvergenceMonitoringConfig(
                averaging_iterations=proto.average_iters,
                iterations_to_consider=proto.analysis_iters,
            )
        return instance


@dataclass
class SurfaceAverageOutputDefinition(CodeRepr):
    """An output that is calculated as an average over surfaces."""

    id: str = ""
    "Unique identifier for the definition. Will be automatically generated by the server for new definitions."
    name: str = ""
    "Display name for the definition."
    quantity: QuantityType = QuantityType.UNSPECIFIED
    "Type of quantity to measure."
    surfaces: list[str] = field(default_factory=list)
    "List of surfaces across which to calculate the output."
    out_surfaces: list[str] = field(default_factory=list)
    "For a `calc_type` of `CalculationType.DIFFERENCE`, a list of surfaces whose aggregate will be subtracted from the aggregate of the `surfaces` list."
    calc_type: CalculationType = CalculationType.AGGREGATE
    "Type of calculation to perform across all surfaces."
    reference_frame_id: str = ""
    "Reference frame to use for directional quantities. Empty string means use the global reference frame."
    include: OutputDefinitionInclusions = field(default_factory=OutputDefinitionInclusions)
    "Configuration for what values to include."
    space_averaging_type: SpaceAveragingType = SpaceAveragingType.AREA
    "Type of spatial averaging to use."
    vector_component: Vector3Component = Vector3Component.UNSPECIFIED
    "For vector quantities, the component to measure."

    def _to_proto(self) -> feoutputpb.OutputNode:
        proto = feoutputpb.OutputNode(
            id=self.id,
            name=self.name,
            in_surfaces=self.surfaces,
            out_surfaces=self.out_surfaces,
            calc_type=_proto_calc_type(self.calc_type),
            frame_id=self.reference_frame_id,
            surface_average=feoutputpb.SurfaceAverageNode(
                quantity_type=self.quantity.value,
                vector_component=self.vector_component.value,
                props=outputpb.SurfaceAverageProperties(
                    averaging_type=self.space_averaging_type.value
                ),
            ),
        )
        self.include._apply_to_proto(proto)
        return proto

    @classmethod
    def _from_proto(cls, proto: feoutputpb.OutputNode) -> "SurfaceAverageOutputDefinition":
        return cls(
            id=proto.id,
            name=proto.name,
            quantity=QuantityType(proto.surface_average.quantity_type),
            surfaces=list(proto.in_surfaces),
            out_surfaces=list(proto.out_surfaces),
            calc_type=CalculationType(proto.calc_type),
            reference_frame_id=proto.frame_id,
            include=OutputDefinitionInclusions._from_proto(proto),
            space_averaging_type=SpaceAveragingType(proto.surface_average.props.averaging_type),
            vector_component=Vector3Component(proto.surface_average.vector_component),
        )


@dataclass
class ForceOutputDefinition(CodeRepr):
    """A force or moment output."""

    id: str = ""
    "Unique identifier for the definition. Will be automatically generated by the server for new definitions."
    name: str = ""
    "Display name for the definition."
    quantity: QuantityType = QuantityType.UNSPECIFIED
    "Type of quantity to measure."
    surfaces: list[str] = field(default_factory=list)
    "List of surfaces across which to calculate the output."
    calc_type: CalculationType = CalculationType.AGGREGATE
    "Type of calculation to perform across all surfaces."
    reference_frame_id: str = ""
    "Reference frame to use for directional quantities. Empty string means use the global reference frame."
    include: OutputDefinitionInclusions = field(default_factory=OutputDefinitionInclusions)
    "Configuration for what values to include."
    force_direction: Vector3 = field(default_factory=Vector3)
    "Direction vector for force calculation."
    moment_center: Vector3 = field(default_factory=Vector3)
    "Center point for moment calculations."
    porous: bool = False
    "Whether the surfaces involved are porous."
    moment_convention_type: MomentConventionType = MomentConventionType.BODY_FRAME
    "Frame type for moment calculations (body or stability frame)."
    # TODO: it seems clumsy to make the user set `porous` correctly. we could probably set it
    # automatically based on the set of surfaces selected.

    def _to_proto(self) -> feoutputpb.OutputNode:
        proto = feoutputpb.OutputNode(
            id=self.id,
            name=self.name,
            in_surfaces=self.surfaces,
            calc_type=_proto_calc_type(self.calc_type),
            frame_id=self.reference_frame_id,
            force=feoutputpb.ForceNode(
                quantity_type=self.quantity.value,
                props=outputpb.ForceProperties(
                    force_direction=self.force_direction._to_ad_proto(),
                    moment_center=self.moment_center._to_ad_proto(),
                    porous=self.porous,
                    moment_convention_type=self.moment_convention_type.value,
                ),
            ),
        )
        self.include._apply_to_proto(proto)
        return proto

    @classmethod
    def _from_proto(cls, proto: feoutputpb.OutputNode) -> "ForceOutputDefinition":
        return cls(
            id=proto.id,
            name=proto.name,
            quantity=QuantityType(proto.force.quantity_type),
            surfaces=list(proto.in_surfaces),
            calc_type=CalculationType(proto.calc_type),
            reference_frame_id=proto.frame_id,
            include=OutputDefinitionInclusions._from_proto(proto),
            force_direction=Vector3.from_ad_proto(proto.force.props.force_direction),
            moment_center=Vector3.from_ad_proto(proto.force.props.moment_center),
            porous=proto.force.props.porous,
            moment_convention_type=MomentConventionType(proto.force.props.moment_convention_type),
        )


@dataclass
class ResidualOutputDefinition(CodeRepr):
    """Configuration of simulation residuals."""

    id: str = ""
    "Unique identifier for the definition. Will be automatically generated by the server for new definitions."
    name: str = ""
    "Display name for the definition."
    include: dict[QuantityType, bool] = field(default_factory=dict)
    "Which residuals to include."
    residual_type: ResidualType = ResidualType.RELATIVE
    "Type of residual calculation."
    physics_id: str = ""
    "Which physics these residuals apply to."

    def _to_proto(self) -> feoutputpb.OutputNode:
        return feoutputpb.OutputNode(
            id=self.id,
            name=self.name,
            residual=feoutputpb.ResidualNode(
                props=outputpb.ResidualProperties(
                    type=self.residual_type.value,
                ),
                res_enabled={k: v for k, v in self.include.items()},
                physics_id=self.physics_id,
            ),
        )

    @classmethod
    def _from_proto(cls, proto: feoutputpb.OutputNode) -> "ResidualOutputDefinition":
        return cls(
            id=proto.id,
            name=proto.name,
            include={QuantityType(k): v for k, v in proto.residual.res_enabled.items()},
            residual_type=ResidualType(proto.residual.props.type),
            physics_id=proto.residual.physics_id,
        )


@dataclass
class InnerIterationOutputDefinition(CodeRepr):
    """The presence of this definition enables reporting of inner iteration counts for transient simulations."""

    id: str = ""
    "Unique identifier for the definition. Will be automatically generated by the server for new definitions."
    name: str = ""
    "Display name for the definition."

    def _to_proto(self) -> feoutputpb.OutputNode:
        return feoutputpb.OutputNode(
            id=self.id,
            name=self.name,
            basic=feoutputpb.BasicNode(
                quantity_type=QuantityType.INNER_ITERATION_COUNT.value,
                props=outputpb.BasicType(),
            ),
        )

    @classmethod
    def _from_proto(cls, proto: feoutputpb.OutputNode) -> "InnerIterationOutputDefinition":
        return cls(
            id=proto.id,
            name=proto.name,
        )


@dataclass
class PointProbeOutputDefinition(CodeRepr):
    """An output that is calculated at specific points in the domain."""

    id: str = ""
    "Unique identifier for the definition. Will be automatically generated by the server for new definitions."
    name: str = ""
    "Display name for the definition."
    quantity: QuantityType = QuantityType.UNSPECIFIED
    "Type of quantity to measure."
    points: list[str] = field(default_factory=list)
    "List of points for which to calculate the output."
    include: OutputDefinitionInclusions = field(default_factory=OutputDefinitionInclusions)
    "Configuration for what values to include."
    vector_component: Vector3Component = Vector3Component.UNSPECIFIED
    "For vector quantities, the component to measure."

    def _to_proto(self) -> feoutputpb.OutputNode:
        proto = feoutputpb.OutputNode(
            id=self.id,
            name=self.name,
            in_surfaces=self.points,
            point_probe=feoutputpb.PointProbeNode(
                quantity_type=self.quantity.value,
                vector_component=self.vector_component.value,
                props=outputpb.PointProbeType(),
            ),
        )
        self.include._apply_to_proto(proto)
        return proto

    @classmethod
    def _from_proto(cls, proto: feoutputpb.OutputNode) -> "PointProbeOutputDefinition":
        return cls(
            id=proto.id,
            name=proto.name,
            quantity=QuantityType(proto.point_probe.quantity_type),
            points=list(proto.in_surfaces),
            include=OutputDefinitionInclusions._from_proto(proto),
            vector_component=Vector3Component(proto.point_probe.vector_component),
        )


@dataclass
class VolumeReductionOutputDefinition(CodeRepr):
    """An output that is calculated as an aggregation over volumes."""

    id: str = ""
    "Unique identifier for the definition. Will be automatically generated by the server for new definitions."
    name: str = ""
    "Display name for the definition."
    quantity: QuantityType = QuantityType.UNSPECIFIED
    "Type of quantity to measure."
    volumes: list[str] = field(default_factory=list)
    "List of volumes across which to calculate the output."
    include: OutputDefinitionInclusions = field(default_factory=OutputDefinitionInclusions)
    "Configuration for what values to include."
    calc_type: CalculationType = CalculationType.AGGREGATE
    "Type of calculation to perform across all volumes."
    volume_reduction_type: VolumeReductionType = VolumeReductionType.AVERAGE
    "Type of reduction to perform for each volume."
    vector_component: Vector3Component = Vector3Component.UNSPECIFIED
    "For vector quantities, the component to measure."

    def _to_proto(self) -> feoutputpb.OutputNode:
        proto = feoutputpb.OutputNode(
            id=self.id,
            name=self.name,
            in_surfaces=self.volumes,
            calc_type=_proto_calc_type(self.calc_type),
            volume_reduction=feoutputpb.VolumeReductionNode(
                quantity_type=self.quantity.value,
                vector_component=self.vector_component.value,
                props=outputpb.VolumeReductionProperties(
                    reduction_type=self.volume_reduction_type.value,
                ),
            ),
        )
        self.include._apply_to_proto(proto)
        return proto

    @classmethod
    def _from_proto(cls, proto: feoutputpb.OutputNode) -> "VolumeReductionOutputDefinition":
        return cls(
            id=proto.id,
            name=proto.name,
            quantity=QuantityType(proto.volume_reduction.quantity_type),
            volumes=list(proto.in_surfaces),
            include=OutputDefinitionInclusions._from_proto(proto),
            calc_type=CalculationType(proto.calc_type),
            volume_reduction_type=VolumeReductionType(proto.volume_reduction.props.reduction_type),
            vector_component=Vector3Component(proto.volume_reduction.vector_component),
        )


@dataclass
class DerivedOutputDefinition(CodeRepr):
    """An output that is expressed as a function of other output definitions."""

    @dataclass
    class ExpressionElement_Dependency:
        """A dependency on another output definition."""

        id: str = ""
        "ID of the output definition to depend on."
        include: OutputDefinitionIncludes = OutputDefinitionIncludes.BASE
        "Which particular OutputDefinition value to depend on."

        def _to_proto(self) -> feoutputpb.ExpressionElement:
            return feoutputpb.ExpressionElement(
                dependency=feoutputpb.DerivedNodeDependency(
                    id=self.id,
                    include=self.include.value,
                )
            )

        @classmethod
        def _from_proto(
            cls, proto: feoutputpb.ExpressionElement
        ) -> "DerivedOutputDefinition.ExpressionElement_Dependency":
            return cls(
                id=proto.dependency.id,
                include=OutputDefinitionIncludes(proto.dependency.include),
            )

    @dataclass
    class ExpressionElement_Substring:
        """A literal substring in the expression."""

        substring: str = ""
        "The substring"

        def _to_proto(self) -> feoutputpb.ExpressionElement:
            return feoutputpb.ExpressionElement(
                substring=self.substring,
            )

        @classmethod
        def _from_proto(
            cls, proto: feoutputpb.ExpressionElement
        ) -> "DerivedOutputDefinition.ExpressionElement_Substring":
            return cls(
                substring=proto.substring,
            )

    id: str = ""
    "Unique identifier for the definition. Will be automatically generated by the server for new definitions."
    name: str = ""
    "Display name for the definition."
    include: OutputDefinitionInclusions = field(default_factory=OutputDefinitionInclusions)
    "Configuration for what values to include."
    elements: list[ExpressionElement_Dependency | ExpressionElement_Substring] = field(
        default_factory=list
    )
    "Ordered list of expression elements."
    errors: list[str] = field(default_factory=list)
    "Validation errors"

    def _to_proto(self) -> feoutputpb.OutputNode:
        proto = feoutputpb.OutputNode(
            id=self.id,
            name=self.name,
            derived=feoutputpb.DerivedNode(
                elements=[el._to_proto() for el in self.elements],
                errors=self.errors,
            ),
        )
        self.include._apply_to_proto(proto)
        return proto

    @classmethod
    def _from_proto(cls, proto: feoutputpb.OutputNode) -> "DerivedOutputDefinition":
        return cls(
            id=proto.id,
            name=proto.name,
            include=OutputDefinitionInclusions._from_proto(proto),
            elements=[
                (
                    DerivedOutputDefinition.ExpressionElement_Dependency._from_proto(el)
                    if el.dependency
                    else DerivedOutputDefinition.ExpressionElement_Substring._from_proto(el)
                )
                for el in proto.derived.elements
            ],
            errors=list(proto.derived.errors),
        )


AnyOutputDefinitionType = (
    SurfaceAverageOutputDefinition
    | ForceOutputDefinition
    | ResidualOutputDefinition
    | InnerIterationOutputDefinition
    | PointProbeOutputDefinition
    | VolumeReductionOutputDefinition
    | DerivedOutputDefinition
)


def _specific_output_definition_type_from_proto(
    proto: feoutputpb.OutputNode,
) -> AnyOutputDefinitionType:
    if proto.HasField("surface_average"):
        return SurfaceAverageOutputDefinition._from_proto(proto)
    elif proto.HasField("force"):
        return ForceOutputDefinition._from_proto(proto)
    elif proto.HasField("residual"):
        return ResidualOutputDefinition._from_proto(proto)
    elif (
        proto.HasField("basic")
        and proto.basic.quantity_type == QuantityType.INNER_ITERATION_COUNT.value
    ):
        return InnerIterationOutputDefinition._from_proto(proto)
    elif proto.HasField("point_probe"):
        return PointProbeOutputDefinition._from_proto(proto)
    elif proto.HasField("derived"):
        return DerivedOutputDefinition._from_proto(proto)
    elif proto.HasField("volume_reduction"):
        return VolumeReductionOutputDefinition._from_proto(proto)
    else:
        raise ValueError(f"Un-mappable output definition: {proto}")


@overload
def create_output_definition(
    simulation_template_id: str,
    output_definition: SurfaceAverageOutputDefinition,
) -> SurfaceAverageOutputDefinition: ...


@overload
def create_output_definition(
    simulation_template_id: str,
    output_definition: ForceOutputDefinition,
) -> ForceOutputDefinition: ...


@overload
def create_output_definition(
    simulation_template_id: str,
    output_definition: ResidualOutputDefinition,
) -> ResidualOutputDefinition: ...


@overload
def create_output_definition(
    simulation_template_id: str,
    output_definition: InnerIterationOutputDefinition,
) -> InnerIterationOutputDefinition: ...


@overload
def create_output_definition(
    simulation_template_id: str,
    output_definition: PointProbeOutputDefinition,
) -> PointProbeOutputDefinition: ...


@overload
def create_output_definition(
    simulation_template_id: str,
    output_definition: VolumeReductionOutputDefinition,
) -> VolumeReductionOutputDefinition: ...


@overload
def create_output_definition(
    simulation_template_id: str,
    output_definition: DerivedOutputDefinition,
) -> DerivedOutputDefinition: ...


def create_output_definition(
    simulation_template_id: str, output_definition: AnyOutputDefinitionType
) -> AnyOutputDefinitionType:
    """
    Create an output definition for a simulation template.

    Parameters
    ----------
    simulation_template_id : str
        Simulation template in which to create the output definition.
    output_definition : AnyOutputDefinitionType
        Output definition to create.
    """

    req = outdefpb.CreateOutputDefinitionRequest(
        output_definition=output_definition._to_proto(),
        simulation_template_id=simulation_template_id,
    )
    res = get_default_client().CreateOutputDefinition(req)
    return _specific_output_definition_type_from_proto(res.output_definition)


def get_output_definition(
    simulation_template_id: str,
    id: str,
) -> AnyOutputDefinitionType:
    """
    Get an output definition by ID.

    Parameters
    ----------
    simulation_template_id : str
        ID of the simulation template that owns the output definition.
    id : str
        ID of the output definition to retrieve.
    """
    req = outdefpb.GetOutputDefinitionRequest(id=id, simulation_template_id=simulation_template_id)
    res = get_default_client().GetOutputDefinition(req)
    return _specific_output_definition_type_from_proto(res.output_definition)


def list_output_definitions(simulation_template_id: str) -> list[AnyOutputDefinitionType]:
    """
    List output definitions for a simulation template.

    Parameters
    ----------
    simulation_template_id : str
        ID of the simulation template to list output definitions for.
    """
    req = outdefpb.ListOutputDefinitionsRequest(simulation_template_id=simulation_template_id)
    res = get_default_client().ListOutputDefinitions(req)
    return [_specific_output_definition_type_from_proto(od) for od in res.output_definitions]


@overload
def update_output_definition(
    simulation_template_id: str,
    id: str,
    output_definition: SurfaceAverageOutputDefinition,
) -> SurfaceAverageOutputDefinition: ...


@overload
def update_output_definition(
    simulation_template_id: str,
    id: str,
    output_definition: ForceOutputDefinition,
) -> ForceOutputDefinition: ...


@overload
def update_output_definition(
    simulation_template_id: str,
    id: str,
    output_definition: ResidualOutputDefinition,
) -> ResidualOutputDefinition: ...


@overload
def update_output_definition(
    simulation_template_id: str,
    id: str,
    output_definition: InnerIterationOutputDefinition,
) -> InnerIterationOutputDefinition: ...


@overload
def update_output_definition(
    simulation_template_id: str,
    id: str,
    output_definition: PointProbeOutputDefinition,
) -> PointProbeOutputDefinition: ...


@overload
def update_output_definition(
    simulation_template_id: str,
    id: str,
    output_definition: VolumeReductionOutputDefinition,
) -> VolumeReductionOutputDefinition: ...


@overload
def update_output_definition(
    simulation_template_id: str,
    id: str,
    output_definition: DerivedOutputDefinition,
) -> DerivedOutputDefinition: ...


def update_output_definition(
    simulation_template_id: str,
    id: str,
    output_definition: AnyOutputDefinitionType,
) -> AnyOutputDefinitionType:
    """
    Update an output definition.

    Parameters
    ----------
    simulation_template_id : str
        ID of the simulation template that owns the output definition.
    id : str
        ID of the output definition to update.
    output_definition : AnyOutputDefinitionType
        Updated output definition. The ID of this output definition must be present in the set of output definitions
        owned by the specified simulation template.
    """

    req = outdefpb.UpdateOutputDefinitionRequest(
        id=id,
        output_definition=output_definition._to_proto(),
        simulation_template_id=simulation_template_id,
    )
    res = get_default_client().UpdateOutputDefinition(req)
    return _specific_output_definition_type_from_proto(res.output_definition)


def delete_output_definition(simulation_template_id: str, id: str) -> None:
    """
    Delete an output definition.

    Parameters
    ----------
    simulation_template_id : str
        ID of the simulation template that owns the output definition.
    id : str
        ID of the output definition to delete.
    """
    req = outdefpb.DeleteOutputDefinitionRequest(
        id=id, simulation_template_id=simulation_template_id
    )
    get_default_client().DeleteOutputDefinition(req)
