/**
 *  @file BasicDefs.h
 *
 *  @brief Top-level definitions needed for the ADH code
 *
 *  Created on: Oct 3, 2014
 *      Author: lyard
 */

#ifndef BASIC_DEFS_H_
#define BASIC_DEFS_H_

//define the basic atomic types: int32, ...
#include <google/protobuf/stubs/common.h>

#include "CoreMessages.pb.h"

// ColoredOutput has been moved to a separate header so that ConfigService can be built without ADHCore
#include "ColoredOutput.h"

#include <memory>

//this is to access atomic types, e.g. int32, int16...
//another solution would to redefine them, but some versions of gcc have trouble to figure out that both definitions are the same -> conflicts
//using namespace google::protobuf;


typedef google::protobuf::int8  int8;
typedef google::protobuf::int16 int16;
typedef google::protobuf::int32 int32;
typedef google::protobuf::int64 int64;
typedef google::protobuf::uint8  uint8;
typedef google::protobuf::uint16 uint16;
typedef google::protobuf::uint32 uint32;
typedef google::protobuf::uint64 uint64;


/** @namespace ADH
 *  @brief all structures definitions are in Protocol Buffer IDL in the data_model directory
 *
 *  To use the files, one should include the header file(s) generated by the meta compiler.
 *  To do this, use "#include <FileName>.pb.h" where <FileName> corresponds to the desired IDL
 *  and replacing .proto with .pb.h (e.g. #include "R1v1.pb.h" for the definition in R1v1.proto),
 */

namespace ADH
{
/** @namespace ADH::Core
 *  @brief base functions of the framework are here, along with a few helper functions
 *
 *  @todo mode the helper functions to a more suitable namespace (utils maybe ?)
 */
namespace Core
{
    //! @brief utility for easily getting current time
    unsigned long long getTimeUSec();

    //! @brief converts a number of bytes to a string representation
    std::string bytesToString(uint64 bytes);

    /** @brief Figure out the enum type corresponding to the input message
     *
     *  This function is static so that it can be used elsewhere without the need to instantiate.
     *
     *  @param message the message which actual type should be figured out
     *  @return the enum corresponding to this message type
     */
    MessageType extractMessageType(const google::protobuf::Message& message);

    /** @brief the opposise of extractMessageType: create a message structure from its type
     *
     *  @param the type of the desired message
     *  @return a base message class, of the desired type
     */
    google::protobuf::Message* newMessageFromType(MessageType type);

    //! @brief Helper function to print the type of messages
    std::string convert_to_string(MessageType type);

    //! @brief Get the build date and time in ISO format
    std::string getIsoBuildTimestamp();

} //namespace Core

} //namespace ADH

#endif /* BASIC_DEFS_H_ */
