#ifndef Huffman_H
#define Huffman_H

/**
 *
 * @file Huffman.h
 *
 * @brief 16-bits Huffman encoder
 *
 * Author: lyard-bretz
 *
 */

#include <string.h>
#include <stdint.h>

#include <set>
#include <string>
#include <vector>
#include <list>

#include <stdexcept>
#include <iostream>

#include <sys/time.h>

#define MAX_SYMBOLS (1<<16)

#define SINGLE_CASE

// ================================================================

namespace Huffman
{

    unsigned long numbytes_from_numbits(unsigned long numbits);

/********************************************************************************
 ********************************************************************************
 ********************************************************************************/
    struct TreeNode
    {
        TreeNode *parent;
        union
        {
            struct
            {
                TreeNode *zero, *one;
            };
            uint16_t symbol;
        };

        size_t count;
        bool isLeaf;

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        TreeNode(uint16_t sym, size_t cnt);

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        TreeNode(TreeNode *n0=0, TreeNode *n1=0);
        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        ~TreeNode();
        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        bool operator() (const TreeNode *hn1, const TreeNode *hn2) const;
    };

/********************************************************************************
 ********************************************************************************
 ********************************************************************************/
         struct Code
        {
            size_t bits;
            uint8_t numbits;

//            Code() {}//: numbits(0)  {}
        };


    struct Encoder
    {

        size_t count;
        Code lut[MAX_SYMBOLS];

        Encoder();
        void initialize(const uint16_t *bufin, size_t bufinlen);

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        bool CreateEncoder(const TreeNode *n, size_t bits=0, uint8_t nbits=0);

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        void WriteCodeTable(std::string &out) const;

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        void Encode(std::string &out, const uint16_t *bufin, size_t bufinlen) const;

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        Encoder(const uint16_t *bufin, size_t bufinlen);
    };

    void startProfile();
    void endProfile();
    void printProfileStats();

/********************************************************************************
 ********************************************************************************
 ********************************************************************************/
    struct Decoder
    {
        uint16_t symbol;
        uint8_t nbits;
        bool isLeaf;

        Decoder *lut;

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        Decoder();

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        ~Decoder();

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        void Set(uint16_t sym, uint8_t n=0, size_t bits=0);

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        void Build(const TreeNode &p, uint64_t bits=0, uint8_t n=0);

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        Decoder(const TreeNode &p);


        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        const uint8_t *Decode(const uint8_t* in_ptr,
                              const uint8_t* in_end,
                              uint16_t*            out_ptr,
                              const uint16_t*      out_end) const;

        /*---------------------------------------------------
         *---------------------------------------------------
         *--------------------------------------------------*/
        Decoder(const uint8_t* bufin, int64_t &pindex);
    };

/********************************************************************************
 ********************************************************************************
 ********************************************************************************/
    bool Encode(std::string &bufout, const uint16_t *bufin, size_t bufinlen);
/********************************************************************************
 ********************************************************************************
 ********************************************************************************/
    int64_t Decode(const uint8_t *bufin,
                   size_t         bufinlen,
                   std::vector<uint16_t> &pbufout);
};

#endif //Huffman_H
