import os 
import sys
import argparse

from .command_invoker import CommandInvoker
from osdu_perf.utils.logger import get_logger

def main():
    """Main CLI entry point for console script."""
    # Ensure gevent is disabled to avoid conflicts
    os.environ['GEVENT_SUPPORT'] = 'False'
    os.environ['NO_GEVENT_MONKEY_PATCH'] = '1'
    logger = get_logger('CLI')
    logger.debug(f"disable gevent monkey patch: {os.environ['NO_GEVENT_MONKEY_PATCH']}")

    parser = create_parser()
    args = parser.parse_args()
    
    if not args.command:
        parser.print_help()
        return
    
    # Create command invoker
    invoker = CommandInvoker(logger)

    # Determine command name
    if args.command == 'run':
        # Handle run subcommands
        command_name = args.run_command  # 'local' or 'azure_load_test'
    else:
        command_name = args.command  # 'init' or 'version'
    
    # Execute command
    exit_code = invoker.execute_command(command_name, args)

    if exit_code != 0:
        sys.exit(exit_code)



def create_parser():
    """Create and return the argument parser for the CLI"""
    parser = argparse.ArgumentParser(
        description="OSDU Performance Testing Framework CLI",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  osdu_perf init storage              # Initialize tests for storage service
  osdu_perf init search --force       # Force overwrite existing files
  osdu_perf version                   # Show version information
  osdu_perf run local --config config.yaml  # Run local performance tests
"""
    )
    
    subparsers = parser.add_subparsers(dest='command', help='Available commands')
    
    # Init command
    init_parser = subparsers.add_parser('init', help='Initialize a new performance testing project')
    init_parser.add_argument('service_name', help='Name of the OSDU service to test (e.g., storage, search)')
    init_parser.add_argument('--force', action='store_true', help='Force overwrite existing files')
    
    # Version command
    version_parser = subparsers.add_parser('version', help='Show version information')
    
    # Run command
    run_parser = subparsers.add_parser('run', help='Run performance tests')
    run_subparsers = run_parser.add_subparsers(dest='run_command', help='Run command options')
    
    # Run local subcommand
    local_parser = run_subparsers.add_parser('local', help='Run local performance tests using bundled locustfiles')
    
    # Configuration (Required)
    local_parser.add_argument('--config', '-c', required=True, help='Path to config.yaml file (required)')
    
    # OSDU Connection Parameters (Optional - overrides config.yaml values)
    local_parser.add_argument('--host', help='OSDU host URL (overrides config.yaml)')
    local_parser.add_argument('--partition', '-p', help='OSDU data partition ID (overrides config.yaml)')
    local_parser.add_argument('--token', required=True, help='Bearer token for OSDU authentication (required)')
    local_parser.add_argument('--app-id', help='Azure AD Application ID (overrides config.yaml)')
    local_parser.add_argument('--sku', help='OSDU SKU for metrics collection (overrides config.yaml, default: Standard)')
    local_parser.add_argument('--version', help='OSDU version for metrics collection (overrides config.yaml, default: 1.0)')
    # Locust Test Parameters (Optional)
    local_parser.add_argument('--users', '-u', type=int, help='Number of concurrent users (default: 100)')
    local_parser.add_argument('--spawn-rate', '-r', type=int, help='User spawn rate per second (default: 5)')
    local_parser.add_argument('--run-time', '-t', help='Test duration (default: 60m)')
    local_parser.add_argument('--engine-instances', '-e', type=int, help='Number of engine instances (default: 10)')
    # Advanced Options
    local_parser.add_argument('--locustfile', '-f', help='Specific locustfile to use (optional)')
    local_parser.add_argument('--list-locustfiles', action='store_true', help='List available bundled locustfiles')
    local_parser.add_argument('--headless', action='store_true', help='Run in headless mode (overrides web UI)')
    local_parser.add_argument('--web-ui', action='store_true', default=True, help='Run with web UI (default)')
    local_parser.add_argument('--verbose', '-v', action='store_true', help='Enable verbose output')
    
    # Azure Load Test subcommand
    azure_parser = run_subparsers.add_parser('azure_load_test', help='Run performance tests on Azure Load Testing service')
    
    # Configuration (Required)
    azure_parser.add_argument('--config', '-c', required=True, help='Path to config.yaml file (required)')
    
    # Azure Configuration (Optional - can be read from config.yaml)
    azure_parser.add_argument('--subscription-id', help='Azure subscription ID (overrides config.yaml)')
    azure_parser.add_argument('--resource-group', help='Azure resource group name (overrides config.yaml)')
    azure_parser.add_argument('--location', help='Azure region (e.g., eastus, westus2) (overrides config.yaml)')
    
    # OSDU Connection Parameters (Optional - overrides config.yaml values)
    azure_parser.add_argument('--host', help='OSDU host URL (overrides config.yaml)')
    azure_parser.add_argument('--partition', '-p', help='OSDU data partition ID (overrides config.yaml)')
    azure_parser.add_argument('--token', required=True, help='Bearer token for OSDU authentication (required)')
    azure_parser.add_argument('--app-id', help='Azure AD Application ID (overrides config.yaml)')
    azure_parser.add_argument('--sku', help='OSDU SKU for metrics collection (overrides config.yaml, default: Standard)')
    azure_parser.add_argument('--version', help='OSDU version for metrics collection (overrides config.yaml, default: 1.0)')
    
    # Azure Load Testing Configuration (Optional)
    azure_parser.add_argument('--loadtest-name', default='osdu-perf-dev', help='Azure Load Testing resource name (default: osdu-perf-dev)')
    azure_parser.add_argument('--test-name', help='Test name (auto-generated if not provided)')
    
    
    # Advanced Options
    azure_parser.add_argument('--directory', '-d', default='.', help='Directory containing perf_*_test.py files (default: current)')
    azure_parser.add_argument('--force', action='store_true', help='Force overwrite existing tests without prompting')
    azure_parser.add_argument('--verbose', '-v', action='store_true', help='Enable verbose output')
    
    return parser


if __name__ == "__main__":
    main()
