from __future__ import annotations

from dataclasses import dataclass
from pathlib import Path
from typing import Dict, Any, Optional, List

from jinja2 import Environment, PackageLoader, select_autoescape


_env: Optional[Environment] = None


def _get_jinja_env() -> Environment:
    global _env
    if _env is None:
        _env = Environment(
            loader=PackageLoader("devboiler", "templates"),
            autoescape=select_autoescape(["html", "xml"]),
            trim_blocks=True,
            lstrip_blocks=True,
        )
    return _env


def _render_template(template_path: str, context: Dict[str, Any]) -> str:
    env = _get_jinja_env()
    template = env.get_template(template_path)
    return template.render(**context)


def _ensure_parent_directory(file_path: Path) -> None:
    file_path.parent.mkdir(parents=True, exist_ok=True)


def _write_text_file(file_path: Path, content: str, *, force: bool = False) -> Path:
    _ensure_parent_directory(file_path)
    if file_path.exists() and not force:
        raise FileExistsError(f"File already exists: {file_path}")
    file_path.write_text(content, encoding="utf-8")
    return file_path


def create_python_class(
    class_name: str,
    *,
    filename: Optional[str] = None,
    directory: Path | str = ".",
    force: bool = False,
) -> Path:
    """Create a Python class skeleton.

    Args:
        class_name: Name of the class to generate.
        filename: Optional explicit filename; defaults to f"{class_name}.py".
        directory: Output directory.
        force: Overwrite files if they exist.
    Returns:
        Path to the generated file.
    """
    directory_path = Path(directory)
    output_filename = filename or f"{class_name}.py"
    output_path = directory_path / output_filename
    content = _render_template(
        "python/class.py.j2",
        {"class_name": class_name},
    )
    return _write_text_file(output_path, content, force=force)


def create_html_page(
    name: str,
    *,
    title: str = "My Homepage",
    directory: Path | str = ".",
    force: bool = False,
) -> Path:
    """Create a minimal HTML page.

    Args:
        name: Base filename (without extension).
        title: HTML <title> value.
        directory: Output directory.
        force: Overwrite files if they exist.
    Returns:
        Path to the generated HTML file.
    """
    directory_path = Path(directory)
    output_path = directory_path / f"{name}.html"
    content = _render_template(
        "html/page.html.j2",
        {"title": title},
    )
    return _write_text_file(output_path, content, force=force)


def create_react_component(
    name: str,
    *,
    type: str = "function",
    extension: str = "jsx",
    directory: Path | str = ".",
    force: bool = False,
) -> Path:
    """Create a React component (function or class)."""
    template_name = (
        "react/component_function.jsx.j2"
        if type == "function"
        else "react/component_class.jsx.j2"
    )
    directory_path = Path(directory)
    output_path = directory_path / f"{name}.{extension}"
    content = _render_template(template_name, {"component_name": name})
    return _write_text_file(output_path, content, force=force)


def create_project(
    name: str,
    *,
    type: str = "python",
    directory: Path | str = ".",
    force: bool = False,
) -> List[Path]:
    """Create a project skeleton.

    Currently supported types:
    - python
    """
    if type != "python":
        raise ValueError(f"Unsupported project type: {type}")

    root = Path(directory) / name
    files_created: List[Path] = []

    # package dir
    package_dir = root / name
    init_content = _render_template("project/python/init.py.j2", {"package_name": name})
    main_content = _render_template("project/python/main.py.j2", {"package_name": name})

    files_created.append(_write_text_file(package_dir / "__init__.py", init_content, force=force))
    files_created.append(_write_text_file(root / "main.py", main_content, force=force))

    # simple README
    readme = (
        f"# {name}\n\nGenerated by devboiler. Run with:\n\n"
        f"```bash\npython {name}/main.py\n```\n"
    )
    files_created.append(_write_text_file(root / "README.md", readme, force=force))

    return files_created


def create_flask_app(
    name: str,
    *,
    filename: str = "app.py",
    directory: Path | str = ".",
    force: bool = False,
) -> Path:
    """Create a minimal Flask app file."""
    directory_path = Path(directory) / name
    output_path = directory_path / filename
    content = _render_template("flask/app.py.j2", {"app_name": name})
    return _write_text_file(output_path, content, force=force)


def create_fastapi_app(
    name: str,
    *,
    filename: str = "main.py",
    directory: Path | str = ".",
    force: bool = False,
) -> Path:
    """Create a minimal FastAPI app file."""
    directory_path = Path(directory) / name
    output_path = directory_path / filename
    content = _render_template("fastapi/main.py.j2", {"app_name": name})
    return _write_text_file(output_path, content, force=force)


def create_node_script(
    name: str,
    *,
    filename: str = "index.js",
    directory: Path | str = ".",
    force: bool = False,
) -> Path:
    """Create a simple Node.js script."""
    directory_path = Path(directory) / name
    output_path = directory_path / filename
    content = _render_template("node/index.js.j2", {"script_name": name})
    return _write_text_file(output_path, content, force=force)


def create_express_app(
    name: str,
    *,
    filename: str = "server.js",
    directory: Path | str = ".",
    force: bool = False,
) -> Path:
    """Create a minimal Express.js app."""
    directory_path = Path(directory) / name
    output_path = directory_path / filename
    content = _render_template("express/server.js.j2", {"app_name": name})
    return _write_text_file(output_path, content, force=force)


def create_python_cli(
    name: str,
    *,
    filename: str = "cli.py",
    directory: Path | str = ".",
    force: bool = False,
) -> Path:
    """Create a Python CLI using argparse."""
    directory_path = Path(directory) / name
    output_path = directory_path / filename
    content = _render_template("python/cli.py.j2", {"cli_name": name})
    return _write_text_file(output_path, content, force=force)


def create_react_component_with_css(
    name: str,
    *,
    extension: str = "jsx",
    directory: Path | str = ".",
    force: bool = False,
) -> list[Path]:
    """Create a React component with a CSS module side-by-side."""
    directory_path = Path(directory)
    component_path = directory_path / f"{name}.{extension}"
    css_path = directory_path / f"{name}.module.css"

    component_content = _render_template(
        "react/component_with_css.jsx.j2", {"component_name": name}
    )
    css_content = _render_template(
        "react/component_css.module.css.j2", {"component_name": name}
    )
    created = [
        _write_text_file(component_path, component_content, force=force),
        _write_text_file(css_path, css_content, force=force),
    ]
    return created



def scaffold_project(
    *,
    name: str,
    framework: str = "fastapi",
    db: str = "none",
    include_docker: bool = True,
    include_tests: bool = True,
    include_linters: bool = True,
    directory: Path | str = ".",
    force: bool = False,
    include_ci: bool = True,
    include_pre_commit: bool = True,
    package_manager: str = "pip",
) -> List[Path]:
    """High-level project scaffolding with optional Docker, tests, linters, CI and pre-commit.

    Args:
        name: Project name (also used as python package name where applicable).
        framework: One of {fastapi, flask, express, python}.
        db: Database preset (currently informational; reserved for future). {none, postgres}.
        include_docker: Generate Dockerfile and docker-compose.
        include_tests: Generate sample tests (pytest for python-based frameworks).
        include_linters: Generate basic linter config (.flake8 for python-based frameworks).
        directory: Output parent directory.
        force: Overwrite files if they already exist.
        include_ci: Generate GitHub Actions CI workflow for python-based frameworks.
        include_pre_commit: Generate .pre-commit-config.yaml.
        package_manager: "pip" or "poetry" for python-based frameworks.

    Returns:
        List of created file paths.
    """
    root = Path(directory)
    files_created: List[Path] = []

    # Base app by framework
    if framework == "fastapi":
        files_created.append(create_fastapi_app(name, directory=root, force=force))
    elif framework == "flask":
        files_created.append(create_flask_app(name, directory=root, force=force))
    elif framework == "express":
        files_created.append(create_express_app(name, directory=root, force=force))
    elif framework == "python":
        files_created.extend(create_project(name, type="python", directory=root, force=force))
    else:
        raise ValueError(f"Unsupported framework: {framework}")

    project_root = root / name

    # README with quickstart
    quickstart_lines: list[str] = [
        f"# {name}",
        "",
        f"Generated with devboiler (framework: {framework}).",
        "",
    ]
    if framework in ("fastapi", "flask"):
        run_hint = (
            f"uvicorn {name}.main:app --host 0.0.0.0 --port 8000"
            if framework == "fastapi"
            else f"flask --app {name}.app run --host=0.0.0.0 --port=8000"
        )
        quickstart_lines += [
            "## Run locally",
            "```bash",
            f"{run_hint}",
            "```",
        ]
    elif framework == "express":
        quickstart_lines += [
            "## Run locally",
            "```bash",
            f"node {name}/server.js",
            "```",
        ]
    else:
        quickstart_lines += [
            "## Run locally",
            "```bash",
            f"python {name}/main.py",
            "```",
        ]

    files_created.append(
        _write_text_file(project_root / "README.md", "\n".join(quickstart_lines) + "\n", force=force)
    )

    # Docker
    if include_docker:
        if framework in ("fastapi", "flask", "python"):
            dockerfile = _render_template(
                "docker/Dockerfile.python.j2",
                {"project_name": name, "framework": framework},
            )
            compose = _render_template(
                "docker/docker-compose.python.yml.j2",
                {"project_name": name, "framework": framework},
            )
        elif framework == "express":
            dockerfile = _render_template(
                "docker/Dockerfile.node.j2", {"project_name": name}
            )
            compose = _render_template(
                "docker/docker-compose.node.yml.j2", {"project_name": name}
            )
        else:
            dockerfile = ""
            compose = ""

        if dockerfile:
            files_created.append(
                _write_text_file(project_root / "Dockerfile", dockerfile, force=force)
            )
        if compose:
            files_created.append(
                _write_text_file(project_root / "docker-compose.yml", compose, force=force)
            )

    # Tests (python frameworks only)
    if include_tests and framework in ("fastapi", "flask", "python"):
        tests_dir = project_root / "tests"
        tests_dir.mkdir(parents=True, exist_ok=True)
        if framework == "fastapi":
            test_content = _render_template(
                "tests/test_app_fastapi.py.j2", {"project_name": name}
            )
            files_created.append(
                _write_text_file(tests_dir / "test_root.py", test_content, force=force)
            )
        elif framework == "flask":
            test_content = _render_template(
                "tests/test_app_flask.py.j2", {"project_name": name}
            )
            files_created.append(
                _write_text_file(tests_dir / "test_root.py", test_content, force=force)
            )
        else:
            test_content = _render_template(
                "tests/test_app_python.py.j2", {"project_name": name}
            )
            files_created.append(
                _write_text_file(tests_dir / "test_main.py", test_content, force=force)
            )

        pytest_ini = _render_template("tests/pytest.ini.j2", {})
        files_created.append(
            _write_text_file(project_root / "pytest.ini", pytest_ini, force=force)
        )

    # Linters (python only for now)
    if include_linters and framework in ("fastapi", "flask", "python"):
        flake8 = _render_template("linters/flake8.j2", {})
        files_created.append(
            _write_text_file(project_root / ".flake8", flake8, force=force)
        )

    # pre-commit config (python only)
    if include_pre_commit and framework in ("fastapi", "flask", "python"):
        pre_commit_cfg = _render_template("precommit/config.yaml.j2", {})
        files_created.append(
            _write_text_file(project_root / ".pre-commit-config.yaml", pre_commit_cfg, force=force)
        )

    # CI workflow (python only)
    if include_ci and framework in ("fastapi", "flask", "python"):
        context = {
            "package_manager": package_manager,
            "include_tests": include_tests,
            "include_linters": include_linters,
        }
        ci_content = _render_template("ci/github/python.yml.j2", context)
        files_created.append(
            _write_text_file(project_root / ".github" / "workflows" / "ci.yml", ci_content, force=force)
        )

    # Poetry support (python only)
    if package_manager == "poetry" and framework in ("fastapi", "flask", "python"):
        poetry_pyproject = _render_template("poetry/pyproject.toml.j2", {"project_name": name})
        files_created.append(
            _write_text_file(project_root / "pyproject.toml", poetry_pyproject, force=force)
        )

    return files_created
