import os
import click
from elog_cli.auth_manager import AuthManager
from elog_cli.commands.common import handle_error, manage_http_error, handle_response
from elog_cli.elog_management_backend_client.types import Response
from elog_cli.elog_management_backend_client.api.logbooks_controller import get_all_logbook, create_tag as create_tag_api
from elog_cli.elog_management_backend_client.models import ApiResultResponseListLogbookDTO, LogbookDTO, ApiResultResponseString, NewTagDTO

ENPOINT_URL = os.environ.get("ENPOINT_URL")

@click.command()
@click.pass_context
def create_tag(ctx):
    """List all logbooks permitting to select one, and on that logbook, permit to create a single tag."""
    auth_manager: AuthManager = ctx.obj["auth_manager"]  # Retrieve shared AuthManager
    client = auth_manager.get_elog_client()  # Get the authenticated client
    with client as client:
        all_logbook_result: Response[ApiResultResponseListLogbookDTO] = get_all_logbook.sync_detailed(client=client)
        if all_logbook_result.status_code == 200:
            if all_logbook_result.parsed.error_code == 0:
                logbooks: list[LogbookDTO] = all_logbook_result.parsed.payload
                selected_logbook = logbook_selection(logbooks)
                if selected_logbook is None:
                    click.echo("No logbook selected.")
                    return
                create_new_tag(client, selected_logbook)
            else:
                handle_error(all_logbook_result.parsed)
        else:
            manage_http_error(all_logbook_result)

def logbook_selection(logbooks: list[LogbookDTO]) -> LogbookDTO:
    """List all logbooks and select one."""
    if len(logbooks) == 0:
        click.echo("No logbooks found.")
        return None

    click.echo("Select a logbook:")
    for i, logbook in enumerate(logbooks):
        click.echo(f"{i + 1}. {logbook.name}")

    choice = click.prompt("Enter the number of the logbook", type=int)
    if 1 <= choice <= len(logbooks):
        selected_logbook = logbooks[choice - 1]
        click.echo(f"You selected: {selected_logbook.name}")
        return selected_logbook
    else:
        click.echo("Invalid choice. Please try again.")

def create_new_tag(client, logbook:LogbookDTO):
    """Create a new tag"""
    new_name = click.prompt("Enter new tag name", default="tag name")
    new_description = click.prompt("Enter new tag description", default="tag description")

    new_tag_dto = NewTagDTO(name=new_name, description=new_description)
    response = create_tag_api.sync_detailed(
        client=client, 
        logbook_id=logbook.id, 
        body=new_tag_dto)

    handle_response(response, creation_tag_usccess_cb)

def creation_tag_usccess_cb(result: ApiResultResponseString):
    click.echo(f"Tag create successfully with id: {result.payload}")