import os
import click
from elog_cli.auth_manager import AuthManager
from elog_cli.commands.common import handle_error, manage_http_error, handle_response
from elog_cli.elog_management_backend_client.types import Response, Unset
from elog_cli.elog_management_backend_client.api.logbooks_controller import get_all_logbook, update_tag as update_tag_api
from elog_cli.elog_management_backend_client.models import ApiResultResponseListLogbookDTO, LogbookDTO, TagDTO, UpdateTagDTO, ApiResultResponseTagDTO

ENPOINT_URL = os.environ.get("ENPOINT_URL")

@click.command()
@click.pass_context
def update_tag(ctx):
    """List all logbooks permitting to select one, and on that logbook, permit to update a single tag."""
    auth_manager: AuthManager = ctx.obj["auth_manager"]  # Retrieve shared AuthManager
    client = auth_manager.get_elog_client()  # Get the authenticated client
    with client as client:
        all_logbook_result: Response[ApiResultResponseListLogbookDTO] = get_all_logbook.sync_detailed(client=client)
        if all_logbook_result.status_code == 200:
            if all_logbook_result.parsed.error_code == 0:
                logbooks: list[LogbookDTO] = all_logbook_result.parsed.payload
                selected_logbook = logbook_selection(logbooks)
                if selected_logbook is None:
                    click.echo("No logbook selected.")
                    return
                selected_tag = tag_selection(selected_logbook)
                if selected_tag is None:
                    click.echo("No tag selected or found.")
                    return
                # update tag
                update_tag_info(client, selected_tag)
            else:
                handle_error(all_logbook_result.parsed)
        else:
            manage_http_error(all_logbook_result)

def logbook_selection(logbooks: list[LogbookDTO]) -> LogbookDTO:
    """List all logbooks and select one."""
    if len(logbooks) == 0:
        click.echo("No logbooks found.")
        return None

    click.echo("Select a logbook:")
    for i, logbook in enumerate(logbooks):
        click.echo(f"{i + 1}. {logbook.name}")

    choice = click.prompt("Enter the number of the logbook", type=int)
    if 1 <= choice <= len(logbooks):
        selected_logbook = logbooks[choice - 1]
        click.echo(f"You selected: {selected_logbook.name}")
        return selected_logbook
    else:
        click.echo("Invalid choice. Please try again.")

def tag_selection(logbook: LogbookDTO) -> TagDTO:
    """List all tags and select one."""
    if not isinstance(logbook.tags, Unset) and len(logbook.tags) == 0:
        return None

    click.echo("Select a tag:")
    for i, tag in enumerate(logbook.tags):
        click.echo(f"{i + 1}. {tag.name}")

    choice = click.prompt("Enter the number of the tag", type=int)
    if 1 <= choice <= len(logbook.tags):
        selected_tag = logbook.tags[choice - 1]
        click.echo(f"You selected: {selected_tag.name}")
        return selected_tag
    else:
        return None

def update_tag_info(client, tag: TagDTO):
    """Update the tag information."""
    click.echo(f"Current tag name: {tag.name}")
    click.echo(f"Current tag description: {tag.description}")

    new_name = click.prompt("Enter new tag name", default=tag.name)
    new_description = click.prompt("Enter new tag description", default=tag.description)

    update_tag_dto = UpdateTagDTO(name=new_name, description=new_description)
    response = update_tag_api.sync_detailed(
        client=client, 
        logbook_id=tag.logbook.id, 
        tag_id=tag.id, 
        body=update_tag_dto)

    handle_response(response, success_callback=lambda result:click.echo(f"Tag udapted successfully: {result.payload}"))

def creation_tag_usccess_cb(result: ApiResultResponseTagDTO):
    click.echo(f"Tag create successfully with id: {result.payload}")
