"""Annotation command-line interface commands."""

from __future__ import annotations

import os
import subprocess
import sys
from pathlib import Path

import click


@click.command("annotate")
@click.argument("data_dir", type=click.Path(exists=True), required=False)
@click.option(
    "--image-dir",
    type=click.Path(exists=True),
    default=None,
    help="Directory containing images (default: auto-detect from data_dir)",
)
@click.option(
    "--port",
    type=int,
    default=8501,
    help="Port for Streamlit server (default: 8501)",
)
@click.option(
    "--config-path",
    type=click.Path(exists=True),
    default=None,
    help="Path to config.yml (optional, uses ./config.yml if present)",
)
@click.pass_context
def annotate_command(
    ctx: click.Context,
    data_dir: str | None,
    image_dir: str | None,
    port: int,
    config_path: str | None,
) -> None:
    """Launch human annotation interface for QA pairs.

    DATA_DIR is the directory containing processed JSON files with QA pairs.
    If not provided, uses config.preprocess.output_dir.

    The annotation tool allows manual verification and labeling of QA pairs
    generated by automatic systems. It provides:

    - Visual display of documents with bounding boxes
    - Review of automatic verifier results
    - Manual approval/rejection with explanations
    - Progress tracking and statistics
    - Navigation between documents and QA pairs

    Examples:
        # Launch with default settings (from config)
        docs2synth annotate

        # Launch with specific directory
        docs2synth annotate data/processed/dev

        # Specify custom image directory
        docs2synth annotate data/processed/dev --image-dir data/images

        # Use custom port
        docs2synth annotate --port 8502
    """
    cfg = ctx.obj.get("config")

    # Get data directory: CLI argument > config.preprocess.output_dir
    if data_dir is None:
        output_dir = cfg.get("preprocess.output_dir")
        if output_dir is None:
            click.echo(
                click.style(
                    "✗ Error: DATA_DIR argument is required, or set config.preprocess.output_dir",
                    fg="red",
                ),
                err=True,
            )
            sys.exit(1)
        data_dir = output_dir

    # Validate data directory exists
    data_path = Path(data_dir)
    if not data_path.exists():
        click.echo(
            click.style(
                f"✗ Error: Data directory does not exist: {data_dir}",
                fg="red",
            ),
            err=True,
        )
        sys.exit(1)

    # Get image directory: CLI option > config.preprocess.input_dir
    if image_dir is None:
        input_dir = cfg.get("preprocess.input_dir")
        if input_dir:
            image_dir = input_dir

    # Get app path
    app_path = Path(__file__).parent.parent.parent / "annotation" / "streamlit_app.py"

    if not app_path.exists():
        click.echo(
            click.style(
                f"✗ Error: Streamlit app not found: {app_path}",
                fg="red",
            ),
            err=True,
        )
        sys.exit(1)

    # Display launch info
    click.echo(click.style("🚀 Launching annotation tool...", fg="cyan", bold=True))
    click.echo(click.style(f"Data directory: {data_dir}", fg="blue"))
    if image_dir:
        click.echo(click.style(f"Image directory: {image_dir}", fg="blue"))
    click.echo(click.style(f"Port: {port}", fg="blue"))
    click.echo(
        click.style(
            f"\n📝 Open in browser: http://localhost:{port}\n", fg="green", bold=True
        )
    )

    # Set environment variables for Streamlit app
    env = os.environ.copy()
    env["DOCS2SYNTH_DATA_DIR"] = str(data_dir)
    if image_dir:
        env["DOCS2SYNTH_IMAGE_DIR"] = str(image_dir)

    # Launch Streamlit
    try:
        subprocess.run(
            [
                sys.executable,
                "-m",
                "streamlit",
                "run",
                str(app_path),
                "--server.port",
                str(port),
                "--server.headless",
                "true",
            ],
            env=env,
            check=True,
        )
    except FileNotFoundError:
        click.echo(
            click.style(
                "✗ Error: Streamlit not found. Please install it with: pip install streamlit",
                fg="red",
            ),
            err=True,
        )
        sys.exit(1)
    except subprocess.CalledProcessError as e:
        click.echo(
            click.style(f"✗ Error: Failed to launch Streamlit: {e}", fg="red"),
            err=True,
        )
        sys.exit(1)
    except KeyboardInterrupt:
        click.echo(click.style("\n✓ Annotation tool closed.", fg="green"))
        sys.exit(0)
