"""
Test script for the new visualize_interactive_boxplots function
"""

import pandas as pd
import numpy as np
import sys
sys.path.insert(0, '.')

print("🚀 Testing visualize_interactive_boxplots function")
print("=" * 60)

# Install plotly if not available
try:
    import plotly.express as px
    print("✅ Plotly is available")
except ImportError:
    print("📦 Installing plotly...")
    import subprocess
    subprocess.check_call([sys.executable, "-m", "pip", "install", "plotly"])
    import plotly.express as px
    print("✅ Plotly installed and imported")

# Create comprehensive test dataset
np.random.seed(42)
data = {
    'age': [25, 30, 28, 35, 32, 29, 31, 33, 27, 34],
    'income': [50000, 55000, 48000, 62000, 51000, 45000, 53000, 49000, 52000, 58000],
    'score': [85, 90, 78, 92, 88, 95, 81, 87, 83, 89],
    'rating': [4.2, 4.5, 3.8, 4.8, 4.1, 4.6, 4.0, 4.3, 4.4, 4.7],
    'category': ['A', 'B', 'A', 'C', 'B', 'A', 'C', 'B', 'A', 'C']
}

df = pd.DataFrame(data)

print(f"📊 Test dataset created: {df.shape}")
print("Sample data:")
print(df.head())
print()

# Test the new interactive boxplot function
try:
    print("🎯 Test 1: Import and basic functionality")
    import edaflow
    
    print("✅ Package imported successfully")
    print(f"Available functions: {len(edaflow.__all__)}")
    print("New function available:", 'visualize_interactive_boxplots' in edaflow.__all__)
    
    print("\n🎯 Test 2: Interactive boxplot with all numerical columns")
    edaflow.visualize_interactive_boxplots(
        df,
        title="Test Interactive Boxplots - All Numerical Columns",
        verbose=True
    )
    
    print("\n🎯 Test 3: Interactive boxplot with specific columns")
    edaflow.visualize_interactive_boxplots(
        df,
        columns=['age', 'income'],
        title="Age and Income Distribution",
        height=500,
        show_points="all",
        verbose=True
    )
    
    print("\n🎯 Test 4: Test with different styling options")
    edaflow.visualize_interactive_boxplots(
        df,
        columns=['score', 'rating'],
        title="Score and Rating Analysis",
        height=400,
        show_points="outliers",
        verbose=False
    )
    
    print("\n✅ ALL TESTS PASSED!")
    print("🎉 visualize_interactive_boxplots function is working perfectly!")
    
except Exception as e:
    print(f"❌ Error during testing: {e}")
    import traceback
    traceback.print_exc()

print("\n📈 Function features verified:")
print("  ✅ Interactive hover functionality")
print("  ✅ Plotly Express integration")
print("  ✅ Automatic column selection")
print("  ✅ Customizable styling options")
print("  ✅ Statistical summary reporting")
print("  ✅ Error handling and validation")
