"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsiiProject = exports.Stability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const consts_1 = require("./consts");
const jsii_docgen_1 = require("./jsii-docgen");
const consts_2 = require("../build/private/consts");
const workflow_steps_1 = require("../github/workflow-steps");
const workflows_model_1 = require("../github/workflows-model");
const javascript_1 = require("../javascript");
const runner_options_1 = require("../runner-options");
const typescript_1 = require("../typescript");
const util_1 = require("../util");
const EMAIL_REGEX = /(?:[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*|"(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21\x23-\x5b\x5d-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])*")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21-\x5a\x53-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])+)\])/;
const URL_REGEX = /((([A-Za-z]{3,9}:(?:\/\/)?)(?:[\-;:&=\+\$,\w]+@)?[A-Za-z0-9\.\-]+|(?:www\.|[\-;:&=\+\$,\w]+@)[A-Za-z0-9\.\-]+)((?:\/[\+~%\/\.\w\-_]*)?\??(?:[\-\+=&;%@\.\w_]*)#?(?:[\.\!\/\\\w]*))?)/;
const REPO_TEMP_DIRECTORY = ".repo";
const BUILD_ARTIFACT_OLD_DIR = "dist.old";
var Stability;
(function (Stability) {
    Stability["EXPERIMENTAL"] = "experimental";
    Stability["STABLE"] = "stable";
    Stability["DEPRECATED"] = "deprecated";
})(Stability || (exports.Stability = Stability = {}));
/**
 * Multi-language jsii library project
 *
 * @pjid jsii
 */
class JsiiProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        const { authorEmail, authorUrl } = parseAuthorAddress(options);
        const jsiiVersion = options.jsiiVersion ?? "~5.8.0";
        const defaultOptions = {
            repository: options.repositoryUrl,
            authorName: options.author,
            authorEmail,
            authorUrl,
        };
        const forcedOptions = {
            releaseToNpm: false, // we have a jsii release workflow
            disableTsconfig: true, // jsii generates its own tsconfig.json
            docgen: false, // we use jsii-docgen here so disable typescript docgen
        };
        const mergedOptions = (0, util_1.deepMerge)([
            defaultOptions,
            options,
            forcedOptions,
        ]);
        super(mergedOptions);
        const srcdir = this.srcdir;
        const libdir = this.libdir;
        this.addFields({ types: `${libdir}/index.d.ts` });
        const compressAssembly = options.compressAssembly ?? false;
        // this is an unhelpful warning
        const jsiiFlags = ["--silence-warnings=reserved-word"];
        if (compressAssembly) {
            jsiiFlags.push("--compress-assembly");
        }
        const compatIgnore = options.compatIgnore ?? ".compatignore";
        this.addFields({ stability: options.stability ?? Stability.STABLE });
        if (options.stability === Stability.DEPRECATED) {
            this.addFields({ deprecated: true });
        }
        const compatTask = this.addTask("compat", {
            description: "Perform API compatibility check against latest version",
            exec: `jsii-diff npm:$(node -p "require(\'./package.json\').name") -k --ignore-file ${compatIgnore} || (echo "\nUNEXPECTED BREAKING CHANGES: add keys such as \'removed:constructs.Node.of\' to ${compatIgnore} to skip.\n" && exit 1)`,
        });
        const compat = options.compat ?? false;
        if (compat) {
            this.compileTask.spawn(compatTask);
        }
        this.compileTask.reset(["jsii", ...jsiiFlags].join(" "));
        this.watchTask.reset(["jsii", "-w", ...jsiiFlags].join(" "));
        // Create a new package:all task, it will be filled with language targets later
        this.packageAllTask = this.addTask("package-all", {
            description: "Packages artifacts for all target languages",
        });
        // in jsii we consider the entire repo (post build) as the build artifact
        // which is then used to create the language bindings in separate jobs.
        // we achieve this by doing a checkout and overwrite with the files from the js package.
        this.packageJsTask = this.addPackagingTask("js");
        // When running inside CI we initially only package js. Other targets are packaged in separate jobs.
        // Outside of CI (i.e locally) we simply package all targets.
        this.packageTask.reset();
        this.packageTask.spawn(this.packageJsTask, {
            // Only run in CI
            condition: `node -e "if (!process.env.CI) process.exit(1)"`,
        });
        this.packageTask.spawn(this.packageAllTask, {
            // Don't run in CI
            condition: `node -e "if (process.env.CI) process.exit(1)"`,
        });
        const targets = {};
        const jsii = {
            outdir: this.artifactsDirectory,
            targets,
            tsc: {
                outDir: libdir,
                rootDir: srcdir,
            },
        };
        if (options.excludeTypescript) {
            jsii.excludeTypescript = options.excludeTypescript;
        }
        this.addFields({ jsii });
        const extraJobOptions = {
            ...this.getJobRunsOnConfig(options),
            ...(options.workflowContainerImage
                ? { container: { image: options.workflowContainerImage } }
                : {}),
        };
        if (options.releaseToNpm != false) {
            const npmjs = {
                registry: this.package.npmRegistry,
                npmTokenSecret: this.package.npmTokenSecret,
                npmProvenance: this.package.npmProvenance,
                codeArtifactOptions: options.codeArtifactOptions,
                trustedPublishing: options.npmTrustedPublishing ?? false,
            };
            this.addTargetToBuild("js", this.packageJsTask, extraJobOptions);
            this.addTargetToRelease("js", this.packageJsTask, npmjs);
        }
        // we cannot call an option `java` because the java code generated by jsii
        // does not compile due to a conflict between this option name and the `java`
        // package (e.g. when `java.util.Objects` is referenced).
        if ("java" in options) {
            throw new Error('the "java" option is now called "publishToMaven"');
        }
        const maven = options.publishToMaven;
        if (maven) {
            targets.java = {
                package: maven.javaPackage,
                maven: {
                    groupId: maven.mavenGroupId,
                    artifactId: maven.mavenArtifactId,
                },
            };
            const task = this.addPackagingTask("java");
            this.addTargetToBuild("java", task, extraJobOptions);
            this.addTargetToRelease("java", task, maven);
        }
        const pypi = options.publishToPypi ?? options.python;
        if (pypi) {
            targets.python = {
                distName: pypi.distName,
                module: pypi.module,
            };
            const task = this.addPackagingTask("python");
            this.addTargetToBuild("python", task, extraJobOptions);
            this.addTargetToRelease("python", task, pypi);
        }
        const nuget = options.publishToNuget ?? options.dotnet;
        if (nuget) {
            targets.dotnet = {
                namespace: nuget.dotNetNamespace,
                packageId: nuget.packageId,
                iconUrl: nuget.iconUrl,
            };
            const task = this.addPackagingTask("dotnet");
            this.addTargetToBuild("dotnet", task, extraJobOptions);
            this.addTargetToRelease("dotnet", task, nuget);
        }
        const golang = options.publishToGo;
        if (golang) {
            targets.go = {
                moduleName: golang.moduleName,
                packageName: golang.packageName,
                versionSuffix: golang.versionSuffix,
            };
            const task = this.addPackagingTask("go");
            this.addTargetToBuild("go", task, extraJobOptions);
            this.addTargetToRelease("go", task, golang);
        }
        // If jsiiVersion is "*", don't specify anything so the user can manage.
        // Otherwise, use `jsiiVersion`
        const jsiiSuffix = jsiiVersion === "*" ? "" : `@${jsiiVersion}`;
        this.addDevDeps(`jsii${jsiiSuffix}`, `jsii-rosetta${jsiiSuffix}`, "jsii-diff", "jsii-pacmak");
        this.gitignore.exclude(".jsii", "tsconfig.json");
        this.npmignore?.include(".jsii");
        if (options.docgen ?? true) {
            // If jsiiVersion is "*", don't specify anything so the user can manage.
            // Otherwise use a version that is compatible with all supported jsii releases.
            const docgenVersion = options.jsiiVersion === "*" ? "*" : "^10.5.0";
            new jsii_docgen_1.JsiiDocgen(this, {
                version: docgenVersion,
                filePath: options.docgenFilePath,
            });
        }
        // jsii updates .npmignore, so we make it writable
        if (this.npmignore) {
            this.npmignore.readonly = false;
        }
    }
    /**
     * Adds a target language to the release workflow.
     * @param language
     * @returns
     */
    addTargetToRelease(language, packTask, target) {
        if (!this.release) {
            return;
        }
        const pacmak = this.pacmakForLanguage(language, packTask);
        const prePublishSteps = [
            ...pacmak.bootstrapSteps,
            workflow_steps_1.WorkflowSteps.checkout({
                with: {
                    path: REPO_TEMP_DIRECTORY,
                    ...(this.github?.downloadLfs ? { lfs: true } : {}),
                },
            }),
            ...pacmak.packagingSteps,
        ];
        const commonPublishOptions = {
            publishTools: pacmak.publishTools,
            prePublishSteps,
        };
        const handler = publishTo[language];
        this.release?.publisher[handler]({
            ...commonPublishOptions,
            ...target,
        });
    }
    /**
     * Adds a target language to the build workflow
     * @param language
     * @returns
     */
    addTargetToBuild(language, packTask, extraJobOptions) {
        if (!this.buildWorkflow) {
            return;
        }
        const pacmak = this.pacmakForLanguage(language, packTask);
        this.buildWorkflow.addPostBuildJob(`package-${language}`, {
            ...(0, runner_options_1.filteredRunsOnOptions)(extraJobOptions.runsOn, extraJobOptions.runsOnGroup),
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
            },
            tools: {
                node: { version: this.nodeVersion ?? "lts/*" },
                ...pacmak.publishTools,
            },
            steps: [
                ...pacmak.bootstrapSteps,
                workflow_steps_1.WorkflowSteps.checkout({
                    with: {
                        path: REPO_TEMP_DIRECTORY,
                        ref: consts_2.PULL_REQUEST_REF,
                        repository: consts_2.PULL_REQUEST_REPOSITORY,
                        ...(this.github?.downloadLfs ? { lfs: true } : {}),
                    },
                }),
                ...pacmak.packagingSteps,
            ],
            ...extraJobOptions,
        });
    }
    addPackagingTask(language) {
        const packageTargetTask = this.tasks.addTask(`package:${language}`, {
            description: `Create ${language} language bindings`,
        });
        const commandParts = ["jsii-pacmak", "-v"];
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            commandParts.push('--pack-command "pnpm pack"');
        }
        commandParts.push(`--target ${language}`);
        packageTargetTask.exec(commandParts.join(" "));
        this.packageAllTask.spawn(packageTargetTask);
        return packageTargetTask;
    }
    pacmakForLanguage(target, packTask) {
        const bootstrapSteps = [];
        const packagingSteps = [];
        // Generic bootstrapping for all target languages
        bootstrapSteps.push(...this.workflowBootstrapSteps);
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            bootstrapSteps.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v4",
                with: { version: this.package.pnpmVersion },
            });
        }
        else if (this.package.packageManager === javascript_1.NodePackageManager.BUN) {
            bootstrapSteps.push({
                name: "Setup bun",
                uses: "oven-sh/setup-bun@v2",
                with: { "bun-version": this.package.bunVersion },
            });
        }
        // Installation steps before packaging, but after checkout
        packagingSteps.push({
            name: "Install Dependencies",
            run: `cd ${REPO_TEMP_DIRECTORY} && ${this.package.installCommand}`,
        }, {
            name: "Extract build artifact",
            run: `tar --strip-components=1 -xzvf ${this.artifactsDirectory}/js/*.tgz -C ${REPO_TEMP_DIRECTORY}`,
        }, {
            name: `Move build artifact out of the way`,
            run: `mv ${this.artifactsDirectory} ${BUILD_ARTIFACT_OLD_DIR}`,
        }, {
            name: `Create ${target} artifact`,
            run: `cd ${REPO_TEMP_DIRECTORY} && ${this.runTaskCommand(packTask)}`,
        }, {
            name: `Collect ${target} artifact`,
            run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
        });
        return {
            publishTools: consts_1.JSII_TOOLCHAIN[target],
            bootstrapSteps,
            packagingSteps,
        };
    }
    /**
     * Generates the runs-on config for Jobs.
     * Throws error if 'runsOn' and 'runsOnGroup' are both set.
     *
     * @param options - 'runsOn' or 'runsOnGroup'.
     */
    getJobRunsOnConfig(options) {
        return options.workflowRunsOnGroup
            ? { runsOnGroup: options.workflowRunsOnGroup }
            : options.workflowRunsOn
                ? { runsOn: options.workflowRunsOn }
                : {};
    }
}
exports.JsiiProject = JsiiProject;
_a = JSII_RTTI_SYMBOL_1;
JsiiProject[_a] = { fqn: "projen.cdk.JsiiProject", version: "0.98.4" };
const publishTo = {
    js: "publishToNpm",
    java: "publishToMaven",
    python: "publishToPyPi",
    dotnet: "publishToNuget",
    go: "publishToGo",
};
function parseAuthorAddress(options) {
    let authorEmail = options.authorEmail;
    let authorUrl = options.authorUrl;
    if (options.authorAddress) {
        if (options.authorEmail && options.authorEmail !== options.authorAddress) {
            throw new Error("authorEmail is deprecated and cannot be used in conjunction with authorAddress");
        }
        if (options.authorUrl && options.authorUrl !== options.authorAddress) {
            throw new Error("authorUrl is deprecated and cannot be used in conjunction with authorAddress.");
        }
        if (EMAIL_REGEX.test(options.authorAddress)) {
            authorEmail = options.authorAddress;
        }
        else if (URL_REGEX.test(options.authorAddress)) {
            authorUrl = options.authorAddress;
        }
        else {
            throw new Error(`authorAddress must be either an email address or a URL: ${options.authorAddress}`);
        }
    }
    return { authorEmail, authorUrl };
}
//# sourceMappingURL=data:application/json;base64,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