"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FILE_MANIFEST = void 0;
exports.cleanup = cleanup;
const fs = require("fs");
const path = require("path");
const glob = require("fast-glob");
const common_1 = require("./common");
const logging = require("./logging");
exports.FILE_MANIFEST = `${common_1.PROJEN_DIR}/files.json`;
function cleanup(dir, newFiles, exclude) {
    try {
        const manifestFiles = getFilesFromManifest(dir);
        if (manifestFiles.length > 0) {
            // Use `FILE_MANIFEST` to remove files that are no longer managed by projen
            removeFiles(findOrphanedFiles(dir, manifestFiles, newFiles));
        }
        else {
            // Remove all files managed by projen with legacy logic
            removeFiles(findGeneratedFiles(dir, exclude));
        }
    }
    catch (e) {
        logging.warn(`warning: failed to clean up generated files: ${e.stack}`);
    }
}
function removeFiles(files) {
    for (const file of files) {
        fs.rmSync(file, { force: true, recursive: true });
    }
}
function findGeneratedFiles(dir, exclude) {
    const ignore = [
        ...readGitIgnore(dir),
        "node_modules/**",
        ...exclude,
        ".git/**",
    ];
    const files = glob.sync("**", {
        ignore,
        cwd: dir,
        dot: true,
        onlyFiles: true,
        followSymbolicLinks: false,
        absolute: true,
    });
    const generated = new Array();
    for (const file of files) {
        const contents = fs.readFileSync(file, "utf-8");
        if (contents.includes(common_1.PROJEN_MARKER)) {
            generated.push(file);
        }
    }
    return generated;
}
function findOrphanedFiles(dir, oldFiles, newFiles) {
    return oldFiles
        .filter((old) => !newFiles.includes(old))
        .map((f) => path.resolve(dir, f));
}
function getFilesFromManifest(dir) {
    try {
        const fileManifestPath = path.resolve(dir, exports.FILE_MANIFEST);
        if (fs.existsSync(fileManifestPath)) {
            const fileManifest = JSON.parse(fs.readFileSync(fileManifestPath, "utf-8"));
            if (fileManifest.files) {
                return fileManifest.files;
            }
        }
    }
    catch (e) {
        logging.warn(`warning: unable to get files to clean from file manifest: ${e.stack}`);
    }
    return [];
}
function readGitIgnore(dir) {
    const filepath = path.join(dir, ".gitignore");
    if (!fs.existsSync(filepath)) {
        return [];
    }
    return fs
        .readFileSync(filepath, "utf-8")
        .split("\n")
        .filter((x) => x?.trim() !== "")
        .filter((x) => !x.startsWith("#") && !x.startsWith("!"))
        .map((x) => x.replace(/^\//, "")) // remove "/" prefix
        .map((x) => `${x}\n${x}/**`)
        .join("\n")
        .split("\n");
}
//# sourceMappingURL=data:application/json;base64,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