"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutoMerge = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
/**
 * Automatically merge Pull Requests using Mergify
 *
 * > [!NOTE]
 * > GitHub now natively provides the same features, so you don't need Mergify
 * > anymore. See `GitHubOptions.mergeQueue` and `MergeQueueOptions.autoQueue`.
 *
 * If `buildJob` is specified, the specified GitHub workflow job ID is required
 * to succeed in order for the PR to be merged.
 *
 * `approvedReviews` specified the number of code review approvals required for
 * the PR to be merged.
 *
 * @see https://mergify.com/
 */
class AutoMerge extends component_1.Component {
    constructor(github, options = {}) {
        super(github.project);
        this.lazyConditions = new Array();
        const mergify = github.mergify;
        if (!mergify) {
            throw new Error("auto merging requires mergify to be enabled");
        }
        const blockingLabels = options.blockingLabels ?? ["do-not-merge"];
        const blockingCondition = blockingLabels?.length
            ? [`-label~=(${blockingLabels.join("|")})`]
            : [];
        const mergeAction = {
            delete_head_branch: {},
            queue: {
                name: "default",
            },
        };
        const approvedReviews = options.approvedReviews ?? 1;
        // add initial conditions (additional conditions can be added later)
        this.addConditions(`#approved-reviews-by>=${approvedReviews}`);
        this.addConditions(...blockingCondition);
        const ruleName = options.ruleName ?? "Automatic merge on approval and successful build";
        const queueName = options.queueName ?? "default";
        mergify.addRule({
            name: ruleName,
            actions: mergeAction,
            conditions: (() => this.renderConditions()),
        });
        mergify.addQueue({
            name: queueName,
            updateMethod: "merge",
            queueConditions: (() => this.renderConditions()),
            // squash all commits into a single commit when merging
            mergeMethod: "squash",
            // use PR title+body as the commit message
            commitMessageTemplate: [
                "{{ title }} (#{{ number }})",
                "",
                "{{ body }}",
            ].join("\n"),
        });
        this.project.addPackageIgnore("/.mergify.yml");
    }
    /**
     * Adds conditions to the auto merge rule.
     * @param conditions The conditions to add (mergify syntax)
     */
    addConditions(...conditions) {
        this.addConditionsLater({ render: () => conditions });
    }
    /**
     * Adds conditions that will be rendered only during synthesis.
     * @param later The later
     */
    addConditionsLater(later) {
        this.lazyConditions.push(later);
    }
    renderConditions() {
        const output = new Array();
        for (const later of this.lazyConditions) {
            output.push(...later.render());
        }
        return output;
    }
}
exports.AutoMerge = AutoMerge;
_a = JSII_RTTI_SYMBOL_1;
AutoMerge[_a] = { fqn: "projen.github.AutoMerge", version: "0.98.4" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXV0by1tZXJnZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9naXRodWIvYXV0by1tZXJnZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUNBLDRDQUF5QztBQTRCekM7Ozs7Ozs7Ozs7Ozs7O0dBY0c7QUFDSCxNQUFhLFNBQVUsU0FBUSxxQkFBUztJQUd0QyxZQUFZLE1BQWMsRUFBRSxVQUE0QixFQUFFO1FBQ3hELEtBQUssQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUM7UUFIUCxtQkFBYyxHQUFHLElBQUksS0FBSyxFQUF1QixDQUFDO1FBS2pFLE1BQU0sT0FBTyxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUM7UUFDL0IsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDO1lBQ2IsTUFBTSxJQUFJLEtBQUssQ0FBQyw2Q0FBNkMsQ0FBQyxDQUFDO1FBQ2pFLENBQUM7UUFFRCxNQUFNLGNBQWMsR0FBRyxPQUFPLENBQUMsY0FBYyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDbEUsTUFBTSxpQkFBaUIsR0FBRyxjQUFjLEVBQUUsTUFBTTtZQUM5QyxDQUFDLENBQUMsQ0FBQyxZQUFZLGNBQWMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQztZQUMzQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBRVAsTUFBTSxXQUFXLEdBQUc7WUFDbEIsa0JBQWtCLEVBQUUsRUFBRTtZQUV0QixLQUFLLEVBQUU7Z0JBQ0wsSUFBSSxFQUFFLFNBQVM7YUFDaEI7U0FDRixDQUFDO1FBRUYsTUFBTSxlQUFlLEdBQUcsT0FBTyxDQUFDLGVBQWUsSUFBSSxDQUFDLENBQUM7UUFFckQsb0VBQW9FO1FBQ3BFLElBQUksQ0FBQyxhQUFhLENBQUMseUJBQXlCLGVBQWUsRUFBRSxDQUFDLENBQUM7UUFDL0QsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLGlCQUFpQixDQUFDLENBQUM7UUFFekMsTUFBTSxRQUFRLEdBQ1osT0FBTyxDQUFDLFFBQVEsSUFBSSxrREFBa0QsQ0FBQztRQUN6RSxNQUFNLFNBQVMsR0FBRyxPQUFPLENBQUMsU0FBUyxJQUFJLFNBQVMsQ0FBQztRQUVqRCxPQUFPLENBQUMsT0FBTyxDQUFDO1lBQ2QsSUFBSSxFQUFFLFFBQVE7WUFDZCxPQUFPLEVBQUUsV0FBVztZQUNwQixVQUFVLEVBQUUsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBUTtTQUNuRCxDQUFDLENBQUM7UUFFSCxPQUFPLENBQUMsUUFBUSxDQUFDO1lBQ2YsSUFBSSxFQUFFLFNBQVM7WUFDZixZQUFZLEVBQUUsT0FBTztZQUNyQixlQUFlLEVBQUUsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBUTtZQUN2RCx1REFBdUQ7WUFDdkQsV0FBVyxFQUFFLFFBQVE7WUFDckIsMENBQTBDO1lBQzFDLHFCQUFxQixFQUFFO2dCQUNyQiw2QkFBNkI7Z0JBQzdCLEVBQUU7Z0JBQ0YsWUFBWTthQUNiLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztTQUNiLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7T0FHRztJQUNJLGFBQWEsQ0FBQyxHQUFHLFVBQW9CO1FBQzFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLE1BQU0sRUFBRSxHQUFHLEVBQUUsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7O09BR0c7SUFDSSxrQkFBa0IsQ0FBQyxLQUEwQjtRQUNsRCxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRU8sZ0JBQWdCO1FBQ3RCLE1BQU0sTUFBTSxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7UUFFbkMsS0FBSyxNQUFNLEtBQUssSUFBSSxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7WUFDeEMsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBQ2pDLENBQUM7UUFFRCxPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDOztBQWpGSCw4QkFrRkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBHaXRIdWIgfSBmcm9tIFwiLi9naXRodWJcIjtcbmltcG9ydCB7IENvbXBvbmVudCB9IGZyb20gXCIuLi9jb21wb25lbnRcIjtcblxuZXhwb3J0IGludGVyZmFjZSBBdXRvTWVyZ2VPcHRpb25zIHtcbiAgLyoqXG4gICAqIE51bWJlciBvZiBhcHByb3ZlZCBjb2RlIHJldmlld3MuXG4gICAqIEBkZWZhdWx0IDFcbiAgICovXG4gIHJlYWRvbmx5IGFwcHJvdmVkUmV2aWV3cz86IG51bWJlcjtcblxuICAvKipcbiAgICogTGlzdCBvZiBsYWJlbHMgdGhhdCB3aWxsIHByZXZlbnQgYXV0by1tZXJnaW5nLlxuICAgKiBAZGVmYXVsdCBbJ2RvLW5vdC1tZXJnZSddXG4gICAqL1xuICByZWFkb25seSBibG9ja2luZ0xhYmVscz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBtZXJnaWZ5IHJ1bGVcbiAgICogQGRlZmF1bHQgJ0F1dG9tYXRpYyBtZXJnZSBvbiBhcHByb3ZhbCBhbmQgc3VjY2Vzc2Z1bCBidWlsZCdcbiAgICovXG4gIHJlYWRvbmx5IHJ1bGVOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBtZXJnaWZ5IHF1ZXVlXG4gICAqIEBkZWZhdWx0ICdkZWZhdWx0J1xuICAgKi9cbiAgcmVhZG9ubHkgcXVldWVOYW1lPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEF1dG9tYXRpY2FsbHkgbWVyZ2UgUHVsbCBSZXF1ZXN0cyB1c2luZyBNZXJnaWZ5XG4gKlxuICogPiBbIU5PVEVdXG4gKiA+IEdpdEh1YiBub3cgbmF0aXZlbHkgcHJvdmlkZXMgdGhlIHNhbWUgZmVhdHVyZXMsIHNvIHlvdSBkb24ndCBuZWVkIE1lcmdpZnlcbiAqID4gYW55bW9yZS4gU2VlIGBHaXRIdWJPcHRpb25zLm1lcmdlUXVldWVgIGFuZCBgTWVyZ2VRdWV1ZU9wdGlvbnMuYXV0b1F1ZXVlYC5cbiAqXG4gKiBJZiBgYnVpbGRKb2JgIGlzIHNwZWNpZmllZCwgdGhlIHNwZWNpZmllZCBHaXRIdWIgd29ya2Zsb3cgam9iIElEIGlzIHJlcXVpcmVkXG4gKiB0byBzdWNjZWVkIGluIG9yZGVyIGZvciB0aGUgUFIgdG8gYmUgbWVyZ2VkLlxuICpcbiAqIGBhcHByb3ZlZFJldmlld3NgIHNwZWNpZmllZCB0aGUgbnVtYmVyIG9mIGNvZGUgcmV2aWV3IGFwcHJvdmFscyByZXF1aXJlZCBmb3JcbiAqIHRoZSBQUiB0byBiZSBtZXJnZWQuXG4gKlxuICogQHNlZSBodHRwczovL21lcmdpZnkuY29tL1xuICovXG5leHBvcnQgY2xhc3MgQXV0b01lcmdlIGV4dGVuZHMgQ29tcG9uZW50IHtcbiAgcHJpdmF0ZSByZWFkb25seSBsYXp5Q29uZGl0aW9ucyA9IG5ldyBBcnJheTxJQWRkQ29uZGl0aW9uc0xhdGVyPigpO1xuXG4gIGNvbnN0cnVjdG9yKGdpdGh1YjogR2l0SHViLCBvcHRpb25zOiBBdXRvTWVyZ2VPcHRpb25zID0ge30pIHtcbiAgICBzdXBlcihnaXRodWIucHJvamVjdCk7XG5cbiAgICBjb25zdCBtZXJnaWZ5ID0gZ2l0aHViLm1lcmdpZnk7XG4gICAgaWYgKCFtZXJnaWZ5KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXCJhdXRvIG1lcmdpbmcgcmVxdWlyZXMgbWVyZ2lmeSB0byBiZSBlbmFibGVkXCIpO1xuICAgIH1cblxuICAgIGNvbnN0IGJsb2NraW5nTGFiZWxzID0gb3B0aW9ucy5ibG9ja2luZ0xhYmVscyA/PyBbXCJkby1ub3QtbWVyZ2VcIl07XG4gICAgY29uc3QgYmxvY2tpbmdDb25kaXRpb24gPSBibG9ja2luZ0xhYmVscz8ubGVuZ3RoXG4gICAgICA/IFtgLWxhYmVsfj0oJHtibG9ja2luZ0xhYmVscy5qb2luKFwifFwiKX0pYF1cbiAgICAgIDogW107XG5cbiAgICBjb25zdCBtZXJnZUFjdGlvbiA9IHtcbiAgICAgIGRlbGV0ZV9oZWFkX2JyYW5jaDoge30sXG5cbiAgICAgIHF1ZXVlOiB7XG4gICAgICAgIG5hbWU6IFwiZGVmYXVsdFwiLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgY29uc3QgYXBwcm92ZWRSZXZpZXdzID0gb3B0aW9ucy5hcHByb3ZlZFJldmlld3MgPz8gMTtcblxuICAgIC8vIGFkZCBpbml0aWFsIGNvbmRpdGlvbnMgKGFkZGl0aW9uYWwgY29uZGl0aW9ucyBjYW4gYmUgYWRkZWQgbGF0ZXIpXG4gICAgdGhpcy5hZGRDb25kaXRpb25zKGAjYXBwcm92ZWQtcmV2aWV3cy1ieT49JHthcHByb3ZlZFJldmlld3N9YCk7XG4gICAgdGhpcy5hZGRDb25kaXRpb25zKC4uLmJsb2NraW5nQ29uZGl0aW9uKTtcblxuICAgIGNvbnN0IHJ1bGVOYW1lID1cbiAgICAgIG9wdGlvbnMucnVsZU5hbWUgPz8gXCJBdXRvbWF0aWMgbWVyZ2Ugb24gYXBwcm92YWwgYW5kIHN1Y2Nlc3NmdWwgYnVpbGRcIjtcbiAgICBjb25zdCBxdWV1ZU5hbWUgPSBvcHRpb25zLnF1ZXVlTmFtZSA/PyBcImRlZmF1bHRcIjtcblxuICAgIG1lcmdpZnkuYWRkUnVsZSh7XG4gICAgICBuYW1lOiBydWxlTmFtZSxcbiAgICAgIGFjdGlvbnM6IG1lcmdlQWN0aW9uLFxuICAgICAgY29uZGl0aW9uczogKCgpID0+IHRoaXMucmVuZGVyQ29uZGl0aW9ucygpKSBhcyBhbnksXG4gICAgfSk7XG5cbiAgICBtZXJnaWZ5LmFkZFF1ZXVlKHtcbiAgICAgIG5hbWU6IHF1ZXVlTmFtZSxcbiAgICAgIHVwZGF0ZU1ldGhvZDogXCJtZXJnZVwiLFxuICAgICAgcXVldWVDb25kaXRpb25zOiAoKCkgPT4gdGhpcy5yZW5kZXJDb25kaXRpb25zKCkpIGFzIGFueSxcbiAgICAgIC8vIHNxdWFzaCBhbGwgY29tbWl0cyBpbnRvIGEgc2luZ2xlIGNvbW1pdCB3aGVuIG1lcmdpbmdcbiAgICAgIG1lcmdlTWV0aG9kOiBcInNxdWFzaFwiLFxuICAgICAgLy8gdXNlIFBSIHRpdGxlK2JvZHkgYXMgdGhlIGNvbW1pdCBtZXNzYWdlXG4gICAgICBjb21taXRNZXNzYWdlVGVtcGxhdGU6IFtcbiAgICAgICAgXCJ7eyB0aXRsZSB9fSAoI3t7IG51bWJlciB9fSlcIixcbiAgICAgICAgXCJcIixcbiAgICAgICAgXCJ7eyBib2R5IH19XCIsXG4gICAgICBdLmpvaW4oXCJcXG5cIiksXG4gICAgfSk7XG5cbiAgICB0aGlzLnByb2plY3QuYWRkUGFja2FnZUlnbm9yZShcIi8ubWVyZ2lmeS55bWxcIik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBjb25kaXRpb25zIHRvIHRoZSBhdXRvIG1lcmdlIHJ1bGUuXG4gICAqIEBwYXJhbSBjb25kaXRpb25zIFRoZSBjb25kaXRpb25zIHRvIGFkZCAobWVyZ2lmeSBzeW50YXgpXG4gICAqL1xuICBwdWJsaWMgYWRkQ29uZGl0aW9ucyguLi5jb25kaXRpb25zOiBzdHJpbmdbXSkge1xuICAgIHRoaXMuYWRkQ29uZGl0aW9uc0xhdGVyKHsgcmVuZGVyOiAoKSA9PiBjb25kaXRpb25zIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgY29uZGl0aW9ucyB0aGF0IHdpbGwgYmUgcmVuZGVyZWQgb25seSBkdXJpbmcgc3ludGhlc2lzLlxuICAgKiBAcGFyYW0gbGF0ZXIgVGhlIGxhdGVyXG4gICAqL1xuICBwdWJsaWMgYWRkQ29uZGl0aW9uc0xhdGVyKGxhdGVyOiBJQWRkQ29uZGl0aW9uc0xhdGVyKSB7XG4gICAgdGhpcy5sYXp5Q29uZGl0aW9ucy5wdXNoKGxhdGVyKTtcbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyQ29uZGl0aW9ucygpIHtcbiAgICBjb25zdCBvdXRwdXQgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuXG4gICAgZm9yIChjb25zdCBsYXRlciBvZiB0aGlzLmxhenlDb25kaXRpb25zKSB7XG4gICAgICBvdXRwdXQucHVzaCguLi5sYXRlci5yZW5kZXIoKSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG91dHB1dDtcbiAgfVxufVxuXG5leHBvcnQgaW50ZXJmYWNlIElBZGRDb25kaXRpb25zTGF0ZXIge1xuICByZW5kZXIoKTogc3RyaW5nW107XG59XG4iXX0=