"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PullRequestBackport = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const auto_approve_1 = require("./auto-approve");
const github_1 = require("./github");
const workflows_1 = require("./workflows");
const component_1 = require("../component");
const json_1 = require("../json");
const release_1 = require("../release");
class PullRequestBackport extends component_1.Component {
    constructor(scope, options = {}) {
        super(scope);
        const workflowEngine = github_1.GitHub.of(this.project);
        if (!workflowEngine) {
            throw new Error(`Cannot add ${new.target.name} to project without GitHub enabled. Please enable GitHub for this project.`);
        }
        const branches = options.branches ?? release_1.Release.of(this.project)?.branches ?? [];
        if (branches.length === 0) {
            this.project.logger.warn("PullRequestBackport could not find any target branches. Backports will not be available. Please add release branches or configure `branches` manually.");
        }
        const targetPrLabelsRaw = options.backportPRLabels ?? ["backport"];
        const targetPrLabels = [...targetPrLabelsRaw];
        const shouldAutoApprove = options.autoApproveBackport ?? true;
        if (shouldAutoApprove) {
            const autoApprove = this.project.components.find((c) => c instanceof auto_approve_1.AutoApprove);
            if (autoApprove?.label) {
                targetPrLabels.push(autoApprove.label);
            }
        }
        const backportBranchNamePrefix = options.backportBranchNamePrefix ?? "backport/";
        const labelPrefix = options.labelPrefix ?? "backport-to-";
        // Configuration
        this.file = new json_1.JsonFile(this, ".backportrc.json", {
            obj: {
                commitConflicts: options.createWithConflicts ?? true,
                targetPRLabels: targetPrLabels,
                backportBranchName: `${backportBranchNamePrefix}{{targetBranch}}-{{refValues}}`,
                prTitle: "{{sourcePullRequest.title}} (backport #{{sourcePullRequest.number}})",
                targetBranchChoices: branches,
            },
            // File needs to be available to the GitHub Workflow
            committed: true,
        });
        this.project.addPackageIgnore(this.file.path);
        // Workflow
        this.workflow = new workflows_1.GithubWorkflow(workflowEngine, options.workflowName ?? "backport");
        this.workflow.on({
            pullRequestTarget: {
                types: ["labeled", "unlabeled", "closed"],
            },
        });
        // condition to detect if the PR is a backport PR
        // we prefer to match the PR using labels, but will fallback to matching the branch name prefix
        const branchCondition = `startsWith(github.head_ref, '${backportBranchNamePrefix}')`;
        const labelConditions = targetPrLabelsRaw.map((label) => `contains(github.event.pull_request.labels.*.name, '${label}')`);
        const isBackportPr = labelConditions.length
            ? `(${labelConditions.join(" && ")})`
            : `${branchCondition})`;
        const checkStep = "check_labels";
        const checkOutput = "matched";
        const labelPrefixEscaped = labelPrefix.replace(/"/g, '\\"');
        this.workflow.addJob("backport", {
            name: "Backport PR",
            runsOn: ["ubuntu-latest"],
            permissions: {},
            // Only ever run this job if the PR is merged and not a backport PR itself
            if: `github.event.pull_request.merged == true && !${isBackportPr}`,
            steps: [
                ...workflowEngine.projenCredentials.setupSteps,
                // We need a custom step to check if the PR has any of the labels that indicate that the PR should be backported.
                // This is not currently possible with GH Actions expression by itself, so we use a bash script.
                {
                    id: checkStep,
                    name: "Check for backport labels",
                    shell: "bash",
                    run: [
                        "labels='${{ toJSON(github.event.pull_request.labels.*.name) }}'",
                        `matched=$(echo $labels | jq '.|map(select(startswith("${labelPrefixEscaped}"))) | length')`,
                        `echo "${checkOutput}=$matched"`,
                        `echo "${checkOutput}=$matched" >> $GITHUB_OUTPUT`,
                    ].join("\n"),
                },
                {
                    name: "Backport Action",
                    uses: "sqren/backport-github-action@v9.5.1",
                    // only run this step if we have found matching labels in the previous step
                    // this is to prevent workflow failures because the action fails when pre-conditions are not met
                    // and causes any PR to be marked with a red X, leading to error blindness.
                    if: `fromJSON(steps.${checkStep}.outputs.${checkOutput}) > 0`,
                    with: {
                        github_token: workflowEngine.projenCredentials.tokenRef,
                        auto_backport_label_prefix: labelPrefix,
                    },
                },
            ],
        });
    }
}
exports.PullRequestBackport = PullRequestBackport;
_a = JSII_RTTI_SYMBOL_1;
PullRequestBackport[_a] = { fqn: "projen.github.PullRequestBackport", version: "0.98.4" };
//# sourceMappingURL=data:application/json;base64,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