"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Projenrc = void 0;
exports.generateJavaOptionNames = generateJavaOptionNames;
exports.getJavaImport = getJavaImport;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const common_1 = require("../common");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
const projenrc_1 = require("../projenrc");
const util_1 = require("../util");
/**
 * Allows writing projenrc files in java.
 *
 * This will install `org.projen/projen` as a Maven dependency and will add a
 * `synth` task which will compile & execute `main()` from
 * `src/main/java/projenrc.java`.
 */
class Projenrc extends projenrc_1.ProjenrcFile {
    constructor(project, pom, options = {}) {
        super(project);
        const projenVersion = options.projenVersion ?? common_1.PROJEN_VERSION;
        this.className = options.className ?? "projenrc";
        this.testScope = options.testScope ?? true;
        const depType = this.testScope
            ? dependencies_1.DependencyType.TEST
            : dependencies_1.DependencyType.RUNTIME;
        const execOpts = this.testScope ? ' -Dexec.classpathScope="test"' : "";
        const compileGoal = this.testScope
            ? "compiler:testCompile"
            : "compiler:compile";
        project.deps.addDependency(`io.github.cdklabs/projen@${projenVersion}`, depType);
        pom.addPlugin("org.codehaus.mojo/exec-maven-plugin@3.0.0");
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        project.defaultTask?.exec(`mvn ${compileGoal} --quiet`);
        project.defaultTask?.exec(`mvn exec:java --quiet -Dexec.mainClass=${this.className}${execOpts}`);
        // if this is a new project, generate a skeleton for projenrc.java
        this.generateProjenrc();
    }
    get javaClass() {
        const split = this.className.split(".");
        if (split.length === 1) {
            return split[0];
        }
        return split[split.length - 1];
    }
    get javaPackage() {
        const split = this.className.split(".");
        if (split.length === 1) {
            return [];
        }
        return split.slice(0, split.length - 2);
    }
    /**
     * The path of the projenrc file.
     */
    get filePath() {
        const dir = this.testScope ? "src/test/java" : "src/main/java";
        const javaFile = (0, path_1.join)(this.project.outdir, dir, ...this.javaPackage, this.javaClass + ".java");
        const relativePath = (0, path_1.relative)(this.project.outdir, javaFile);
        return (0, util_1.normalizePersistedPath)(relativePath);
    }
    generateProjenrc() {
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = (0, inventory_1.readJsiiManifest)(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const javaTarget = jsiiManifest.targets.java;
        const optionsTypeFqn = jsiiType.initializer?.parameters?.[0].type?.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const jsiiOptionsType = jsiiManifest.types[optionsTypeFqn];
        if (!jsiiOptionsType) {
            this.project.logger.warn(`cannot find jsii type for project options: ${optionsTypeFqn}`);
            return;
        }
        // skip if file exists
        if ((0, fs_1.existsSync)(this.filePath)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        const openBlock = (line = "") => {
            emit(line + " {");
            indent++;
        };
        const closeBlock = () => {
            indent--;
            emit("}");
        };
        const optionFqns = generateJavaOptionNames(bootstrap.type.options, jsiiManifest);
        if (this.javaPackage.length > 0) {
            emit(`package ${this.javaPackage.join(".")};`);
            emit();
        }
        const { renderedOptions, imports } = renderJavaOptions(2, jsiiOptionsType.name, optionFqns, bootstrap.args);
        emit(`import ${getJavaImport(jsiiType, jsiiManifest)};`);
        emit(`import ${getJavaImport(jsiiOptionsType, jsiiManifest)};`);
        for (const optionTypeName of imports) {
            emit(`import ${javaTarget.package}.${optionTypeName};`);
        }
        emit();
        openBlock(`public class ${this.javaClass}`);
        openBlock("public static void main(String[] args)");
        emit(`${jsiiType.name} project = new ${jsiiType.name}(${renderedOptions});`);
        emit("project.synth();");
        closeBlock();
        closeBlock();
        const filePath = (0, path_1.join)(this.project.outdir, this.filePath);
        (0, fs_1.mkdirSync)((0, path_1.dirname)(filePath), { recursive: true });
        (0, fs_1.writeFileSync)(filePath, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${this.filePath}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.java.Projenrc", version: "0.98.4" };
function generateJavaOptionNames(options, jsiiManifest) {
    const optionFqns = {};
    for (const option of options) {
        if (option.fqn && jsiiManifest.types[option.fqn]) {
            optionFqns[option.name] = toJavaFullTypeName(jsiiManifest.types[option.fqn]);
        }
    }
    return optionFqns;
}
function renderJavaOptions(indent, optionsTypeName, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        return { renderedOptions: "", imports }; // no options
    }
    const lines = [`${optionsTypeName}.builder()`];
    for (const [name, value] of Object.entries(initOptions)) {
        const { javaValue, importName } = toJavaValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`.${toJavaProperty(name)}(${javaValue})`);
    }
    lines.push(".build()");
    const renderedOptions = lines.join(`\n${" ".repeat((indent + 1) * 4)}`);
    return { renderedOptions, imports };
}
function toJavaProperty(prop) {
    return prop;
}
function toJavaValue(value, name, optionFqns) {
    if (typeof value === "string" && optionFqns[name] !== undefined) {
        const parts = optionFqns[name].split(".");
        const base = parts[parts.length - 1];
        const choice = String(value).toUpperCase().replace(/-/g, "_");
        return { javaValue: `${base}.${choice}`, importName: optionFqns[name] };
    }
    else {
        return { javaValue: JSON.stringify(value) };
    }
}
function toJavaFullTypeName(jsiiType) {
    return [jsiiType.namespace, jsiiType.name].filter((x) => x).join(".");
}
function getJavaImport(jsiiType, jsiiManifest) {
    const packageName = jsiiManifest?.submodules?.[`${jsiiType.assembly}.${jsiiType?.namespace}`]
        ?.targets?.java?.package ||
        [jsiiManifest.targets.java.package, jsiiType.namespace]
            .filter((x) => x)
            .join(".");
    return `${packageName}.${jsiiType.name}`;
}
//# sourceMappingURL=data:application/json;base64,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