"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeProject = exports.AutoRelease = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const bundler_1 = require("./bundler");
const jest_1 = require("./jest");
const license_checker_1 = require("./license-checker");
const node_package_1 = require("./node-package");
const projenrc_1 = require("./projenrc");
const build_1 = require("../build");
const consts_1 = require("../build/private/consts");
const common_1 = require("../common");
const dependencies_1 = require("../dependencies");
const github_1 = require("../github");
const biome_1 = require("./biome/biome");
const constants_1 = require("../github/constants");
const util_1 = require("../github/private/util");
const workflows_model_1 = require("../github/workflows-model");
const ignore_file_1 = require("../ignore-file");
const javascript_1 = require("../javascript");
const license_1 = require("../license");
const projenrc_json_1 = require("../projenrc-json");
const release_1 = require("../release");
const runner_options_1 = require("../runner-options");
const util_2 = require("../util");
const path_2 = require("../util/path");
const PROJEN_SCRIPT = "projen";
/**
 * Automatic bump modes
 */
var AutoRelease;
(function (AutoRelease) {
    /**
     * Automatically bump & release a new version for every commit to "main"
     */
    AutoRelease[AutoRelease["EVERY_COMMIT"] = 0] = "EVERY_COMMIT";
    /**
     * Automatically bump & release a new version on a daily basis.
     */
    AutoRelease[AutoRelease["DAILY"] = 1] = "DAILY";
})(AutoRelease || (exports.AutoRelease = AutoRelease = {}));
/**
 * Node.js project.
 *
 * @pjid node
 */
class NodeProject extends github_1.GitHubProject {
    /**
     * The .npmrc file
     */
    get npmrc() {
        if (!this._npmrc) {
            this._npmrc = new javascript_1.NpmConfig(this, { omitEmpty: true });
        }
        return this._npmrc;
    }
    /**
     * @deprecated use `package.allowLibraryDependencies`
     */
    get allowLibraryDependencies() {
        return this.package.allowLibraryDependencies;
    }
    /**
     * @deprecated use `package.entrypoint`
     */
    get entrypoint() {
        return this.package.entrypoint;
    }
    /**
     * The minimum node version required by this package to function.
     *
     * This value indicates the package is incompatible with older versions.
     */
    get minNodeVersion() {
        return this.package.minNodeVersion;
    }
    /**
     * Maximum node version supported by this package.
     *
     * The value indicates the package is incompatible with newer versions.
     */
    get maxNodeVersion() {
        return this.package.maxNodeVersion;
    }
    /**
     * The package manager to use.
     *
     * @deprecated use `package.packageManager`
     */
    get packageManager() {
        return this.package.packageManager;
    }
    /**
     * @deprecated use `package.addField(x, y)`
     */
    get manifest() {
        return this.package.manifest;
    }
    constructor(options) {
        super({
            ...options,
            // Node projects have the specific projen version locked via lockfile, so we can skip the @<VERSION> part of the top-level project
            projenCommand: options.projenCommand ?? "npx projen",
        });
        this.package = new node_package_1.NodePackage(this, options);
        this.workflowBootstrapSteps = options.workflowBootstrapSteps ?? [];
        this.workflowGitIdentity =
            options.workflowGitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.workflowPackageCache = options.workflowPackageCache ?? false;
        this.artifactsDirectory =
            options.artifactsDirectory ?? consts_1.DEFAULT_ARTIFACTS_DIRECTORY;
        (0, util_1.ensureNotHiddenPath)(this.artifactsDirectory, "artifactsDirectory");
        const normalizedArtifactsDirectory = (0, util_2.normalizePersistedPath)(this.artifactsDirectory);
        this.artifactsJavascriptDirectory = path_1.posix.join(normalizedArtifactsDirectory, "js");
        this.runScriptCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.NPM:
                    return "npm run";
                case node_package_1.NodePackageManager.YARN:
                case node_package_1.NodePackageManager.YARN2:
                case node_package_1.NodePackageManager.YARN_CLASSIC:
                case node_package_1.NodePackageManager.YARN_BERRY:
                    return "yarn run";
                case node_package_1.NodePackageManager.PNPM:
                    return "pnpm run";
                case node_package_1.NodePackageManager.BUN:
                    return "bun run";
                default:
                    throw new Error(`unexpected package manager ${this.packageManager}`);
            }
        })();
        const envCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.PNPM:
                    return '$(pnpm -c exec "node --print process.env.PATH")';
                case node_package_1.NodePackageManager.BUN:
                    return '$(bun --eval "console.log(process.env.PATH)")';
                default:
                    return '$(npx -c "node --print process.env.PATH")';
            }
        })();
        this.nodeVersion =
            options.workflowNodeVersion ?? this.package.minNodeVersion;
        // add PATH for all tasks which includes the project's npm .bin list
        this.tasks.addEnvironment("PATH", envCommand);
        this.addLicense(options);
        if (options.npmignoreEnabled ?? true) {
            this.npmignore = new ignore_file_1.IgnoreFile(this, ".npmignore", options.npmIgnoreOptions);
        }
        this.addDefaultGitIgnore();
        if (options.gitignore?.length) {
            for (const i of options.gitignore) {
                this.gitignore.exclude(i);
            }
        }
        if (options.npmignore?.length) {
            if (!this.npmignore) {
                throw new Error('.npmignore is not defined for an APP project type. Add "npmIgnore: true" to override this');
            }
            for (const i of options.npmignore) {
                this.npmignore.exclude(i);
            }
        }
        if (!this.ejected) {
            this.setScript(PROJEN_SCRIPT, this.package.projenCommand);
        }
        this.npmignore?.exclude(`/${common_1.PROJEN_DIR}/`);
        const projen = options.projenDevDependency ?? (this.parent ? false : true);
        if (projen && !this.ejected) {
            const postfix = options.projenVersion ? `@${options.projenVersion}` : "";
            this.addDevDeps(`projen${postfix}`);
            if (!this.deps.isDependencySatisfied("constructs", dependencies_1.DependencyType.BUILD, "^10.0.0")) {
                this.addDevDeps(`constructs@^10.0.0`);
            }
        }
        if (!options.defaultReleaseBranch) {
            throw new Error('"defaultReleaseBranch" is temporarily a required option while we migrate its default value from "master" to "main"');
        }
        const buildEnabled = options.buildWorkflow ?? (this.parent ? false : true);
        // configure jest if enabled
        // must be before the build/release workflows
        if (options.jest ?? true) {
            this.jest = new jest_1.Jest(this, options.jestOptions);
        }
        const workflowPermissions = {
            idToken: this.determineIdTokenPermissions(options),
        };
        const buildWorkflowOptions = options.buildWorkflowOptions ?? {};
        if (buildEnabled && (this.github || github_1.GitHub.of(this.root))) {
            this.buildWorkflow = new build_1.BuildWorkflow(this, {
                buildTask: this.buildTask,
                artifactsDirectory: this.artifactsDirectory,
                containerImage: options.workflowContainerImage,
                gitIdentity: this.workflowGitIdentity,
                mutableBuild: options.mutableBuild,
                workflowTriggers: options.buildWorkflowTriggers,
                permissions: workflowPermissions,
                ...buildWorkflowOptions,
                preBuildSteps: this.renderWorkflowSetup({
                    installStepConfiguration: {
                        workingDirectory: this.determineInstallWorkingDirectory(),
                    },
                    mutable: buildWorkflowOptions.mutableBuild ?? options.mutableBuild ?? true,
                }).concat(buildWorkflowOptions.preBuildSteps ?? []),
                postBuildSteps: [...(options.postBuildSteps ?? [])],
                ...(0, runner_options_1.filteredRunsOnOptions)(options.workflowRunsOn, options.workflowRunsOnGroup),
            });
            this.buildWorkflow.addPostBuildSteps(...this.renderUploadCoverageJobStep(options));
            this.maybeAddCodecovIgnores(options);
        }
        const release = options.release ??
            options.releaseWorkflow ??
            (this.parent ? false : true);
        if (release) {
            this.release = new release_1.Release(this, {
                versionFile: "package.json", // this is where "version" is set after bump
                task: this.buildTask,
                branch: options.defaultReleaseBranch ?? "main",
                ...options,
                artifactsDirectory: this.artifactsDirectory,
                releaseWorkflowSetupSteps: [
                    ...this.renderWorkflowSetup({
                        installStepConfiguration: {
                            workingDirectory: this.determineInstallWorkingDirectory(),
                        },
                        mutable: false,
                    }),
                    ...(options.releaseWorkflowSetupSteps ?? []),
                ],
                postBuildSteps: [
                    ...(options.postBuildSteps ?? []),
                    ...this.renderUploadCoverageJobStep(options),
                ],
                workflowNodeVersion: this.nodeVersion,
                workflowPermissions,
            });
            this.maybeAddCodecovIgnores(options);
            this.publisher = this.release.publisher;
            const nodePackageToReleaseCodeArtifactAuthProviderMapping = {
                [node_package_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR]: release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                [node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC]: release_1.CodeArtifactAuthProvider.GITHUB_OIDC,
            };
            if (options.releaseToNpm ?? false) {
                const codeArtifactOptions = (0, release_1.isAwsCodeArtifactRegistry)(this.package.npmRegistry)
                    ? {
                        accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret,
                        secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret,
                        roleToAssume: options.codeArtifactOptions?.roleToAssume,
                        authProvider: options.codeArtifactOptions?.authProvider
                            ? nodePackageToReleaseCodeArtifactAuthProviderMapping[options.codeArtifactOptions.authProvider]
                            : release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                    }
                    : {};
                this.release.publisher.publishToNpm({
                    registry: this.package.npmRegistry,
                    npmTokenSecret: this.package.npmTokenSecret,
                    npmProvenance: this.package.npmProvenance,
                    trustedPublishing: options.npmTrustedPublishing ?? false,
                    codeArtifactOptions,
                });
            }
        }
        else {
            // validate that no release options are selected if the release workflow is disabled.
            if (options.releaseToNpm) {
                throw new Error('"releaseToNpm" is not supported if "release" is not set');
            }
            if (options.releaseEveryCommit) {
                throw new Error('"releaseEveryCommit" is not supported if "release" is not set');
            }
            if (options.releaseSchedule) {
                throw new Error('"releaseSchedule" is not supported if "release" is not set');
            }
        }
        if ((options.autoMerge ?? true) &&
            this.github?.mergify &&
            this.buildWorkflow?.buildJobIds) {
            this.autoMerge = new github_1.AutoMerge(this.github, options.autoMergeOptions);
            this.autoMerge.addConditionsLater({
                render: () => this.buildWorkflow?.buildJobIds.map((id) => `status-success=${id}`) ??
                    [],
            });
        }
        const dependabot = options.dependabot ?? false;
        const depsUpgrade = options.depsUpgrade ?? !dependabot;
        if (dependabot && depsUpgrade) {
            throw new Error("'dependabot' cannot be configured together with 'depsUpgrade'");
        }
        const depsAutoApprove = options.autoApproveUpgrades ?? false;
        if (depsAutoApprove && !this.autoApprove && this.github) {
            throw new Error("Automatic approval of dependencies upgrades requires configuring `autoApproveOptions`");
        }
        const autoApproveLabel = (condition) => condition && this.autoApprove?.label
            ? [this.autoApprove.label]
            : undefined;
        if (dependabot) {
            const defaultOptions = {
                labels: autoApproveLabel(depsAutoApprove),
            };
            this.github?.addDependabot((0, util_2.deepMerge)([defaultOptions, options.dependabotOptions ?? {}]));
        }
        if (depsUpgrade) {
            const defaultOptions = {
                workflowOptions: {
                    container: options.workflowContainerImage
                        ? {
                            image: options.workflowContainerImage,
                        }
                        : undefined,
                    labels: autoApproveLabel(depsAutoApprove),
                    gitIdentity: this.workflowGitIdentity,
                    permissions: workflowPermissions,
                },
            };
            this.upgradeWorkflow = new javascript_1.UpgradeDependencies(this, (0, util_2.deepMerge)([defaultOptions, options.depsUpgradeOptions ?? {}]));
        }
        if (options.pullRequestTemplate ?? true) {
            this.github?.addPullRequestTemplate(...(options.pullRequestTemplateContents ?? []));
        }
        const projenrcJs = options.projenrcJs ?? !options.projenrcJson;
        if (!this.parent && projenrcJs) {
            const projenrcJsFile = new projenrc_1.Projenrc(this, options.projenrcJsOptions);
            this.npmignore?.exclude(`/${projenrcJsFile.filePath}`);
        }
        else if (options.projenrcJson) {
            const projenrcJsonFile = projenrc_json_1.ProjenrcJson.of(this);
            if (projenrcJsonFile) {
                this.npmignore?.exclude(`/${projenrcJsonFile.filePath}`);
            }
        }
        // add a bundler component - this enables things like Lambda bundling and in the future web bundling.
        this.bundler = new bundler_1.Bundler(this, options.bundlerOptions);
        const shouldPackage = options.package ?? true;
        if (release && !shouldPackage) {
            this.logger.warn("When `release` is enabled, `package` must also be enabled as it is required by release. Force enabling `package`.");
        }
        if (release || shouldPackage) {
            this.packageTask.exec(`mkdir -p ${this.artifactsJavascriptDirectory}`);
            const pkgMgr = this.package.packageManager === node_package_1.NodePackageManager.PNPM
                ? "pnpm"
                : "npm";
            this.packageTask.exec(`${pkgMgr} pack --pack-destination ${this.artifactsJavascriptDirectory}`);
        }
        if ((0, util_2.multipleSelected)([options.biome, options.prettier])) {
            throw new Error("Only one of biome and prettier can be enabled.");
        }
        if (options.biome ?? false) {
            this.biome = new biome_1.Biome(this, { ...options.biomeOptions });
        }
        if (options.prettier ?? false) {
            this.prettier = new javascript_1.Prettier(this, { ...options.prettierOptions });
        }
        // For PNPM, the default resolution mode is "lowest", which leads to any non-versioned (ie '*') dependencies being
        // resolved to the lowest available version, which is unlikely to be expected behaviour for users. We set resolution
        // mode to "highest" to match the behaviour of yarn and npm.
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            this.npmrc.addConfig("resolution-mode", "highest");
        }
        if (options.checkLicenses) {
            new license_checker_1.LicenseChecker(this, options.checkLicenses);
        }
    }
    determineInstallWorkingDirectory() {
        if (this.parent) {
            return (0, path_2.ensureRelativePathStartsWithDot)((0, path_1.relative)(".", this.root.outdir));
        }
        return;
    }
    determineIdTokenPermissions(options) {
        const { codeCovTokenSecret, scopedPackagesOptions, codeArtifactOptions } = options;
        const useCodeCoveOidc = this.useCodecov(options) && !codeCovTokenSecret;
        const hasScopedPackages = (scopedPackagesOptions ?? []).length > 0;
        const useCodeArtifactOidc = codeArtifactOptions?.authProvider ===
            release_1.CodeArtifactAuthProvider.GITHUB_OIDC;
        if (useCodeCoveOidc || (useCodeArtifactOidc && hasScopedPackages)) {
            return workflows_model_1.JobPermission.WRITE;
        }
        return undefined;
    }
    useCodecov(options) {
        // Use Codecov when it is enabled or if or a secret token name is passed in
        // AND jest must be configured
        return (options.codeCov || options.codeCovTokenSecret) && this.jest?.config;
    }
    maybeAddCodecovIgnores(options) {
        if (this.useCodecov(options)) {
            this.addGitIgnore("codecov");
            this.addGitIgnore("codecov.*");
        }
    }
    renderUploadCoverageJobStep(options) {
        // run codecov if enabled or a secret token name is passed in
        // AND jest must be configured
        if (this.useCodecov(options)) {
            return [
                {
                    name: "Upload coverage to Codecov",
                    uses: "codecov/codecov-action@v5",
                    with: options.codeCovTokenSecret
                        ? {
                            token: `\${{ secrets.${options.codeCovTokenSecret} }}`,
                            directory: this.jest?.config.coverageDirectory,
                        }
                        : {
                            directory: this.jest?.config.coverageDirectory,
                            use_oidc: true,
                        },
                },
            ];
        }
        else {
            return [];
        }
    }
    addBins(bins) {
        this.package.addBin(bins);
    }
    /**
     * Replaces the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.package.setScript(name, command);
    }
    /**
     * Replaces the contents of multiple npm package.json scripts.
     * @param scripts The scripts to set
     */
    addScripts(scripts) {
        for (const [name, command] of Object.entries(scripts)) {
            this.package.setScript(name, command);
        }
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.package.removeScript(name);
    }
    /**
     * Indicates if a script by the name name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.package.hasScript(name);
    }
    /**
     * DEPRECATED
     * @deprecated use `project.compileTask.exec()`
     */
    addCompileCommand(...commands) {
        for (const c of commands) {
            this.compileTask.exec(c);
        }
    }
    /**
     * DEPRECATED
     * @deprecated use `project.testTask.exec()`
     */
    addTestCommand(...commands) {
        for (const c of commands) {
            this.testTask.exec(c);
        }
    }
    /**
     * Directly set fields in `package.json`.
     * @param fields The fields to set
     */
    addFields(fields) {
        for (const [name, value] of Object.entries(fields)) {
            this.package.addField(name, value);
        }
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        this.package.addKeywords(...keywords);
    }
    /**
     * Get steps for scoped package access
     *
     * @param codeArtifactOptions Details of logging in to AWS
     * @returns array of job steps required for each private scoped packages
     */
    getScopedPackageSteps(codeArtifactOptions) {
        const parsedCodeArtifactOptions = {
            accessKeyIdSecret: codeArtifactOptions?.accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID",
            secretAccessKeySecret: codeArtifactOptions?.secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY",
            roleToAssume: codeArtifactOptions?.roleToAssume,
            authProvider: codeArtifactOptions?.authProvider,
        };
        if (parsedCodeArtifactOptions.authProvider ===
            node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v5",
                    with: {
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        if (parsedCodeArtifactOptions.roleToAssume) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v5",
                    with: {
                        "aws-access-key-id": (0, util_1.secretToString)(parsedCodeArtifactOptions.accessKeyIdSecret),
                        "aws-secret-access-key": (0, util_1.secretToString)(parsedCodeArtifactOptions.secretAccessKeySecret),
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        return [
            {
                name: "AWS CodeArtifact Login",
                run: `${this.runScriptCommand} ca:login`,
                env: {
                    AWS_ACCESS_KEY_ID: (0, util_1.secretToString)(parsedCodeArtifactOptions.accessKeyIdSecret),
                    AWS_SECRET_ACCESS_KEY: (0, util_1.secretToString)(parsedCodeArtifactOptions.secretAccessKeySecret),
                },
            },
        ];
    }
    /**
     * Returns the set of workflow steps which should be executed to bootstrap a
     * workflow.
     *
     * @param options Options.
     * @returns Job steps
     */
    renderWorkflowSetup(options = {}) {
        const install = new Array();
        // first run the workflow bootstrap steps
        install.push(...this.workflowBootstrapSteps);
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            install.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v4",
                with: { version: this.package.pnpmVersion },
            });
        }
        else if (this.package.packageManager === node_package_1.NodePackageManager.BUN) {
            install.push({
                name: "Setup bun",
                uses: "oven-sh/setup-bun@v2",
                with: { "bun-version": this.package.bunVersion },
            });
        }
        if (this.package.packageManager !== node_package_1.NodePackageManager.BUN) {
            if (this.nodeVersion || this.workflowPackageCache) {
                const cache = this.package.packageManager === node_package_1.NodePackageManager.YARN
                    ? "yarn"
                    : this.package.packageManager === node_package_1.NodePackageManager.YARN2
                        ? "yarn"
                        : this.package.packageManager === node_package_1.NodePackageManager.YARN_CLASSIC
                            ? "yarn"
                            : this.package.packageManager === node_package_1.NodePackageManager.YARN_BERRY
                                ? "yarn"
                                : this.packageManager === node_package_1.NodePackageManager.BUN
                                    ? "bun"
                                    : this.package.packageManager === node_package_1.NodePackageManager.PNPM
                                        ? "pnpm"
                                        : "npm";
                install.push({
                    name: "Setup Node.js",
                    uses: "actions/setup-node@v5",
                    with: {
                        ...(this.nodeVersion && {
                            "node-version": this.nodeVersion,
                        }),
                        ...(this.workflowPackageCache && {
                            cache,
                        }),
                    },
                });
            }
        }
        const mutable = options.mutable ?? false;
        if (this.package.scopedPackagesOptions) {
            install.push(...this.getScopedPackageSteps(this.package.codeArtifactOptions));
        }
        install.push({
            name: "Install dependencies",
            run: mutable
                ? this.package.installAndUpdateLockfileCommand
                : this.package.installCommand,
            ...(options.installStepConfiguration ?? {}),
        });
        return install;
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        return this.package.addDeps(...deps);
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        return this.package.addDevDeps(...deps);
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        return this.package.addPeerDeps(...deps);
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        return this.package.addBundledDeps(...deps);
    }
    /**
     * Adds patterns to be ignored by npm.
     *
     * @param pattern The pattern to ignore.
     *
     * @remarks
     * If you are having trouble getting an ignore to populate, try using your construct or component's preSynthesize method to properly delay calling this method.
     */
    addPackageIgnore(pattern) {
        this.npmignore?.addPatterns(pattern);
    }
    addLicense(options) {
        if (this.package.license) {
            new license_1.License(this, {
                spdx: this.package.license,
                copyrightOwner: options.copyrightOwner ?? options.authorName,
                copyrightPeriod: options.copyrightPeriod,
            });
        }
    }
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Logs", "logs", "*.log", "npm-debug.log*", "yarn-debug.log*", "yarn-error.log*", "lerna-debug.log*", "# Diagnostic reports (https://nodejs.org/api/report.html)", "report.[0-9]*.[0-9]*.[0-9]*.[0-9]*.json", "# Runtime data", "pids", "*.pid", "*.seed", "*.pid.lock", "# Directory for instrumented libs generated by jscoverage/JSCover", "lib-cov", "# Coverage directory used by tools like istanbul", "coverage", "*.lcov", "# nyc test coverage", ".nyc_output", "# Compiled binary addons (https://nodejs.org/api/addons.html)", "build/Release", "# Dependency directories", "node_modules/", "jspm_packages/", "# TypeScript cache", "*.tsbuildinfo", "# Optional eslint cache", ".eslintcache", "# Output of 'npm pack'", "*.tgz", "# Yarn Integrity file", ".yarn-integrity", "# parcel-bundler cache (https://parceljs.org/)", ".cache");
    }
    /**
     * Returns the shell command to execute in order to run a task. This will
     * typically be `npx projen TASK`.
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `${this.package.projenCommand} ${task.name}`;
    }
    /**
     * The job ID of the build workflow.
     */
    get buildWorkflowJobId() {
        return this.buildWorkflow?.buildJobIds[0];
    }
}
exports.NodeProject = NodeProject;
_a = JSII_RTTI_SYMBOL_1;
NodeProject[_a] = { fqn: "projen.javascript.NodeProject", version: "0.98.4" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibm9kZS1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2phdmFzY3JpcHQvbm9kZS1wcm9qZWN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsK0JBQXVDO0FBQ3ZDLHVDQUFvRDtBQUNwRCxpQ0FBMkM7QUFDM0MsdURBQTBFO0FBQzFFLGlEQU13QjtBQUN4Qix5Q0FBdUQ7QUFDdkQsb0NBQXFFO0FBQ3JFLG9EQUFzRTtBQUN0RSxzQ0FBdUM7QUFDdkMsa0RBQWlEO0FBQ2pELHNDQU9tQjtBQUNuQix5Q0FBb0Q7QUFDcEQsbURBQWtFO0FBQ2xFLGlEQUE2RTtBQUM3RSwrREFNbUM7QUFDbkMsZ0RBQStEO0FBQy9ELDhDQU11QjtBQUN2Qix3Q0FBcUM7QUFDckMsb0RBQWdEO0FBQ2hELHdDQVFvQjtBQUNwQixzREFBMEQ7QUFFMUQsa0NBQThFO0FBQzlFLHVDQUErRDtBQUUvRCxNQUFNLGFBQWEsR0FBRyxRQUFRLENBQUM7QUEwVC9COztHQUVHO0FBQ0gsSUFBWSxXQVVYO0FBVkQsV0FBWSxXQUFXO0lBQ3JCOztPQUVHO0lBQ0gsNkRBQVksQ0FBQTtJQUVaOztPQUVHO0lBQ0gsK0NBQUssQ0FBQTtBQUNQLENBQUMsRUFWVyxXQUFXLDJCQUFYLFdBQVcsUUFVdEI7QUFFRDs7OztHQUlHO0FBQ0gsTUFBYSxXQUFZLFNBQVEsc0JBQWE7SUFXNUM7O09BRUc7SUFDSCxJQUFXLEtBQUs7UUFDZCxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ2pCLElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxzQkFBUyxDQUFDLElBQUksRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ3pELENBQUM7UUFDRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUM7SUFDckIsQ0FBQztJQUdEOztPQUVHO0lBQ0gsSUFBVyx3QkFBd0I7UUFDakMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLHdCQUF3QixDQUFDO0lBQy9DLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsVUFBVTtRQUNuQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDO0lBQ2pDLENBQUM7SUF5QkQ7Ozs7T0FJRztJQUNILElBQVcsY0FBYztRQUN2QixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsSUFBVyxjQUFjO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUM7SUFDckMsQ0FBQztJQUlEOzs7O09BSUc7SUFDSCxJQUFXLGNBQWM7UUFDdkIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQztJQUNyQyxDQUFDO0lBWUQ7O09BRUc7SUFDSCxJQUFXLFFBQVE7UUFDakIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQztJQUMvQixDQUFDO0lBMkJELFlBQVksT0FBMkI7UUFDckMsS0FBSyxDQUFDO1lBQ0osR0FBRyxPQUFPO1lBQ1Ysa0lBQWtJO1lBQ2xJLGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYSxJQUFJLFlBQVk7U0FDckQsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLDBCQUFXLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQzlDLElBQUksQ0FBQyxzQkFBc0IsR0FBRyxPQUFPLENBQUMsc0JBQXNCLElBQUksRUFBRSxDQUFDO1FBQ25FLElBQUksQ0FBQyxtQkFBbUI7WUFDdEIsT0FBTyxDQUFDLG1CQUFtQixJQUFJLHVDQUEyQixDQUFDO1FBQzdELElBQUksQ0FBQyxvQkFBb0IsR0FBRyxPQUFPLENBQUMsb0JBQW9CLElBQUksS0FBSyxDQUFDO1FBQ2xFLElBQUksQ0FBQyxrQkFBa0I7WUFDckIsT0FBTyxDQUFDLGtCQUFrQixJQUFJLG9DQUEyQixDQUFDO1FBQzVELElBQUEsMEJBQW1CLEVBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFLG9CQUFvQixDQUFDLENBQUM7UUFDbkUsTUFBTSw0QkFBNEIsR0FBRyxJQUFBLDZCQUFzQixFQUN6RCxJQUFJLENBQUMsa0JBQWtCLENBQ3hCLENBQUM7UUFDRixJQUFJLENBQUMsNEJBQTRCLEdBQUcsWUFBSyxDQUFDLElBQUksQ0FDNUMsNEJBQTRCLEVBQzVCLElBQUksQ0FDTCxDQUFDO1FBRUYsSUFBSSxDQUFDLGdCQUFnQixHQUFHLENBQUMsR0FBRyxFQUFFO1lBQzVCLFFBQVEsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDO2dCQUM1QixLQUFLLGlDQUFrQixDQUFDLEdBQUc7b0JBQ3pCLE9BQU8sU0FBUyxDQUFDO2dCQUNuQixLQUFLLGlDQUFrQixDQUFDLElBQUksQ0FBQztnQkFDN0IsS0FBSyxpQ0FBa0IsQ0FBQyxLQUFLLENBQUM7Z0JBQzlCLEtBQUssaUNBQWtCLENBQUMsWUFBWSxDQUFDO2dCQUNyQyxLQUFLLGlDQUFrQixDQUFDLFVBQVU7b0JBQ2hDLE9BQU8sVUFBVSxDQUFDO2dCQUNwQixLQUFLLGlDQUFrQixDQUFDLElBQUk7b0JBQzFCLE9BQU8sVUFBVSxDQUFDO2dCQUNwQixLQUFLLGlDQUFrQixDQUFDLEdBQUc7b0JBQ3pCLE9BQU8sU0FBUyxDQUFDO2dCQUNuQjtvQkFDRSxNQUFNLElBQUksS0FBSyxDQUFDLDhCQUE4QixJQUFJLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQztZQUN6RSxDQUFDO1FBQ0gsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUVMLE1BQU0sVUFBVSxHQUFHLENBQUMsR0FBRyxFQUFFO1lBQ3ZCLFFBQVEsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDO2dCQUM1QixLQUFLLGlDQUFrQixDQUFDLElBQUk7b0JBQzFCLE9BQU8saURBQWlELENBQUM7Z0JBQzNELEtBQUssaUNBQWtCLENBQUMsR0FBRztvQkFDekIsT0FBTywrQ0FBK0MsQ0FBQztnQkFDekQ7b0JBQ0UsT0FBTywyQ0FBMkMsQ0FBQztZQUN2RCxDQUFDO1FBQ0gsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUVMLElBQUksQ0FBQyxXQUFXO1lBQ2QsT0FBTyxDQUFDLG1CQUFtQixJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDO1FBRTdELG9FQUFvRTtRQUNwRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFOUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUV6QixJQUFJLE9BQU8sQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLEVBQUUsQ0FBQztZQUNyQyxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksd0JBQVUsQ0FDN0IsSUFBSSxFQUNKLFlBQVksRUFDWixPQUFPLENBQUMsZ0JBQWdCLENBQ3pCLENBQUM7UUFDSixDQUFDO1FBRUQsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUM7UUFFM0IsSUFBSSxPQUFPLENBQUMsU0FBUyxFQUFFLE1BQU0sRUFBRSxDQUFDO1lBQzlCLEtBQUssTUFBTSxDQUFDLElBQUksT0FBTyxDQUFDLFNBQVMsRUFBRSxDQUFDO2dCQUNsQyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM1QixDQUFDO1FBQ0gsQ0FBQztRQUVELElBQUksT0FBTyxDQUFDLFNBQVMsRUFBRSxNQUFNLEVBQUUsQ0FBQztZQUM5QixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxDQUFDO2dCQUNwQixNQUFNLElBQUksS0FBSyxDQUNiLDJGQUEyRixDQUM1RixDQUFDO1lBQ0osQ0FBQztZQUVELEtBQUssTUFBTSxDQUFDLElBQUksT0FBTyxDQUFDLFNBQVMsRUFBRSxDQUFDO2dCQUNsQyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM1QixDQUFDO1FBQ0gsQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDbEIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxhQUFhLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUM1RCxDQUFDO1FBRUQsSUFBSSxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsSUFBSSxtQkFBVSxHQUFHLENBQUMsQ0FBQztRQUUzQyxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsbUJBQW1CLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzNFLElBQUksTUFBTSxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDO1lBQzVCLE1BQU0sT0FBTyxHQUFHLE9BQU8sQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLElBQUksT0FBTyxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7WUFDekUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxTQUFTLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFFcEMsSUFDRSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQzlCLFlBQVksRUFDWiw2QkFBYyxDQUFDLEtBQUssRUFDcEIsU0FBUyxDQUNWLEVBQ0QsQ0FBQztnQkFDRCxJQUFJLENBQUMsVUFBVSxDQUFDLG9CQUFvQixDQUFDLENBQUM7WUFDeEMsQ0FBQztRQUNILENBQUM7UUFFRCxJQUFJLENBQUMsT0FBTyxDQUFDLG9CQUFvQixFQUFFLENBQUM7WUFDbEMsTUFBTSxJQUFJLEtBQUssQ0FDYixvSEFBb0gsQ0FDckgsQ0FBQztRQUNKLENBQUM7UUFFRCxNQUFNLFlBQVksR0FBRyxPQUFPLENBQUMsYUFBYSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUUzRSw0QkFBNEI7UUFDNUIsNkNBQTZDO1FBQzdDLElBQUksT0FBTyxDQUFDLElBQUksSUFBSSxJQUFJLEVBQUUsQ0FBQztZQUN6QixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksV0FBSSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDbEQsQ0FBQztRQUVELE1BQU0sbUJBQW1CLEdBQW1CO1lBQzFDLE9BQU8sRUFBRSxJQUFJLENBQUMsMkJBQTJCLENBQUMsT0FBTyxDQUFDO1NBQ25ELENBQUM7UUFFRixNQUFNLG9CQUFvQixHQUN4QixPQUFPLENBQUMsb0JBQW9CLElBQUksRUFBRSxDQUFDO1FBRXJDLElBQUksWUFBWSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sSUFBSSxlQUFNLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFLENBQUM7WUFDMUQsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLHFCQUFhLENBQUMsSUFBSSxFQUFFO2dCQUMzQyxTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVM7Z0JBQ3pCLGtCQUFrQixFQUFFLElBQUksQ0FBQyxrQkFBa0I7Z0JBQzNDLGNBQWMsRUFBRSxPQUFPLENBQUMsc0JBQXNCO2dCQUM5QyxXQUFXLEVBQUUsSUFBSSxDQUFDLG1CQUFtQjtnQkFDckMsWUFBWSxFQUFFLE9BQU8sQ0FBQyxZQUFZO2dCQUNsQyxnQkFBZ0IsRUFBRSxPQUFPLENBQUMscUJBQXFCO2dCQUMvQyxXQUFXLEVBQUUsbUJBQW1CO2dCQUNoQyxHQUFHLG9CQUFvQjtnQkFDdkIsYUFBYSxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQztvQkFDdEMsd0JBQXdCLEVBQUU7d0JBQ3hCLGdCQUFnQixFQUFFLElBQUksQ0FBQyxnQ0FBZ0MsRUFBRTtxQkFDMUQ7b0JBQ0QsT0FBTyxFQUNMLG9CQUFvQixDQUFDLFlBQVksSUFBSSxPQUFPLENBQUMsWUFBWSxJQUFJLElBQUk7aUJBQ3BFLENBQUMsQ0FBQyxNQUFNLENBQUMsb0JBQW9CLENBQUMsYUFBYSxJQUFJLEVBQUUsQ0FBQztnQkFDbkQsY0FBYyxFQUFFLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLElBQUksRUFBRSxDQUFDLENBQUM7Z0JBQ25ELEdBQUcsSUFBQSxzQ0FBcUIsRUFDdEIsT0FBTyxDQUFDLGNBQWMsRUFDdEIsT0FBTyxDQUFDLG1CQUFtQixDQUM1QjthQUNGLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxhQUFhLENBQUMsaUJBQWlCLENBQ2xDLEdBQUcsSUFBSSxDQUFDLDJCQUEyQixDQUFDLE9BQU8sQ0FBQyxDQUM3QyxDQUFDO1lBQ0YsSUFBSSxDQUFDLHNCQUFzQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ3ZDLENBQUM7UUFFRCxNQUFNLE9BQU8sR0FDWCxPQUFPLENBQUMsT0FBTztZQUNmLE9BQU8sQ0FBQyxlQUFlO1lBQ3ZCLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMvQixJQUFJLE9BQU8sRUFBRSxDQUFDO1lBQ1osSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLGlCQUFPLENBQUMsSUFBSSxFQUFFO2dCQUMvQixXQUFXLEVBQUUsY0FBYyxFQUFFLDRDQUE0QztnQkFDekUsSUFBSSxFQUFFLElBQUksQ0FBQyxTQUFTO2dCQUNwQixNQUFNLEVBQUUsT0FBTyxDQUFDLG9CQUFvQixJQUFJLE1BQU07Z0JBQzlDLEdBQUcsT0FBTztnQkFFVixrQkFBa0IsRUFBRSxJQUFJLENBQUMsa0JBQWtCO2dCQUMzQyx5QkFBeUIsRUFBRTtvQkFDekIsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUM7d0JBQzFCLHdCQUF3QixFQUFFOzRCQUN4QixnQkFBZ0IsRUFBRSxJQUFJLENBQUMsZ0NBQWdDLEVBQUU7eUJBQzFEO3dCQUNELE9BQU8sRUFBRSxLQUFLO3FCQUNmLENBQUM7b0JBQ0YsR0FBRyxDQUFDLE9BQU8sQ0FBQyx5QkFBeUIsSUFBSSxFQUFFLENBQUM7aUJBQzdDO2dCQUNELGNBQWMsRUFBRTtvQkFDZCxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsSUFBSSxFQUFFLENBQUM7b0JBQ2pDLEdBQUcsSUFBSSxDQUFDLDJCQUEyQixDQUFDLE9BQU8sQ0FBQztpQkFDN0M7Z0JBRUQsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLFdBQVc7Z0JBQ3JDLG1CQUFtQjthQUNwQixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsc0JBQXNCLENBQUMsT0FBTyxDQUFDLENBQUM7WUFDckMsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQztZQUV4QyxNQUFNLG1EQUFtRCxHQUdyRDtnQkFDRixDQUFDLHVDQUFtQyxDQUFDLDBCQUEwQixDQUFDLEVBQzlELGtDQUErQixDQUFDLDBCQUEwQjtnQkFDNUQsQ0FBQyx1Q0FBbUMsQ0FBQyxXQUFXLENBQUMsRUFDL0Msa0NBQStCLENBQUMsV0FBVzthQUM5QyxDQUFDO1lBRUYsSUFBSSxPQUFPLENBQUMsWUFBWSxJQUFJLEtBQUssRUFBRSxDQUFDO2dCQUNsQyxNQUFNLG1CQUFtQixHQUN2QixJQUFBLG1DQUF5QixFQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDO29CQUNqRCxDQUFDLENBQUM7d0JBQ0UsaUJBQWlCLEVBQ2YsT0FBTyxDQUFDLG1CQUFtQixFQUFFLGlCQUFpQjt3QkFDaEQscUJBQXFCLEVBQ25CLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxxQkFBcUI7d0JBQ3BELFlBQVksRUFBRSxPQUFPLENBQUMsbUJBQW1CLEVBQUUsWUFBWTt3QkFDdkQsWUFBWSxFQUFFLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxZQUFZOzRCQUNyRCxDQUFDLENBQUMsbURBQW1ELENBQ2pELE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxZQUFZLENBQ3pDOzRCQUNILENBQUMsQ0FBQyxrQ0FBK0IsQ0FBQywwQkFBMEI7cUJBQy9EO29CQUNILENBQUMsQ0FBQyxFQUFFLENBQUM7Z0JBQ1QsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDO29CQUNsQyxRQUFRLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXO29CQUNsQyxjQUFjLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjO29CQUMzQyxhQUFhLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhO29CQUN6QyxpQkFBaUIsRUFBRSxPQUFPLENBQUMsb0JBQW9CLElBQUksS0FBSztvQkFDeEQsbUJBQW1CO2lCQUNwQixDQUFDLENBQUM7WUFDTCxDQUFDO1FBQ0gsQ0FBQzthQUFNLENBQUM7WUFDTixxRkFBcUY7WUFDckYsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFLENBQUM7Z0JBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQ2IseURBQXlELENBQzFELENBQUM7WUFDSixDQUFDO1lBRUQsSUFBSSxPQUFPLENBQUMsa0JBQWtCLEVBQUUsQ0FBQztnQkFDL0IsTUFBTSxJQUFJLEtBQUssQ0FDYiwrREFBK0QsQ0FDaEUsQ0FBQztZQUNKLENBQUM7WUFFRCxJQUFJLE9BQU8sQ0FBQyxlQUFlLEVBQUUsQ0FBQztnQkFDNUIsTUFBTSxJQUFJLEtBQUssQ0FDYiw0REFBNEQsQ0FDN0QsQ0FBQztZQUNKLENBQUM7UUFDSCxDQUFDO1FBRUQsSUFDRSxDQUFDLE9BQU8sQ0FBQyxTQUFTLElBQUksSUFBSSxDQUFDO1lBQzNCLElBQUksQ0FBQyxNQUFNLEVBQUUsT0FBTztZQUNwQixJQUFJLENBQUMsYUFBYSxFQUFFLFdBQVcsRUFDL0IsQ0FBQztZQUNELElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxrQkFBUyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLGdCQUFnQixDQUFDLENBQUM7WUFDdEUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQztnQkFDaEMsTUFBTSxFQUFFLEdBQUcsRUFBRSxDQUNYLElBQUksQ0FBQyxhQUFhLEVBQUUsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsa0JBQWtCLEVBQUUsRUFBRSxDQUFDO29CQUNuRSxFQUFFO2FBQ0wsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztRQUVELE1BQU0sVUFBVSxHQUFHLE9BQU8sQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDO1FBQy9DLE1BQU0sV0FBVyxHQUFHLE9BQU8sQ0FBQyxXQUFXLElBQUksQ0FBQyxVQUFVLENBQUM7UUFFdkQsSUFBSSxVQUFVLElBQUksV0FBVyxFQUFFLENBQUM7WUFDOUIsTUFBTSxJQUFJLEtBQUssQ0FDYiwrREFBK0QsQ0FDaEUsQ0FBQztRQUNKLENBQUM7UUFFRCxNQUFNLGVBQWUsR0FBRyxPQUFPLENBQUMsbUJBQW1CLElBQUksS0FBSyxDQUFDO1FBRTdELElBQUksZUFBZSxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUM7WUFDeEQsTUFBTSxJQUFJLEtBQUssQ0FDYix1RkFBdUYsQ0FDeEYsQ0FBQztRQUNKLENBQUM7UUFFRCxNQUFNLGdCQUFnQixHQUFHLENBQUMsU0FBa0IsRUFBRSxFQUFFLENBQzlDLFNBQVMsSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFLEtBQUs7WUFDbEMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUM7WUFDMUIsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVoQixJQUFJLFVBQVUsRUFBRSxDQUFDO1lBQ2YsTUFBTSxjQUFjLEdBQUc7Z0JBQ3JCLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxlQUFlLENBQUM7YUFDMUMsQ0FBQztZQUNGLElBQUksQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUN4QixJQUFBLGdCQUFTLEVBQUMsQ0FBQyxjQUFjLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQzdELENBQUM7UUFDSixDQUFDO1FBRUQsSUFBSSxXQUFXLEVBQUUsQ0FBQztZQUNoQixNQUFNLGNBQWMsR0FBK0I7Z0JBQ2pELGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUUsT0FBTyxDQUFDLHNCQUFzQjt3QkFDdkMsQ0FBQyxDQUFDOzRCQUNFLEtBQUssRUFBRSxPQUFPLENBQUMsc0JBQXNCO3lCQUN0Qzt3QkFDSCxDQUFDLENBQUMsU0FBUztvQkFDYixNQUFNLEVBQUUsZ0JBQWdCLENBQUMsZUFBZSxDQUFDO29CQUN6QyxXQUFXLEVBQUUsSUFBSSxDQUFDLG1CQUFtQjtvQkFDckMsV0FBVyxFQUFFLG1CQUFtQjtpQkFDakM7YUFDRixDQUFDO1lBQ0YsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLGdDQUFtQixDQUM1QyxJQUFJLEVBQ0osSUFBQSxnQkFBUyxFQUFDLENBQUMsY0FBYyxFQUFFLE9BQU8sQ0FBQyxrQkFBa0IsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUM5RCxDQUFDO1FBQ0osQ0FBQztRQUVELElBQUksT0FBTyxDQUFDLG1CQUFtQixJQUFJLElBQUksRUFBRSxDQUFDO1lBQ3hDLElBQUksQ0FBQyxNQUFNLEVBQUUsc0JBQXNCLENBQ2pDLEdBQUcsQ0FBQyxPQUFPLENBQUMsMkJBQTJCLElBQUksRUFBRSxDQUFDLENBQy9DLENBQUM7UUFDSixDQUFDO1FBRUQsTUFBTSxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUM7UUFDL0QsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLElBQUksVUFBVSxFQUFFLENBQUM7WUFDL0IsTUFBTSxjQUFjLEdBQUcsSUFBSSxtQkFBUSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztZQUVyRSxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxJQUFJLGNBQWMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ3pELENBQUM7YUFBTSxJQUFJLE9BQU8sQ0FBQyxZQUFZLEVBQUUsQ0FBQztZQUNoQyxNQUFNLGdCQUFnQixHQUFHLDRCQUFZLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQy9DLElBQUksZ0JBQWdCLEVBQUUsQ0FBQztnQkFDckIsSUFBSSxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsSUFBSSxnQkFBZ0IsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1lBQzNELENBQUM7UUFDSCxDQUFDO1FBRUQscUdBQXFHO1FBQ3JHLElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxpQkFBTyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFekQsTUFBTSxhQUFhLEdBQUcsT0FBTyxDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUM7UUFDOUMsSUFBSSxPQUFPLElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUM5QixJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDZCxtSEFBbUgsQ0FDcEgsQ0FBQztRQUNKLENBQUM7UUFDRCxJQUFJLE9BQU8sSUFBSSxhQUFhLEVBQUUsQ0FBQztZQUM3QixJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxZQUFZLElBQUksQ0FBQyw0QkFBNEIsRUFBRSxDQUFDLENBQUM7WUFFdkUsTUFBTSxNQUFNLEdBQ1YsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEtBQUssaUNBQWtCLENBQUMsSUFBSTtnQkFDckQsQ0FBQyxDQUFDLE1BQU07Z0JBQ1IsQ0FBQyxDQUFDLEtBQUssQ0FBQztZQUNaLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUNuQixHQUFHLE1BQU0sNEJBQTRCLElBQUksQ0FBQyw0QkFBNEIsRUFBRSxDQUN6RSxDQUFDO1FBQ0osQ0FBQztRQUVELElBQUksSUFBQSx1QkFBZ0IsRUFBQyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLEVBQUUsQ0FBQztZQUN4RCxNQUFNLElBQUksS0FBSyxDQUFDLGdEQUFnRCxDQUFDLENBQUM7UUFDcEUsQ0FBQztRQUVELElBQUksT0FBTyxDQUFDLEtBQUssSUFBSSxLQUFLLEVBQUUsQ0FBQztZQUMzQixJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksYUFBSyxDQUFDLElBQUksRUFBRSxFQUFFLEdBQUcsT0FBTyxDQUFDLFlBQVksRUFBRSxDQUFDLENBQUM7UUFDNUQsQ0FBQztRQUVELElBQUksT0FBTyxDQUFDLFFBQVEsSUFBSSxLQUFLLEVBQUUsQ0FBQztZQUM5QixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUkscUJBQVEsQ0FBQyxJQUFJLEVBQUUsRUFBRSxHQUFHLE9BQU8sQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDO1FBQ3JFLENBQUM7UUFFRCxrSEFBa0g7UUFDbEgsb0hBQW9IO1FBQ3BILDREQUE0RDtRQUM1RCxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxLQUFLLGlDQUFrQixDQUFDLElBQUksRUFBRSxDQUFDO1lBQzVELElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLGlCQUFpQixFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3JELENBQUM7UUFFRCxJQUFJLE9BQU8sQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUMxQixJQUFJLGdDQUFjLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUNsRCxDQUFDO0lBQ0gsQ0FBQztJQUVPLGdDQUFnQztRQUN0QyxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQztZQUNoQixPQUFPLElBQUEsc0NBQStCLEVBQUMsSUFBQSxlQUFRLEVBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztRQUMxRSxDQUFDO1FBQ0QsT0FBTztJQUNULENBQUM7SUFFTywyQkFBMkIsQ0FDakMsT0FBMkI7UUFFM0IsTUFBTSxFQUFFLGtCQUFrQixFQUFFLHFCQUFxQixFQUFFLG1CQUFtQixFQUFFLEdBQ3RFLE9BQU8sQ0FBQztRQUVWLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQztRQUN4RSxNQUFNLGlCQUFpQixHQUFHLENBQUMscUJBQXFCLElBQUksRUFBRSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztRQUNuRSxNQUFNLG1CQUFtQixHQUN2QixtQkFBbUIsRUFBRSxZQUFZO1lBQ2pDLGtDQUF3QixDQUFDLFdBQVcsQ0FBQztRQUV2QyxJQUFJLGVBQWUsSUFBSSxDQUFDLG1CQUFtQixJQUFJLGlCQUFpQixDQUFDLEVBQUUsQ0FBQztZQUNsRSxPQUFPLCtCQUFhLENBQUMsS0FBSyxDQUFDO1FBQzdCLENBQUM7UUFFRCxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRU8sVUFBVSxDQUFDLE9BQTJCO1FBQzVDLDJFQUEyRTtRQUMzRSw4QkFBOEI7UUFDOUIsT0FBTyxDQUFDLE9BQU8sQ0FBQyxPQUFPLElBQUksT0FBTyxDQUFDLGtCQUFrQixDQUFDLElBQUksSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLENBQUM7SUFDOUUsQ0FBQztJQUVPLHNCQUFzQixDQUFDLE9BQTJCO1FBQ3hELElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQzdCLElBQUksQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDN0IsSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUNqQyxDQUFDO0lBQ0gsQ0FBQztJQUVPLDJCQUEyQixDQUFDLE9BQTJCO1FBQzdELDZEQUE2RDtRQUM3RCw4QkFBOEI7UUFDOUIsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDN0IsT0FBTztnQkFDTDtvQkFDRSxJQUFJLEVBQUUsNEJBQTRCO29CQUNsQyxJQUFJLEVBQUUsMkJBQTJCO29CQUNqQyxJQUFJLEVBQUUsT0FBTyxDQUFDLGtCQUFrQjt3QkFDOUIsQ0FBQyxDQUFDOzRCQUNFLEtBQUssRUFBRSxnQkFBZ0IsT0FBTyxDQUFDLGtCQUFrQixLQUFLOzRCQUN0RCxTQUFTLEVBQUUsSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsaUJBQWlCO3lCQUMvQzt3QkFDSCxDQUFDLENBQUM7NEJBQ0UsU0FBUyxFQUFFLElBQUksQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLGlCQUFpQjs0QkFDOUMsUUFBUSxFQUFFLElBQUk7eUJBQ2Y7aUJBQ047YUFDRixDQUFDO1FBQ0osQ0FBQzthQUFNLENBQUM7WUFDTixPQUFPLEVBQUUsQ0FBQztRQUNaLENBQUM7SUFDSCxDQUFDO0lBRU0sT0FBTyxDQUFDLElBQTRCO1FBQ3pDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzVCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLFNBQVMsQ0FBQyxJQUFZLEVBQUUsT0FBZTtRQUM1QyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFVBQVUsQ0FBQyxPQUFtQztRQUNuRCxLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQ3RELElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN4QyxDQUFDO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFlBQVksQ0FBQyxJQUFZO1FBQzlCLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksU0FBUyxDQUFDLElBQVk7UUFDM0IsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksaUJBQWlCLENBQUMsR0FBRyxRQUFrQjtRQUM1QyxLQUFLLE1BQU0sQ0FBQyxJQUFJLFFBQVEsRUFBRSxDQUFDO1lBQ3pCLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzNCLENBQUM7SUFDSCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksY0FBYyxDQUFDLEdBQUcsUUFBa0I7UUFDekMsS0FBSyxNQUFNLENBQUMsSUFBSSxRQUFRLEVBQUUsQ0FBQztZQUN6QixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN4QixDQUFDO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFNBQVMsQ0FBQyxNQUErQjtRQUM5QyxLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDO1lBQ25ELElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNyQyxDQUFDO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFdBQVcsQ0FBQyxHQUFHLFFBQWtCO1FBQ3RDLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0sscUJBQXFCLENBQzNCLG1CQUFvRDtRQUVwRCxNQUFNLHlCQUF5QixHQUFHO1lBQ2hDLGlCQUFpQixFQUNmLG1CQUFtQixFQUFFLGlCQUFpQixJQUFJLG1CQUFtQjtZQUMvRCxxQkFBcUIsRUFDbkIsbUJBQW1CLEVBQUUscUJBQXFCLElBQUksdUJBQXVCO1lBQ3ZFLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxZQUFZO1lBQy9DLFlBQVksRUFBRSxtQkFBbUIsRUFBRSxZQUFZO1NBQ2hELENBQUM7UUFFRixJQUNFLHlCQUF5QixDQUFDLFlBQVk7WUFDdEMsdUNBQW1DLENBQUMsV0FBVyxFQUMvQyxDQUFDO1lBQ0QsT0FBTztnQkFDTDtvQkFDRSxJQUFJLEVBQUUsMkJBQTJCO29CQUNqQyxJQUFJLEVBQUUsMENBQTBDO29CQUNoRCxJQUFJLEVBQUU7d0JBQ0osWUFBWSxFQUFFLFdBQVc7d0JBQ3pCLGdCQUFnQixFQUFFLHlCQUF5QixDQUFDLFlBQVk7d0JBQ3hELHVCQUF1QixFQUFFLEdBQUc7cUJBQzdCO2lCQUNGO2dCQUNEO29CQUNFLElBQUksRUFBRSx3QkFBd0I7b0JBQzlCLEdBQUcsRUFBRSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsV0FBVztpQkFDekM7YUFDRixDQUFDO1FBQ0osQ0FBQztRQUVELElBQUkseUJBQXlCLENBQUMsWUFBWSxFQUFFLENBQUM7WUFDM0MsT0FBTztnQkFDTDtvQkFDRSxJQUFJLEVBQUUsMkJBQTJCO29CQUNqQyxJQUFJLEVBQUUsMENBQTBDO29CQUNoRCxJQUFJLEVBQUU7d0JBQ0osbUJBQW1CLEVBQUUsSUFBQSxxQkFBYyxFQUNqQyx5QkFBeUIsQ0FBQyxpQkFBaUIsQ0FDNUM7d0JBQ0QsdUJBQXVCLEVBQUUsSUFBQSxxQkFBYyxFQUNyQyx5QkFBeUIsQ0FBQyxxQkFBcUIsQ0FDaEQ7d0JBQ0QsWUFBWSxFQUFFLFdBQVc7d0JBQ3pCLGdCQUFnQixFQUFFLHlCQUF5QixDQUFDLFlBQVk7d0JBQ3hELHVCQUF1QixFQUFFLEdBQUc7cUJBQzdCO2lCQUNGO2dCQUNEO29CQUNFLElBQUksRUFBRSx3QkFBd0I7b0JBQzlCLEdBQUcsRUFBRSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsV0FBVztpQkFDekM7YUFDRixDQUFDO1FBQ0osQ0FBQztRQUVELE9BQU87WUFDTDtnQkFDRSxJQUFJLEVBQUUsd0JBQXdCO2dCQUM5QixHQUFHLEVBQUUsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLFdBQVc7Z0JBQ3hDLEdBQUcsRUFBRTtvQkFDSCxpQkFBaUIsRUFBRSxJQUFBLHFCQUFjLEVBQy9CLHlCQUF5QixDQUFDLGlCQUFpQixDQUM1QztvQkFDRCxxQkFBcUIsRUFBRSxJQUFBLHFCQUFjLEVBQ25DLHlCQUF5QixDQUFDLHFCQUFxQixDQUNoRDtpQkFDRjthQUNGO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUIsQ0FDeEIsVUFBc0MsRUFBRTtRQUV4QyxNQUFNLE9BQU8sR0FBRyxJQUFJLEtBQUssRUFBVyxDQUFDO1FBRXJDLHlDQUF5QztRQUN6QyxPQUFPLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFFN0MsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsS0FBSyxpQ0FBa0IsQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUM1RCxPQUFPLENBQUMsSUFBSSxDQUFDO2dCQUNYLElBQUksRUFBRSxZQUFZO2dCQUNsQixJQUFJLEVBQUUsc0JBQXNCO2dCQUM1QixJQUFJLEVBQUUsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUU7YUFDNUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQzthQUFNLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEtBQUssaUNBQWtCLENBQUMsR0FBRyxFQUFFLENBQUM7WUFDbEUsT0FBTyxDQUFDLElBQUksQ0FBQztnQkFDWCxJQUFJLEVBQUUsV0FBVztnQkFDakIsSUFBSSxFQUFFLHNCQUFzQjtnQkFDNUIsSUFBSSxFQUFFLEVBQUUsYUFBYSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxFQUFFO2FBQ2pELENBQUMsQ0FBQztRQUNMLENBQUM7UUFFRCxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxLQUFLLGlDQUFrQixDQUFDLEdBQUcsRUFBRSxDQUFDO1lBQzNELElBQUksSUFBSSxDQUFDLFdBQVcsSUFBSSxJQUFJLENBQUMsb0JBQW9CLEVBQUUsQ0FBQztnQkFDbEQsTUFBTSxLQUFLLEdBQ1QsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEtBQUssaUNBQWtCLENBQUMsSUFBSTtvQkFDckQsQ0FBQyxDQUFDLE1BQU07b0JBQ1IsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxLQUFLLGlDQUFrQixDQUFDLEtBQUs7d0JBQzFELENBQUMsQ0FBQyxNQUFNO3dCQUNSLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsS0FBSyxpQ0FBa0IsQ0FBQyxZQUFZOzRCQUNqRSxDQUFDLENBQUMsTUFBTTs0QkFDUixDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEtBQUssaUNBQWtCLENBQUMsVUFBVTtnQ0FDL0QsQ0FBQyxDQUFDLE1BQU07Z0NBQ1IsQ0FBQyxDQUFDLElBQUksQ0FBQyxjQUFjLEtBQUssaUNBQWtCLENBQUMsR0FBRztvQ0FDaEQsQ0FBQyxDQUFDLEtBQUs7b0NBQ1AsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxLQUFLLGlDQUFrQixDQUFDLElBQUk7d0NBQ3pELENBQUMsQ0FBQyxNQUFNO3dDQUNSLENBQUMsQ0FBQyxLQUFLLENBQUM7Z0JBQ1osT0FBTyxDQUFDLElBQUksQ0FBQztvQkFDWCxJQUFJLEVBQUUsZUFBZTtvQkFDckIsSUFBSSxFQUFFLHVCQUF1QjtvQkFDN0IsSUFBSSxFQUFFO3dCQUNKLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxJQUFJOzRCQUN0QixjQUFjLEVBQUUsSUFBSSxDQUFDLFdBQVc7eUJBQ2pDLENBQUM7d0JBQ0YsR0FBRyxDQUFDLElBQUksQ0FBQyxvQkFBb0IsSUFBSTs0QkFDL0IsS0FBSzt5QkFDTixDQUFDO3FCQUNIO2lCQUNGLENBQUMsQ0FBQztZQUNMLENBQUM7UUFDSCxDQUFDO1FBRUQsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLE9BQU8sSUFBSSxLQUFLLENBQUM7UUFFekMsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLHFCQUFxQixFQUFFLENBQUM7WUFDdkMsT0FBTyxDQUFDLElBQUksQ0FDVixHQUFHLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLG1CQUFtQixDQUFDLENBQ2hFLENBQUM7UUFDSixDQUFDO1FBRUQsT0FBTyxDQUFDLElBQUksQ0FBQztZQUNYLElBQUksRUFBRSxzQkFBc0I7WUFDNUIsR0FBRyxFQUFFLE9BQU87Z0JBQ1YsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsK0JBQStCO2dCQUM5QyxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjO1lBQy9CLEdBQUcsQ0FBQyxPQUFPLENBQUMsd0JBQXdCLElBQUksRUFBRSxDQUFDO1NBQzVDLENBQUMsQ0FBQztRQUVILE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLE9BQU8sQ0FBQyxHQUFHLElBQWM7UUFDOUIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLFVBQVUsQ0FBQyxHQUFHLElBQWM7UUFDakMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxXQUFXLENBQUMsR0FBRyxJQUFjO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxjQUFjLENBQUMsR0FBRyxJQUFjO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNhLGdCQUFnQixDQUFDLE9BQWU7UUFDOUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVPLFVBQVUsQ0FBQyxPQUEyQjtRQUM1QyxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDekIsSUFBSSxpQkFBTyxDQUFDLElBQUksRUFBRTtnQkFDaEIsSUFBSSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTztnQkFDMUIsY0FBYyxFQUFFLE9BQU8sQ0FBQyxjQUFjLElBQUksT0FBTyxDQUFDLFVBQVU7Z0JBQzVELGVBQWUsRUFBRSxPQUFPLENBQUMsZUFBZTthQUN6QyxDQUFDLENBQUM7UUFDTCxDQUFDO0lBQ0gsQ0FBQztJQUVPLG1CQUFtQjtRQUN6QixJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FDcEIsUUFBUSxFQUNSLE1BQU0sRUFDTixPQUFPLEVBQ1AsZ0JBQWdCLEVBQ2hCLGlCQUFpQixFQUNqQixpQkFBaUIsRUFDakIsa0JBQWtCLEVBRWxCLDJEQUEyRCxFQUMzRCx5Q0FBeUMsRUFFekMsZ0JBQWdCLEVBQ2hCLE1BQU0sRUFDTixPQUFPLEVBQ1AsUUFBUSxFQUNSLFlBQVksRUFFWixtRUFBbUUsRUFDbkUsU0FBUyxFQUVULGtEQUFrRCxFQUNsRCxVQUFVLEVBQ1YsUUFBUSxFQUVSLHFCQUFxQixFQUNyQixhQUFhLEVBRWIsK0RBQStELEVBQy9ELGVBQWUsRUFFZiwwQkFBMEIsRUFDMUIsZUFBZSxFQUNmLGdCQUFnQixFQUVoQixvQkFBb0IsRUFDcEIsZUFBZSxFQUVmLHlCQUF5QixFQUN6QixjQUFjLEVBRWQsd0JBQXdCLEVBQ3hCLE9BQU8sRUFFUCx1QkFBdUIsRUFDdkIsaUJBQWlCLEVBRWpCLGdEQUFnRCxFQUNoRCxRQUFRLENBQ1QsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGNBQWMsQ0FBQyxJQUFVO1FBQzlCLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDdEQsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBVyxrQkFBa0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsYUFBYSxFQUFFLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUM1QyxDQUFDOztBQTE3Qkgsa0NBMjdCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IHJlbGF0aXZlLCBwb3NpeCB9IGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBCdW5kbGVyLCBCdW5kbGVyT3B0aW9ucyB9IGZyb20gXCIuL2J1bmRsZXJcIjtcbmltcG9ydCB7IEplc3QsIEplc3RPcHRpb25zIH0gZnJvbSBcIi4vamVzdFwiO1xuaW1wb3J0IHsgTGljZW5zZUNoZWNrZXIsIExpY2Vuc2VDaGVja2VyT3B0aW9ucyB9IGZyb20gXCIuL2xpY2Vuc2UtY2hlY2tlclwiO1xuaW1wb3J0IHtcbiAgQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyIGFzIE5vZGVQYWNrYWdlQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyLFxuICBDb2RlQXJ0aWZhY3RPcHRpb25zLFxuICBOb2RlUGFja2FnZSxcbiAgTm9kZVBhY2thZ2VNYW5hZ2VyLFxuICBOb2RlUGFja2FnZU9wdGlvbnMsXG59IGZyb20gXCIuL25vZGUtcGFja2FnZVwiO1xuaW1wb3J0IHsgUHJvamVucmMsIFByb2plbnJjT3B0aW9ucyB9IGZyb20gXCIuL3Byb2plbnJjXCI7XG5pbXBvcnQgeyBCdWlsZFdvcmtmbG93LCBCdWlsZFdvcmtmbG93Q29tbW9uT3B0aW9ucyB9IGZyb20gXCIuLi9idWlsZFwiO1xuaW1wb3J0IHsgREVGQVVMVF9BUlRJRkFDVFNfRElSRUNUT1JZIH0gZnJvbSBcIi4uL2J1aWxkL3ByaXZhdGUvY29uc3RzXCI7XG5pbXBvcnQgeyBQUk9KRU5fRElSIH0gZnJvbSBcIi4uL2NvbW1vblwiO1xuaW1wb3J0IHsgRGVwZW5kZW5jeVR5cGUgfSBmcm9tIFwiLi4vZGVwZW5kZW5jaWVzXCI7XG5pbXBvcnQge1xuICBBdXRvTWVyZ2UsXG4gIERlcGVuZGFib3RPcHRpb25zLFxuICBHaXRIdWIsXG4gIEdpdEh1YlByb2plY3QsXG4gIEdpdEh1YlByb2plY3RPcHRpb25zLFxuICBHaXRJZGVudGl0eSxcbn0gZnJvbSBcIi4uL2dpdGh1YlwiO1xuaW1wb3J0IHsgQmlvbWUsIEJpb21lT3B0aW9ucyB9IGZyb20gXCIuL2Jpb21lL2Jpb21lXCI7XG5pbXBvcnQgeyBERUZBVUxUX0dJVEhVQl9BQ1RJT05TX1VTRVIgfSBmcm9tIFwiLi4vZ2l0aHViL2NvbnN0YW50c1wiO1xuaW1wb3J0IHsgZW5zdXJlTm90SGlkZGVuUGF0aCwgc2VjcmV0VG9TdHJpbmcgfSBmcm9tIFwiLi4vZ2l0aHViL3ByaXZhdGUvdXRpbFwiO1xuaW1wb3J0IHtcbiAgSm9iUGVybWlzc2lvbixcbiAgSm9iUGVybWlzc2lvbnMsXG4gIEpvYlN0ZXAsXG4gIEpvYlN0ZXBDb25maWd1cmF0aW9uLFxuICBUcmlnZ2Vycyxcbn0gZnJvbSBcIi4uL2dpdGh1Yi93b3JrZmxvd3MtbW9kZWxcIjtcbmltcG9ydCB7IElnbm9yZUZpbGUsIElnbm9yZUZpbGVPcHRpb25zIH0gZnJvbSBcIi4uL2lnbm9yZS1maWxlXCI7XG5pbXBvcnQge1xuICBOcG1Db25maWcsXG4gIFByZXR0aWVyLFxuICBQcmV0dGllck9wdGlvbnMsXG4gIFVwZ3JhZGVEZXBlbmRlbmNpZXMsXG4gIFVwZ3JhZGVEZXBlbmRlbmNpZXNPcHRpb25zLFxufSBmcm9tIFwiLi4vamF2YXNjcmlwdFwiO1xuaW1wb3J0IHsgTGljZW5zZSB9IGZyb20gXCIuLi9saWNlbnNlXCI7XG5pbXBvcnQgeyBQcm9qZW5yY0pzb24gfSBmcm9tIFwiLi4vcHJvamVucmMtanNvblwiO1xuaW1wb3J0IHtcbiAgQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyIGFzIFJlbGVhc2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIsXG4gIENvZGVBcnRpZmFjdEF1dGhQcm92aWRlcixcbiAgaXNBd3NDb2RlQXJ0aWZhY3RSZWdpc3RyeSxcbiAgTnBtUHVibGlzaE9wdGlvbnMsXG4gIFB1Ymxpc2hlcixcbiAgUmVsZWFzZSxcbiAgUmVsZWFzZVByb2plY3RPcHRpb25zLFxufSBmcm9tIFwiLi4vcmVsZWFzZVwiO1xuaW1wb3J0IHsgZmlsdGVyZWRSdW5zT25PcHRpb25zIH0gZnJvbSBcIi4uL3J1bm5lci1vcHRpb25zXCI7XG5pbXBvcnQgeyBUYXNrIH0gZnJvbSBcIi4uL3Rhc2tcIjtcbmltcG9ydCB7IGRlZXBNZXJnZSwgbXVsdGlwbGVTZWxlY3RlZCwgbm9ybWFsaXplUGVyc2lzdGVkUGF0aCB9IGZyb20gXCIuLi91dGlsXCI7XG5pbXBvcnQgeyBlbnN1cmVSZWxhdGl2ZVBhdGhTdGFydHNXaXRoRG90IH0gZnJvbSBcIi4uL3V0aWwvcGF0aFwiO1xuXG5jb25zdCBQUk9KRU5fU0NSSVBUID0gXCJwcm9qZW5cIjtcblxuZXhwb3J0IGludGVyZmFjZSBOb2RlUHJvamVjdE9wdGlvbnNcbiAgZXh0ZW5kcyBHaXRIdWJQcm9qZWN0T3B0aW9ucyxcbiAgICBOb2RlUGFja2FnZU9wdGlvbnMsXG4gICAgUmVsZWFzZVByb2plY3RPcHRpb25zIHtcbiAgLyoqXG4gICAqIExpY2Vuc2UgY29weXJpZ2h0IG93bmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHRzIHRvIHRoZSB2YWx1ZSBvZiBhdXRob3JOYW1lIG9yIFwiXCIgaWYgYGF1dGhvck5hbWVgIGlzIHVuZGVmaW5lZC5cbiAgICovXG4gIHJlYWRvbmx5IGNvcHlyaWdodE93bmVyPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgY29weXJpZ2h0IHllYXJzIHRvIHB1dCBpbiB0aGUgTElDRU5TRSBmaWxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGN1cnJlbnQgeWVhclxuICAgKi9cbiAgcmVhZG9ubHkgY29weXJpZ2h0UGVyaW9kPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBWZXJzaW9uIG9mIHByb2plbiB0byBpbnN0YWxsLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlZmF1bHRzIHRvIHRoZSBsYXRlc3QgdmVyc2lvbi5cbiAgICovXG4gIHJlYWRvbmx5IHByb2plblZlcnNpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyBvZiBcInByb2plblwiIHNob3VsZCBiZSBpbnN0YWxsZWQgYXMgYSBkZXZEZXBlbmRlbmN5LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWUgaWYgbm90IGEgc3VicHJvamVjdFxuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVuRGV2RGVwZW5kZW5jeT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIERlZmluZSBhIEdpdEh1YiB3b3JrZmxvdyBmb3IgYnVpbGRpbmcgUFJzLlxuICAgKiBAZGVmYXVsdCAtIHRydWUgaWYgbm90IGEgc3VicHJvamVjdFxuICAgKi9cbiAgcmVhZG9ubHkgYnVpbGRXb3JrZmxvdz86IGJvb2xlYW47XG4gIC8qKlxuICAgKiBPcHRpb25zIGZvciBQUiBidWlsZCB3b3JrZmxvdy5cbiAgICovXG4gIHJlYWRvbmx5IGJ1aWxkV29ya2Zsb3dPcHRpb25zPzogQnVpbGRXb3JrZmxvd09wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgdXBkYXRlIGZpbGVzIG1vZGlmaWVkIGR1cmluZyBidWlsZHMgdG8gcHVsbC1yZXF1ZXN0IGJyYW5jaGVzLiBUaGlzIG1lYW5zXG4gICAqIHRoYXQgYW55IGZpbGVzIHN5bnRoZXNpemVkIGJ5IHByb2plbiBvciBlLmcuIHRlc3Qgc25hcHNob3RzIHdpbGwgYWx3YXlzIGJlIHVwLXRvLWRhdGVcbiAgICogYmVmb3JlIGEgUFIgaXMgbWVyZ2VkLlxuICAgKlxuICAgKiBJbXBsaWVzIHRoYXQgUFIgYnVpbGRzIGRvIG5vdCBoYXZlIGFudGktdGFtcGVyIGNoZWNrcy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCAtIFVzZSBgYnVpbGRXb3JrZmxvd09wdGlvbnMubXV0YWJsZUJ1aWxkYFxuICAgKi9cbiAgcmVhZG9ubHkgbXV0YWJsZUJ1aWxkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRGVmaW5lIGEgR2l0SHViIHdvcmtmbG93IHN0ZXAgZm9yIHNlbmRpbmcgY29kZSBjb3ZlcmFnZSBtZXRyaWNzIHRvIGh0dHBzOi8vY29kZWNvdi5pby9cbiAgICogVXNlcyBjb2RlY292L2NvZGVjb3YtYWN0aW9uQHY1XG4gICAqIEJ5IGRlZmF1bHQsIE9JREMgYXV0aCBpcyB1c2VkLiBBbHRlcm5hdGl2ZWx5IGEgdG9rZW4gY2FuIGJlIHByb3ZpZGVkIHZpYSBgY29kZUNvdlRva2VuU2VjcmV0YC5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGNvZGVDb3Y/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBEZWZpbmUgdGhlIHNlY3JldCBuYW1lIGZvciBhIHNwZWNpZmllZCBodHRwczovL2NvZGVjb3YuaW8vIHRva2VuXG5cbiAgICogQGRlZmF1bHQgLSBPSURDIGF1dGggaXMgdXNlZFxuICAgKi9cbiAgcmVhZG9ubHkgY29kZUNvdlRva2VuU2VjcmV0Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBERVBSRUNBVEVEOiByZW5hbWVkIHRvIGByZWxlYXNlYC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0cnVlIGlmIG5vdCBhIHN1YnByb2plY3RcbiAgICogQGRlcHJlY2F0ZWQgc2VlIGByZWxlYXNlYC5cbiAgICovXG4gIHJlYWRvbmx5IHJlbGVhc2VXb3JrZmxvdz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEFkZCByZWxlYXNlIG1hbmFnZW1lbnQgdG8gdGhpcyBwcm9qZWN0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWUgKGZhbHNlIGZvciBzdWJwcm9qZWN0cylcbiAgICovXG4gIHJlYWRvbmx5IHJlbGVhc2U/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgbWFpbiByZWxlYXNlIGJyYW5jaC5cbiAgICpcbiAgICogQGRlZmF1bHQgXCJtYWluXCJcbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRSZWxlYXNlQnJhbmNoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdvcmtmbG93IHN0ZXBzIHRvIHVzZSBpbiBvcmRlciB0byBib290c3RyYXAgdGhpcyByZXBvLlxuICAgKlxuICAgKiBAZGVmYXVsdCBcInlhcm4gaW5zdGFsbCAtLWZyb3plbi1sb2NrZmlsZSAmJiB5YXJuIHByb2plblwiXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd0Jvb3RzdHJhcFN0ZXBzPzogSm9iU3RlcFtdO1xuXG4gIC8qKlxuICAgKiBUaGUgZ2l0IGlkZW50aXR5IHRvIHVzZSBpbiB3b3JrZmxvd3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBHaXRIdWIgQWN0aW9ucyB1c2VyXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd0dpdElkZW50aXR5PzogR2l0SWRlbnRpdHk7XG5cbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgcmVsZWFzZSB0byBucG0gd2hlbiBuZXcgdmVyc2lvbnMgYXJlIGludHJvZHVjZWQuXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSByZWxlYXNlVG9OcG0/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbm9kZSB2ZXJzaW9uIHVzZWQgaW4gR2l0SHViIEFjdGlvbnMgd29ya2Zsb3dzLlxuICAgKlxuICAgKiBBbHdheXMgdXNlIHRoaXMgb3B0aW9uIGlmIHlvdXIgR2l0SHViIEFjdGlvbnMgd29ya2Zsb3dzIHJlcXVpcmUgYSBzcGVjaWZpYyB0byBydW4uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYG1pbk5vZGVWZXJzaW9uYCBpZiBzZXQsIG90aGVyd2lzZSBgbHRzLypgLlxuICAgKi9cbiAgcmVhZG9ubHkgd29ya2Zsb3dOb2RlVmVyc2lvbj86IHN0cmluZztcblxuICAvKipcbiAgICogRW5hYmxlIE5vZGUuanMgcGFja2FnZSBjYWNoZSBpbiBHaXRIdWIgd29ya2Zsb3dzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgd29ya2Zsb3dQYWNrYWdlQ2FjaGU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBVc2UgZGVwZW5kYWJvdCB0byBoYW5kbGUgZGVwZW5kZW5jeSB1cGdyYWRlcy5cbiAgICogQ2Fubm90IGJlIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCBgZGVwc1VwZ3JhZGVgLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVwZW5kYWJvdD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIGRlcGVuZGFib3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBvcHRpb25zXG4gICAqL1xuICByZWFkb25seSBkZXBlbmRhYm90T3B0aW9ucz86IERlcGVuZGFib3RPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBVc2UgdGFza3MgYW5kIGdpdGh1YiB3b3JrZmxvd3MgdG8gaGFuZGxlIGRlcGVuZGVuY3kgdXBncmFkZXMuXG4gICAqIENhbm5vdCBiZSB1c2VkIGluIGNvbmp1bmN0aW9uIHdpdGggYGRlcGVuZGFib3RgLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBkZXBzVXBncmFkZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIGBVcGdyYWRlRGVwZW5kZW5jaWVzYC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IG9wdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IGRlcHNVcGdyYWRlT3B0aW9ucz86IFVwZ3JhZGVEZXBlbmRlbmNpZXNPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IGFwcHJvdmUgZGVwcyB1cGdyYWRlIFBScywgYWxsb3dpbmcgdGhlbSB0byBiZVxuICAgKiBtZXJnZWQgYnkgbWVyZ2lmeSAoaWYgY29uZmlndWVkKS5cbiAgICpcbiAgICogVGhyb3cgaWYgc2V0IHRvIHRydWUgYnV0IGBhdXRvQXBwcm92ZU9wdGlvbnNgIGFyZSBub3QgZGVmaW5lZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0cnVlXG4gICAqL1xuICByZWFkb25seSBhdXRvQXBwcm92ZVVwZ3JhZGVzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRGVmaW5lcyBhbiAubnBtaWdub3JlIGZpbGUuIE5vcm1hbGx5IHRoaXMgaXMgb25seSBuZWVkZWQgZm9yIGxpYnJhcmllcyB0aGF0XG4gICAqIGFyZSBwYWNrYWdlZCBhcyB0YXJiYWxscy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgbnBtaWdub3JlRW5hYmxlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyYXRpb24gb3B0aW9ucyBmb3IgLm5wbWlnbm9yZSBmaWxlXG4gICAqL1xuICByZWFkb25seSBucG1JZ25vcmVPcHRpb25zPzogSWdub3JlRmlsZU9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEFkZGl0aW9uYWwgZW50cmllcyB0byAubnBtaWdub3JlLlxuICAgKiBAZGVwcmVjYXRlZCAtIHVzZSBgcHJvamVjdC5hZGRQYWNrYWdlSWdub3JlYFxuICAgKi9cbiAgcmVhZG9ubHkgbnBtaWdub3JlPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIEluY2x1ZGUgYSBHaXRIdWIgcHVsbCByZXF1ZXN0IHRlbXBsYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBwdWxsUmVxdWVzdFRlbXBsYXRlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIGNvbnRlbnRzIG9mIHRoZSBwdWxsIHJlcXVlc3QgdGVtcGxhdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBjb250ZW50XG4gICAqL1xuICByZWFkb25seSBwdWxsUmVxdWVzdFRlbXBsYXRlQ29udGVudHM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogU2V0dXAgcHJldHRpZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBwcmV0dGllcj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFByZXR0aWVyIG9wdGlvbnNcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IG9wdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IHByZXR0aWVyT3B0aW9ucz86IFByZXR0aWVyT3B0aW9ucztcblxuICAvKipcbiAgICogQWRkaXRpb25hbCBlbnRyaWVzIHRvIC5naXRpZ25vcmVcbiAgICovXG4gIHJlYWRvbmx5IGdpdGlnbm9yZT86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBTZXR1cCBqZXN0IHVuaXQgdGVzdHNcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgamVzdD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEplc3Qgb3B0aW9uc1xuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHQgb3B0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgamVzdE9wdGlvbnM/OiBKZXN0T3B0aW9ucztcblxuICAvKipcbiAgICogR2VuZXJhdGUgKG9uY2UpIC5wcm9qZW5yYy5qcyAoaW4gSmF2YVNjcmlwdCkuIFNldCB0byBgZmFsc2VgIGluIG9yZGVyIHRvIGRpc2FibGVcbiAgICogLnByb2plbnJjLmpzIGdlbmVyYXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdHJ1ZSBpZiBwcm9qZW5yY0pzb24gaXMgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHByb2plbnJjSnM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBPcHRpb25zIGZvciAucHJvamVucmMuanNcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IG9wdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IHByb2plbnJjSnNPcHRpb25zPzogUHJvamVucmNPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBPcHRpb25zIGZvciBgQnVuZGxlcmAuXG4gICAqL1xuICByZWFkb25seSBidW5kbGVyT3B0aW9ucz86IEJ1bmRsZXJPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBBIGRpcmVjdG9yeSB3aGljaCB3aWxsIGNvbnRhaW4gYnVpbGQgYXJ0aWZhY3RzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBcImRpc3RcIlxuICAgKi9cbiAgcmVhZG9ubHkgYXJ0aWZhY3RzRGlyZWN0b3J5Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGEgYHBhY2thZ2VgIHRhc2sgdGhhdCB3aWxsIHByb2R1Y2UgYW4gbnBtIHRhcmJhbGwgdW5kZXIgdGhlXG4gICAqIGFydGlmYWN0cyBkaXJlY3RvcnkgKGUuZy4gYGRpc3RgKS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcGFja2FnZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEJ1aWxkIHdvcmtmbG93IHRyaWdnZXJzXG4gICAqIEBkZWZhdWx0IFwieyBwdWxsUmVxdWVzdDoge30sIHdvcmtmbG93RGlzcGF0Y2g6IHt9IH1cIlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCAtIFVzZSBgYnVpbGRXb3JrZmxvd09wdGlvbnMud29ya2Zsb3dUcmlnZ2Vyc2BcbiAgICovXG4gIHJlYWRvbmx5IGJ1aWxkV29ya2Zsb3dUcmlnZ2Vycz86IFRyaWdnZXJzO1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmUgd2hpY2ggbGljZW5zZXMgc2hvdWxkIGJlIGRlZW1lZCBhY2NlcHRhYmxlIGZvciB1c2UgYnkgZGVwZW5kZW5jaWVzXG4gICAqXG4gICAqIFRoaXMgc2V0dGluZyB3aWxsIGNhdXNlIHRoZSBidWlsZCB0byBmYWlsLCBpZiBhbnkgcHJvaGliaXRlZCBvciBub3QgYWxsb3dlZCBsaWNlbnNlcyBhcmVzIGVuY291bnRlcmVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGxpY2Vuc2UgY2hlY2tzIGFyZSBydW4gZHVyaW5nIHRoZSBidWlsZCBhbmQgYWxsIGxpY2Vuc2VzIHdpbGwgYmUgYWNjZXB0ZWRcbiAgICovXG4gIHJlYWRvbmx5IGNoZWNrTGljZW5zZXM/OiBMaWNlbnNlQ2hlY2tlck9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIFNldHVwIEJpb21lLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgYmlvbWU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBCaW9tZSBvcHRpb25zXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBvcHRpb25zXG4gICAqL1xuICByZWFkb25seSBiaW9tZU9wdGlvbnM/OiBCaW9tZU9wdGlvbnM7XG59XG5cbi8qKlxuICogQnVpbGQgd29ya2Zsb3cgb3B0aW9ucyBmb3IgTm9kZVByb2plY3RcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBCdWlsZFdvcmtmbG93T3B0aW9ucyBleHRlbmRzIEJ1aWxkV29ya2Zsb3dDb21tb25PcHRpb25zIHtcbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgdXBkYXRlIGZpbGVzIG1vZGlmaWVkIGR1cmluZyBidWlsZHMgdG8gcHVsbC1yZXF1ZXN0IGJyYW5jaGVzLlxuICAgKiBUaGlzIG1lYW5zIHRoYXQgYW55IGZpbGVzIHN5bnRoZXNpemVkIGJ5IHByb2plbiBvciBlLmcuIHRlc3Qgc25hcHNob3RzIHdpbGxcbiAgICogYWx3YXlzIGJlIHVwLXRvLWRhdGUgYmVmb3JlIGEgUFIgaXMgbWVyZ2VkLlxuICAgKlxuICAgKiBJbXBsaWVzIHRoYXQgUFIgYnVpbGRzIGRvIG5vdCBoYXZlIGFudGktdGFtcGVyIGNoZWNrcy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgbXV0YWJsZUJ1aWxkPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBBdXRvbWF0aWMgYnVtcCBtb2Rlc1xuICovXG5leHBvcnQgZW51bSBBdXRvUmVsZWFzZSB7XG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IGJ1bXAgJiByZWxlYXNlIGEgbmV3IHZlcnNpb24gZm9yIGV2ZXJ5IGNvbW1pdCB0byBcIm1haW5cIlxuICAgKi9cbiAgRVZFUllfQ09NTUlULFxuXG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IGJ1bXAgJiByZWxlYXNlIGEgbmV3IHZlcnNpb24gb24gYSBkYWlseSBiYXNpcy5cbiAgICovXG4gIERBSUxZLFxufVxuXG4vKipcbiAqIE5vZGUuanMgcHJvamVjdC5cbiAqXG4gKiBAcGppZCBub2RlXG4gKi9cbmV4cG9ydCBjbGFzcyBOb2RlUHJvamVjdCBleHRlbmRzIEdpdEh1YlByb2plY3Qge1xuICAvKipcbiAgICogQVBJIGZvciBtYW5hZ2luZyB0aGUgbm9kZSBwYWNrYWdlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHBhY2thZ2U6IE5vZGVQYWNrYWdlO1xuXG4gIC8qKlxuICAgKiBUaGUgLm5wbWlnbm9yZSBmaWxlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG5wbWlnbm9yZT86IElnbm9yZUZpbGU7XG5cbiAgLyoqXG4gICAqIFRoZSAubnBtcmMgZmlsZVxuICAgKi9cbiAgcHVibGljIGdldCBucG1yYygpOiBOcG1Db25maWcge1xuICAgIGlmICghdGhpcy5fbnBtcmMpIHtcbiAgICAgIHRoaXMuX25wbXJjID0gbmV3IE5wbUNvbmZpZyh0aGlzLCB7IG9taXRFbXB0eTogdHJ1ZSB9KTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuX25wbXJjO1xuICB9XG4gIHByaXZhdGUgX25wbXJjPzogTnBtQ29uZmlnO1xuXG4gIC8qKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYHBhY2thZ2UuYWxsb3dMaWJyYXJ5RGVwZW5kZW5jaWVzYFxuICAgKi9cbiAgcHVibGljIGdldCBhbGxvd0xpYnJhcnlEZXBlbmRlbmNpZXMoKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5hbGxvd0xpYnJhcnlEZXBlbmRlbmNpZXM7XG4gIH1cblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwYWNrYWdlLmVudHJ5cG9pbnRgXG4gICAqL1xuICBwdWJsaWMgZ2V0IGVudHJ5cG9pbnQoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLmVudHJ5cG9pbnQ7XG4gIH1cblxuICAvKipcbiAgICogQ29tcG9uZW50IHRoYXQgc2V0cyB1cCBtZXJnaWZ5IGZvciBtZXJnaW5nIGFwcHJvdmVkIHB1bGwgcmVxdWVzdHMuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXV0b01lcmdlPzogQXV0b01lcmdlO1xuXG4gIC8qKlxuICAgKiBUaGUgUFIgYnVpbGQgR2l0SHViIHdvcmtmbG93LiBgdW5kZWZpbmVkYCBpZiBgYnVpbGRXb3JrZmxvd2AgaXMgZGlzYWJsZWQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYnVpbGRXb3JrZmxvdz86IEJ1aWxkV29ya2Zsb3c7XG5cbiAgLyoqXG4gICAqIFBhY2thZ2UgcHVibGlzaGVyLiBUaGlzIHdpbGwgYmUgYHVuZGVmaW5lZGAgaWYgdGhlIHByb2plY3QgZG9lcyBub3QgaGF2ZSBhXG4gICAqIHJlbGVhc2Ugd29ya2Zsb3cuXG4gICAqXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcmVsZWFzZS5wdWJsaXNoZXJgLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHB1Ymxpc2hlcj86IFB1Ymxpc2hlcjtcblxuICAvKipcbiAgICogUmVsZWFzZSBtYW5hZ2VtZW50LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlbGVhc2U/OiBSZWxlYXNlO1xuXG4gIC8qKlxuICAgKiBUaGUgbWluaW11bSBub2RlIHZlcnNpb24gcmVxdWlyZWQgYnkgdGhpcyBwYWNrYWdlIHRvIGZ1bmN0aW9uLlxuICAgKlxuICAgKiBUaGlzIHZhbHVlIGluZGljYXRlcyB0aGUgcGFja2FnZSBpcyBpbmNvbXBhdGlibGUgd2l0aCBvbGRlciB2ZXJzaW9ucy5cbiAgICovXG4gIHB1YmxpYyBnZXQgbWluTm9kZVZlcnNpb24oKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLm1pbk5vZGVWZXJzaW9uO1xuICB9XG5cbiAgLyoqXG4gICAqIE1heGltdW0gbm9kZSB2ZXJzaW9uIHN1cHBvcnRlZCBieSB0aGlzIHBhY2thZ2UuXG4gICAqXG4gICAqIFRoZSB2YWx1ZSBpbmRpY2F0ZXMgdGhlIHBhY2thZ2UgaXMgaW5jb21wYXRpYmxlIHdpdGggbmV3ZXIgdmVyc2lvbnMuXG4gICAqL1xuICBwdWJsaWMgZ2V0IG1heE5vZGVWZXJzaW9uKCk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5tYXhOb2RlVmVyc2lvbjtcbiAgfVxuXG4gIHByb3RlY3RlZCByZWFkb25seSBub2RlVmVyc2lvbj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHBhY2thZ2UgbWFuYWdlciB0byB1c2UuXG4gICAqXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcGFja2FnZS5wYWNrYWdlTWFuYWdlcmBcbiAgICovXG4gIHB1YmxpYyBnZXQgcGFja2FnZU1hbmFnZXIoKTogTm9kZVBhY2thZ2VNYW5hZ2VyIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBjb21tYW5kIHRvIHVzZSB0byBydW4gc2NyaXB0cyAoZS5nLiBgeWFybiBydW5gIG9yIGBucG0gcnVuYCBkZXBlbmRzIG9uIHRoZSBwYWNrYWdlIG1hbmFnZXIpLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJ1blNjcmlwdENvbW1hbmQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEplc3QgY29uZmlndXJhdGlvbiAoaWYgZW5hYmxlZClcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBqZXN0PzogSmVzdDtcblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwYWNrYWdlLmFkZEZpZWxkKHgsIHkpYFxuICAgKi9cbiAgcHVibGljIGdldCBtYW5pZmVzdCgpIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLm1hbmlmZXN0O1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IGJ1bmRsZXI6IEJ1bmRsZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBidWlsZCBvdXRwdXQgZGlyZWN0b3J5LiBBbiBucG0gdGFyYmFsbCB3aWxsIGJlIGNyZWF0ZWQgdW5kZXIgdGhlIGBqc2BcbiAgICogc3ViZGlyZWN0b3J5LiBGb3IgZXhhbXBsZSwgaWYgdGhpcyBpcyBzZXQgdG8gYGRpc3RgICh0aGUgZGVmYXVsdCksIHRoZSBucG1cbiAgICogdGFyYmFsbCB3aWxsIGJlIHBsYWNlZCB1bmRlciBgZGlzdC9qcy9ib29tLWJvb20tMS4yLjMudGdgLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFydGlmYWN0c0RpcmVjdG9yeTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbG9jYXRpb24gb2YgdGhlIG5wbSB0YXJiYWxsIGFmdGVyIGJ1aWxkIChgJHthcnRpZmFjdHNEaXJlY3Rvcnl9L2pzYCkuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXJ0aWZhY3RzSmF2YXNjcmlwdERpcmVjdG9yeTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgdXBncmFkZSB3b3JrZmxvdy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB1cGdyYWRlV29ya2Zsb3c/OiBVcGdyYWRlRGVwZW5kZW5jaWVzO1xuXG4gIHByb3RlY3RlZCByZWFkb25seSB3b3JrZmxvd0Jvb3RzdHJhcFN0ZXBzOiBKb2JTdGVwW107XG4gIHByaXZhdGUgcmVhZG9ubHkgd29ya2Zsb3dHaXRJZGVudGl0eTogR2l0SWRlbnRpdHk7XG4gIHByb3RlY3RlZCByZWFkb25seSB3b3JrZmxvd1BhY2thZ2VDYWNoZTogYm9vbGVhbjtcbiAgcHVibGljIHJlYWRvbmx5IHByZXR0aWVyPzogUHJldHRpZXI7XG4gIHB1YmxpYyByZWFkb25seSBiaW9tZT86IEJpb21lO1xuXG4gIGNvbnN0cnVjdG9yKG9wdGlvbnM6IE5vZGVQcm9qZWN0T3B0aW9ucykge1xuICAgIHN1cGVyKHtcbiAgICAgIC4uLm9wdGlvbnMsXG4gICAgICAvLyBOb2RlIHByb2plY3RzIGhhdmUgdGhlIHNwZWNpZmljIHByb2plbiB2ZXJzaW9uIGxvY2tlZCB2aWEgbG9ja2ZpbGUsIHNvIHdlIGNhbiBza2lwIHRoZSBAPFZFUlNJT04+IHBhcnQgb2YgdGhlIHRvcC1sZXZlbCBwcm9qZWN0XG4gICAgICBwcm9qZW5Db21tYW5kOiBvcHRpb25zLnByb2plbkNvbW1hbmQgPz8gXCJucHggcHJvamVuXCIsXG4gICAgfSk7XG5cbiAgICB0aGlzLnBhY2thZ2UgPSBuZXcgTm9kZVBhY2thZ2UodGhpcywgb3B0aW9ucyk7XG4gICAgdGhpcy53b3JrZmxvd0Jvb3RzdHJhcFN0ZXBzID0gb3B0aW9ucy53b3JrZmxvd0Jvb3RzdHJhcFN0ZXBzID8/IFtdO1xuICAgIHRoaXMud29ya2Zsb3dHaXRJZGVudGl0eSA9XG4gICAgICBvcHRpb25zLndvcmtmbG93R2l0SWRlbnRpdHkgPz8gREVGQVVMVF9HSVRIVUJfQUNUSU9OU19VU0VSO1xuICAgIHRoaXMud29ya2Zsb3dQYWNrYWdlQ2FjaGUgPSBvcHRpb25zLndvcmtmbG93UGFja2FnZUNhY2hlID8/IGZhbHNlO1xuICAgIHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5ID1cbiAgICAgIG9wdGlvbnMuYXJ0aWZhY3RzRGlyZWN0b3J5ID8/IERFRkFVTFRfQVJUSUZBQ1RTX0RJUkVDVE9SWTtcbiAgICBlbnN1cmVOb3RIaWRkZW5QYXRoKHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5LCBcImFydGlmYWN0c0RpcmVjdG9yeVwiKTtcbiAgICBjb25zdCBub3JtYWxpemVkQXJ0aWZhY3RzRGlyZWN0b3J5ID0gbm9ybWFsaXplUGVyc2lzdGVkUGF0aChcbiAgICAgIHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5XG4gICAgKTtcbiAgICB0aGlzLmFydGlmYWN0c0phdmFzY3JpcHREaXJlY3RvcnkgPSBwb3NpeC5qb2luKFxuICAgICAgbm9ybWFsaXplZEFydGlmYWN0c0RpcmVjdG9yeSxcbiAgICAgIFwianNcIlxuICAgICk7XG5cbiAgICB0aGlzLnJ1blNjcmlwdENvbW1hbmQgPSAoKCkgPT4ge1xuICAgICAgc3dpdGNoICh0aGlzLnBhY2thZ2VNYW5hZ2VyKSB7XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLk5QTTpcbiAgICAgICAgICByZXR1cm4gXCJucG0gcnVuXCI7XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLllBUk46XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLllBUk4yOlxuICAgICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5ZQVJOX0NMQVNTSUM6XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLllBUk5fQkVSUlk6XG4gICAgICAgICAgcmV0dXJuIFwieWFybiBydW5cIjtcbiAgICAgICAgY2FzZSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTTpcbiAgICAgICAgICByZXR1cm4gXCJwbnBtIHJ1blwiO1xuICAgICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5CVU46XG4gICAgICAgICAgcmV0dXJuIFwiYnVuIHJ1blwiO1xuICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgdW5leHBlY3RlZCBwYWNrYWdlIG1hbmFnZXIgJHt0aGlzLnBhY2thZ2VNYW5hZ2VyfWApO1xuICAgICAgfVxuICAgIH0pKCk7XG5cbiAgICBjb25zdCBlbnZDb21tYW5kID0gKCgpID0+IHtcbiAgICAgIHN3aXRjaCAodGhpcy5wYWNrYWdlTWFuYWdlcikge1xuICAgICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5QTlBNOlxuICAgICAgICAgIHJldHVybiAnJChwbnBtIC1jIGV4ZWMgXCJub2RlIC0tcHJpbnQgcHJvY2Vzcy5lbnYuUEFUSFwiKSc7XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLkJVTjpcbiAgICAgICAgICByZXR1cm4gJyQoYnVuIC0tZXZhbCBcImNvbnNvbGUubG9nKHByb2Nlc3MuZW52LlBBVEgpXCIpJztcbiAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICByZXR1cm4gJyQobnB4IC1jIFwibm9kZSAtLXByaW50IHByb2Nlc3MuZW52LlBBVEhcIiknO1xuICAgICAgfVxuICAgIH0pKCk7XG5cbiAgICB0aGlzLm5vZGVWZXJzaW9uID1cbiAgICAgIG9wdGlvbnMud29ya2Zsb3dOb2RlVmVyc2lvbiA/PyB0aGlzLnBhY2thZ2UubWluTm9kZVZlcnNpb247XG5cbiAgICAvLyBhZGQgUEFUSCBmb3IgYWxsIHRhc2tzIHdoaWNoIGluY2x1ZGVzIHRoZSBwcm9qZWN0J3MgbnBtIC5iaW4gbGlzdFxuICAgIHRoaXMudGFza3MuYWRkRW52aXJvbm1lbnQoXCJQQVRIXCIsIGVudkNvbW1hbmQpO1xuXG4gICAgdGhpcy5hZGRMaWNlbnNlKG9wdGlvbnMpO1xuXG4gICAgaWYgKG9wdGlvbnMubnBtaWdub3JlRW5hYmxlZCA/PyB0cnVlKSB7XG4gICAgICB0aGlzLm5wbWlnbm9yZSA9IG5ldyBJZ25vcmVGaWxlKFxuICAgICAgICB0aGlzLFxuICAgICAgICBcIi5ucG1pZ25vcmVcIixcbiAgICAgICAgb3B0aW9ucy5ucG1JZ25vcmVPcHRpb25zXG4gICAgICApO1xuICAgIH1cblxuICAgIHRoaXMuYWRkRGVmYXVsdEdpdElnbm9yZSgpO1xuXG4gICAgaWYgKG9wdGlvbnMuZ2l0aWdub3JlPy5sZW5ndGgpIHtcbiAgICAgIGZvciAoY29uc3QgaSBvZiBvcHRpb25zLmdpdGlnbm9yZSkge1xuICAgICAgICB0aGlzLmdpdGlnbm9yZS5leGNsdWRlKGkpO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmIChvcHRpb25zLm5wbWlnbm9yZT8ubGVuZ3RoKSB7XG4gICAgICBpZiAoIXRoaXMubnBtaWdub3JlKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAnLm5wbWlnbm9yZSBpcyBub3QgZGVmaW5lZCBmb3IgYW4gQVBQIHByb2plY3QgdHlwZS4gQWRkIFwibnBtSWdub3JlOiB0cnVlXCIgdG8gb3ZlcnJpZGUgdGhpcydcbiAgICAgICAgKTtcbiAgICAgIH1cblxuICAgICAgZm9yIChjb25zdCBpIG9mIG9wdGlvbnMubnBtaWdub3JlKSB7XG4gICAgICAgIHRoaXMubnBtaWdub3JlLmV4Y2x1ZGUoaSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKCF0aGlzLmVqZWN0ZWQpIHtcbiAgICAgIHRoaXMuc2V0U2NyaXB0KFBST0pFTl9TQ1JJUFQsIHRoaXMucGFja2FnZS5wcm9qZW5Db21tYW5kKTtcbiAgICB9XG5cbiAgICB0aGlzLm5wbWlnbm9yZT8uZXhjbHVkZShgLyR7UFJPSkVOX0RJUn0vYCk7XG5cbiAgICBjb25zdCBwcm9qZW4gPSBvcHRpb25zLnByb2plbkRldkRlcGVuZGVuY3kgPz8gKHRoaXMucGFyZW50ID8gZmFsc2UgOiB0cnVlKTtcbiAgICBpZiAocHJvamVuICYmICF0aGlzLmVqZWN0ZWQpIHtcbiAgICAgIGNvbnN0IHBvc3RmaXggPSBvcHRpb25zLnByb2plblZlcnNpb24gPyBgQCR7b3B0aW9ucy5wcm9qZW5WZXJzaW9ufWAgOiBcIlwiO1xuICAgICAgdGhpcy5hZGREZXZEZXBzKGBwcm9qZW4ke3Bvc3RmaXh9YCk7XG5cbiAgICAgIGlmIChcbiAgICAgICAgIXRoaXMuZGVwcy5pc0RlcGVuZGVuY3lTYXRpc2ZpZWQoXG4gICAgICAgICAgXCJjb25zdHJ1Y3RzXCIsXG4gICAgICAgICAgRGVwZW5kZW5jeVR5cGUuQlVJTEQsXG4gICAgICAgICAgXCJeMTAuMC4wXCJcbiAgICAgICAgKVxuICAgICAgKSB7XG4gICAgICAgIHRoaXMuYWRkRGV2RGVwcyhgY29uc3RydWN0c0BeMTAuMC4wYCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKCFvcHRpb25zLmRlZmF1bHRSZWxlYXNlQnJhbmNoKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICdcImRlZmF1bHRSZWxlYXNlQnJhbmNoXCIgaXMgdGVtcG9yYXJpbHkgYSByZXF1aXJlZCBvcHRpb24gd2hpbGUgd2UgbWlncmF0ZSBpdHMgZGVmYXVsdCB2YWx1ZSBmcm9tIFwibWFzdGVyXCIgdG8gXCJtYWluXCInXG4gICAgICApO1xuICAgIH1cblxuICAgIGNvbnN0IGJ1aWxkRW5hYmxlZCA9IG9wdGlvbnMuYnVpbGRXb3JrZmxvdyA/PyAodGhpcy5wYXJlbnQgPyBmYWxzZSA6IHRydWUpO1xuXG4gICAgLy8gY29uZmlndXJlIGplc3QgaWYgZW5hYmxlZFxuICAgIC8vIG11c3QgYmUgYmVmb3JlIHRoZSBidWlsZC9yZWxlYXNlIHdvcmtmbG93c1xuICAgIGlmIChvcHRpb25zLmplc3QgPz8gdHJ1ZSkge1xuICAgICAgdGhpcy5qZXN0ID0gbmV3IEplc3QodGhpcywgb3B0aW9ucy5qZXN0T3B0aW9ucyk7XG4gICAgfVxuXG4gICAgY29uc3Qgd29ya2Zsb3dQZXJtaXNzaW9uczogSm9iUGVybWlzc2lvbnMgPSB7XG4gICAgICBpZFRva2VuOiB0aGlzLmRldGVybWluZUlkVG9rZW5QZXJtaXNzaW9ucyhvcHRpb25zKSxcbiAgICB9O1xuXG4gICAgY29uc3QgYnVpbGRXb3JrZmxvd09wdGlvbnM6IEJ1aWxkV29ya2Zsb3dPcHRpb25zID1cbiAgICAgIG9wdGlvbnMuYnVpbGRXb3JrZmxvd09wdGlvbnMgPz8ge307XG5cbiAgICBpZiAoYnVpbGRFbmFibGVkICYmICh0aGlzLmdpdGh1YiB8fCBHaXRIdWIub2YodGhpcy5yb290KSkpIHtcbiAgICAgIHRoaXMuYnVpbGRXb3JrZmxvdyA9IG5ldyBCdWlsZFdvcmtmbG93KHRoaXMsIHtcbiAgICAgICAgYnVpbGRUYXNrOiB0aGlzLmJ1aWxkVGFzayxcbiAgICAgICAgYXJ0aWZhY3RzRGlyZWN0b3J5OiB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSxcbiAgICAgICAgY29udGFpbmVySW1hZ2U6IG9wdGlvbnMud29ya2Zsb3dDb250YWluZXJJbWFnZSxcbiAgICAgICAgZ2l0SWRlbnRpdHk6IHRoaXMud29ya2Zsb3dHaXRJZGVudGl0eSxcbiAgICAgICAgbXV0YWJsZUJ1aWxkOiBvcHRpb25zLm11dGFibGVCdWlsZCxcbiAgICAgICAgd29ya2Zsb3dUcmlnZ2Vyczogb3B0aW9ucy5idWlsZFdvcmtmbG93VHJpZ2dlcnMsXG4gICAgICAgIHBlcm1pc3Npb25zOiB3b3JrZmxvd1Blcm1pc3Npb25zLFxuICAgICAgICAuLi5idWlsZFdvcmtmbG93T3B0aW9ucyxcbiAgICAgICAgcHJlQnVpbGRTdGVwczogdGhpcy5yZW5kZXJXb3JrZmxvd1NldHVwKHtcbiAgICAgICAgICBpbnN0YWxsU3RlcENvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgIHdvcmtpbmdEaXJlY3Rvcnk6IHRoaXMuZGV0ZXJtaW5lSW5zdGFsbFdvcmtpbmdEaXJlY3RvcnkoKSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIG11dGFibGU6XG4gICAgICAgICAgICBidWlsZFdvcmtmbG93T3B0aW9ucy5tdXRhYmxlQnVpbGQgPz8gb3B0aW9ucy5tdXRhYmxlQnVpbGQgPz8gdHJ1ZSxcbiAgICAgICAgfSkuY29uY2F0KGJ1aWxkV29ya2Zsb3dPcHRpb25zLnByZUJ1aWxkU3RlcHMgPz8gW10pLFxuICAgICAgICBwb3N0QnVpbGRTdGVwczogWy4uLihvcHRpb25zLnBvc3RCdWlsZFN0ZXBzID8/IFtdKV0sXG4gICAgICAgIC4uLmZpbHRlcmVkUnVuc09uT3B0aW9ucyhcbiAgICAgICAgICBvcHRpb25zLndvcmtmbG93UnVuc09uLFxuICAgICAgICAgIG9wdGlvbnMud29ya2Zsb3dSdW5zT25Hcm91cFxuICAgICAgICApLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuYnVpbGRXb3JrZmxvdy5hZGRQb3N0QnVpbGRTdGVwcyhcbiAgICAgICAgLi4udGhpcy5yZW5kZXJVcGxvYWRDb3ZlcmFnZUpvYlN0ZXAob3B0aW9ucylcbiAgICAgICk7XG4gICAgICB0aGlzLm1heWJlQWRkQ29kZWNvdklnbm9yZXMob3B0aW9ucyk7XG4gICAgfVxuXG4gICAgY29uc3QgcmVsZWFzZSA9XG4gICAgICBvcHRpb25zLnJlbGVhc2UgPz9cbiAgICAgIG9wdGlvbnMucmVsZWFzZVdvcmtmbG93ID8/XG4gICAgICAodGhpcy5wYXJlbnQgPyBmYWxzZSA6IHRydWUpO1xuICAgIGlmIChyZWxlYXNlKSB7XG4gICAgICB0aGlzLnJlbGVhc2UgPSBuZXcgUmVsZWFzZSh0aGlzLCB7XG4gICAgICAgIHZlcnNpb25GaWxlOiBcInBhY2thZ2UuanNvblwiLCAvLyB0aGlzIGlzIHdoZXJlIFwidmVyc2lvblwiIGlzIHNldCBhZnRlciBidW1wXG4gICAgICAgIHRhc2s6IHRoaXMuYnVpbGRUYXNrLFxuICAgICAgICBicmFuY2g6IG9wdGlvbnMuZGVmYXVsdFJlbGVhc2VCcmFuY2ggPz8gXCJtYWluXCIsXG4gICAgICAgIC4uLm9wdGlvbnMsXG5cbiAgICAgICAgYXJ0aWZhY3RzRGlyZWN0b3J5OiB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSxcbiAgICAgICAgcmVsZWFzZVdvcmtmbG93U2V0dXBTdGVwczogW1xuICAgICAgICAgIC4uLnRoaXMucmVuZGVyV29ya2Zsb3dTZXR1cCh7XG4gICAgICAgICAgICBpbnN0YWxsU3RlcENvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgICAgd29ya2luZ0RpcmVjdG9yeTogdGhpcy5kZXRlcm1pbmVJbnN0YWxsV29ya2luZ0RpcmVjdG9yeSgpLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIG11dGFibGU6IGZhbHNlLFxuICAgICAgICAgIH0pLFxuICAgICAgICAgIC4uLihvcHRpb25zLnJlbGVhc2VXb3JrZmxvd1NldHVwU3RlcHMgPz8gW10pLFxuICAgICAgICBdLFxuICAgICAgICBwb3N0QnVpbGRTdGVwczogW1xuICAgICAgICAgIC4uLihvcHRpb25zLnBvc3RCdWlsZFN0ZXBzID8/IFtdKSxcbiAgICAgICAgICAuLi50aGlzLnJlbmRlclVwbG9hZENvdmVyYWdlSm9iU3RlcChvcHRpb25zKSxcbiAgICAgICAgXSxcblxuICAgICAgICB3b3JrZmxvd05vZGVWZXJzaW9uOiB0aGlzLm5vZGVWZXJzaW9uLFxuICAgICAgICB3b3JrZmxvd1Blcm1pc3Npb25zLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMubWF5YmVBZGRDb2RlY292SWdub3JlcyhvcHRpb25zKTtcbiAgICAgIHRoaXMucHVibGlzaGVyID0gdGhpcy5yZWxlYXNlLnB1Ymxpc2hlcjtcblxuICAgICAgY29uc3Qgbm9kZVBhY2thZ2VUb1JlbGVhc2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXJNYXBwaW5nOiBSZWNvcmQ8XG4gICAgICAgIE5vZGVQYWNrYWdlQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyLFxuICAgICAgICBSZWxlYXNlQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyXG4gICAgICA+ID0ge1xuICAgICAgICBbTm9kZVBhY2thZ2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIuQUNDRVNTX0FORF9TRUNSRVRfS0VZX1BBSVJdOlxuICAgICAgICAgIFJlbGVhc2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIuQUNDRVNTX0FORF9TRUNSRVRfS0VZX1BBSVIsXG4gICAgICAgIFtOb2RlUGFja2FnZUNvZGVBcnRpZmFjdEF1dGhQcm92aWRlci5HSVRIVUJfT0lEQ106XG4gICAgICAgICAgUmVsZWFzZUNvZGVBcnRpZmFjdEF1dGhQcm92aWRlci5HSVRIVUJfT0lEQyxcbiAgICAgIH07XG5cbiAgICAgIGlmIChvcHRpb25zLnJlbGVhc2VUb05wbSA/PyBmYWxzZSkge1xuICAgICAgICBjb25zdCBjb2RlQXJ0aWZhY3RPcHRpb25zOiBOcG1QdWJsaXNoT3B0aW9uc1tcImNvZGVBcnRpZmFjdE9wdGlvbnNcIl0gPVxuICAgICAgICAgIGlzQXdzQ29kZUFydGlmYWN0UmVnaXN0cnkodGhpcy5wYWNrYWdlLm5wbVJlZ2lzdHJ5KVxuICAgICAgICAgICAgPyB7XG4gICAgICAgICAgICAgICAgYWNjZXNzS2V5SWRTZWNyZXQ6XG4gICAgICAgICAgICAgICAgICBvcHRpb25zLmNvZGVBcnRpZmFjdE9wdGlvbnM/LmFjY2Vzc0tleUlkU2VjcmV0LFxuICAgICAgICAgICAgICAgIHNlY3JldEFjY2Vzc0tleVNlY3JldDpcbiAgICAgICAgICAgICAgICAgIG9wdGlvbnMuY29kZUFydGlmYWN0T3B0aW9ucz8uc2VjcmV0QWNjZXNzS2V5U2VjcmV0LFxuICAgICAgICAgICAgICAgIHJvbGVUb0Fzc3VtZTogb3B0aW9ucy5jb2RlQXJ0aWZhY3RPcHRpb25zPy5yb2xlVG9Bc3N1bWUsXG4gICAgICAgICAgICAgICAgYXV0aFByb3ZpZGVyOiBvcHRpb25zLmNvZGVBcnRpZmFjdE9wdGlvbnM/LmF1dGhQcm92aWRlclxuICAgICAgICAgICAgICAgICAgPyBub2RlUGFja2FnZVRvUmVsZWFzZUNvZGVBcnRpZmFjdEF1dGhQcm92aWRlck1hcHBpbmdbXG4gICAgICAgICAgICAgICAgICAgICAgb3B0aW9ucy5jb2RlQXJ0aWZhY3RPcHRpb25zLmF1dGhQcm92aWRlclxuICAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgICA6IFJlbGVhc2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIuQUNDRVNTX0FORF9TRUNSRVRfS0VZX1BBSVIsXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDoge307XG4gICAgICAgIHRoaXMucmVsZWFzZS5wdWJsaXNoZXIucHVibGlzaFRvTnBtKHtcbiAgICAgICAgICByZWdpc3RyeTogdGhpcy5wYWNrYWdlLm5wbVJlZ2lzdHJ5LFxuICAgICAgICAgIG5wbVRva2VuU2VjcmV0OiB0aGlzLnBhY2thZ2UubnBtVG9rZW5TZWNyZXQsXG4gICAgICAgICAgbnBtUHJvdmVuYW5jZTogdGhpcy5wYWNrYWdlLm5wbVByb3ZlbmFuY2UsXG4gICAgICAgICAgdHJ1c3RlZFB1Ymxpc2hpbmc6IG9wdGlvbnMubnBtVHJ1c3RlZFB1Ymxpc2hpbmcgPz8gZmFsc2UsXG4gICAgICAgICAgY29kZUFydGlmYWN0T3B0aW9ucyxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIHZhbGlkYXRlIHRoYXQgbm8gcmVsZWFzZSBvcHRpb25zIGFyZSBzZWxlY3RlZCBpZiB0aGUgcmVsZWFzZSB3b3JrZmxvdyBpcyBkaXNhYmxlZC5cbiAgICAgIGlmIChvcHRpb25zLnJlbGVhc2VUb05wbSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgJ1wicmVsZWFzZVRvTnBtXCIgaXMgbm90IHN1cHBvcnRlZCBpZiBcInJlbGVhc2VcIiBpcyBub3Qgc2V0J1xuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICBpZiAob3B0aW9ucy5yZWxlYXNlRXZlcnlDb21taXQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICdcInJlbGVhc2VFdmVyeUNvbW1pdFwiIGlzIG5vdCBzdXBwb3J0ZWQgaWYgXCJyZWxlYXNlXCIgaXMgbm90IHNldCdcbiAgICAgICAgKTtcbiAgICAgIH1cblxuICAgICAgaWYgKG9wdGlvbnMucmVsZWFzZVNjaGVkdWxlKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAnXCJyZWxlYXNlU2NoZWR1bGVcIiBpcyBub3Qgc3VwcG9ydGVkIGlmIFwicmVsZWFzZVwiIGlzIG5vdCBzZXQnXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKFxuICAgICAgKG9wdGlvbnMuYXV0b01lcmdlID8/IHRydWUpICYmXG4gICAgICB0aGlzLmdpdGh1Yj8ubWVyZ2lmeSAmJlxuICAgICAgdGhpcy5idWlsZFdvcmtmbG93Py5idWlsZEpvYklkc1xuICAgICkge1xuICAgICAgdGhpcy5hdXRvTWVyZ2UgPSBuZXcgQXV0b01lcmdlKHRoaXMuZ2l0aHViLCBvcHRpb25zLmF1dG9NZXJnZU9wdGlvbnMpO1xuICAgICAgdGhpcy5hdXRvTWVyZ2UuYWRkQ29uZGl0aW9uc0xhdGVyKHtcbiAgICAgICAgcmVuZGVyOiAoKSA9PlxuICAgICAgICAgIHRoaXMuYnVpbGRXb3JrZmxvdz8uYnVpbGRKb2JJZHMubWFwKChpZCkgPT4gYHN0YXR1cy1zdWNjZXNzPSR7aWR9YCkgPz9cbiAgICAgICAgICBbXSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGNvbnN0IGRlcGVuZGFib3QgPSBvcHRpb25zLmRlcGVuZGFib3QgPz8gZmFsc2U7XG4gICAgY29uc3QgZGVwc1VwZ3JhZGUgPSBvcHRpb25zLmRlcHNVcGdyYWRlID8/ICFkZXBlbmRhYm90O1xuXG4gICAgaWYgKGRlcGVuZGFib3QgJiYgZGVwc1VwZ3JhZGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCInZGVwZW5kYWJvdCcgY2Fubm90IGJlIGNvbmZpZ3VyZWQgdG9nZXRoZXIgd2l0aCAnZGVwc1VwZ3JhZGUnXCJcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgZGVwc0F1dG9BcHByb3ZlID0gb3B0aW9ucy5hdXRvQXBwcm92ZVVwZ3JhZGVzID8/IGZhbHNlO1xuXG4gICAgaWYgKGRlcHNBdXRvQXBwcm92ZSAmJiAhdGhpcy5hdXRvQXBwcm92ZSAmJiB0aGlzLmdpdGh1Yikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIkF1dG9tYXRpYyBhcHByb3ZhbCBvZiBkZXBlbmRlbmNpZXMgdXBncmFkZXMgcmVxdWlyZXMgY29uZmlndXJpbmcgYGF1dG9BcHByb3ZlT3B0aW9uc2BcIlxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCBhdXRvQXBwcm92ZUxhYmVsID0gKGNvbmRpdGlvbjogYm9vbGVhbikgPT5cbiAgICAgIGNvbmRpdGlvbiAmJiB0aGlzLmF1dG9BcHByb3ZlPy5sYWJlbFxuICAgICAgICA/IFt0aGlzLmF1dG9BcHByb3ZlLmxhYmVsXVxuICAgICAgICA6IHVuZGVmaW5lZDtcblxuICAgIGlmIChkZXBlbmRhYm90KSB7XG4gICAgICBjb25zdCBkZWZhdWx0T3B0aW9ucyA9IHtcbiAgICAgICAgbGFiZWxzOiBhdXRvQXBwcm92ZUxhYmVsKGRlcHNBdXRvQXBwcm92ZSksXG4gICAgICB9O1xuICAgICAgdGhpcy5naXRodWI/LmFkZERlcGVuZGFib3QoXG4gICAgICAgIGRlZXBNZXJnZShbZGVmYXVsdE9wdGlvbnMsIG9wdGlvbnMuZGVwZW5kYWJvdE9wdGlvbnMgPz8ge31dKVxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAoZGVwc1VwZ3JhZGUpIHtcbiAgICAgIGNvbnN0IGRlZmF1bHRPcHRpb25zOiBVcGdyYWRlRGVwZW5kZW5jaWVzT3B0aW9ucyA9IHtcbiAgICAgICAgd29ya2Zsb3dPcHRpb25zOiB7XG4gICAgICAgICAgY29udGFpbmVyOiBvcHRpb25zLndvcmtmbG93Q29udGFpbmVySW1hZ2VcbiAgICAgICAgICAgID8ge1xuICAgICAgICAgICAgICAgIGltYWdlOiBvcHRpb25zLndvcmtmbG93Q29udGFpbmVySW1hZ2UsXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICAgIGxhYmVsczogYXV0b0FwcHJvdmVMYWJlbChkZXBzQXV0b0FwcHJvdmUpLFxuICAgICAgICAgIGdpdElkZW50aXR5OiB0aGlzLndvcmtmbG93R2l0SWRlbnRpdHksXG4gICAgICAgICAgcGVybWlzc2lvbnM6IHdvcmtmbG93UGVybWlzc2lvbnMsXG4gICAgICAgIH0sXG4gICAgICB9O1xuICAgICAgdGhpcy51cGdyYWRlV29ya2Zsb3cgPSBuZXcgVXBncmFkZURlcGVuZGVuY2llcyhcbiAgICAgICAgdGhpcyxcbiAgICAgICAgZGVlcE1lcmdlKFtkZWZhdWx0T3B0aW9ucywgb3B0aW9ucy5kZXBzVXBncmFkZU9wdGlvbnMgPz8ge31dKVxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAob3B0aW9ucy5wdWxsUmVxdWVzdFRlbXBsYXRlID8/IHRydWUpIHtcbiAgICAgIHRoaXMuZ2l0aHViPy5hZGRQdWxsUmVxdWVzdFRlbXBsYXRlKFxuICAgICAgICAuLi4ob3B0aW9ucy5wdWxsUmVxdWVzdFRlbXBsYXRlQ29udGVudHMgPz8gW10pXG4gICAgICApO1xuICAgIH1cblxuICAgIGNvbnN0IHByb2plbnJjSnMgPSBvcHRpb25zLnByb2plbnJjSnMgPz8gIW9wdGlvbnMucHJvamVucmNKc29uO1xuICAgIGlmICghdGhpcy5wYXJlbnQgJiYgcHJvamVucmNKcykge1xuICAgICAgY29uc3QgcHJvamVucmNKc0ZpbGUgPSBuZXcgUHJvamVucmModGhpcywgb3B0aW9ucy5wcm9qZW5yY0pzT3B0aW9ucyk7XG5cbiAgICAgIHRoaXMubnBtaWdub3JlPy5leGNsdWRlKGAvJHtwcm9qZW5yY0pzRmlsZS5maWxlUGF0aH1gKTtcbiAgICB9IGVsc2UgaWYgKG9wdGlvbnMucHJvamVucmNKc29uKSB7XG4gICAgICBjb25zdCBwcm9qZW5yY0pzb25GaWxlID0gUHJvamVucmNKc29uLm9mKHRoaXMpO1xuICAgICAgaWYgKHByb2plbnJjSnNvbkZpbGUpIHtcbiAgICAgICAgdGhpcy5ucG1pZ25vcmU/LmV4Y2x1ZGUoYC8ke3Byb2plbnJjSnNvbkZpbGUuZmlsZVBhdGh9YCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gYWRkIGEgYnVuZGxlciBjb21wb25lbnQgLSB0aGlzIGVuYWJsZXMgdGhpbmdzIGxpa2UgTGFtYmRhIGJ1bmRsaW5nIGFuZCBpbiB0aGUgZnV0dXJlIHdlYiBidW5kbGluZy5cbiAgICB0aGlzLmJ1bmRsZXIgPSBuZXcgQnVuZGxlcih0aGlzLCBvcHRpb25zLmJ1bmRsZXJPcHRpb25zKTtcblxuICAgIGNvbnN0IHNob3VsZFBhY2thZ2UgPSBvcHRpb25zLnBhY2thZ2UgPz8gdHJ1ZTtcbiAgICBpZiAocmVsZWFzZSAmJiAhc2hvdWxkUGFja2FnZSkge1xuICAgICAgdGhpcy5sb2dnZXIud2FybihcbiAgICAgICAgXCJXaGVuIGByZWxlYXNlYCBpcyBlbmFibGVkLCBgcGFja2FnZWAgbXVzdCBhbHNvIGJlIGVuYWJsZWQgYXMgaXQgaXMgcmVxdWlyZWQgYnkgcmVsZWFzZS4gRm9yY2UgZW5hYmxpbmcgYHBhY2thZ2VgLlwiXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAocmVsZWFzZSB8fCBzaG91bGRQYWNrYWdlKSB7XG4gICAgICB0aGlzLnBhY2thZ2VUYXNrLmV4ZWMoYG1rZGlyIC1wICR7dGhpcy5hcnRpZmFjdHNKYXZhc2NyaXB0RGlyZWN0b3J5fWApO1xuXG4gICAgICBjb25zdCBwa2dNZ3IgPVxuICAgICAgICB0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIgPT09IE5vZGVQYWNrYWdlTWFuYWdlci5QTlBNXG4gICAgICAgICAgPyBcInBucG1cIlxuICAgICAgICAgIDogXCJucG1cIjtcbiAgICAgIHRoaXMucGFja2FnZVRhc2suZXhlYyhcbiAgICAgICAgYCR7cGtnTWdyfSBwYWNrIC0tcGFjay1kZXN0aW5hdGlvbiAke3RoaXMuYXJ0aWZhY3RzSmF2YXNjcmlwdERpcmVjdG9yeX1gXG4gICAgICApO1xuICAgIH1cblxuICAgIGlmIChtdWx0aXBsZVNlbGVjdGVkKFtvcHRpb25zLmJpb21lLCBvcHRpb25zLnByZXR0aWVyXSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcIk9ubHkgb25lIG9mIGJpb21lIGFuZCBwcmV0dGllciBjYW4gYmUgZW5hYmxlZC5cIik7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMuYmlvbWUgPz8gZmFsc2UpIHtcbiAgICAgIHRoaXMuYmlvbWUgPSBuZXcgQmlvbWUodGhpcywgeyAuLi5vcHRpb25zLmJpb21lT3B0aW9ucyB9KTtcbiAgICB9XG5cbiAgICBpZiAob3B0aW9ucy5wcmV0dGllciA/PyBmYWxzZSkge1xuICAgICAgdGhpcy5wcmV0dGllciA9IG5ldyBQcmV0dGllcih0aGlzLCB7IC4uLm9wdGlvbnMucHJldHRpZXJPcHRpb25zIH0pO1xuICAgIH1cblxuICAgIC8vIEZvciBQTlBNLCB0aGUgZGVmYXVsdCByZXNvbHV0aW9uIG1vZGUgaXMgXCJsb3dlc3RcIiwgd2hpY2ggbGVhZHMgdG8gYW55IG5vbi12ZXJzaW9uZWQgKGllICcqJykgZGVwZW5kZW5jaWVzIGJlaW5nXG4gICAgLy8gcmVzb2x2ZWQgdG8gdGhlIGxvd2VzdCBhdmFpbGFibGUgdmVyc2lvbiwgd2hpY2ggaXMgdW5saWtlbHkgdG8gYmUgZXhwZWN0ZWQgYmVoYXZpb3VyIGZvciB1c2Vycy4gV2Ugc2V0IHJlc29sdXRpb25cbiAgICAvLyBtb2RlIHRvIFwiaGlnaGVzdFwiIHRvIG1hdGNoIHRoZSBiZWhhdmlvdXIgb2YgeWFybiBhbmQgbnBtLlxuICAgIGlmICh0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIgPT09IE5vZGVQYWNrYWdlTWFuYWdlci5QTlBNKSB7XG4gICAgICB0aGlzLm5wbXJjLmFkZENvbmZpZyhcInJlc29sdXRpb24tbW9kZVwiLCBcImhpZ2hlc3RcIik7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMuY2hlY2tMaWNlbnNlcykge1xuICAgICAgbmV3IExpY2Vuc2VDaGVja2VyKHRoaXMsIG9wdGlvbnMuY2hlY2tMaWNlbnNlcyk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBkZXRlcm1pbmVJbnN0YWxsV29ya2luZ0RpcmVjdG9yeSgpOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICAgIGlmICh0aGlzLnBhcmVudCkge1xuICAgICAgcmV0dXJuIGVuc3VyZVJlbGF0aXZlUGF0aFN0YXJ0c1dpdGhEb3QocmVsYXRpdmUoXCIuXCIsIHRoaXMucm9vdC5vdXRkaXIpKTtcbiAgICB9XG4gICAgcmV0dXJuO1xuICB9XG5cbiAgcHJpdmF0ZSBkZXRlcm1pbmVJZFRva2VuUGVybWlzc2lvbnMoXG4gICAgb3B0aW9uczogTm9kZVByb2plY3RPcHRpb25zXG4gICk6IEpvYlBlcm1pc3Npb24gfCB1bmRlZmluZWQge1xuICAgIGNvbnN0IHsgY29kZUNvdlRva2VuU2VjcmV0LCBzY29wZWRQYWNrYWdlc09wdGlvbnMsIGNvZGVBcnRpZmFjdE9wdGlvbnMgfSA9XG4gICAgICBvcHRpb25zO1xuXG4gICAgY29uc3QgdXNlQ29kZUNvdmVPaWRjID0gdGhpcy51c2VDb2RlY292KG9wdGlvbnMpICYmICFjb2RlQ292VG9rZW5TZWNyZXQ7XG4gICAgY29uc3QgaGFzU2NvcGVkUGFja2FnZXMgPSAoc2NvcGVkUGFja2FnZXNPcHRpb25zID8/IFtdKS5sZW5ndGggPiAwO1xuICAgIGNvbnN0IHVzZUNvZGVBcnRpZmFjdE9pZGMgPVxuICAgICAgY29kZUFydGlmYWN0T3B0aW9ucz8uYXV0aFByb3ZpZGVyID09PVxuICAgICAgQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyLkdJVEhVQl9PSURDO1xuXG4gICAgaWYgKHVzZUNvZGVDb3ZlT2lkYyB8fCAodXNlQ29kZUFydGlmYWN0T2lkYyAmJiBoYXNTY29wZWRQYWNrYWdlcykpIHtcbiAgICAgIHJldHVybiBKb2JQZXJtaXNzaW9uLldSSVRFO1xuICAgIH1cblxuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cblxuICBwcml2YXRlIHVzZUNvZGVjb3Yob3B0aW9uczogTm9kZVByb2plY3RPcHRpb25zKTogYm9vbGVhbiB7XG4gICAgLy8gVXNlIENvZGVjb3Ygd2hlbiBpdCBpcyBlbmFibGVkIG9yIGlmIG9yIGEgc2VjcmV0IHRva2VuIG5hbWUgaXMgcGFzc2VkIGluXG4gICAgLy8gQU5EIGplc3QgbXVzdCBiZSBjb25maWd1cmVkXG4gICAgcmV0dXJuIChvcHRpb25zLmNvZGVDb3YgfHwgb3B0aW9ucy5jb2RlQ292VG9rZW5TZWNyZXQpICYmIHRoaXMuamVzdD8uY29uZmlnO1xuICB9XG5cbiAgcHJpdmF0ZSBtYXliZUFkZENvZGVjb3ZJZ25vcmVzKG9wdGlvbnM6IE5vZGVQcm9qZWN0T3B0aW9ucykge1xuICAgIGlmICh0aGlzLnVzZUNvZGVjb3Yob3B0aW9ucykpIHtcbiAgICAgIHRoaXMuYWRkR2l0SWdub3JlKFwiY29kZWNvdlwiKTtcbiAgICAgIHRoaXMuYWRkR2l0SWdub3JlKFwiY29kZWNvdi4qXCIpO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyVXBsb2FkQ292ZXJhZ2VKb2JTdGVwKG9wdGlvbnM6IE5vZGVQcm9qZWN0T3B0aW9ucyk6IEpvYlN0ZXBbXSB7XG4gICAgLy8gcnVuIGNvZGVjb3YgaWYgZW5hYmxlZCBvciBhIHNlY3JldCB0b2tlbiBuYW1lIGlzIHBhc3NlZCBpblxuICAgIC8vIEFORCBqZXN0IG11c3QgYmUgY29uZmlndXJlZFxuICAgIGlmICh0aGlzLnVzZUNvZGVjb3Yob3B0aW9ucykpIHtcbiAgICAgIHJldHVybiBbXG4gICAgICAgIHtcbiAgICAgICAgICBuYW1lOiBcIlVwbG9hZCBjb3ZlcmFnZSB0byBDb2RlY292XCIsXG4gICAgICAgICAgdXNlczogXCJjb2RlY292L2NvZGVjb3YtYWN0aW9uQHY1XCIsXG4gICAgICAgICAgd2l0aDogb3B0aW9ucy5jb2RlQ292VG9rZW5TZWNyZXRcbiAgICAgICAgICAgID8ge1xuICAgICAgICAgICAgICAgIHRva2VuOiBgXFwke3sgc2VjcmV0cy4ke29wdGlvbnMuY29kZUNvdlRva2VuU2VjcmV0fSB9fWAsXG4gICAgICAgICAgICAgICAgZGlyZWN0b3J5OiB0aGlzLmplc3Q/LmNvbmZpZy5jb3ZlcmFnZURpcmVjdG9yeSxcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgOiB7XG4gICAgICAgICAgICAgICAgZGlyZWN0b3J5OiB0aGlzLmplc3Q/LmNvbmZpZy5jb3ZlcmFnZURpcmVjdG9yeSxcbiAgICAgICAgICAgICAgICB1c2Vfb2lkYzogdHJ1ZSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIF07XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBbXTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgYWRkQmlucyhiaW5zOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+KSB7XG4gICAgdGhpcy5wYWNrYWdlLmFkZEJpbihiaW5zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXBsYWNlcyB0aGUgY29udGVudHMgb2YgYW4gbnBtIHBhY2thZ2UuanNvbiBzY3JpcHQuXG4gICAqXG4gICAqIEBwYXJhbSBuYW1lIFRoZSBzY3JpcHQgbmFtZVxuICAgKiBAcGFyYW0gY29tbWFuZCBUaGUgY29tbWFuZCB0byBleGVjdXRlXG4gICAqL1xuICBwdWJsaWMgc2V0U2NyaXB0KG5hbWU6IHN0cmluZywgY29tbWFuZDogc3RyaW5nKSB7XG4gICAgdGhpcy5wYWNrYWdlLnNldFNjcmlwdChuYW1lLCBjb21tYW5kKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXBsYWNlcyB0aGUgY29udGVudHMgb2YgbXVsdGlwbGUgbnBtIHBhY2thZ2UuanNvbiBzY3JpcHRzLlxuICAgKiBAcGFyYW0gc2NyaXB0cyBUaGUgc2NyaXB0cyB0byBzZXRcbiAgICovXG4gIHB1YmxpYyBhZGRTY3JpcHRzKHNjcmlwdHM6IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9KSB7XG4gICAgZm9yIChjb25zdCBbbmFtZSwgY29tbWFuZF0gb2YgT2JqZWN0LmVudHJpZXMoc2NyaXB0cykpIHtcbiAgICAgIHRoaXMucGFja2FnZS5zZXRTY3JpcHQobmFtZSwgY29tbWFuZCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZXMgdGhlIG5wbSBzY3JpcHQgKGFsd2F5cyBzdWNjZXNzZnVsKS5cbiAgICogQHBhcmFtIG5hbWUgVGhlIG5hbWUgb2YgdGhlIHNjcmlwdC5cbiAgICovXG4gIHB1YmxpYyByZW1vdmVTY3JpcHQobmFtZTogc3RyaW5nKSB7XG4gICAgdGhpcy5wYWNrYWdlLnJlbW92ZVNjcmlwdChuYW1lKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgaWYgYSBzY3JpcHQgYnkgdGhlIG5hbWUgbmFtZSBpcyBkZWZpbmVkLlxuICAgKiBAcGFyYW0gbmFtZSBUaGUgbmFtZSBvZiB0aGUgc2NyaXB0XG4gICAqIEBkZXByZWNhdGVkIFVzZSBgcHJvamVjdC50YXNrcy50cnlGaW5kKG5hbWUpYFxuICAgKi9cbiAgcHVibGljIGhhc1NjcmlwdChuYW1lOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLmhhc1NjcmlwdChuYW1lKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBERVBSRUNBVEVEXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcHJvamVjdC5jb21waWxlVGFzay5leGVjKClgXG4gICAqL1xuICBwdWJsaWMgYWRkQ29tcGlsZUNvbW1hbmQoLi4uY29tbWFuZHM6IHN0cmluZ1tdKSB7XG4gICAgZm9yIChjb25zdCBjIG9mIGNvbW1hbmRzKSB7XG4gICAgICB0aGlzLmNvbXBpbGVUYXNrLmV4ZWMoYyk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIERFUFJFQ0FURURcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwcm9qZWN0LnRlc3RUYXNrLmV4ZWMoKWBcbiAgICovXG4gIHB1YmxpYyBhZGRUZXN0Q29tbWFuZCguLi5jb21tYW5kczogc3RyaW5nW10pIHtcbiAgICBmb3IgKGNvbnN0IGMgb2YgY29tbWFuZHMpIHtcbiAgICAgIHRoaXMudGVzdFRhc2suZXhlYyhjKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogRGlyZWN0bHkgc2V0IGZpZWxkcyBpbiBgcGFja2FnZS5qc29uYC5cbiAgICogQHBhcmFtIGZpZWxkcyBUaGUgZmllbGRzIHRvIHNldFxuICAgKi9cbiAgcHVibGljIGFkZEZpZWxkcyhmaWVsZHM6IHsgW25hbWU6IHN0cmluZ106IGFueSB9KSB7XG4gICAgZm9yIChjb25zdCBbbmFtZSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKGZpZWxkcykpIHtcbiAgICAgIHRoaXMucGFja2FnZS5hZGRGaWVsZChuYW1lLCB2YWx1ZSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMga2V5d29yZHMgdG8gcGFja2FnZS5qc29uIChkZWR1cGxpY2F0ZWQpXG4gICAqIEBwYXJhbSBrZXl3b3JkcyBUaGUga2V5d29yZHMgdG8gYWRkXG4gICAqL1xuICBwdWJsaWMgYWRkS2V5d29yZHMoLi4ua2V5d29yZHM6IHN0cmluZ1tdKSB7XG4gICAgdGhpcy5wYWNrYWdlLmFkZEtleXdvcmRzKC4uLmtleXdvcmRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXQgc3RlcHMgZm9yIHNjb3BlZCBwYWNrYWdlIGFjY2Vzc1xuICAgKlxuICAgKiBAcGFyYW0gY29kZUFydGlmYWN0T3B0aW9ucyBEZXRhaWxzIG9mIGxvZ2dpbmcgaW4gdG8gQVdTXG4gICAqIEByZXR1cm5zIGFycmF5IG9mIGpvYiBzdGVwcyByZXF1aXJlZCBmb3IgZWFjaCBwcml2YXRlIHNjb3BlZCBwYWNrYWdlc1xuICAgKi9cbiAgcHJpdmF0ZSBnZXRTY29wZWRQYWNrYWdlU3RlcHMoXG4gICAgY29kZUFydGlmYWN0T3B0aW9uczogQ29kZUFydGlmYWN0T3B0aW9ucyB8IHVuZGVmaW5lZFxuICApOiBKb2JTdGVwW10ge1xuICAgIGNvbnN0IHBhcnNlZENvZGVBcnRpZmFjdE9wdGlvbnMgPSB7XG4gICAgICBhY2Nlc3NLZXlJZFNlY3JldDpcbiAgICAgICAgY29kZUFydGlmYWN0T3B0aW9ucz8uYWNjZXNzS2V5SWRTZWNyZXQgPz8gXCJBV1NfQUNDRVNTX0tFWV9JRFwiLFxuICAgICAgc2VjcmV0QWNjZXNzS2V5U2VjcmV0OlxuICAgICAgICBjb2RlQXJ0aWZhY3RPcHRpb25zPy5zZWNyZXRBY2Nlc3NLZXlTZWNyZXQgPz8gXCJBV1NfU0VDUkVUX0FDQ0VTU19LRVlcIixcbiAgICAgIHJvbGVUb0Fzc3VtZTogY29kZUFydGlmYWN0T3B0aW9ucz8ucm9sZVRvQXNzdW1lLFxuICAgICAgYXV0aFByb3ZpZGVyOiBjb2RlQXJ0aWZhY3RPcHRpb25zPy5hdXRoUHJvdmlkZXIsXG4gICAgfTtcblxuICAgIGlmIChcbiAgICAgIHBhcnNlZENvZGVBcnRpZmFjdE9wdGlvbnMuYXV0aFByb3ZpZGVyID09PVxuICAgICAgTm9kZVBhY2thZ2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIuR0lUSFVCX09JRENcbiAgICApIHtcbiAgICAgIHJldHVybiBbXG4gICAgICAgIHtcbiAgICAgICAgICBuYW1lOiBcIkNvbmZpZ3VyZSBBV1MgQ3JlZGVudGlhbHNcIixcbiAgICAgICAgICB1c2VzOiBcImF3cy1hY3Rpb25zL2NvbmZpZ3VyZS1hd3MtY3JlZGVudGlhbHNAdjVcIixcbiAgICAgICAgICB3aXRoOiB7XG4gICAgICAgICAgICBcImF3cy1yZWdpb25cIjogXCJ1cy1lYXN0LTJcIixcbiAgICAgICAgICAgIFwicm9sZS10by1hc3N1bWVcIjogcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5yb2xlVG9Bc3N1bWUsXG4gICAgICAgICAgICBcInJvbGUtZHVyYXRpb24tc2Vjb25kc1wiOiA5MDAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiQVdTIENvZGVBcnRpZmFjdCBMb2dpblwiLFxuICAgICAgICAgIHJ1bjogYCR7dGhpcy5ydW5TY3JpcHRDb21tYW5kfSBjYTpsb2dpbmAsXG4gICAgICAgIH0sXG4gICAgICBdO1xuICAgIH1cblxuICAgIGlmIChwYXJzZWRDb2RlQXJ0aWZhY3RPcHRpb25zLnJvbGVUb0Fzc3VtZSkge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiQ29uZmlndXJlIEFXUyBDcmVkZW50aWFsc1wiLFxuICAgICAgICAgIHVzZXM6IFwiYXdzLWFjdGlvbnMvY29uZmlndXJlLWF3cy1jcmVkZW50aWFsc0B2NVwiLFxuICAgICAgICAgIHdpdGg6IHtcbiAgICAgICAgICAgIFwiYXdzLWFjY2Vzcy1rZXktaWRcIjogc2VjcmV0VG9TdHJpbmcoXG4gICAgICAgICAgICAgIHBhcnNlZENvZGVBcnRpZmFjdE9wdGlvbnMuYWNjZXNzS2V5SWRTZWNyZXRcbiAgICAgICAgICAgICksXG4gICAgICAgICAgICBcImF3cy1zZWNyZXQtYWNjZXNzLWtleVwiOiBzZWNyZXRUb1N0cmluZyhcbiAgICAgICAgICAgICAgcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5zZWNyZXRBY2Nlc3NLZXlTZWNyZXRcbiAgICAgICAgICAgICksXG4gICAgICAgICAgICBcImF3cy1yZWdpb25cIjogXCJ1cy1lYXN0LTJcIixcbiAgICAgICAgICAgIFwicm9sZS10by1hc3N1bWVcIjogcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5yb2xlVG9Bc3N1bWUsXG4gICAgICAgICAgICBcInJvbGUtZHVyYXRpb24tc2Vjb25kc1wiOiA5MDAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiQVdTIENvZGVBcnRpZmFjdCBMb2dpblwiLFxuICAgICAgICAgIHJ1bjogYCR7dGhpcy5ydW5TY3JpcHRDb21tYW5kfSBjYTpsb2dpbmAsXG4gICAgICAgIH0sXG4gICAgICBdO1xuICAgIH1cblxuICAgIHJldHVybiBbXG4gICAgICB7XG4gICAgICAgIG5hbWU6IFwiQVdTIENvZGVBcnRpZmFjdCBMb2dpblwiLFxuICAgICAgICBydW46IGAke3RoaXMucnVuU2NyaXB0Q29tbWFuZH0gY2E6bG9naW5gLFxuICAgICAgICBlbnY6IHtcbiAgICAgICAgICBBV1NfQUNDRVNTX0tFWV9JRDogc2VjcmV0VG9TdHJpbmcoXG4gICAgICAgICAgICBwYXJzZWRDb2RlQXJ0aWZhY3RPcHRpb25zLmFjY2Vzc0tleUlkU2VjcmV0XG4gICAgICAgICAgKSxcbiAgICAgICAgICBBV1NfU0VDUkVUX0FDQ0VTU19LRVk6IHNlY3JldFRvU3RyaW5nKFxuICAgICAgICAgICAgcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5zZWNyZXRBY2Nlc3NLZXlTZWNyZXRcbiAgICAgICAgICApLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICBdO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIHNldCBvZiB3b3JrZmxvdyBzdGVwcyB3aGljaCBzaG91bGQgYmUgZXhlY3V0ZWQgdG8gYm9vdHN0cmFwIGFcbiAgICogd29ya2Zsb3cuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIE9wdGlvbnMuXG4gICAqIEByZXR1cm5zIEpvYiBzdGVwc1xuICAgKi9cbiAgcHVibGljIHJlbmRlcldvcmtmbG93U2V0dXAoXG4gICAgb3B0aW9uczogUmVuZGVyV29ya2Zsb3dTZXR1cE9wdGlvbnMgPSB7fVxuICApOiBKb2JTdGVwW10ge1xuICAgIGNvbnN0IGluc3RhbGwgPSBuZXcgQXJyYXk8Sm9iU3RlcD4oKTtcblxuICAgIC8vIGZpcnN0IHJ1biB0aGUgd29ya2Zsb3cgYm9vdHN0cmFwIHN0ZXBzXG4gICAgaW5zdGFsbC5wdXNoKC4uLnRoaXMud29ya2Zsb3dCb290c3RyYXBTdGVwcyk7XG5cbiAgICBpZiAodGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyID09PSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTSkge1xuICAgICAgaW5zdGFsbC5wdXNoKHtcbiAgICAgICAgbmFtZTogXCJTZXR1cCBwbnBtXCIsXG4gICAgICAgIHVzZXM6IFwicG5wbS9hY3Rpb24tc2V0dXBAdjRcIixcbiAgICAgICAgd2l0aDogeyB2ZXJzaW9uOiB0aGlzLnBhY2thZ2UucG5wbVZlcnNpb24gfSxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSBpZiAodGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyID09PSBOb2RlUGFja2FnZU1hbmFnZXIuQlVOKSB7XG4gICAgICBpbnN0YWxsLnB1c2goe1xuICAgICAgICBuYW1lOiBcIlNldHVwIGJ1blwiLFxuICAgICAgICB1c2VzOiBcIm92ZW4tc2gvc2V0dXAtYnVuQHYyXCIsXG4gICAgICAgIHdpdGg6IHsgXCJidW4tdmVyc2lvblwiOiB0aGlzLnBhY2thZ2UuYnVuVmVyc2lvbiB9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlciAhPT0gTm9kZVBhY2thZ2VNYW5hZ2VyLkJVTikge1xuICAgICAgaWYgKHRoaXMubm9kZVZlcnNpb24gfHwgdGhpcy53b3JrZmxvd1BhY2thZ2VDYWNoZSkge1xuICAgICAgICBjb25zdCBjYWNoZSA9XG4gICAgICAgICAgdGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyID09PSBOb2RlUGFja2FnZU1hbmFnZXIuWUFSTlxuICAgICAgICAgICAgPyBcInlhcm5cIlxuICAgICAgICAgICAgOiB0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIgPT09IE5vZGVQYWNrYWdlTWFuYWdlci5ZQVJOMlxuICAgICAgICAgICAgPyBcInlhcm5cIlxuICAgICAgICAgICAgOiB0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIgPT09IE5vZGVQYWNrYWdlTWFuYWdlci5ZQVJOX0NMQVNTSUNcbiAgICAgICAgICAgID8gXCJ5YXJuXCJcbiAgICAgICAgICAgIDogdGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyID09PSBOb2RlUGFja2FnZU1hbmFnZXIuWUFSTl9CRVJSWVxuICAgICAgICAgICAgPyBcInlhcm5cIlxuICAgICAgICAgICAgOiB0aGlzLnBhY2thZ2VNYW5hZ2VyID09PSBOb2RlUGFja2FnZU1hbmFnZXIuQlVOXG4gICAgICAgICAgICA/IFwiYnVuXCJcbiAgICAgICAgICAgIDogdGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyID09PSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTVxuICAgICAgICAgICAgPyBcInBucG1cIlxuICAgICAgICAgICAgOiBcIm5wbVwiO1xuICAgICAgICBpbnN0YWxsLnB1c2goe1xuICAgICAgICAgIG5hbWU6IFwiU2V0dXAgTm9kZS5qc1wiLFxuICAgICAgICAgIHVzZXM6IFwiYWN0aW9ucy9zZXR1cC1ub2RlQHY1XCIsXG4gICAgICAgICAgd2l0aDoge1xuICAgICAgICAgICAgLi4uKHRoaXMubm9kZVZlcnNpb24gJiYge1xuICAgICAgICAgICAgICBcIm5vZGUtdmVyc2lvblwiOiB0aGlzLm5vZGVWZXJzaW9uLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgICAuLi4odGhpcy53b3JrZmxvd1BhY2thZ2VDYWNoZSAmJiB7XG4gICAgICAgICAgICAgIGNhY2hlLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgbXV0YWJsZSA9IG9wdGlvbnMubXV0YWJsZSA/PyBmYWxzZTtcblxuICAgIGlmICh0aGlzLnBhY2thZ2Uuc2NvcGVkUGFja2FnZXNPcHRpb25zKSB7XG4gICAgICBpbnN0YWxsLnB1c2goXG4gICAgICAgIC4uLnRoaXMuZ2V0U2NvcGVkUGFja2FnZVN0ZXBzKHRoaXMucGFja2FnZS5jb2RlQXJ0aWZhY3RPcHRpb25zKVxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpbnN0YWxsLnB1c2goe1xuICAgICAgbmFtZTogXCJJbnN0YWxsIGRlcGVuZGVuY2llc1wiLFxuICAgICAgcnVuOiBtdXRhYmxlXG4gICAgICAgID8gdGhpcy5wYWNrYWdlLmluc3RhbGxBbmRVcGRhdGVMb2NrZmlsZUNvbW1hbmRcbiAgICAgICAgOiB0aGlzLnBhY2thZ2UuaW5zdGFsbENvbW1hbmQsXG4gICAgICAuLi4ob3B0aW9ucy5pbnN0YWxsU3RlcENvbmZpZ3VyYXRpb24gPz8ge30pLFxuICAgIH0pO1xuXG4gICAgcmV0dXJuIGluc3RhbGw7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBub3JtYWwgZGVwZW5kZW5jaWVzLlxuICAgKlxuICAgKiBAcGFyYW0gZGVwcyBOYW1lcyBtb2R1bGVzIHRvIGluc3RhbGwuIEJ5IGRlZmF1bHQsIHRoZSB0aGUgZGVwZW5kZW5jeSB3aWxsXG4gICAqIGJlIGluc3RhbGxlZCBpbiB0aGUgbmV4dCBgbnB4IHByb2plbmAgcnVuIGFuZCB0aGUgdmVyc2lvbiB3aWxsIGJlIHJlY29yZGVkXG4gICAqIGluIHlvdXIgYHBhY2thZ2UuanNvbmAgZmlsZS4gWW91IGNhbiB1cGdyYWRlIG1hbnVhbGx5IG9yIHVzaW5nIGB5YXJuXG4gICAqIGFkZC91cGdyYWRlYC4gSWYgeW91IHdpc2ggdG8gc3BlY2lmeSBhIHZlcnNpb24gcmFuZ2UgdXNlIHRoaXMgc3ludGF4OlxuICAgKiBgbW9kdWxlQF43YC5cbiAgICovXG4gIHB1YmxpYyBhZGREZXBzKC4uLmRlcHM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5hZGREZXBzKC4uLmRlcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlZmluZXMgZGV2ZWxvcG1lbnQvdGVzdCBkZXBlbmRlbmNpZXMuXG4gICAqXG4gICAqIEBwYXJhbSBkZXBzIE5hbWVzIG1vZHVsZXMgdG8gaW5zdGFsbC4gQnkgZGVmYXVsdCwgdGhlIHRoZSBkZXBlbmRlbmN5IHdpbGxcbiAgICogYmUgaW5zdGFsbGVkIGluIHRoZSBuZXh0IGBucHggcHJvamVuYCBydW4gYW5kIHRoZSB2ZXJzaW9uIHdpbGwgYmUgcmVjb3JkZWRcbiAgICogaW4geW91ciBgcGFja2FnZS5qc29uYCBmaWxlLiBZb3UgY2FuIHVwZ3JhZGUgbWFudWFsbHkgb3IgdXNpbmcgYHlhcm5cbiAgICogYWRkL3VwZ3JhZGVgLiBJZiB5b3Ugd2lzaCB0byBzcGVjaWZ5IGEgdmVyc2lvbiByYW5nZSB1c2UgdGhpcyBzeW50YXg6XG4gICAqIGBtb2R1bGVAXjdgLlxuICAgKi9cbiAgcHVibGljIGFkZERldkRlcHMoLi4uZGVwczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLmFkZERldkRlcHMoLi4uZGVwcyk7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBwZWVyIGRlcGVuZGVuY2llcy5cbiAgICpcbiAgICogV2hlbiBhZGRpbmcgcGVlciBkZXBlbmRlbmNpZXMsIGEgZGV2RGVwZW5kZW5jeSB3aWxsIGFsc28gYmUgYWRkZWQgb24gdGhlXG4gICAqIHBpbm5lZCB2ZXJzaW9uIG9mIHRoZSBkZWNsYXJlZCBwZWVyLiBUaGlzIHdpbGwgZW5zdXJlIHRoYXQgeW91IGFyZSB0ZXN0aW5nXG4gICAqIHlvdXIgY29kZSBhZ2FpbnN0IHRoZSBtaW5pbXVtIHZlcnNpb24gcmVxdWlyZWQgZnJvbSB5b3VyIGNvbnN1bWVycy5cbiAgICpcbiAgICogQHBhcmFtIGRlcHMgTmFtZXMgbW9kdWxlcyB0byBpbnN0YWxsLiBCeSBkZWZhdWx0LCB0aGUgdGhlIGRlcGVuZGVuY3kgd2lsbFxuICAgKiBiZSBpbnN0YWxsZWQgaW4gdGhlIG5leHQgYG5weCBwcm9qZW5gIHJ1biBhbmQgdGhlIHZlcnNpb24gd2lsbCBiZSByZWNvcmRlZFxuICAgKiBpbiB5b3VyIGBwYWNrYWdlLmpzb25gIGZpbGUuIFlvdSBjYW4gdXBncmFkZSBtYW51YWxseSBvciB1c2luZyBgeWFyblxuICAgKiBhZGQvdXBncmFkZWAuIElmIHlvdSB3aXNoIHRvIHNwZWNpZnkgYSB2ZXJzaW9uIHJhbmdlIHVzZSB0aGlzIHN5bnRheDpcbiAgICogYG1vZHVsZUBeN2AuXG4gICAqL1xuICBwdWJsaWMgYWRkUGVlckRlcHMoLi4uZGVwczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLmFkZFBlZXJEZXBzKC4uLmRlcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYnVuZGxlZCBkZXBlbmRlbmNpZXMuXG4gICAqXG4gICAqIEJ1bmRsZWQgZGVwZW5kZW5jaWVzIHdpbGwgYmUgYWRkZWQgYXMgbm9ybWFsIGRlcGVuZGVuY2llcyBhcyB3ZWxsIGFzIHRvIHRoZVxuICAgKiBgYnVuZGxlZERlcGVuZGVuY2llc2Agc2VjdGlvbiBvZiB5b3VyIGBwYWNrYWdlLmpzb25gLlxuICAgKlxuICAgKiBAcGFyYW0gZGVwcyBOYW1lcyBtb2R1bGVzIHRvIGluc3RhbGwuIEJ5IGRlZmF1bHQsIHRoZSB0aGUgZGVwZW5kZW5jeSB3aWxsXG4gICAqIGJlIGluc3RhbGxlZCBpbiB0aGUgbmV4dCBgbnB4IHByb2plbmAgcnVuIGFuZCB0aGUgdmVyc2lvbiB3aWxsIGJlIHJlY29yZGVkXG4gICAqIGluIHlvdXIgYHBhY2thZ2UuanNvbmAgZmlsZS4gWW91IGNhbiB1cGdyYWRlIG1hbnVhbGx5IG9yIHVzaW5nIGB5YXJuXG4gICAqIGFkZC91cGdyYWRlYC4gSWYgeW91IHdpc2ggdG8gc3BlY2lmeSBhIHZlcnNpb24gcmFuZ2UgdXNlIHRoaXMgc3ludGF4OlxuICAgKiBgbW9kdWxlQF43YC5cbiAgICovXG4gIHB1YmxpYyBhZGRCdW5kbGVkRGVwcyguLi5kZXBzOiBzdHJpbmdbXSkge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UuYWRkQnVuZGxlZERlcHMoLi4uZGVwcyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBwYXR0ZXJucyB0byBiZSBpZ25vcmVkIGJ5IG5wbS5cbiAgICpcbiAgICogQHBhcmFtIHBhdHRlcm4gVGhlIHBhdHRlcm4gdG8gaWdub3JlLlxuICAgKlxuICAgKiBAcmVtYXJrc1xuICAgKiBJZiB5b3UgYXJlIGhhdmluZyB0cm91YmxlIGdldHRpbmcgYW4gaWdub3JlIHRvIHBvcHVsYXRlLCB0cnkgdXNpbmcgeW91ciBjb25zdHJ1Y3Qgb3IgY29tcG9uZW50J3MgcHJlU3ludGhlc2l6ZSBtZXRob2QgdG8gcHJvcGVybHkgZGVsYXkgY2FsbGluZyB0aGlzIG1ldGhvZC5cbiAgICovXG4gIHB1YmxpYyBvdmVycmlkZSBhZGRQYWNrYWdlSWdub3JlKHBhdHRlcm46IHN0cmluZyk6IHZvaWQge1xuICAgIHRoaXMubnBtaWdub3JlPy5hZGRQYXR0ZXJucyhwYXR0ZXJuKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkTGljZW5zZShvcHRpb25zOiBOb2RlUHJvamVjdE9wdGlvbnMpIHtcbiAgICBpZiAodGhpcy5wYWNrYWdlLmxpY2Vuc2UpIHtcbiAgICAgIG5ldyBMaWNlbnNlKHRoaXMsIHtcbiAgICAgICAgc3BkeDogdGhpcy5wYWNrYWdlLmxpY2Vuc2UsXG4gICAgICAgIGNvcHlyaWdodE93bmVyOiBvcHRpb25zLmNvcHlyaWdodE93bmVyID8/IG9wdGlvbnMuYXV0aG9yTmFtZSxcbiAgICAgICAgY29weXJpZ2h0UGVyaW9kOiBvcHRpb25zLmNvcHlyaWdodFBlcmlvZCxcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgYWRkRGVmYXVsdEdpdElnbm9yZSgpIHtcbiAgICB0aGlzLmdpdGlnbm9yZS5leGNsdWRlKFxuICAgICAgXCIjIExvZ3NcIixcbiAgICAgIFwibG9nc1wiLFxuICAgICAgXCIqLmxvZ1wiLFxuICAgICAgXCJucG0tZGVidWcubG9nKlwiLFxuICAgICAgXCJ5YXJuLWRlYnVnLmxvZypcIixcbiAgICAgIFwieWFybi1lcnJvci5sb2cqXCIsXG4gICAgICBcImxlcm5hLWRlYnVnLmxvZypcIixcblxuICAgICAgXCIjIERpYWdub3N0aWMgcmVwb3J0cyAoaHR0cHM6Ly9ub2RlanMub3JnL2FwaS9yZXBvcnQuaHRtbClcIixcbiAgICAgIFwicmVwb3J0LlswLTldKi5bMC05XSouWzAtOV0qLlswLTldKi5qc29uXCIsXG5cbiAgICAgIFwiIyBSdW50aW1lIGRhdGFcIixcbiAgICAgIFwicGlkc1wiLFxuICAgICAgXCIqLnBpZFwiLFxuICAgICAgXCIqLnNlZWRcIixcbiAgICAgIFwiKi5waWQubG9ja1wiLFxuXG4gICAgICBcIiMgRGlyZWN0b3J5IGZvciBpbnN0cnVtZW50ZWQgbGlicyBnZW5lcmF0ZWQgYnkganNjb3ZlcmFnZS9KU0NvdmVyXCIsXG4gICAgICBcImxpYi1jb3ZcIixcblxuICAgICAgXCIjIENvdmVyYWdlIGRpcmVjdG9yeSB1c2VkIGJ5IHRvb2xzIGxpa2UgaXN0YW5idWxcIixcbiAgICAgIFwiY292ZXJhZ2VcIixcbiAgICAgIFwiKi5sY292XCIsXG5cbiAgICAgIFwiIyBueWMgdGVzdCBjb3ZlcmFnZVwiLFxuICAgICAgXCIubnljX291dHB1dFwiLFxuXG4gICAgICBcIiMgQ29tcGlsZWQgYmluYXJ5IGFkZG9ucyAoaHR0cHM6Ly9ub2RlanMub3JnL2FwaS9hZGRvbnMuaHRtbClcIixcbiAgICAgIFwiYnVpbGQvUmVsZWFzZVwiLFxuXG4gICAgICBcIiMgRGVwZW5kZW5jeSBkaXJlY3Rvcmllc1wiLFxuICAgICAgXCJub2RlX21vZHVsZXMvXCIsXG4gICAgICBcImpzcG1fcGFja2FnZXMvXCIsXG5cbiAgICAgIFwiIyBUeXBlU2NyaXB0IGNhY2hlXCIsXG4gICAgICBcIioudHNidWlsZGluZm9cIixcblxuICAgICAgXCIjIE9wdGlvbmFsIGVzbGludCBjYWNoZVwiLFxuICAgICAgXCIuZXNsaW50Y2FjaGVcIixcblxuICAgICAgXCIjIE91dHB1dCBvZiAnbnBtIHBhY2snXCIsXG4gICAgICBcIioudGd6XCIsXG5cbiAgICAgIFwiIyBZYXJuIEludGVncml0eSBmaWxlXCIsXG4gICAgICBcIi55YXJuLWludGVncml0eVwiLFxuXG4gICAgICBcIiMgcGFyY2VsLWJ1bmRsZXIgY2FjaGUgKGh0dHBzOi8vcGFyY2VsanMub3JnLylcIixcbiAgICAgIFwiLmNhY2hlXCJcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIHNoZWxsIGNvbW1hbmQgdG8gZXhlY3V0ZSBpbiBvcmRlciB0byBydW4gYSB0YXNrLiBUaGlzIHdpbGxcbiAgICogdHlwaWNhbGx5IGJlIGBucHggcHJvamVuIFRBU0tgLlxuICAgKlxuICAgKiBAcGFyYW0gdGFzayBUaGUgdGFzayBmb3Igd2hpY2ggdGhlIGNvbW1hbmQgaXMgcmVxdWlyZWRcbiAgICovXG4gIHB1YmxpYyBydW5UYXNrQ29tbWFuZCh0YXNrOiBUYXNrKSB7XG4gICAgcmV0dXJuIGAke3RoaXMucGFja2FnZS5wcm9qZW5Db21tYW5kfSAke3Rhc2submFtZX1gO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBqb2IgSUQgb2YgdGhlIGJ1aWxkIHdvcmtmbG93LlxuICAgKi9cbiAgcHVibGljIGdldCBidWlsZFdvcmtmbG93Sm9iSWQoKSB7XG4gICAgcmV0dXJuIHRoaXMuYnVpbGRXb3JrZmxvdz8uYnVpbGRKb2JJZHNbMF07XG4gIH1cbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgcmVuZGVyV29ya2Zsb3dTZXR1cCgpYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSZW5kZXJXb3JrZmxvd1NldHVwT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBDb25maWd1cmUgdGhlIGluc3RhbGwgc3RlcCBpbiB0aGUgd29ya2Zsb3cgc2V0dXAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYHsgbmFtZTogXCJJbnN0YWxsIGRlcGVuZGVuY2llc1wiIH1gXG4gICAqXG4gICAqIEBleGFtcGxlIC0geyB3b3JraW5nRGlyZWN0b3J5OiBcInJvb3Rwcm9qZWN0LWRpclwiIH0gZm9yIHN1YnByb2plY3RzIGluc3RhbGxpbmcgZnJvbSByb290LlxuICAgKiBAZXhhbXBsZSAtIHsgZW52OiB7IE5QTV9UT0tFTjogXCJ0b2tlblwiIH19IGZvciBpbnN0YWxsaW5nIGZyb20gcHJpdmF0ZSBucG0gcmVnaXN0cnkuXG4gICAqL1xuICByZWFkb25seSBpbnN0YWxsU3RlcENvbmZpZ3VyYXRpb24/OiBKb2JTdGVwQ29uZmlndXJhdGlvbjtcbiAgLyoqXG4gICAqIFNob3VsZCB0aGUgcGFja2FnZSBsb2NrZmlsZSBiZSB1cGRhdGVkP1xuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgbXV0YWJsZT86IGJvb2xlYW47XG59XG4iXX0=