"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpgradeDependenciesSchedule = exports.UpgradeDependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const workflows_model_1 = require("../github/workflows-model");
const javascript_1 = require("../javascript");
const release_1 = require("../release");
const runner_options_1 = require("../runner-options");
const CREATE_PATCH_STEP_ID = "create_patch";
const PATCH_CREATED_OUTPUT = "patch_created";
/**
 * Upgrade node project dependencies.
 */
class UpgradeDependencies extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        /**
         * The workflows that execute the upgrades. One workflow per branch.
         */
        this.workflows = [];
        this._project = project;
        this.options = options;
        this.depTypes = this.options.types ?? [
            dependencies_1.DependencyType.BUILD,
            dependencies_1.DependencyType.BUNDLED,
            dependencies_1.DependencyType.DEVENV,
            dependencies_1.DependencyType.PEER,
            dependencies_1.DependencyType.RUNTIME,
            dependencies_1.DependencyType.TEST,
            dependencies_1.DependencyType.OPTIONAL,
        ];
        this.upgradeTarget = this.options.target ?? "minor";
        this.satisfyPeerDependencies = this.options.satisfyPeerDependencies ?? true;
        this.includeDeprecatedVersions =
            this.options.includeDeprecatedVersions ?? false;
        this.pullRequestTitle = options.pullRequestTitle ?? "upgrade dependencies";
        this.gitIdentity =
            options.workflowOptions?.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.permissions = {
            contents: workflows_model_1.JobPermission.READ,
            ...options.workflowOptions?.permissions,
        };
        this.postBuildSteps = [];
        this.containerOptions = options.workflowOptions?.container;
        this.postUpgradeTask =
            project.tasks.tryFind("post-upgrade") ??
                project.tasks.addTask("post-upgrade", {
                    description: "Runs after upgrading dependencies",
                });
        this.upgradeTask = project.addTask(options.taskName ?? "upgrade", {
            // this task should not run in CI mode because its designed to
            // update package.json and lock files.
            env: { CI: "0" },
            description: this.pullRequestTitle,
            steps: { toJSON: () => this.renderTaskSteps() },
        });
        this.upgradeTask.lock(); // this task is a lazy value, so make it readonly
        if (this.upgradeTask && project.github && (options.workflow ?? true)) {
            if (options.workflowOptions?.branches) {
                for (const branch of options.workflowOptions.branches) {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                }
            }
            else if (release_1.Release.of(project)) {
                const release = release_1.Release.of(project);
                release._forEachBranch((branch) => {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                });
            }
            else {
                // represents the default repository branch.
                // just like not specifying anything.
                const defaultBranch = undefined;
                this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, defaultBranch));
            }
        }
    }
    /**
     * Add steps to execute a successful build.
     * @param steps workflow steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    renderTaskSteps() {
        function executeCommand(packageManager) {
            switch (packageManager) {
                case javascript_1.NodePackageManager.NPM:
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN_CLASSIC:
                    return "npx";
                case javascript_1.NodePackageManager.PNPM:
                    return "pnpm dlx";
                case javascript_1.NodePackageManager.YARN2:
                case javascript_1.NodePackageManager.YARN_BERRY:
                    return "yarn dlx";
                case javascript_1.NodePackageManager.BUN:
                    return "bunx";
            }
        }
        const steps = new Array();
        // Package Manager upgrade should always include all deps
        const includeForPackageManagerUpgrade = this.buildDependencyList(true);
        if (includeForPackageManagerUpgrade.length === 0) {
            return [{ exec: "echo No dependencies to upgrade." }];
        }
        // Removing `npm-check-updates` from our dependency tree because it depends on a package
        // that uses an npm-specific feature that causes an invalid dependency tree when using Yarn 1.
        // See https://github.com/projen/projen/pull/3136 for more details.
        const includeForNcu = this.buildDependencyList(false);
        const ncuCommand = [
            `${executeCommand(this._project.package.packageManager)} npm-check-updates@16`,
            "--upgrade",
            `--target=${this.upgradeTarget}`,
            `--${this.satisfyPeerDependencies ? "peer" : "no-peer"}`,
            `--${this.includeDeprecatedVersions ? "deprecated" : "no-deprecated"}`,
            `--dep=${this.renderNcuDependencyTypes(this.depTypes)}`,
            `--filter=${includeForNcu.join(",")}`,
        ];
        // bump versions in package.json
        if (includeForNcu.length) {
            steps.push({ exec: ncuCommand.join(" ") });
        }
        // run "yarn/npm install" to update the lockfile and install any deps (such as projen)
        steps.push({ exec: this._project.package.installAndUpdateLockfileCommand });
        // run upgrade command to upgrade transitive deps as well
        steps.push({
            exec: this.renderUpgradePackagesCommand(includeForPackageManagerUpgrade),
        });
        // run "projen" to give projen a chance to update dependencies (it will also run "yarn install")
        steps.push({ exec: this._project.projenCommand });
        steps.push({ spawn: this.postUpgradeTask.name });
        return steps;
    }
    /**
     * Render projen dependencies types to a list of ncu compatible types
     */
    renderNcuDependencyTypes(types) {
        return Array.from(new Set(types
            .map((type) => {
            switch (type) {
                case dependencies_1.DependencyType.PEER:
                    return "peer";
                case dependencies_1.DependencyType.RUNTIME:
                    return "prod";
                case dependencies_1.DependencyType.OPTIONAL:
                    return "optional";
                case dependencies_1.DependencyType.TEST:
                case dependencies_1.DependencyType.DEVENV:
                case dependencies_1.DependencyType.BUILD:
                    return "dev";
                case dependencies_1.DependencyType.BUNDLED:
                default:
                    return false;
            }
        })
            .filter((type) => Boolean(type)))).join(",");
    }
    /**
     * Render a package manager specific command to upgrade all requested dependencies.
     */
    renderUpgradePackagesCommand(include) {
        function upgradePackages(command) {
            return () => {
                return `${command} ${include.join(" ")}`;
            };
        }
        const packageManager = this._project.package.packageManager;
        let lazy = undefined;
        switch (packageManager) {
            case javascript_1.NodePackageManager.YARN:
            case javascript_1.NodePackageManager.YARN_CLASSIC:
                lazy = upgradePackages("yarn upgrade");
                break;
            case javascript_1.NodePackageManager.YARN2:
            case javascript_1.NodePackageManager.YARN_BERRY:
                lazy = upgradePackages("yarn up");
                break;
            case javascript_1.NodePackageManager.NPM:
                lazy = upgradePackages("npm update");
                break;
            case javascript_1.NodePackageManager.PNPM:
                lazy = upgradePackages("pnpm update");
                break;
            case javascript_1.NodePackageManager.BUN:
                lazy = upgradePackages("bun update");
                break;
            default:
                throw new Error(`unexpected package manager ${packageManager}`);
        }
        // return a lazy function so that dependencies include ones that were
        // added post project instantiation (i.e using project.addDeps)
        return lazy;
    }
    buildDependencyList(includeDependenciesWithConstraint) {
        return Array.from(new Set(this.options.include ??
            this.filterDependencies(includeDependenciesWithConstraint)));
    }
    filterDependencies(includeConstraint) {
        const dependencies = [];
        const deps = this.project.deps.all
            // remove those that have a constraint version (unless includeConstraint is true)
            .filter((d) => includeConstraint || this.packageCanBeUpgradedInPackageJson(d.version))
            // remove override dependencies
            .filter((d) => d.type !== dependencies_1.DependencyType.OVERRIDE);
        for (const type of this.depTypes) {
            dependencies.push(...deps
                .filter((d) => d.type === type)
                .filter((d) => !(this.options.exclude ?? []).includes(d.name)));
        }
        return dependencies.map((d) => d.name);
    }
    /**
     * Projen can alter a package's version in package.json when either the version is omitted, or set to "*".
     * Otherwise, the exact version selected is placed in the package.json file and upgrading is handled through the package manager
     * rather than npm-check-updates.
     *
     * @param version semver from DependencyCoordinates.version, may be undefined
     * @returns whether the version is the default versioning behavior
     */
    packageCanBeUpgradedInPackageJson(version) {
        // No version means "latest"
        return !version || version === "*";
    }
    createWorkflow(task, github, branch) {
        const schedule = this.options.workflowOptions?.schedule ??
            UpgradeDependenciesSchedule.DAILY;
        const workflowName = `${task.name}${branch ? `-${branch.replace(/\//g, "-")}` : ""}`;
        const workflow = github.addWorkflow(workflowName);
        const triggers = {
            workflowDispatch: {},
            schedule: schedule.cron.length > 0
                ? schedule.cron.map((e) => ({ cron: e }))
                : undefined,
        };
        workflow.on(triggers);
        const upgrade = this.createUpgrade(task, github, branch);
        const pr = this.createPr(workflow, upgrade);
        const jobs = {};
        jobs[upgrade.jobId] = upgrade.job;
        jobs[pr.jobId] = pr.job;
        workflow.addJobs(jobs);
        return workflow;
    }
    createUpgrade(task, github, branch) {
        const with_ = {
            ...(branch ? { ref: branch } : {}),
            ...(github.downloadLfs ? { lfs: true } : {}),
        };
        const steps = [
            github_1.WorkflowSteps.checkout({ with: with_ }),
            ...this._project.renderWorkflowSetup({ mutable: false }),
            {
                name: "Upgrade dependencies",
                run: this._project.runTaskCommand(task),
            },
        ];
        steps.push(...this.postBuildSteps);
        steps.push(...workflow_actions_1.WorkflowActions.uploadGitPatch({
            stepId: CREATE_PATCH_STEP_ID,
            outputName: PATCH_CREATED_OUTPUT,
        }));
        return {
            job: {
                name: "Upgrade",
                container: this.containerOptions,
                permissions: this.permissions,
                env: this.options.workflowOptions?.env,
                ...(0, runner_options_1.filteredRunsOnOptions)(this.options.workflowOptions?.runsOn, this.options.workflowOptions?.runsOnGroup),
                steps: steps,
                outputs: {
                    [PATCH_CREATED_OUTPUT]: {
                        stepId: CREATE_PATCH_STEP_ID,
                        outputName: PATCH_CREATED_OUTPUT,
                    },
                },
            },
            jobId: "upgrade",
            ref: branch,
        };
    }
    createPr(workflow, upgrade) {
        const credentials = this.options.workflowOptions?.projenCredentials ??
            workflow.projenCredentials;
        const semanticCommit = this.options.semanticCommit ?? "chore";
        return {
            job: github_1.WorkflowJobs.pullRequestFromPatch({
                patch: {
                    jobId: upgrade.jobId,
                    outputName: PATCH_CREATED_OUTPUT,
                    ref: upgrade.ref,
                },
                workflowName: workflow.name,
                credentials,
                ...(0, runner_options_1.filteredRunsOnOptions)(this.options.workflowOptions?.runsOn, this.options.workflowOptions?.runsOnGroup),
                pullRequestTitle: `${semanticCommit}(deps): ${this.pullRequestTitle}`,
                pullRequestDescription: "Upgrades project dependencies.",
                gitIdentity: this.gitIdentity,
                assignees: this.options.workflowOptions?.assignees,
                labels: this.options.workflowOptions?.labels,
                signoff: this.options.signoff,
            }),
            jobId: "pr",
        };
    }
}
exports.UpgradeDependencies = UpgradeDependencies;
_a = JSII_RTTI_SYMBOL_1;
UpgradeDependencies[_a] = { fqn: "projen.javascript.UpgradeDependencies", version: "0.98.4" };
/**
 * How often to check for new versions and raise pull requests for version upgrades.
 */
class UpgradeDependenciesSchedule {
    /**
     * Create a schedule from a raw cron expression.
     */
    static expressions(cron) {
        return new UpgradeDependenciesSchedule(cron);
    }
    constructor(cron) {
        this.cron = cron;
    }
}
exports.UpgradeDependenciesSchedule = UpgradeDependenciesSchedule;
_b = JSII_RTTI_SYMBOL_1;
UpgradeDependenciesSchedule[_b] = { fqn: "projen.javascript.UpgradeDependenciesSchedule", version: "0.98.4" };
/**
 * Disables automatic upgrades.
 */
UpgradeDependenciesSchedule.NEVER = new UpgradeDependenciesSchedule([]);
/**
 * At 00:00.
 */
UpgradeDependenciesSchedule.DAILY = new UpgradeDependenciesSchedule(["0 0 * * *"]);
/**
 * At 00:00 on every day-of-week from Monday through Friday.
 */
UpgradeDependenciesSchedule.WEEKDAY = new UpgradeDependenciesSchedule([
    "0 0 * * 1-5",
]);
/**
 * At 00:00 on Monday.
 */
UpgradeDependenciesSchedule.WEEKLY = new UpgradeDependenciesSchedule([
    "0 0 * * 1",
]);
/**
 * At 00:00 on day-of-month 1.
 */
UpgradeDependenciesSchedule.MONTHLY = new UpgradeDependenciesSchedule([
    "0 0 1 * *",
]);
//# sourceMappingURL=data:application/json;base64,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