"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Release = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const publisher_1 = require("./publisher");
const release_trigger_1 = require("./release-trigger");
const consts_1 = require("../build/private/consts");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const util_1 = require("../github/private/util");
const workflows_model_1 = require("../github/workflows-model");
const runner_options_1 = require("../runner-options");
const util_2 = require("../util");
const name_1 = require("../util/name");
const path_1 = require("../util/path");
const version_1 = require("../version");
const BUILD_JOBID = "release";
const GIT_REMOTE_STEPID = "git_remote";
const TAG_EXISTS_STEPID = "check_tag_exists";
const LATEST_COMMIT_OUTPUT = "latest_commit";
const TAG_EXISTS_OUTPUT = "tag_exists";
/**
 * Conditional (Github Workflow Job `if`) to check if a release job should be run.
 */
const DEPENDENT_JOB_CONDITIONAL = `needs.${BUILD_JOBID}.outputs.${TAG_EXISTS_OUTPUT} != 'true' && needs.${BUILD_JOBID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha`;
/**
 * Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 */
class Release extends component_1.Component {
    /**
     * Returns the `Release` component of a project or `undefined` if the project
     * does not have a Release component.
     */
    static of(project) {
        const isRelease = (c) => c instanceof Release;
        return project.components.find(isRelease);
    }
    /**
     * @param scope should be part of the project the Release belongs to.
     * @param options options to configure the Release Component.
     */
    constructor(scope, options) {
        super(scope);
        this._branches = new Array();
        this.jobs = {};
        if (Array.isArray(options.releaseBranches)) {
            throw new Error('"releaseBranches" is no longer an array. See type annotations');
        }
        this.github = github_1.GitHub.of(this.project.root);
        this.buildTask = options.task;
        this.preBuildSteps = options.releaseWorkflowSetupSteps ?? [];
        this.postBuildSteps = options.postBuildSteps ?? [];
        this.artifactsDirectory =
            options.artifactsDirectory ?? consts_1.DEFAULT_ARTIFACTS_DIRECTORY;
        (0, util_1.ensureNotHiddenPath)(this.artifactsDirectory, "artifactsDirectory");
        this.versionFile = options.versionFile;
        this.releaseTrigger = options.releaseTrigger ?? release_trigger_1.ReleaseTrigger.continuous();
        this.containerImage = options.workflowContainerImage;
        this.workflowRunsOn = options.workflowRunsOn;
        this.workflowRunsOnGroup = options.workflowRunsOnGroup;
        this.workflowPermissions = {
            contents: workflows_model_1.JobPermission.WRITE,
            ...options.workflowPermissions,
        };
        this.releaseWorkflowEnv = options.releaseWorkflowEnv;
        this._branchHooks = [];
        /**
         * Use manual releases with no changelog if releaseEveryCommit is explicitly
         * disabled and no other trigger is set.
         *
         * TODO: Remove this when releaseEveryCommit and releaseSchedule are removed
         */
        if (!((options.releaseEveryCommit ?? true) ||
            options.releaseSchedule ||
            options.releaseTrigger)) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.manual({ changelog: false });
        }
        if (options.releaseSchedule) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.scheduled({
                schedule: options.releaseSchedule,
            });
        }
        this.version = new version_1.Version(this.project, {
            versionInputFile: this.versionFile,
            artifactsDirectory: this.artifactsDirectory,
            versionrcOptions: options.versionrcOptions,
            tagPrefix: options.releaseTagPrefix,
            releasableCommits: options.releasableCommits,
            bumpPackage: options.bumpPackage,
            nextVersionCommand: options.nextVersionCommand,
        });
        this.releaseTagFilePath = path.posix.normalize(path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName));
        this.publisher = new publisher_1.Publisher(this.project, {
            artifactName: this.artifactsDirectory,
            condition: DEPENDENT_JOB_CONDITIONAL,
            buildJobId: BUILD_JOBID,
            jsiiReleaseVersion: options.jsiiReleaseVersion,
            failureIssue: options.releaseFailureIssue,
            failureIssueLabel: options.releaseFailureIssueLabel,
            ...(0, runner_options_1.filteredWorkflowRunsOnOptions)(options.workflowRunsOn, options.workflowRunsOnGroup),
            publishTasks: options.publishTasks,
            dryRun: options.publishDryRun,
            workflowNodeVersion: options.workflowNodeVersion,
            workflowContainerImage: options.workflowContainerImage,
        });
        const githubRelease = options.githubRelease ?? true;
        if (githubRelease) {
            this.publisher.publishToGitHubReleases({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
            });
        }
        // add the default branch (we need the internal method which does not require majorVersion)
        this.defaultBranch = this._addBranch(options.branch, {
            prerelease: options.prerelease,
            majorVersion: options.majorVersion,
            minMajorVersion: options.minMajorVersion,
            workflowName: options.releaseWorkflowName ??
                (0, name_1.workflowNameForProject)("release", this.project),
            environment: options.releaseEnvironment,
            tagPrefix: options.releaseTagPrefix,
            npmDistTag: options.npmDistTag,
        });
        for (const [name, opts] of Object.entries(options.releaseBranches ?? {})) {
            this.addBranch(name, {
                environment: options.releaseEnvironment,
                ...opts,
            });
        }
    }
    /**
     * Add a hook that should be run for every branch (including those that will
     * be added by future `addBranch` calls).
     * @internal
     */
    _forEachBranch(hook) {
        for (const branch of this._branches) {
            hook(branch.name);
        }
        this._branchHooks.push(hook);
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    addBranch(branch, options) {
        this._addBranch(branch, options);
        // run all branch hooks
        for (const hook of this._branchHooks) {
            hook(branch);
        }
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    _addBranch(branch, options) {
        if (this._branches.find((b) => b.name === branch)) {
            throw new Error(`The release branch ${branch} is already defined`);
        }
        // if we add a branch, we require that the default branch will also define a
        // major version.
        if (this.defaultBranch &&
            options.majorVersion &&
            this.defaultBranch.majorVersion === undefined) {
            throw new Error('you must specify "majorVersion" for the default branch when adding multiple release branches');
        }
        const releaseBranch = {
            name: branch,
            ...options,
            workflow: this.createWorkflow(branch, options),
        };
        this._branches.push(releaseBranch);
        return releaseBranch;
    }
    preSynthesize() {
        for (const branch of this._branches) {
            if (!branch.workflow) {
                continue;
            }
            branch.workflow.addJobs(this.publisher._renderJobsForBranch(branch.name, branch));
            branch.workflow.addJobs(this.jobs);
        }
    }
    /**
     * Adds jobs to all release workflows.
     * @param jobs The jobs to add (name => job)
     */
    addJobs(jobs) {
        for (const [name, job] of Object.entries(jobs)) {
            this.jobs[name] = job;
        }
    }
    /**
     * Retrieve all release branch names
     */
    get branches() {
        return this._branches.map((b) => b.name);
    }
    /**
     * @returns a workflow or `undefined` if github integration is disabled.
     */
    createWorkflow(branchName, branch) {
        const workflowName = branch.workflowName ??
            (0, name_1.workflowNameForProject)(`release-${branchName}`, this.project);
        // to avoid race conditions between two commits trying to release the same
        // version, we check if the head sha is identical to the remote sha. if
        // not, we will skip the release and just finish the build.
        const noNewCommits = `\${{ steps.${GIT_REMOTE_STEPID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha }}`;
        // The arrays are being cloned to avoid accumulating values from previous branches
        const preBuildSteps = [...this.preBuildSteps];
        const env = {
            RELEASE: "true",
            ...this.version.envForBranch({
                majorVersion: branch.majorVersion,
                minorVersion: branch.minorVersion,
                minMajorVersion: branch.minMajorVersion,
                prerelease: branch.prerelease,
                tagPrefix: branch.tagPrefix,
            }),
        };
        // the "release" task prepares a release but does not publish anything. the
        // output of the release task is: `dist`, `.version.txt`, and
        // `.changelog.md`. this is what publish tasks expect.
        // if this is the release for "main" or "master", just call it "release".
        // otherwise, "release:BRANCH"
        const releaseTaskName = branchName === "main" || branchName === "master"
            ? "release"
            : `release:${branchName}`;
        const releaseTask = this.project.addTask(releaseTaskName, {
            description: `Prepare a release from "${branchName}" branch`,
            env,
        });
        releaseTask.exec(`rm -fr ${this.artifactsDirectory}`);
        releaseTask.spawn(this.version.bumpTask);
        releaseTask.spawn(this.buildTask);
        releaseTask.spawn(this.version.unbumpTask);
        // anti-tamper check (fails if there were changes to committed files)
        // this will identify any non-committed files generated during build (e.g. test snapshots)
        releaseTask.exec(Release.ANTI_TAMPER_CMD);
        if (this.releaseTrigger.isManual) {
            const publishTask = this.publisher.publishToGit({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
                projectChangelogFile: this.releaseTrigger.changelogPath,
                gitBranch: branchName,
                gitPushCommand: this.releaseTrigger.gitPushCommand,
            });
            releaseTask.spawn(publishTask);
        }
        const postBuildSteps = [...this.postBuildSteps];
        // Read the releasetag, then check if it already exists.
        // If it does, we will cancel this release
        postBuildSteps.push(github_1.WorkflowSteps.tagExists(`$(cat ${this.releaseTagFilePath})`, {
            name: "Check if version has already been tagged",
            id: TAG_EXISTS_STEPID,
        }));
        // check if new commits were pushed to the repo while we were building.
        // if new commits have been pushed, we will cancel this release
        postBuildSteps.push({
            name: "Check for new commits",
            id: GIT_REMOTE_STEPID,
            shell: "bash",
            run: [
                `echo "${LATEST_COMMIT_OUTPUT}=$(git ls-remote origin -h \${{ github.ref }} | cut -f1)" >> $GITHUB_OUTPUT`,
                "cat $GITHUB_OUTPUT",
            ].join("\n"),
        });
        const projectPathRelativeToRoot = path.relative(this.project.root.outdir, this.project.outdir);
        const normalizedProjectPathRelativeToRoot = (0, util_2.normalizePersistedPath)(projectPathRelativeToRoot);
        postBuildSteps.push({
            name: "Backup artifact permissions",
            if: noNewCommits,
            continueOnError: true,
            run: `cd ${this.artifactsDirectory} && getfacl -R . > ${constants_1.PERMISSION_BACKUP_FILE}`,
        }, github_1.WorkflowSteps.uploadArtifact({
            if: noNewCommits,
            with: {
                name: constants_1.BUILD_ARTIFACT_NAME,
                path: normalizedProjectPathRelativeToRoot.length > 0
                    ? `${normalizedProjectPathRelativeToRoot}/${this.artifactsDirectory}`
                    : this.artifactsDirectory,
            },
        }));
        if (this.github && !this.releaseTrigger.isManual) {
            // Use target (possible parent) GitHub to create the workflow
            const workflow = new github_1.GithubWorkflow(this.github, workflowName, {
                // see https://github.com/projen/projen/issues/3761
                limitConcurrency: true,
            });
            workflow.on({
                schedule: this.releaseTrigger.schedule
                    ? [{ cron: this.releaseTrigger.schedule }]
                    : undefined,
                push: this.releaseTrigger.isContinuous
                    ? { branches: [branchName], paths: this.releaseTrigger.paths }
                    : undefined,
                workflowDispatch: {}, // allow manual triggering
            });
            // Create job based on child (only?) project GitHub
            const taskjob = new github_1.TaskWorkflowJob(this, releaseTask, {
                outputs: {
                    [LATEST_COMMIT_OUTPUT]: {
                        stepId: GIT_REMOTE_STEPID,
                        outputName: LATEST_COMMIT_OUTPUT,
                    },
                    [TAG_EXISTS_OUTPUT]: {
                        stepId: TAG_EXISTS_STEPID,
                        outputName: "exists",
                    },
                },
                container: this.containerImage
                    ? { image: this.containerImage }
                    : undefined,
                env: {
                    CI: "true",
                    ...this.releaseWorkflowEnv,
                },
                permissions: this.workflowPermissions,
                checkoutWith: {
                    // fetch-depth= indicates all history for all branches and tags
                    // we must use this in order to fetch all tags
                    // and to inspect the history to decide if we should release
                    fetchDepth: 0,
                },
                preBuildSteps,
                postBuildSteps,
                jobDefaults: normalizedProjectPathRelativeToRoot.length > 0 // is subproject
                    ? {
                        run: {
                            workingDirectory: (0, path_1.ensureRelativePathStartsWithDot)(normalizedProjectPathRelativeToRoot),
                        },
                    }
                    : undefined,
                ...(0, runner_options_1.filteredRunsOnOptions)(this.workflowRunsOn, this.workflowRunsOnGroup),
            });
            workflow.addJob(BUILD_JOBID, taskjob);
            return workflow;
        }
        else {
            return undefined;
        }
    }
}
exports.Release = Release;
_a = JSII_RTTI_SYMBOL_1;
Release[_a] = { fqn: "projen.release.Release", version: "0.98.4" };
Release.ANTI_TAMPER_CMD = "git diff --ignore-space-at-eol --exit-code";
//# sourceMappingURL=data:application/json;base64,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