"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const render_options_1 = require("../javascript/render-options");
const projenrc_1 = require("../projenrc");
const DEFAULT_FILENAME = ".projenrc.ts";
/**
 * Sets up a typescript project to use TypeScript for projenrc.
 */
class Projenrc extends projenrc_1.ProjenrcFile {
    constructor(project, options = {}) {
        super(project);
        this._tsProject = project;
        this.filePath = options.filename ?? DEFAULT_FILENAME;
        this._projenCodeDir = options.projenCodeDir ?? "projenrc";
        this._swc = options.swc ?? false;
        this.addDefaultTask();
        this.generateProjenrc();
    }
    addDefaultTask() {
        const deps = ["ts-node"];
        if (this._swc) {
            deps.push("@swc/core");
        }
        // this is the task projen executes when running `projen` without a
        // specific task (if this task is not defined, projen falls back to
        // running "node .projenrc.js").
        this._tsProject.addDevDeps(...deps);
        const tsNode = this._swc ? "ts-node --swc" : "ts-node";
        // we use "tsconfig.dev.json" here to allow projen source files to reside
        // anywhere in the project tree.
        this._tsProject.defaultTask?.exec(`${tsNode} --project ${this._tsProject.tsconfigDev.fileName} ${this.filePath}`);
    }
    preSynthesize() {
        super.preSynthesize();
        this._tsProject.addPackageIgnore(`/${this.filePath}`);
        this._tsProject.addPackageIgnore(`/${this._projenCodeDir}`);
        this._tsProject.tsconfigDev.addInclude(this.filePath);
        this._tsProject.tsconfigDev.addInclude(`${this._projenCodeDir}/**/*.ts`);
        this._tsProject.eslint?.addLintPattern(this._projenCodeDir);
        this._tsProject.eslint?.addLintPattern(this.filePath);
        this._tsProject.eslint?.allowDevDeps(this.filePath);
        this._tsProject.eslint?.allowDevDeps(`${this._projenCodeDir}/**/*.ts`);
        this._tsProject.eslint?.addIgnorePattern(`!${this.filePath}`);
        this._tsProject.eslint?.addIgnorePattern(`!${this._projenCodeDir}/**/*.ts`);
        this._tsProject.eslint?.addOverride({
            files: [this.filePath],
            rules: {
                "@typescript-eslint/no-require-imports": "off",
                "import/no-extraneous-dependencies": "off",
            },
        });
        this._tsProject.biome?.addFilePattern(this.filePath);
        this._tsProject.biome?.addFilePattern(`${this._projenCodeDir}/**`);
        this._tsProject.jest?.discoverTestMatchPatternsForDirs([this._projenCodeDir], {
            fileExtensionPattern: this._tsProject.tsconfig?.compilerOptions?.allowJs
                ? undefined
                : "ts?(x)",
        });
    }
    generateProjenrc() {
        const rcfile = (0, path_1.resolve)(this.project.outdir, this.filePath);
        if ((0, fs_1.existsSync)(rcfile)) {
            return; // already exists
        }
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const parts = bootstrap.fqn.split(".");
        const moduleName = parts[0];
        const importName = parts[1];
        const className = parts.slice(1).join(".");
        const { renderedOptions, imports } = (0, render_options_1.renderJavaScriptOptions)({
            args: bootstrap.args,
            type: bootstrap.type,
            comments: bootstrap.comments,
        });
        imports.add(moduleName, importName);
        const lines = new Array();
        lines.push(...imports.asEsmImports());
        lines.push();
        lines.push(`const project = new ${className}(${renderedOptions});`);
        lines.push();
        lines.push("project.synth();");
        (0, fs_1.mkdirSync)((0, path_1.dirname)(rcfile), { recursive: true });
        (0, fs_1.writeFileSync)(rcfile, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${rcfile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.typescript.Projenrc", version: "0.98.4" };
//# sourceMappingURL=data:application/json;base64,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