#pragma once

#include <iostream>
#include <unordered_set>
#include <cstdint>

#include "../mmcfilters/utils/Common.hpp"
#include "../mmcfilters/trees/NodeMT.hpp"
#include "../mmcfilters/trees/MorphologicalTree.hpp"


#include <iomanip>
#include <fstream>
#include <iostream>


#include "../external/stb/stb_image.h"
#include "../external/stb/stb_image_write.h"
using namespace mmcfilters;


inline ImageUInt8Ptr openImage(std::string filename){
    //std::cout << "filename:" << filename << std::endl;
    int nchannels;
    int numCols, numRows;
    unsigned char* data = stbi_load(filename.c_str(), &numCols, &numRows, &nchannels, 1);
    
    ImageUInt8Ptr img = ImageUInt8::create(numRows, numCols);
    for (int i = 0; i < numCols * numRows; i++) {
        (*img)[i] = static_cast<int>(data[i]);  // Converte de `unsigned char` para `int`
    }
    stbi_image_free(data); // Liberar a memória da imagem carregada

    return img;
}


/**
 * @brief Imprime uma representação em árvore (ASCII) a partir do nó raiz.
 *
 * @param root Nó raiz (ou sub-raiz) a ser impresso.
 * @param prefix Prefixo usado na recursão para formatar a árvore.
 * @param isLast Indica se o nó atual é o último filho do seu pai (formatação).
 */
inline void printTree(NodeMT root, std::string prefix = "", bool isLast = true) {
    std::cout << prefix;
    std::cout << (isLast ? "└──" : "├──");
    std::cout << "ID: " << root.getIndex() << ", repNode:" << root.getRepNode() <<  ", Level: " <<  root.getLevel()<< ", Area: " << root.getArea() << std::endl;

    prefix += (isLast ? "   " : "│  ");

    const int n = root.getNumChildren();
    int i = 0;
    for (auto child : root.getChildren()) {
        printTree(child, prefix, i == n - 1);
        ++i;
    }

}

/**
 * @brief Imprime o mapeamento de cada pixel para o seu Small Component (SC).
 *
 * Para árvores baseadas em Pixels, utiliza `getSC(p)` diretamente. Para árvores
 * baseadas em FlatZones, atribui a cada pixel o índice do nó que contém sua flatzone.
 * A saída pode ir para stdout ou para arquivo.
 *
 * @param tree Ponteiro para a árvore de componentes.
 * @param setw Largura de coluna para formatação da tabela.
 * @param nomeArquivo Caminho de saída; vazio para imprimir em stdout.
 */
inline void printMappingSC(MorphologicalTreePtr tree, int setw=4, std::string nomeArquivo = "") {

    int numRows = tree->getNumRowsOfImage();
    int numCols = tree->getNumColsOfImage();
    int n = numRows*numCols;
    std::unique_ptr<int[]> map = std::unique_ptr<int[]>(new int[n]);

    for (int p=0; p < n; p++){
        map[p] = tree->getSC(p).getIndex();
    }


    std::ostream* streamSaida;
    std::ofstream arquivoSaida;
    if (nomeArquivo.empty()) {
        streamSaida = &std::cout;
    } else {
        arquivoSaida.open(nomeArquivo);
        if (!arquivoSaida.is_open()) {
            std::cerr << "Erro ao abrir o arquivo para escrita." << std::endl;
            return;
        }
        streamSaida = &arquivoSaida;
    }
    if(tree->isMaxtree()){
        *streamSaida << "---- Mapping from pixel to small component of max-tree ----" << "\n";
    }else{
        *streamSaida << "---- Mapping from pixel to small component of min-tree ----" << "\n";
    }

    // Imprime o cabeçalho de colunas
    *streamSaida << std::setw(setw) << " "; // espaço para a primeira coluna (índice da linha)
    for (int col = 0; col < numCols; col++) {
        *streamSaida << std::setw(setw) << col;
    }
    *streamSaida << "\n";

    // Impressão bidimensional com índice de linha
    for (int row = 0; row < numRows; row++) {
        *streamSaida << std::setw(setw) << row; // índice da linha
        for (int col = 0; col < numCols; col++) {
            *streamSaida << std::setw(setw) << map[ImageUtils::to1D(row, col, numCols)]; // ou ImageUtils::to1D(row, col, numCols)
        }
        *streamSaida << "\n";
    }
    if (streamSaida != &std::cout){
        dynamic_cast<std::ofstream*>(streamSaida)->close(); // std::cout não precisa ser fechado explicitamente
    }

}

/**
 * @brief Imprime uma visão 2D da componente conexa de um nó.
 *
 * Mostra metadados do nó (ID, nível, filhos, CNPs, flatzones, área) e um grid
 * com marcadores distintos para representantes, CNPs e a CC completa.
 * A saída pode ir para stdout ou para arquivo.
 *
 * @param node Nó alvo a ser inspecionado/impressos seus dados.
 * @param tree Árvore à qual o nó pertence (para recuperar dimensões e pixels).
 * @param nomeArquivo Caminho de saída; vazio para imprimir em stdout.
 */
inline  void printConnectedComponent(NodeMT node, MorphologicalTreePtr tree, std::string nomeArquivo = "") {
    int numRows = tree->getNumRowsOfImage();
    int numCols = tree->getNumColsOfImage();
    int n = numRows*numCols;
    std::ostream* streamSaida;
    std::ofstream arquivoSaida;

    if (nomeArquivo.empty()) {
        streamSaida = &std::cout;
    } else {
        arquivoSaida.open(nomeArquivo); 
        if (!arquivoSaida.is_open()) {
            std::cerr << "Erro ao abrir o arquivo para escrita." << std::endl;
            return;
        }
        streamSaida = &arquivoSaida;
    }
    int setw=3;
    *streamSaida << "printCC: ---- "
                 << "ID: " << node
                 << ", parentID: " << node.getParent() 
                 << ", repNode: " << node.getRepNode()
                 << ", level:" << node.getLevel() 
                 << ", |children|:" << node.getNumChildren() 
                 << ", |cnps|:" << node.getNumCNPs()
                 << ", Area:" << node.getArea()  
                 << ", numDescendentes:" << node.getNumDescendants()
                 << " ---\n";
    
    // Imprime o cabeçalho de colunas
    *streamSaida << std::setw(setw) << " "; // espaço para a primeira coluna (índice da linha)
    for (int col = 0; col < numCols; col++) {
        *streamSaida << std::setw(setw) << col;
    }
    *streamSaida << "\n";                 
    // Impressão bidimensional
    std::vector<bool> imageReps(n, false);
    imageReps[node.getRepNode()] = true;
    std::vector<bool> imageCNPs(n, false);
    for (int p : tree->getCNPsById(node)) {
        imageCNPs[p] = true;
    }
    std::vector<bool> imageCC(n, false);
    for (int p : tree->getPixelsOfCCById(node)) {
        imageCC[p] = true;
    }

    for (int row = 0; row < numRows; ++row) {
        *streamSaida << std::setw(setw) << row; // índice da linha
        for (int col = 0; col < numCols; ++col) {
            int index = ImageUtils::to1D(row, col, numCols);
            if (imageReps[index]) {
                *streamSaida << std::setw(setw) << "  🅡";
            }
            else if (imageCNPs[index]) {
                *streamSaida << std::setw(setw) << "  🅲";
            }
            else if (imageCC[index]){
                *streamSaida << std::setw(setw) << "  🅇";
            }
            else{
                *streamSaida << std::setw(setw) << "  ·";
            }
        }
        *streamSaida << "\n";
    }
    if (streamSaida != &std::cout){
        dynamic_cast<std::ofstream*>(streamSaida)->close(); // std::cout não precisa ser fechado explicitamente
    }

}


inline void printConnectedComponents(MorphologicalTreePtr tree, int setw=3, std::string nomeArquivo = ""){
    int numRows = tree->getNumRowsOfImage();
    int numCols = tree->getNumColsOfImage();
    int n = numRows*numCols;
    std::ostream* streamSaida;
    std::ofstream arquivoSaida;

    if (nomeArquivo.empty()) {
        streamSaida = &std::cout;
    } else {
        arquivoSaida.open(nomeArquivo); 
        if (!arquivoSaida.is_open()) {
            std::cerr << "Erro ao abrir o arquivo para escrita." << std::endl;
            return;
        }
        streamSaida = &arquivoSaida;
    }
    auto printTreeType = [](int tipo) {
        if (tipo == 0)
            return "max-tree";
        else if (tipo == 1)
            return "min-tree";
        else
            return "tree of shapes";
    };
    *streamSaida << "----------- Connected components of "<< printTreeType(tree->getTreeType()) <<"  -----------\n" << std::endl;
    for(NodeId nodeId: tree->getIteratorBreadthFirstTraversalById()){
        NodeMT node = tree->proxy(nodeId);
        *streamSaida << "printCC: ---- "
                    << "ID: " << node
                    << ", parentID: " << node.getParent() 
                    << ", repNode: " << node.getRepNode()
                    << ", level:" << node.getLevel() 
                    << ", |children|:" << node.getNumChildren() 
                    << ", |cnps|:" << node.getNumCNPs()
                    << ", Area:" << node.getArea()  
                    << ", numDescendentes:" << node.getNumDescendants()
                    << " ---\n";
        
        // Imprime o cabeçalho de colunas
        *streamSaida << std::setw(setw) << " "; // espaço para a primeira coluna (índice da linha)
        for (int col = 0; col < numCols; col++) {
            *streamSaida << std::setw(setw) << col;
        }
        *streamSaida << "\n";                 
        // Impressão bidimensional
        std::vector<bool> imageReps(n, false);
        imageReps[node.getRepNode()] = true;
        std::vector<bool> imageCNPs(n, false);
        for (int p : tree->getCNPsById(node)) {
            imageCNPs[p] = true;
        }
        std::vector<bool> imageCC(n, false);
        for (int p : tree->getPixelsOfCCById(node)) {
            imageCC[p] = true;
        }

        for (int row = 0; row < numRows; ++row) {
            *streamSaida << std::setw(setw) << row; // índice da linha
            for (int col = 0; col < numCols; ++col) {
                int index = ImageUtils::to1D(row, col, numCols);
                if (imageReps[index]) {
                    *streamSaida << std::setw(setw) << "  🅡";
                }
                else if (imageCNPs[index]) {
                    *streamSaida << std::setw(setw) << "  🅲";
                }
                else if (imageCC[index]){
                    *streamSaida << std::setw(setw) << "  🅇";
                }
                else{
                    *streamSaida << std::setw(setw) << "  ·";
                }
            }
            *streamSaida << "\n";
        }
        
    }
    if (streamSaida != &std::cout){
        dynamic_cast<std::ofstream*>(streamSaida)->close(); // std::cout não precisa ser fechado explicitamente
    }
}



/**
 * @brief Imprime os valores de uma imagem `uint8` com índices de linha/coluna.
 *
 * A saída pode ir para stdout ou para arquivo.
 *
 * @param imgPtr Ponteiro para a imagem (uint8).
 * @param setw Largura de coluna para formatação da tabela.
 * @param nomeArquivo Caminho de saída; vazio para imprimir em stdout.
 */
inline void printImage(ImageUInt8Ptr imgPtr, int setw=4, std::string nomeArquivo = "") {
    auto img = imgPtr->rawData();
    std::ostream* streamSaida;
    std::ofstream arquivoSaida;
    
    if (nomeArquivo.empty()) {
        streamSaida = &std::cout;
    } else {
        arquivoSaida.open(nomeArquivo);
        if (!arquivoSaida.is_open()) {
            std::cerr << "Erro ao abrir o arquivo para escrita." << std::endl;
            return;
        }
        streamSaida = &arquivoSaida;
    }
    *streamSaida << "Image size: " << imgPtr->getNumCols() << "x" << imgPtr->getNumRows() << "\n";
    // Imprime o cabeçalho de colunas
    *streamSaida << std::setw(setw) << " "; // espaço para a primeira coluna (índice da linha)
    for (int col = 0; col < imgPtr->getNumCols(); col++) {
        *streamSaida << std::setw(setw) << col;
    }
    *streamSaida << "\n";

    // Impressão bidimensional com índice de linha
    for (int row = 0; row < imgPtr->getNumRows(); row++) {
        *streamSaida << std::setw(setw) << row; // índice da linha
        for (int col = 0; col < imgPtr->getNumCols(); col++) {
            *streamSaida << std::setw(setw) << static_cast<int>(img[ImageUtils::to1D(row, col, imgPtr->getNumCols())]); // ou ImageUtils::to1D(row, col, numCols)
        }
        *streamSaida << "\n";
    }

    if (streamSaida != &std::cout) {
        dynamic_cast<std::ofstream*>(streamSaida)->close();
    }
}


/**
 * @brief Executa testes básicos de sanidade em uma ComponentTree.
 *
 * Valida existência da raiz, contagem de nós, relações pai/filho e o mapeamento
 * de cada pixel para seu Small Component compatível com os níveis da imagem.
 * Alguns testes adicionais encontram-se comentados (TODOs).
 *
 * @tparam CNPsType Pixels ou FlatZones da árvore.
 * @tparam ImageType Tipo de pixel da imagem de entrada (ex.: uint8_t).
 * @param tree Ponteiro para a árvore de componentes sob teste.
 * @param treeType Rótulo amigável (ex.: "max-tree" ou "min-tree").
 * @param imgPtr Imagem base usada na construção/validação da árvore.
 */
 inline  void testComponentTree(MorphologicalTreePtr tree, const std::string& treeType, ImageUInt8Ptr imgPtr) {
    std::cout << "🔍 Testando " << treeType << "..." << std::endl;

    //int numRows = imgPtr->getNumRows();
    //int numCols = imgPtr->getNumCols();
    
    if (!tree) {
        std::cerr << "❌ Erro: " << treeType << " não foi criada corretamente!" << std::endl;
        return;
    }

    if(MorphologicalTree::validateStructure(tree)){
        std::cout << "✅ Estrutura da árvore está consistente no arena" << std::endl;
    }else{
        std::cerr << "❌ Erro: Estrutura da árvore está inconsistente no arena" << std::endl;
    }

    //Teste: Verifica se o Iterator getPixelsOfCC está correto
    int area = tree->getRoot().getArea();
    int count_area = 0;
    for(int p : tree->getPixelsOfCCById(tree->getRoot())){
        count_area++;
        (void)p;
    }
    if (area == count_area) {
        std::cout << "✅ Iterator getPixelsOfCC da " << treeType << " está correto." << std::endl;
    }else{
        std::cerr << "❌ Erro: Iterator getPixelsOfCC da " << treeType << ". Valor de count_area:" << count_area << ", area:"<< area<< std::endl;
    }
        
    
    //Teste: Verifica se o Iterator getCNPs está correto
    int num_cnps = tree->getRoot().getNumCNPs();
    int count_cnps= 0;
    for(int p : tree->getRoot().getCNPs()){
        if(p >= 0)
            count_cnps++;
    }
    if (num_cnps == count_cnps) {
        std::cout << "✅ Iterator getCNPs da" << treeType << " está correto." << std::endl;
    }else{
        std::cerr << "❌ Erro: Iterator getCNPs da" << treeType << ". Valor de count_cnps:" << count_cnps << std::endl;
        return;
    } 


    // Teste: Raiz da árvore não nula
    NodeMT root = tree->getRoot();
    if (!root) {
        std::cerr << "❌ Erro: Raiz da " << treeType << " é nula!" << std::endl;
        return;
    }
    std::cout << "✅ Raiz da " << treeType << " existe." << std::endl;
    
    // Teste: Número de nós na árvore
    int numNodes = tree->getNumNodes();
    if (numNodes <= 0) {
        std::cerr << "❌ Erro: " << treeType << " tem número inválido de nós!" << std::endl;
        return;
    }
    std::cout << "✅ " << treeType << " tem " << numNodes << " nós." << std::endl;

    // Teste: Verificando se todos os nós possuem um pai correto (exceto a raiz)
    bool allParentsCorrect = true;
    for (NodeMT node : tree->getRoot().getIteratorBreadthFirstTraversal()) {
        if (node != root && !node.getParent()) {
            std::cerr << "❌ Erro: Nó (id:"<< node <<") sem pai encontrado na " << treeType << "!" << std::endl;
            allParentsCorrect = false;
        }
    }
    if (allParentsCorrect) {
        std::cout << "✅ Todos os nós possuem pais corretos na " << treeType << "." << std::endl;
    }

    // Teste: Verificando se cada nó tem filhos corretos
    bool allChildrenCorrect = true;
    for (NodeMT node : tree->getRoot().getIteratorBreadthFirstTraversal()) {
        for (NodeMT child : node.getChildren()) {
            if (child.getParent() != node) {
                std::cerr << "❌ Erro: Nó com filho sem referência ao pai na " << treeType << "!" << std::endl;
                allChildrenCorrect = false;
            }
        }
    }
    if (allChildrenCorrect) {
        std::cout << "✅ Todos os nós possuem filhos corretamente associados na " << treeType << "." << std::endl;
    }

    
    // Teste: Verificando se os pixels estão corretamente armazenados nos nós
    bool allPixelsCorrect = true;
    for (NodeMT node : tree->getRoot().getIteratorBreadthFirstTraversal()) {
        int p = node.getRepNode();
        if (tree->getSC(p) != node) {
            std::cerr << "❌ Erro: Pixel " << p << " não está corretamente associado ao nó (id:"<< node <<") na " << treeType << "!" << std::endl;
            allPixelsCorrect = false;
        }
    
    }
    if (allPixelsCorrect) {
        std::cout << "✅ Todos os pixels estão corretamente mapeados para os nós na " << treeType << "." << std::endl;
    }


    // Teste: Verificando se todos os pixels da imagem estão mapeados corretamente na ComponentTree
    bool allMappedCorrectly = true;
    for (int p: tree->getAllRepCNPs()) {
        NodeMT mappedNode = tree->getSC(p);
        int level = (*imgPtr)[p];
        if (!mappedNode) {
            std::cerr << "❌ Erro: Pixel " << p << " não foi mapeado para nenhum nó na " << treeType << "!" << std::endl;
            allMappedCorrectly = false;
        } else if (mappedNode.getLevel() != level) {
            std::cerr << "❌ Erro: Pixel " << p << " está associado a um nó (id:"<< mappedNode <<") de nível " << mappedNode.getLevel()
                      << " mas deveria estar em " << level << " na " << treeType << "!" << std::endl;
            allMappedCorrectly = false;
        }
    }
    if (allMappedCorrectly) {
        std::cout << "✅ Todos os pixels da imagem foram corretamente mapeados na " << treeType << "." << std::endl;
    }
    

}

/**
 * @brief Localiza um nó da árvore pelo índice (BFS).
 *
 * @tparam CNPsType Pixels ou FlatZones.
 * @param tree Ponteiro para a árvore.
 * @param index Índice do nó desejado.
 * @return O nó correspondente, ou nullptr se não encontrado.
 */
inline NodeMT getNodeByIndex(MorphologicalTreePtr tree, int index){
	for (NodeMT node : tree->getRoot().getIteratorBreadthFirstTraversal()) {
		if(node.getIndex() == index){
			return node;
		}
	}
	return NodeMT();
}

/**
 * @brief Retorna uma imagem de teste 12x12 (valores uint8) pequena.
 * @return Ponteiro para imagem 12x12.
 */
inline ImageUInt8Ptr getSmallImage(){
    auto img = new uint8_t[144]{
        2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0,
        2, 5, 5, 5, 3, 3, 3, 1, 1, 1, 3, 0,
        2, 5, 6, 5, 3, 3, 3, 1, 5, 1, 3, 0,
        2, 5, 6, 5, 3, 3, 3, 1, 6, 1, 3, 0,
        2, 5, 6, 5, 3, 3, 3, 1, 5, 1, 3, 0,
        2, 5, 5, 5, 3, 3, 3, 1, 1, 1, 3, 0,
        3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0,
        2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 0,
        6, 6, 6, 6, 6, 2, 4, 4, 4, 4, 4, 0,
        6, 4, 6, 4, 6, 2, 4, 7, 4, 7, 4, 0,
        6, 6, 6, 6, 6, 2, 4, 4, 4, 4, 4, 0,
        2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0
    };
    return ImageUInt8::fromRaw(img, 12, 12);
}

/**
 * @brief Retorna uma imagem de teste 12x12 variante (DGMM25).
 * @return Ponteiro para imagem 12x12.
 */
inline ImageUInt8Ptr getDgmm25Image(){
    auto img = new uint8_t[144]{
        2, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1,
        2, 5, 5, 5, 3, 3, 3, 2, 2, 2, 3, 1,
        2, 5, 6, 5, 3, 3, 3, 2, 5, 2, 3, 1,
        2, 5, 6, 5, 3, 3, 3, 2, 6, 2, 3, 1,
        2, 5, 6, 5, 3, 3, 3, 2, 5, 2, 3, 1,
        2, 5, 5, 5, 3, 3, 3, 2, 2, 2, 3, 1,
        3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 1,
        2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 1,
        6, 6, 6, 6, 6, 2, 4, 4, 4, 4, 4, 1,
        7, 3, 6, 4, 6, 2, 4, 8, 4, 8, 4, 1,
        7, 8, 6, 6, 6, 2, 4, 4, 4, 4, 4, 1,
        2, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1};
        return ImageUInt8::fromRaw(img, 12, 12);
}

/**
 * @brief Retorna uma imagem de teste maior (68x81) "Wonder".
 * @return Ponteiro para imagem 68x81.
 */
inline ImageUInt8Ptr getWonderImage(){
    auto img = new uint8_t[625]{
        203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,
        203,203,203, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78,203,203,203,203,203,203,203,203,
        203,203, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78,203,203,203,203,203,203,203,
        203,203, 78, 78,126,126,126,126,126,126,126, 78, 78, 78, 78, 78, 78, 78,203,203,203, 54, 54,203,203,
        203,203, 78, 78,126, 38, 38, 38,126,126,126, 78, 78, 78, 78, 78, 78, 78,203,203, 54, 54, 54, 54,203,
        203,203, 78, 78,126, 38, 38, 38,126,126,126, 78, 78, 78, 78, 78, 78, 78,203, 54, 54, 54, 80, 54,203,
        203,203, 78, 78,126, 38, 38, 38,126, 78, 78, 78, 78, 78, 78, 78, 78, 78,203, 54, 80, 54, 54, 54,203,
        203, 78, 78, 78,126, 38, 38,126,126, 78, 78, 78,203,203,203,203,203,203,203, 54, 54, 54, 54,203,203,
        203, 78, 78, 78,126,126,126,126, 78, 78,203,203,203,203,203,203,203,203, 54, 54, 54, 54, 54,203,203,
        203, 78, 78, 78,126,126,126, 78, 78, 78,203,203,253,253,253,203,203,203, 54, 80, 54, 54, 54,203,203,
        203, 78, 78, 78,126,126,126, 78, 78, 78,203,203,253,253,253,203,203, 54, 54, 54, 54, 54,203,203,203,
        203, 78, 78, 78,126,126,126, 78, 78, 78,203,203,253,253,253,203,203,203, 54, 54, 54,203,203,203,203,
        203, 78, 78, 78,126,126,126, 78, 78, 78, 78,203,203,203,203,203,203,203,203,203,203,203,203,203,203,
        203, 78, 78, 78,126,126,126, 78, 78, 78, 78,203,203,203,126,126,126,126,203,203,203,203,203,203,203,
        203, 78, 78, 78,126,126,126, 78, 78, 78, 78,203,203,126,126,126,126,126,126,126,126,126,203,203,203,
        203, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78,203,203,126,126,126,126,126,126, 72,126,126,203,203,203,
        203, 78, 78, 78, 78, 78, 78,161,161,161, 78, 78,203,126,126,126,126,126,126, 72, 72,126,126,126,203,
        203, 78, 78, 78, 78, 78,161,161,161,161,161, 78,203,126,126,126,126,126, 72, 72, 72,126,126,126,203,
        203, 78, 78, 78, 78, 78,161, 30, 30, 30,161, 78,203,203,126,126,126, 72, 72, 72, 72, 72,126,126,203,
        203, 78, 78, 78, 78, 78,161, 30, 90, 30,161, 78, 78,203,126,126, 72, 72, 72, 72, 72, 72, 72,126,203,
        203, 78, 78, 78, 78, 78,161, 30, 30, 30,161, 78, 78,203,203,126,126,126,126,126,126,126,126,126,203,
        203, 78, 78, 78, 78, 78,161,161,161,161,161, 78, 78, 78,203,203,126,126,126,126,126,126,126,203,203,
        203, 78, 78, 78, 78, 78,161,161,161,161,161, 78, 78, 78,203,203,203,203,126,126,126,126,203,203,203,
        203,203, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78,203,203,203,203,203,203,203,203,203,203,
        203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203};
    return ImageUInt8::fromRaw(img, 25, 25);
}



inline ImageUInt8Ptr getPassatImage(){
    auto img= new uint8_t[6164]{
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,5,5,5,5,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,7,7,7,7,7,7,7,7,7,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,5,5,5,5,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,7,7,7,7,7,7,7,7,7,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,5,5,5,5,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,7,7,7,7,7,7,7,7,7,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,5,5,5,5,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,7,7,7,7,7,7,7,7,7,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,5,5,5,5,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,7,7,7,7,7,7,7,7,7,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,5,5,5,5,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,7,7,7,7,7,7,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,6,5,5,5,5,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,7,7,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,6,5,5,5,5,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,3,
        6,6,6,6,6,6,6,1,1,1,1,1,1,6,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,5,5,5,5,5,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,3,
        6,6,6,6,6,6,6,1,1,1,1,1,1,1,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,5,5,5,5,5,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,3,
        6,6,6,6,6,6,6,1,1,1,1,1,1,1,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,5,5,5,5,5,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,3,
        6,6,6,6,6,6,6,1,1,1,1,1,1,1,6,6,6,6,6,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,1,1,1,1,1,6,6,6,6,6,6,6,6,6,6,5,5,5,5,5,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,3,
        6,6,6,6,6,6,6,1,1,1,1,1,1,1,6,6,6,6,6,1,1,1,1,1,1,0,0,0,0,0,0,0,0,1,1,1,1,1,1,5,5,5,5,6,1,1,1,1,1,5,5,5,5,5,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,3,
        6,6,6,6,6,6,6,1,1,1,1,1,1,1,6,6,6,6,6,1,1,1,1,1,1,0,0,0,0,0,0,0,0,1,1,1,1,1,1,5,5,5,5,5,1,1,1,1,1,5,5,5,5,5,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,3,
        6,6,6,6,6,6,6,1,4,4,4,4,4,6,6,6,6,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,5,5,5,5,5,1,1,1,1,1,5,5,5,5,5,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,3,
        6,6,6,6,6,6,6,4,4,4,4,4,4,6,6,6,6,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,5,5,5,5,5,1,1,1,1,1,5,5,5,5,5,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,3,
        6,6,6,6,6,6,6,4,4,4,4,4,4,6,6,6,6,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,5,5,5,5,5,5,1,1,1,5,5,5,5,5,5,5,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,3,3,3,3,3,3,3,3,3,
        6,6,6,6,6,6,6,4,4,4,4,4,4,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,1,1,1,1,1,1,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,5,5,5,3,3,3,3,3,3,
        6,6,6,6,6,6,6,4,4,4,4,4,4,4,0,0,0,0,0,0,0,0,0,6,6,6,6,6,6,0,0,0,0,1,1,1,1,1,1,5,5,5,5,5,5,5,5,5,5,5,5,5,5,1,1,1,1,1,1,1,1,2,2,2,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,5,5,5,5,5,3,3,3,3,
        6,6,6,6,6,6,6,4,4,4,4,4,4,4,0,0,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,0,0,1,1,1,1,1,1,5,5,5,5,5,5,5,5,5,5,5,5,5,1,1,1,1,1,1,1,1,1,3,3,3,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,5,5,5,5,5,5,5,3,3,
        6,6,6,6,6,6,6,4,4,4,4,4,1,0,0,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,0,0,0,5,5,5,5,5,5,5,5,5,1,1,1,1,1,1,1,1,1,1,1,3,3,3,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,1,0,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,0,0,0,0,0,0,5,5,5,5,5,1,1,1,1,1,1,1,1,1,1,1,3,3,3,3,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,1,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,0,0,0,0,0,0,0,5,5,5,5,1,1,1,1,1,1,1,1,1,1,3,3,3,3,3,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,1,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,0,0,0,0,0,0,0,5,5,5,5,1,1,1,1,1,1,1,1,1,1,3,3,3,3,3,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,1,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,0,0,0,0,0,0,0,5,5,5,5,1,1,1,1,1,1,1,5,5,5,5,3,3,3,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,1,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,0,0,0,0,0,2,2,2,2,2,2,1,1,1,1,5,5,5,5,5,5,5,3,3,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,0,0,0,2,2,2,2,2,2,2,2,2,1,1,5,5,5,5,5,5,5,5,3,3,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,2,2,2,2,2,2,2,2,2,2,2,2,2,2,5,5,5,5,5,5,8,8,8,8,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,0,0,0,0,6,6,6,6,6,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,5,5,5,5,8,8,8,8,8,8,8,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,6,6,6,6,6,6,6,6,6,0,0,0,0,0,0,0,6,6,6,6,6,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,5,8,8,8,8,8,8,8,8,8,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,6,6,6,6,6,6,6,0,0,0,0,0,0,0,0,0,6,6,6,6,6,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,8,8,8,8,8,8,8,8,8,8,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,6,6,6,6,6,6,6,0,0,0,0,0,0,0,0,6,6,6,6,6,6,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,8,8,8,8,8,8,8,8,8,8,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,1,1,1,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,6,6,6,6,6,6,0,0,0,0,0,0,0,6,6,6,6,6,6,6,6,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,8,8,8,8,8,8,8,8,8,8,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,6,6,6,6,6,6,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,8,8,8,8,8,8,8,8,8,8,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,5,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,6,6,6,6,6,6,6,6,0,0,0,6,6,6,6,6,6,6,6,6,6,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,8,8,8,8,8,8,8,8,8,8,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,5,5,5,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,8,8,8,8,8,8,8,8,8,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,5,
        6,6,6,6,6,6,6,1,1,1,1,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,3,8,8,8,8,8,8,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
        6,6,6,6,6,6,6,1,1,1,1,0,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,3,3,3,3,8,8,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,1,1,1,1,1,1,1,1,1,1,1,1,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,3,3,3,3,3,3,8,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,1,1,1,1,1,1,1,1,1,1,1,
        6,6,6,6,6,6,6,1,1,1,1,1,0,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,8,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,4,1,1,1,1,1,1,1,1,1,
        6,6,6,6,6,6,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,6,6,6,6,6,6,6,6,6,6,1,1,1,1,1,1,1,1,7,7,7,2,2,2,2,2,2,2,8,8,8,8,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,4,4,4,4,4,1,1,1,1,1,1,1,
        1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,6,0,0,0,0,1,1,1,1,1,1,1,7,7,7,7,7,3,3,3,3,8,8,8,8,8,8,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,4,4,4,4,4,4,4,1,1,1,1,1,1,
        1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,7,7,7,7,7,7,7,7,3,3,3,3,3,3,8,8,8,8,8,8,8,3,3,3,0,0,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,1,1,1,1,
        1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,7,7,7,7,7,7,7,3,3,3,3,3,3,3,8,8,8,8,8,8,8,8,8,8,0,0,0,0,0,0,0,0,4,4,4,4,4,4,4,4,4,4,1,1,1,1,4,4,4,4,4,4,4,4,1,1,
        1,1,1,1,1,1,1,1,1,1,1,1,1,1,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,7,7,7,7,7,7,7,3,3,3,3,3,3,3,8,8,8,8,8,8,8,8,8,8,8,0,0,0,0,0,4,4,4,4,4,4,4,4,4,4,1,1,1,1,1,1,1,1,4,4,4,4,4,4,4,4,
        1,1,1,1,1,1,1,1,1,1,1,1,1,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,7,7,7,7,6,6,6,3,3,3,3,3,3,3,6,8,8,8,8,8,8,8,8,8,8,8,0,0,0,4,4,4,4,4,4,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,4,4,4,4,4,4,
        1,1,1,1,1,1,1,1,1,1,1,1,1,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,0,7,7,7,7,7,6,6,6,6,3,3,3,3,3,3,3,6,8,8,8,8,8,8,8,8,2,2,2,2,2,2,4,4,4,4,4,4,4,4,1,1,1,1,1,1,1,1,1,1,1,1,4,4,4,4,4,4,4,
        1,1,1,1,1,1,6,6,6,1,3,3,3,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,7,7,7,7,6,6,6,6,6,6,3,3,3,3,3,6,6,6,6,8,8,8,8,8,8,2,2,2,2,2,2,4,4,4,4,4,4,4,1,1,1,1,1,1,1,1,1,1,1,1,1,1,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,6,6,3,3,3,3,4,4,4,3,3,3,3,3,3,7,7,0,0,0,0,0,0,0,0,0,7,7,7,7,6,6,6,6,6,6,6,6,3,3,6,6,6,6,6,6,0,8,8,8,8,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,6,6,6,6,6,6,1,1,1,1,1,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,6,6,3,3,3,4,4,4,4,4,3,3,3,3,3,7,7,7,7,0,0,0,0,0,0,0,7,7,7,7,6,6,6,6,6,6,6,6,6,6,6,6,6,6,0,0,0,0,0,8,8,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,6,6,6,6,6,6,1,1,1,1,1,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,6,3,3,3,3,4,4,4,4,4,4,3,3,3,3,7,7,7,7,0,0,0,0,0,0,0,7,7,7,7,6,6,6,6,6,6,6,6,6,6,6,6,6,0,0,0,0,0,0,0,0,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,6,6,6,6,6,6,1,1,1,1,1,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,6,3,3,3,3,4,4,4,4,4,4,3,3,3,3,7,7,7,7,0,0,0,0,0,0,0,7,7,7,7,3,3,3,3,3,6,6,6,6,6,6,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,6,6,6,6,6,6,1,1,1,1,1,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,6,3,3,3,3,4,4,4,4,4,4,3,3,3,7,7,7,7,7,0,0,0,0,0,0,0,7,7,7,3,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,6,6,6,6,6,6,1,1,1,1,1,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,6,3,3,3,3,4,4,4,4,4,4,3,3,3,7,7,7,7,7,0,0,0,0,0,0,0,2,2,2,3,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,6,6,6,6,6,6,1,1,1,1,4,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,6,3,3,3,3,4,4,4,4,4,4,3,3,3,7,7,7,7,7,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,5,5,5,5,5,5,1,1,1,1,4,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,6,3,3,3,3,4,4,4,4,4,4,3,3,3,7,7,7,7,7,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,5,5,5,5,5,5,1,1,1,1,4,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,6,3,3,3,3,4,4,4,4,4,4,3,3,3,7,7,7,7,7,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,5,5,5,5,5,5,1,1,1,1,4,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,6,3,3,3,3,4,4,4,4,4,4,3,3,3,7,7,7,7,7,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,5,5,5,5,5,5,1,1,1,1,4,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,6,3,3,3,3,4,4,4,4,4,4,3,3,3,7,7,7,7,7,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,5,5,5,5,5,5,5,1,1,1,1,4,4,4,4,4,4,4,
        6,6,6,6,6,6,6,6,3,3,3,3,3,4,4,4,4,4,4,3,3,3,7,7,7,7,7,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,5,5,5,5,5,5,5,1,1,1,1,4,4,4,4,4,4,4,
        1,1,1,1,1,1,1,1,3,3,3,3,3,3,4,4,4,4,3,3,3,3,7,7,7,7,7,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,5,5,5,5,5,5,1,1,1,1,4,4,4,4,4,4,4,
        1,1,1,1,1,1,1,1,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1,1,1,1,1,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,1,1,1,1,1,1,1,1,1,1,4,4,4,4,4,4,4,
        1,1,1,1,1,1,1,1,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1,1,1,1,1,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,4,1,1,1,1,1,1,1,1,1,1,1,1,1,1,4,4,4,4,4,4,4,
        1,1,1,1,1,1,1,1,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1,1,1,1,1,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,4,4,4,4,4,4,4,
        1,1,1,1,1,1,1,1,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1,1,1,1,1,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,4,4,4,4,4,4,4,
        1,1,1,1,1,1,1,1,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1,1,1,1,1,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,4,4,4,4,4,4,4,
        1,1,1,1,1,1,1,1,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1,1,1,1,1,0,0,0,0,0,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,4,4,4,4,4,4,4};
        return ImageUInt8::fromRaw(img, 67, 92);
}

inline ImageUInt8Ptr getCameramanImage(){
    auto img = new uint8_t[16384]{
        157,157,157,158,158,158,159,159,161,161,162,163,162,163,164,164,164,164,166,166,165,168,168,169,169,169,169,170,170,170,170,172,172,174,176,178,178,180,180,179,180,180,181,182,182,182,181,183,182,181,182,184,186,185,184,184,183,183,185,186,185,185,185,185,183,183,184,182,182,183,184,183,182,182,180,179,179,179,180,179,179,177,177,176,176,175,174,174,174,172,170,170,169,168,169,168,168,167,167,166,168,168,167,168,167,166,165,162,163,161,161,161,161,160,158,156,156,157,156,155,155,155,154,155,155,155,153,153,
        157,157,158,158,158,157,159,159,161,160,162,162,161,163,164,164,164,164,165,165,165,168,168,169,169,170,169,170,169,169,170,171,172,174,176,178,178,180,179,178,180,180,181,181,182,182,181,182,182,181,182,184,186,185,184,184,183,183,185,185,185,185,185,185,184,184,184,183,183,184,184,183,183,183,181,179,180,179,180,180,178,178,177,176,176,175,174,175,174,172,170,170,169,169,169,168,168,167,167,167,169,168,167,168,167,166,165,163,163,161,161,162,161,160,158,156,156,157,156,155,155,155,155,155,155,155,153,153,
        155,156,157,156,156,155,157,159,159,159,160,160,159,161,162,163,163,165,165,165,167,167,168,168,168,169,169,169,169,169,169,171,173,174,175,177,178,178,179,178,178,179,181,180,182,183,183,182,181,183,183,185,186,187,185,185,185,186,186,186,188,188,188,187,187,186,185,186,186,185,185,185,183,184,183,182,182,182,182,181,179,179,178,177,178,177,175,176,174,173,172,171,169,168,169,168,169,168,168,168,168,168,167,167,167,166,164,163,162,162,161,161,160,159,159,158,157,156,155,153,153,153,153,153,153,150,150,152,
        157,156,156,157,158,157,158,159,160,161,161,160,160,162,163,164,166,167,168,169,169,170,170,171,171,172,172,172,173,173,173,175,176,176,176,178,179,178,179,179,180,180,182,183,183,183,183,183,183,184,185,186,188,187,186,186,187,187,187,187,190,189,189,188,188,187,186,188,187,185,185,185,184,185,184,183,182,182,182,181,180,178,178,178,177,177,176,174,173,172,171,171,169,167,169,169,169,169,167,166,166,166,167,167,167,164,163,161,160,160,160,159,157,158,157,157,156,155,154,152,152,151,152,151,151,150,149,149,
        156,157,158,159,159,159,159,160,161,161,159,160,161,162,165,166,168,168,169,169,169,169,168,170,171,171,171,171,172,172,173,175,175,175,175,176,177,177,176,176,178,178,179,180,181,180,179,180,179,181,182,183,183,185,183,182,183,184,184,185,186,187,186,186,185,185,185,185,185,184,183,182,181,181,181,180,180,180,181,180,179,178,177,177,176,175,174,174,173,172,171,169,169,168,168,168,169,169,167,165,165,164,166,166,166,165,164,162,161,159,159,159,158,159,157,156,156,156,154,152,154,152,152,151,151,150,150,150,
        154,156,156,157,158,159,158,159,160,159,159,160,160,161,163,165,166,167,167,166,166,166,166,168,169,169,169,170,169,171,171,173,172,172,173,174,175,175,175,177,176,176,176,177,179,179,180,180,180,181,181,182,183,184,184,183,182,183,184,184,186,187,185,186,184,184,184,186,186,184,184,183,183,182,181,180,181,183,183,182,181,180,179,179,178,177,176,176,175,175,173,171,170,170,169,169,170,170,168,168,168,166,167,166,167,167,166,165,163,160,160,160,160,161,159,157,156,156,155,154,154,154,154,153,153,153,152,152,
        154,156,157,158,158,159,159,160,161,161,161,162,162,163,166,166,167,169,169,168,167,168,168,170,172,171,172,172,172,172,173,174,175,175,174,173,175,176,177,177,178,178,178,179,181,181,182,181,181,183,183,184,185,185,185,185,184,184,184,186,187,186,186,185,186,186,186,186,186,186,185,185,184,184,183,184,184,184,184,182,181,180,180,179,177,178,177,177,176,175,173,171,170,170,169,168,170,169,168,168,167,166,165,166,166,166,166,165,164,162,161,160,160,161,159,158,156,156,156,155,153,154,154,153,153,152,152,151,
        156,158,157,157,159,159,160,161,162,163,164,163,163,165,165,166,168,171,170,170,169,170,170,170,171,172,173,173,173,173,173,176,177,175,174,175,176,176,177,177,178,178,180,180,181,182,181,181,182,182,183,183,184,183,184,183,182,183,183,185,185,185,184,183,183,183,184,183,183,184,183,181,181,182,180,180,181,181,180,180,179,177,176,175,174,174,173,173,173,172,173,169,167,167,167,167,169,168,167,166,166,165,165,165,166,165,164,164,163,163,161,161,161,160,159,159,157,157,158,156,154,154,155,155,154,151,151,152,
        154,156,156,157,157,157,158,159,161,162,163,162,162,161,163,166,168,168,167,168,168,168,168,170,171,172,172,172,172,172,172,175,176,174,174,176,176,177,177,178,178,179,180,181,180,181,181,181,181,182,182,182,184,184,184,183,183,184,184,185,186,186,186,184,183,183,183,182,183,183,181,181,181,180,180,180,180,181,179,180,179,177,176,176,175,175,174,173,173,172,173,171,169,168,168,169,168,170,170,168,168,166,166,166,167,167,166,164,164,164,163,162,162,161,161,161,159,158,160,157,155,154,156,156,156,153,152,153,
        153,155,156,155,156,157,158,160,162,162,161,162,162,161,162,165,167,168,167,167,167,168,169,170,171,173,174,175,174,174,175,175,176,177,177,178,179,179,179,180,180,179,180,181,182,182,183,183,184,183,183,185,186,187,187,187,187,187,187,187,187,188,188,186,184,185,185,185,185,185,184,183,182,182,183,183,182,181,180,180,178,178,177,178,175,176,174,172,173,171,172,170,170,168,168,170,168,169,169,169,167,166,166,166,166,166,166,164,164,163,162,162,162,162,161,160,157,155,156,156,155,156,156,156,155,153,153,153,
        154,155,157,158,158,159,160,161,162,163,162,162,162,164,164,165,167,167,168,168,168,168,169,171,173,173,174,174,175,174,175,175,176,176,177,178,177,179,178,178,178,178,179,181,180,182,182,183,184,184,184,183,185,186,187,186,185,185,185,185,184,185,185,184,184,183,183,183,184,183,184,183,180,180,180,180,180,179,178,178,177,176,175,175,174,173,173,172,171,170,170,168,168,167,166,167,168,168,167,168,166,165,165,165,165,165,165,163,161,161,161,161,161,160,159,159,156,156,156,155,155,155,156,157,157,155,154,153,
        152,154,155,156,157,158,159,159,161,161,161,161,161,161,162,164,165,165,166,166,167,167,166,169,170,170,171,171,172,172,171,172,173,173,174,175,175,176,175,176,176,176,178,179,180,182,181,182,181,182,182,183,184,184,185,186,185,185,184,185,184,184,185,184,184,184,184,184,185,185,185,184,183,181,180,182,181,179,179,178,178,178,176,176,177,175,174,173,172,172,170,169,169,169,168,168,167,169,168,168,167,167,167,166,165,164,164,163,163,162,161,161,161,161,160,159,158,158,158,157,157,156,157,158,158,157,156,156,
        154,154,155,156,157,158,159,159,160,159,160,161,160,162,163,163,165,165,166,168,167,166,167,169,169,170,170,172,172,171,171,172,173,173,174,175,174,174,175,177,177,177,179,180,181,181,181,182,182,183,184,184,185,185,187,187,187,187,187,187,188,187,188,188,187,187,188,188,188,188,187,188,187,186,185,184,183,183,181,180,179,178,177,177,177,176,175,174,174,173,171,170,170,168,168,168,167,169,167,168,168,167,166,165,165,162,162,163,162,161,161,160,159,159,158,158,158,156,156,155,155,156,157,155,154,154,154,154,
        158,156,157,157,158,160,161,161,163,162,163,163,163,165,165,166,168,169,169,169,169,170,171,173,172,173,174,175,174,173,173,173,173,174,174,176,175,175,176,177,178,178,180,179,181,181,180,180,181,183,183,184,185,186,186,187,187,187,187,188,189,189,189,188,186,186,187,187,188,187,188,187,187,186,185,183,184,183,182,180,179,178,177,177,176,176,174,173,172,172,170,169,169,168,168,167,166,168,168,168,166,166,165,165,165,162,162,162,160,159,160,160,159,159,157,157,156,155,155,154,156,155,155,154,152,152,153,153,
        158,158,157,158,159,161,161,163,163,163,164,164,163,164,164,165,166,167,169,169,169,171,171,171,171,172,174,174,172,173,173,173,172,173,174,174,174,175,175,177,177,177,178,179,179,179,178,179,180,181,181,183,183,183,184,185,185,186,185,186,187,188,187,186,184,183,185,186,186,186,186,186,186,185,184,184,183,182,182,180,179,178,178,178,178,177,177,176,176,175,173,171,171,171,170,168,169,169,170,169,168,167,167,166,165,164,163,163,161,160,161,161,160,160,160,158,157,156,156,156,156,155,154,155,154,154,153,153,
        155,156,157,158,159,160,161,162,162,162,164,164,164,164,164,165,168,168,168,170,170,171,172,172,172,172,174,174,174,174,173,173,174,176,177,176,176,176,176,176,178,178,179,179,180,180,179,179,180,181,183,183,183,184,185,186,186,187,188,187,187,187,186,185,183,183,184,185,184,186,186,187,186,185,184,184,183,182,181,182,181,180,179,179,178,177,178,178,177,176,175,173,172,171,171,170,170,170,170,169,168,168,167,166,166,165,164,162,161,161,159,158,159,159,160,158,158,155,155,154,154,153,153,153,154,152,152,151,
        156,157,157,157,160,161,163,162,162,163,164,166,167,168,168,168,170,171,169,172,173,174,174,173,174,175,175,177,177,177,176,176,176,177,179,179,178,178,177,177,180,181,181,181,180,181,182,182,183,184,185,184,185,186,187,188,188,188,188,187,186,186,185,184,184,182,183,181,181,183,183,184,184,183,182,182,182,182,181,181,180,179,178,177,176,176,176,177,175,174,173,172,172,172,171,171,170,168,168,167,167,167,165,165,164,163,164,161,161,159,159,157,157,158,158,157,156,154,153,152,152,153,153,153,153,151,151,151,
        157,157,158,158,160,162,162,162,163,163,164,165,166,168,168,168,169,170,170,172,172,173,174,175,175,176,176,176,175,176,175,176,177,178,178,178,177,178,177,179,180,180,179,179,180,180,181,182,183,184,186,187,187,189,191,191,191,191,189,188,187,185,185,184,183,182,181,180,180,181,182,182,182,180,181,180,180,180,180,179,179,177,175,176,176,176,175,174,173,172,172,173,173,173,171,170,170,170,168,168,168,167,166,165,165,164,164,162,161,161,161,160,160,159,158,158,157,155,154,153,153,154,155,155,155,153,152,152,
        155,155,156,158,159,160,159,161,163,163,162,163,164,166,166,166,168,170,170,171,173,172,173,174,174,174,175,176,175,175,176,176,178,178,178,176,177,178,178,179,180,180,180,182,182,183,183,183,183,184,187,190,180,160,128,103,108,132,154,169,182,190,190,189,186,184,183,182,182,183,184,183,182,182,182,180,180,180,180,179,179,178,177,177,176,176,175,174,173,172,172,173,172,172,170,169,168,170,169,169,168,167,167,166,166,166,164,163,162,161,161,160,160,160,159,159,158,156,154,154,154,154,154,153,154,153,152,151,
        157,156,157,158,159,160,161,161,163,163,163,164,166,168,168,169,169,171,171,172,173,173,174,175,176,176,176,177,178,178,178,178,180,180,180,181,180,180,180,181,182,182,182,183,183,184,185,185,186,187,190,148,86,45,22,11,13,20,28,39,68,130,184,190,188,186,186,184,184,184,183,182,182,181,181,180,180,180,180,179,179,177,176,174,174,175,174,172,173,173,171,171,170,170,169,167,168,169,168,167,166,166,166,166,164,164,163,162,161,160,158,158,157,156,155,155,155,154,154,152,152,152,152,151,152,151,150,149,
        157,159,159,160,162,163,162,163,163,164,164,164,168,170,170,170,170,171,171,172,173,175,175,175,176,176,177,177,177,177,176,177,179,179,180,180,180,179,180,182,181,181,181,182,183,182,183,184,187,181,124,39,17,16,13,11,11,11,11,11,11,25,83,154,181,181,183,184,182,182,182,182,181,181,179,179,180,180,179,179,178,176,174,174,173,174,173,173,173,172,171,170,170,169,168,168,170,169,168,167,166,166,166,165,163,162,163,161,160,160,158,158,157,156,157,156,155,153,153,153,152,152,152,152,152,151,150,150,
        157,158,158,159,161,162,161,162,162,162,164,165,166,168,168,167,169,170,170,171,172,173,174,174,175,174,176,176,175,175,176,177,177,177,178,178,178,177,179,180,178,180,180,182,181,182,181,183,186,176,77,15,14,13,11,11,12,12,12,11,12,11,13,55,134,143,159,185,184,183,184,184,183,182,181,181,180,181,180,179,178,178,176,175,176,176,175,174,174,174,172,171,171,171,170,171,170,169,169,169,169,167,167,166,165,164,164,163,161,160,160,159,158,158,158,157,156,154,152,153,152,152,152,150,151,151,151,151,
        158,160,159,160,162,162,162,163,164,164,167,167,167,168,170,170,171,170,172,172,173,173,174,173,176,176,177,177,178,177,177,177,178,180,179,180,179,179,178,179,181,181,182,183,183,183,183,185,189,171,61,12,10,9,9,10,13,12,12,12,13,12,11,12,47,80,101,173,185,184,186,185,184,183,182,181,181,182,181,180,180,178,177,177,177,177,176,175,174,174,172,171,172,171,172,171,171,170,169,167,167,166,165,165,164,163,163,162,161,160,159,158,157,157,158,156,153,152,151,151,151,151,151,150,150,150,149,150,
        160,161,161,164,164,165,164,164,165,167,168,169,169,170,171,172,171,172,174,174,174,174,174,174,176,176,177,178,178,179,178,178,179,180,180,181,181,180,178,179,179,180,183,182,182,184,183,185,186,106,18,10,10,9,9,10,13,13,12,13,13,13,13,11,11,17,42,133,183,184,183,182,182,181,181,180,180,179,179,179,178,177,177,176,176,176,175,175,174,174,172,170,170,170,169,170,170,170,169,167,166,166,164,164,165,164,163,162,161,159,158,158,158,157,157,155,154,153,151,151,151,151,152,151,152,151,148,150,
        160,160,161,162,162,164,164,166,166,166,166,166,167,168,169,168,169,171,172,172,173,174,174,175,175,176,177,178,178,178,177,176,177,178,180,181,180,179,178,179,179,178,180,181,181,181,182,184,174,55,11,10,9,9,9,10,11,11,12,11,11,11,10,9,10,10,18,79,171,183,183,183,181,180,181,180,179,179,178,178,178,177,177,177,177,176,175,175,175,174,173,171,171,171,170,171,170,171,170,169,168,168,167,166,165,164,163,162,161,160,159,158,157,157,156,155,153,152,151,152,151,151,151,151,152,151,150,150,
        160,161,161,161,162,164,165,166,165,167,166,166,167,169,169,169,171,171,173,174,174,174,176,176,176,176,177,179,180,179,179,179,180,181,182,183,182,182,181,180,181,181,182,183,183,183,184,186,156,31,9,9,9,9,9,9,11,11,12,11,9,9,9,9,10,10,11,38,155,184,184,184,182,180,181,181,180,179,179,179,178,178,177,177,175,175,175,175,174,173,173,172,171,172,171,171,172,170,169,168,168,167,167,166,164,163,163,162,161,160,159,158,157,155,155,154,152,151,151,151,149,149,149,149,148,150,149,147,
        162,163,162,163,165,166,167,168,168,169,169,169,169,170,172,172,173,173,174,175,175,174,176,174,176,176,178,178,178,180,183,180,180,181,182,183,183,181,181,181,181,182,182,184,185,185,186,189,140,23,10,10,9,9,9,9,10,11,11,10,9,10,12,17,11,9,10,56,152,173,179,182,181,180,179,178,178,177,177,176,176,176,175,174,173,173,174,173,172,171,171,171,170,170,169,168,167,166,166,166,165,166,165,164,162,162,163,162,160,159,159,158,157,155,155,155,152,150,151,150,149,150,150,150,148,148,147,147,
        162,161,162,164,165,167,167,168,169,168,169,169,169,169,171,172,173,173,174,174,173,173,174,174,177,176,175,178,178,178,180,179,180,179,180,180,181,180,179,178,179,180,182,182,183,184,184,176,87,16,10,10,9,10,18,17,10,10,10,10,15,35,61,88,32,16,17,59,98,101,144,165,179,181,180,179,179,177,177,178,177,176,175,174,174,174,174,174,172,171,170,169,169,169,169,169,168,167,166,166,166,166,166,164,163,162,163,162,160,161,159,158,157,156,157,156,154,153,153,152,151,151,151,151,150,150,148,148,
        161,161,162,162,164,167,168,168,168,169,168,168,169,170,170,171,172,173,174,173,173,175,174,175,177,177,177,179,179,180,180,180,181,181,182,181,182,181,181,181,181,180,183,184,184,184,187,138,25,9,9,9,9,34,69,41,13,10,10,31,89,143,161,157,94,53,26,31,43,52,67,99,151,181,182,181,181,180,179,179,177,177,176,175,175,175,174,174,173,172,171,170,170,169,168,168,168,169,167,166,166,165,165,163,162,161,162,161,160,159,158,157,156,156,156,155,154,153,151,150,150,150,150,151,150,149,148,147,
        160,162,163,163,164,166,168,168,172,179,171,170,170,172,172,172,173,174,175,175,176,177,176,177,176,177,178,179,180,181,181,180,181,182,182,182,183,183,182,181,182,181,183,184,185,186,189,128,22,10,9,10,13,88,127,57,31,12,28,122,170,155,122,91,106,66,56,67,72,74,75,82,113,169,177,176,178,180,178,177,177,177,177,176,174,174,173,173,173,172,170,169,168,167,167,168,168,167,166,165,165,164,163,162,161,160,159,160,159,156,156,154,154,154,154,153,152,151,149,149,148,149,149,149,148,147,147,146,
        161,162,162,162,164,166,168,168,169,172,171,170,169,170,172,172,173,173,173,174,175,175,175,175,175,177,178,178,178,179,179,179,179,180,181,181,182,181,180,180,181,181,182,182,184,186,183,86,15,10,9,10,14,89,117,110,112,25,59,167,169,117,57,61,101,86,76,95,82,66,65,65,69,82,113,87,103,172,176,176,176,176,176,175,175,175,174,173,172,172,169,169,170,169,168,168,168,168,166,165,165,164,164,163,161,162,160,160,159,157,157,156,155,154,154,153,152,151,151,149,149,149,150,149,148,147,147,146,
        160,161,160,160,162,165,166,168,168,168,167,169,168,169,170,171,171,170,172,173,174,173,174,174,175,177,178,178,178,178,179,179,179,180,182,183,184,183,183,182,183,183,184,185,187,188,153,40,13,10,9,10,15,71,118,105,112,34,77,164,184,170,129,171,180,101,79,82,75,89,110,112,104,124,127,73,71,147,166,173,164,176,177,175,175,174,174,174,174,173,172,172,171,170,170,170,169,169,168,166,165,165,165,163,163,162,161,161,159,157,157,156,155,154,153,152,151,150,150,149,148,147,148,148,148,145,146,145,
        160,161,162,162,163,164,165,167,168,169,170,169,168,169,170,172,172,171,173,174,174,175,175,176,177,177,179,179,178,179,180,181,182,185,183,176,171,165,165,167,169,165,165,161,147,122,52,12,12,11,11,15,19,39,126,128,112,97,125,158,172,175,174,185,200,158,59,56,79,138,171,181,173,189,119,57,114,165,191,199,188,177,175,175,174,173,172,173,172,172,172,170,169,168,168,168,167,167,167,164,164,163,163,162,162,162,160,158,157,156,157,155,153,152,151,151,149,148,147,147,147,146,146,147,148,145,145,145,
        163,163,164,165,164,164,165,167,169,169,169,167,167,169,171,172,171,173,174,174,174,175,175,176,177,176,177,177,177,178,179,181,184,164,99,65,53,43,42,46,49,42,40,38,31,23,14,10,13,20,37,52,34,24,49,88,111,138,150,159,169,174,159,147,170,160,56,51,67,118,87,142,103,122,83,46,117,104,116,126,122,168,175,174,173,173,173,172,172,172,171,169,169,169,169,169,168,167,165,164,165,164,164,163,163,162,160,159,158,157,158,156,155,154,153,152,150,150,149,150,149,149,149,149,148,147,146,147,
        162,163,163,163,163,163,164,166,167,167,166,166,167,168,170,171,171,172,172,173,173,173,174,175,176,176,176,177,177,179,180,182,170,74,18,14,17,19,19,19,18,18,18,17,16,15,13,9,16,62,66,80,73,34,31,45,88,123,142,155,170,158,143,120,88,77,47,56,82,80,67,154,75,79,103,71,122,83,79,107,144,173,174,174,173,172,173,172,172,172,171,170,170,169,169,171,172,168,166,165,164,163,163,163,162,162,161,160,158,156,157,157,155,155,153,152,152,151,149,149,148,148,148,147,147,147,147,146,
        162,162,163,164,165,165,165,166,168,168,168,168,168,169,171,172,172,173,174,174,174,175,175,176,177,177,178,180,179,180,182,182,112,22,14,14,15,18,18,18,17,17,16,16,15,14,13,9,14,101,114,94,114,80,52,25,47,86,117,134,148,126,110,122,140,77,50,88,89,82,56,110,54,67,118,111,138,124,102,147,175,174,167,174,173,172,172,171,171,170,170,169,168,169,169,168,168,167,165,163,162,163,162,162,161,160,160,160,158,156,156,156,154,155,155,153,152,151,150,149,147,147,147,146,147,146,146,145,
        164,164,165,165,166,166,167,168,168,169,170,170,168,170,171,172,173,173,174,174,175,175,175,176,177,178,178,179,180,182,185,149,44,16,14,13,15,17,18,18,17,16,16,16,15,15,13,9,11,93,177,123,113,92,55,21,19,41,80,105,112,111,94,111,160,83,51,58,47,41,39,55,44,62,127,100,139,125,121,168,171,161,118,102,145,173,171,171,172,170,170,169,169,169,169,167,167,167,165,163,162,163,163,163,161,161,161,160,158,157,157,156,155,155,154,154,153,151,150,150,147,147,147,147,147,147,146,145,
        164,165,165,167,167,167,169,170,170,171,170,171,169,170,173,173,174,174,175,175,175,175,176,178,178,179,179,181,183,185,159,65,18,16,16,14,15,16,17,17,16,16,17,16,16,16,16,10,9,64,192,170,120,91,60,24,16,17,40,74,103,116,102,158,169,84,51,39,40,40,42,43,56,95,130,100,158,166,189,202,198,189,103,62,133,174,173,172,173,171,171,170,170,170,169,168,168,167,166,165,164,165,165,164,163,162,162,161,159,157,157,156,155,155,154,154,152,151,150,150,148,148,147,147,146,147,147,146,
        163,165,166,165,166,167,168,169,170,172,171,171,170,171,173,173,174,174,175,176,176,177,178,179,180,180,181,182,184,157,65,19,16,16,16,14,14,15,15,16,15,16,16,15,15,15,16,9,8,28,139,156,159,133,96,39,17,14,15,34,82,112,145,182,170,87,50,38,39,40,54,45,57,72,80,71,124,96,80,106,94,87,42,21,101,173,173,172,172,171,171,169,169,168,167,167,166,166,165,164,164,165,165,164,163,162,160,160,159,158,156,155,154,155,154,152,151,151,150,149,148,147,146,146,146,146,145,145,
        163,165,165,166,168,168,168,168,171,172,171,170,170,172,172,172,174,174,175,175,175,175,178,180,180,180,183,184,153,63,18,15,16,16,15,14,14,14,15,15,14,15,16,15,16,15,11,9,8,13,79,122,150,131,113,53,14,11,11,12,47,106,162,180,168,83,52,42,40,37,50,69,86,74,49,49,105,106,115,119,115,110,100,96,136,172,171,169,169,169,170,168,168,167,167,167,167,167,166,164,164,164,164,163,163,162,161,159,158,158,157,156,155,154,154,153,151,151,151,150,149,147,147,146,146,146,146,145,
        162,164,165,166,168,168,167,169,169,169,168,168,169,170,170,170,172,173,173,173,174,174,177,179,180,181,178,134,53,18,15,15,16,16,15,14,13,13,14,14,13,14,15,15,16,15,12,11,9,13,63,134,116,83,76,38,10,8,8,9,52,167,181,180,169,76,56,41,40,51,63,85,111,74,44,48,66,103,176,174,172,172,171,171,170,171,170,170,170,169,170,168,168,168,167,167,167,168,166,165,165,164,164,164,163,164,162,160,159,158,157,157,156,156,155,153,151,151,152,150,150,149,148,148,148,147,146,146,
        163,164,164,165,167,167,167,168,167,169,169,168,168,168,169,171,171,173,174,175,175,176,177,180,183,169,99,36,17,16,15,15,16,16,16,15,14,13,12,13,13,13,14,15,15,15,15,15,12,18,29,102,90,58,55,21,9,8,8,9,29,153,182,179,179,99,59,59,54,65,64,62,70,60,53,55,46,85,168,173,172,171,170,169,169,170,170,169,169,169,168,167,167,166,166,166,166,167,165,165,165,163,162,162,162,162,160,159,158,156,155,155,154,154,153,152,150,150,149,149,147,148,148,147,147,146,146,145,
        163,164,165,164,165,168,168,167,169,169,169,168,169,170,170,172,173,174,175,176,175,177,179,183,163,81,23,14,13,13,13,15,14,13,14,13,13,13,12,13,13,13,15,15,15,19,17,15,12,17,21,96,91,28,17,14,11,9,9,9,16,102,178,178,178,150,62,41,40,50,41,39,53,62,32,24,22,38,122,170,170,168,167,167,167,167,167,166,166,167,166,166,166,165,164,164,164,165,165,164,164,162,161,161,161,162,160,157,156,155,154,155,153,152,152,151,151,150,149,149,147,148,148,146,146,145,144,143,
        161,164,165,165,165,166,166,166,168,169,168,168,168,167,169,169,171,172,174,175,176,177,179,149,65,20,16,14,12,11,11,14,14,12,11,12,12,12,12,13,13,13,14,15,17,36,26,15,10,11,17,76,68,17,17,18,18,12,9,11,19,46,138,154,183,174,141,92,91,91,98,120,133,106,29,20,14,16,98,170,169,169,168,168,167,168,167,167,167,168,167,167,166,166,166,167,166,166,166,165,165,163,163,165,163,163,162,161,158,158,156,156,155,153,154,152,152,151,151,150,150,149,148,148,147,146,146,145,
        162,163,163,163,163,163,164,165,166,166,166,166,166,167,168,169,170,171,173,174,176,177,139,52,17,15,15,16,14,12,11,11,14,14,11,12,12,12,12,12,12,12,13,14,14,16,19,15,9,9,13,77,81,20,18,18,20,17,27,75,102,110,133,112,181,135,132,113,120,126,125,141,157,139,56,37,34,52,145,170,170,169,168,167,168,168,168,167,167,168,168,167,166,165,166,167,167,166,165,165,165,164,164,165,163,162,161,160,158,157,156,156,154,154,154,151,151,150,151,151,149,148,148,148,147,146,145,145,
        163,163,163,163,164,164,165,165,165,166,167,166,165,168,168,170,171,172,173,176,178,135,46,16,16,16,15,15,16,15,14,12,13,15,12,11,12,12,12,12,12,12,13,14,14,12,13,15,10,9,14,69,43,21,21,19,16,35,101,84,48,39,72,103,126,60,59,82,189,224,224,222,216,166,43,25,25,68,161,168,167,167,167,166,166,166,166,167,166,166,167,166,166,165,165,165,165,165,165,164,163,163,163,162,162,161,161,159,158,156,156,156,154,153,153,151,150,149,148,149,149,148,147,146,145,145,145,144,
        162,162,163,163,163,163,164,164,164,166,166,165,165,167,167,168,171,173,174,176,138,47,16,15,16,16,16,16,16,17,17,15,14,15,13,12,12,12,13,13,12,12,13,14,14,13,13,15,11,9,13,28,23,21,16,14,34,98,62,19,13,16,60,130,126,113,110,116,217,223,236,239,199,172,65,53,49,100,169,169,167,166,166,166,166,165,166,167,167,167,167,165,165,164,164,164,165,165,165,164,164,163,162,162,163,162,162,161,159,157,156,157,155,153,153,152,150,150,149,149,150,148,146,146,146,146,145,145,
        159,160,161,160,161,161,163,164,163,163,164,164,164,166,167,169,171,173,176,136,48,17,16,16,17,18,18,18,17,17,16,16,15,15,14,11,12,13,14,13,13,13,14,15,15,15,16,15,12,10,12,18,17,13,11,30,98,65,18,15,14,10,24,139,175,177,173,207,234,213,237,238,146,138,40,27,34,114,172,169,168,167,167,167,167,168,168,168,169,167,167,166,166,165,165,165,166,166,166,165,163,164,164,162,162,163,161,160,159,157,157,156,155,153,153,152,151,150,150,149,150,148,147,147,146,146,145,145,
        159,160,161,162,163,163,163,164,164,164,164,165,164,166,168,169,170,174,141,49,16,15,16,17,17,17,17,16,16,16,14,13,13,13,12,10,11,12,12,12,12,13,14,14,13,11,11,11,11,11,13,12,11,12,29,99,73,16,12,14,13,10,13,100,179,170,188,217,228,229,237,196,129,110,16,10,27,126,170,168,168,166,165,166,166,166,167,166,166,166,166,165,164,163,164,164,164,165,165,163,163,163,164,162,162,161,160,158,158,156,156,155,154,153,154,152,150,148,148,149,149,147,146,145,145,144,143,143,
        159,159,161,162,162,162,163,163,164,165,165,165,165,165,166,168,172,152,58,17,16,16,14,13,12,12,12,13,13,14,13,12,12,13,12,11,12,12,12,12,11,12,13,14,13,11,10,10,10,12,11,10,12,26,96,80,19,12,11,12,14,10,9,52,169,180,179,182,187,224,225,159,138,46,12,19,84,163,170,168,166,164,164,164,165,164,165,165,165,165,165,164,163,162,163,163,162,164,164,162,162,163,163,161,161,160,160,158,158,157,156,156,155,154,154,151,150,148,148,149,148,147,146,145,144,143,143,143,
        159,160,160,162,162,164,163,163,163,164,164,164,164,166,166,169,166,84,19,15,15,13,11,12,13,12,12,13,12,11,12,13,13,14,12,12,13,13,12,13,12,13,13,13,14,13,12,11,12,11,11,11,20,80,78,22,15,14,11,10,11,10,9,24,125,154,132,118,135,211,214,147,52,22,19,62,155,170,170,168,167,164,164,164,164,164,164,164,164,165,165,163,163,162,163,162,161,163,163,163,163,162,161,160,159,160,160,159,158,157,156,156,155,154,153,152,151,149,149,148,149,147,146,145,144,143,143,143,
        159,161,161,161,163,164,165,165,166,166,166,165,165,166,168,170,130,30,15,14,13,11,10,11,12,14,15,14,12,10,9,10,11,11,12,13,13,12,12,12,12,13,13,12,12,12,12,12,12,11,11,19,77,73,21,15,14,13,12,10,9,9,9,11,33,39,51,59,46,71,80,36,19,18,37,134,171,169,167,167,167,165,164,163,164,163,163,163,162,163,163,162,161,162,161,160,159,161,160,161,161,160,159,159,158,158,158,158,156,155,155,155,153,154,153,153,151,150,149,148,146,146,146,145,144,143,142,141,
        160,159,160,161,161,163,164,164,164,165,165,164,163,165,168,167,88,14,12,13,15,13,14,15,15,15,15,13,13,12,11,9,9,10,12,12,12,12,11,12,12,13,12,12,11,11,12,13,12,11,18,80,85,22,15,14,13,13,12,10,9,9,9,9,21,13,24,33,11,11,11,12,15,13,37,148,172,169,167,166,164,164,164,163,164,163,163,163,163,163,163,162,161,162,161,161,161,161,158,158,161,160,159,159,159,160,159,158,157,156,156,156,155,154,153,153,150,150,149,148,145,147,147,146,145,143,143,141,
        159,158,159,160,162,162,162,162,163,163,163,163,163,164,166,164,77,11,11,11,14,14,14,14,13,12,11,12,12,14,14,10,10,11,12,13,12,11,11,11,12,12,15,14,10,10,11,12,12,15,73,94,25,15,14,13,13,13,13,12,9,9,9,21,93,28,10,13,12,15,11,12,11,10,14,99,171,170,168,167,166,165,165,165,164,164,164,163,163,164,163,161,161,162,162,160,160,162,158,165,179,166,159,160,159,160,159,159,158,157,156,155,154,153,153,152,150,149,148,148,148,148,147,146,145,143,143,141,
        158,159,160,161,162,162,164,163,164,163,164,164,164,164,165,167,106,16,9,9,10,10,11,14,14,14,14,15,16,16,16,14,14,14,13,13,11,11,11,11,12,13,14,12,10,10,10,10,13,66,100,29,11,10,11,12,12,13,13,13,10,9,9,46,116,25,18,22,17,19,12,11,11,9,10,44,153,169,167,165,165,165,164,164,164,165,164,162,163,163,163,161,161,162,161,160,160,161,152,173,204,181,160,159,159,158,158,158,156,156,156,155,154,153,153,152,150,150,149,148,148,148,148,147,145,143,143,142,
        160,161,160,162,162,164,164,164,165,164,164,165,164,164,166,168,153,55,11,10,9,9,9,12,15,16,16,16,16,16,14,14,14,15,16,16,13,12,11,11,12,12,10,10,12,13,18,15,53,105,35,12,12,11,12,12,13,13,14,12,9,9,9,28,75,88,79,72,91,95,47,12,11,11,10,18,115,168,166,166,165,164,163,163,163,163,162,160,162,162,162,160,161,161,160,159,146,136,133,134,169,170,159,159,158,158,158,157,156,156,155,155,155,154,154,153,152,151,150,150,150,149,148,148,146,145,143,144,
        160,161,160,161,161,163,163,164,163,162,162,163,165,165,165,166,168,131,40,11,10,9,8,9,10,12,12,12,12,13,13,14,15,16,17,17,17,16,14,12,12,12,10,9,15,47,55,20,50,39,12,11,12,13,13,12,13,15,14,12,9,8,9,19,78,150,61,82,155,151,92,16,9,11,11,18,109,169,168,166,165,164,163,163,163,164,163,161,162,161,161,161,161,161,161,159,135,116,125,111,141,160,159,159,159,159,159,158,158,157,156,156,156,155,155,153,152,151,150,150,149,149,149,148,146,144,142,144,
        161,161,161,161,162,162,162,163,163,164,163,163,164,164,165,167,168,168,126,38,10,9,8,9,9,9,10,10,10,11,12,14,15,16,16,15,17,16,16,15,13,11,11,13,26,43,24,15,14,15,15,15,15,12,12,14,15,14,13,11,9,9,10,22,52,91,38,77,127,103,68,16,11,10,15,65,151,168,167,166,165,164,164,163,163,164,163,161,161,161,160,160,160,160,159,155,166,169,173,169,174,171,160,159,158,159,159,158,158,158,156,155,154,154,154,153,151,150,149,150,148,147,148,148,146,144,143,142,
        162,162,163,163,163,164,164,165,164,165,165,165,165,165,166,167,168,169,168,122,34,10,9,9,10,11,11,11,11,11,12,13,14,18,25,16,15,15,16,17,17,16,27,31,26,15,15,13,11,12,13,13,11,10,12,14,13,13,13,10,9,9,9,11,28,20,15,25,24,50,48,21,16,13,61,149,167,167,166,165,164,164,163,162,162,162,162,161,161,160,159,160,159,159,159,151,176,186,185,186,185,178,161,159,159,158,160,160,159,159,157,156,154,154,154,153,151,150,150,150,148,148,148,148,146,145,144,143,
        160,162,162,163,163,164,164,164,162,163,162,163,163,163,165,165,166,166,167,167,120,38,11,10,11,11,11,11,11,10,10,10,10,13,17,15,14,15,16,17,18,20,37,43,42,30,17,12,12,11,12,12,10,11,11,12,13,14,12,9,9,9,9,18,85,29,10,10,10,24,54,31,28,32,133,170,168,167,166,164,164,163,163,162,163,162,163,163,163,162,161,160,160,160,158,151,177,187,186,186,184,179,162,160,160,159,160,159,159,159,158,157,155,155,153,153,151,150,150,149,149,148,147,147,147,145,144,144,
        161,162,162,163,164,164,163,161,162,163,163,164,163,164,165,165,166,166,168,170,169,131,50,13,9,10,10,10,10,9,9,10,10,10,11,14,14,14,14,15,17,19,38,44,46,53,49,34,25,14,11,11,11,12,13,14,15,13,10,8,9,9,8,29,98,23,18,36,18,12,17,16,35,36,106,170,168,167,166,166,165,164,164,163,163,163,163,163,163,162,161,160,161,161,159,152,175,187,187,185,186,180,161,160,159,160,159,159,159,158,158,157,156,155,153,152,151,150,150,148,148,148,147,147,146,145,145,144,
        162,162,162,164,165,164,163,163,163,164,164,165,164,164,164,165,166,167,168,168,169,171,148,55,9,9,10,9,9,9,9,10,10,11,10,10,11,11,12,14,15,22,41,43,47,55,52,40,25,14,12,12,13,14,15,16,15,12,10,9,9,9,9,29,78,28,82,161,73,16,18,13,20,27,64,162,169,167,166,165,165,164,163,163,163,163,164,163,162,161,161,160,161,160,159,150,173,186,186,186,187,181,162,158,159,161,161,159,158,158,158,156,155,155,154,152,152,151,149,149,148,148,147,148,147,146,145,146,
        162,162,163,164,163,163,162,162,162,163,164,163,163,163,163,163,164,165,166,167,168,171,168,68,10,10,10,10,9,9,9,9,9,10,10,10,11,11,12,13,14,24,35,33,37,44,38,22,14,14,13,13,14,15,15,15,13,11,9,8,8,8,9,39,51,40,132,203,97,16,13,43,22,19,46,140,170,168,166,165,166,164,164,163,163,164,164,164,163,162,161,161,160,159,159,151,173,187,186,185,185,180,161,160,160,160,161,159,159,158,157,157,156,156,155,153,153,153,151,149,149,148,149,149,147,146,145,146,
        163,162,163,163,163,163,163,162,162,162,163,164,163,164,165,165,165,165,166,167,168,171,161,43,9,9,10,10,10,10,10,10,10,11,12,13,13,12,12,13,14,22,27,26,33,38,33,30,27,16,14,14,15,16,15,13,12,10,9,8,8,9,10,60,36,45,144,207,92,22,65,143,74,15,36,98,170,169,168,167,166,165,163,163,164,164,164,163,163,162,162,162,160,160,161,152,174,185,184,184,184,180,162,159,159,159,160,159,159,157,156,156,156,156,155,153,152,152,152,150,150,150,150,149,148,147,146,146,
        163,162,164,163,163,164,163,163,164,164,163,163,164,164,166,164,166,166,167,168,169,172,147,29,9,9,9,9,10,11,11,12,13,14,14,14,14,14,13,13,12,18,22,26,32,32,31,24,25,21,16,15,16,16,14,13,14,13,9,8,8,9,12,56,28,46,146,207,93,84,153,170,126,24,27,56,152,170,168,166,165,164,163,163,164,165,164,163,162,161,161,161,160,161,159,150,172,183,184,184,184,181,163,159,159,159,159,159,158,157,155,155,155,156,155,154,154,152,152,151,150,149,150,149,149,147,146,146,
        164,163,162,162,163,163,162,162,163,163,162,163,163,164,165,164,165,165,166,167,168,173,121,18,9,9,9,9,9,9,9,10,11,12,13,14,14,13,13,10,10,12,15,19,24,27,30,27,17,14,15,15,15,15,13,12,51,75,18,8,8,9,21,57,22,47,148,210,115,152,167,166,158,60,16,38,109,168,168,167,165,164,163,163,164,164,163,163,162,162,161,160,160,161,159,150,171,183,182,182,185,181,163,161,160,159,160,158,157,157,157,161,160,155,155,155,154,152,152,151,151,150,149,149,148,146,145,146,
        162,162,161,162,163,163,162,162,163,163,163,163,162,164,164,165,164,165,166,167,168,174,101,11,9,9,9,9,8,8,8,8,9,10,10,11,12,12,10,8,8,9,12,14,17,21,17,22,24,24,21,16,14,13,12,16,100,159,93,26,9,9,29,65,15,48,154,213,128,157,166,165,166,112,18,29,65,157,168,166,164,164,163,164,163,162,163,163,163,162,161,161,160,160,160,151,171,182,180,180,184,181,163,159,159,158,159,157,157,160,173,184,181,166,161,157,152,152,150,150,151,149,148,149,148,146,145,145,
        162,163,163,162,163,163,164,163,162,163,164,164,163,165,165,165,165,165,167,168,171,171,80,9,9,9,9,9,9,8,8,8,8,9,9,9,9,10,10,10,11,17,9,10,15,28,22,15,13,27,35,19,14,13,12,31,140,168,163,120,42,10,42,46,18,49,155,215,140,157,162,163,164,151,46,18,44,123,167,167,170,170,167,164,162,161,161,161,161,162,161,161,160,159,157,149,170,181,179,178,182,179,161,158,157,157,157,157,158,172,187,173,177,187,183,169,153,151,151,150,150,149,148,148,148,147,146,146,
        162,162,163,163,163,163,164,163,162,163,163,164,165,165,164,164,166,165,167,169,172,160,48,10,10,10,10,10,10,9,8,8,8,9,9,9,9,11,13,13,12,15,11,10,10,13,21,24,14,16,19,16,13,13,12,69,161,166,168,171,96,13,68,47,85,62,153,213,139,155,162,162,164,165,98,16,34,82,172,184,189,193,193,186,173,163,161,160,161,160,159,160,158,158,158,150,171,182,181,180,184,181,161,158,157,157,158,157,157,160,163,159,158,160,160,161,153,151,151,150,150,149,149,149,148,148,147,146,
        161,162,162,162,163,164,163,163,163,163,164,164,165,165,165,165,165,166,167,170,174,116,21,11,11,12,12,12,12,11,10,9,9,9,9,10,11,14,14,13,12,15,16,15,13,10,10,14,15,13,14,14,13,12,19,112,169,167,168,167,79,14,67,57,133,70,154,212,138,155,163,164,164,164,137,33,22,49,150,191,190,194,200,201,197,181,165,161,165,164,162,161,162,161,159,151,170,181,181,183,185,182,160,157,157,157,158,158,159,157,150,152,150,146,150,159,153,151,151,150,149,149,149,149,149,147,146,147,
        160,162,162,163,164,162,163,164,163,163,164,164,165,166,165,165,166,167,169,172,160,55,14,13,13,12,12,12,12,13,13,12,11,10,10,11,14,15,15,14,13,16,17,17,16,14,13,14,15,15,14,15,14,13,41,148,167,167,168,159,49,23,69,70,142,71,155,213,139,155,162,163,162,144,119,60,14,34,96,179,184,188,191,193,194,192,179,164,170,171,168,166,166,165,159,150,168,179,180,181,184,182,162,157,157,157,157,158,158,158,152,151,148,145,147,158,153,151,151,150,150,149,149,148,147,146,146,147,
        162,163,163,162,162,159,163,164,165,164,164,164,165,166,165,166,168,168,171,174,110,19,13,13,14,15,14,13,13,13,13,13,13,12,11,12,14,15,15,14,13,16,17,16,16,15,15,15,15,15,15,15,14,14,80,164,166,167,169,146,31,32,72,89,147,73,156,214,140,154,162,162,152,101,98,100,24,25,54,152,179,184,186,187,188,190,189,180,156,149,147,145,144,151,158,150,167,179,178,180,184,183,163,159,158,157,158,159,158,157,151,150,150,149,149,157,155,153,152,152,151,149,149,149,148,147,146,146,
        151,154,151,146,146,151,157,161,162,162,163,164,165,166,166,165,166,167,171,158,49,12,12,12,13,14,16,15,14,14,14,13,13,13,12,12,13,14,15,14,13,16,16,15,16,14,14,16,15,15,15,14,13,24,123,166,166,167,170,119,18,46,60,107,149,74,157,212,141,154,161,159,127,89,100,119,53,15,36,104,167,171,173,176,182,184,188,194,190,181,169,158,156,158,158,149,164,176,178,180,184,183,163,158,157,157,157,158,158,157,152,150,148,149,149,156,153,151,151,151,150,149,148,148,147,147,147,147,
        126,127,125,122,124,129,131,142,151,160,163,164,166,166,166,166,167,169,174,112,18,11,12,12,13,13,14,16,16,15,15,13,14,13,13,12,13,14,15,13,14,16,16,15,15,14,14,15,15,15,15,14,13,50,152,167,173,175,182,101,11,70,57,136,149,75,157,212,140,154,159,155,99,82,107,123,95,19,28,61,160,179,173,168,175,184,190,194,198,198,191,165,144,150,157,147,164,178,177,178,183,184,163,157,157,157,157,157,157,155,148,148,147,147,149,154,152,151,151,150,150,149,148,148,147,147,146,146,
        114,114,113,116,116,115,116,127,143,158,163,164,165,166,167,167,168,171,162,55,11,10,10,10,11,12,13,15,15,16,15,15,15,14,14,15,14,15,17,13,14,16,15,16,15,14,15,15,15,15,15,14,17,95,162,170,180,183,179,73,12,71,59,153,149,77,158,211,139,146,145,156,107,100,130,138,133,47,19,43,133,176,142,151,166,179,186,187,188,190,192,184,158,149,155,146,166,183,179,175,183,186,164,157,156,156,157,157,157,156,146,147,147,147,148,154,152,151,151,150,151,150,149,150,149,148,147,147,
        109,109,110,112,110,106,110,120,131,156,163,164,166,167,168,170,173,172,119,20,11,12,12,11,11,11,10,11,12,14,15,15,14,13,13,14,14,17,20,14,15,16,15,15,14,14,15,15,14,15,14,14,36,146,169,164,167,170,160,41,24,76,75,175,164,80,161,212,141,142,141,156,107,102,128,134,143,92,16,35,84,170,165,173,176,180,179,175,174,173,178,176,177,162,158,147,165,181,178,177,183,184,163,157,156,155,155,156,155,155,146,147,146,146,150,148,150,152,151,152,152,152,153,153,154,151,150,149,
        108,109,110,107,106,107,112,123,135,149,158,168,190,185,173,184,192,179,67,10,11,12,12,12,12,12,13,12,11,10,11,13,15,14,13,13,13,14,11,12,15,16,15,14,13,14,14,14,14,14,14,14,83,184,190,188,190,194,157,28,36,91,102,195,175,82,161,213,144,152,156,167,121,117,135,136,133,133,35,23,51,147,180,180,181,180,180,182,178,172,174,171,175,176,171,163,168,175,174,175,178,177,165,158,157,153,149,146,145,147,149,150,152,155,159,152,157,161,159,162,162,163,165,163,163,161,161,159,
        104,99,101,94,113,123,119,129,130,136,150,164,163,144,131,133,146,137,30,9,11,12,12,12,12,12,13,14,14,12,11,10,11,12,13,13,14,14,13,11,13,17,16,14,14,14,15,14,14,13,14,22,129,202,201,202,204,207,133,18,45,73,127,205,183,84,159,211,146,162,175,189,183,179,171,161,130,148,80,15,35,97,171,171,171,168,168,169,166,163,162,160,155,154,161,161,149,140,148,154,159,159,157,153,154,156,156,156,159,159,164,162,164,164,163,155,144,143,141,141,141,139,141,141,140,136,139,136,
        127,121,102,95,151,160,157,164,160,146,158,149,116,90,85,89,90,57,12,9,11,12,12,12,12,11,12,13,13,13,12,12,11,11,11,12,14,14,13,12,13,15,13,13,15,15,15,14,12,11,14,35,171,210,209,210,211,210,107,12,68,59,150,198,180,84,160,211,148,173,186,194,198,195,171,156,143,147,103,21,26,50,140,163,162,160,159,157,157,156,154,151,146,145,153,150,117,102,125,137,145,151,155,159,167,172,173,173,175,174,176,176,178,177,177,175,167,157,149,139,132,122,124,134,145,132,145,133,
        114,121,104,103,136,133,136,129,123,115,118,125,128,120,82,70,66,26,9,9,11,12,12,12,12,11,11,12,12,12,12,12,12,12,12,13,14,14,13,11,11,11,13,13,12,13,13,12,9,10,14,35,145,166,163,160,159,156,61,12,81,54,129,146,133,77,160,212,144,147,154,154,164,179,177,155,154,148,131,52,15,32,102,148,128,153,123,142,127,123,141,113,160,169,164,133,113,127,146,156,160,164,169,177,180,181,183,182,183,183,184,182,185,187,184,183,176,170,164,143,122,106,86,103,125,87,105,89,
        78,76,74,75,81,79,75,74,73,73,74,74,80,82,72,69,52,16,10,10,12,12,13,13,13,13,12,13,13,13,13,13,12,13,13,14,15,14,13,11,15,15,15,14,14,14,15,14,13,12,13,54,153,161,157,158,161,147,36,20,78,60,149,158,143,79,159,212,150,156,158,146,137,149,168,131,147,130,125,95,16,22,55,101,56,118,57,90,75,64,134,85,155,182,157,101,91,104,124,147,167,177,183,184,183,182,186,184,186,186,185,186,187,185,183,174,157,140,126,113,96,91,81,97,107,93,89,76,
        67,64,63,63,62,63,64,64,64,65,64,63,64,65,66,68,37,13,10,11,12,12,13,14,14,14,13,13,14,14,14,14,13,13,13,14,16,15,13,11,15,15,15,15,14,14,15,14,13,11,14,52,100,100,92,90,91,79,20,31,84,43,79,76,69,66,158,211,141,99,104,111,140,143,146,138,147,146,141,127,40,16,37,77,53,126,64,87,80,64,150,100,146,136,66,34,33,39,47,65,99,143,173,184,188,190,190,188,187,189,189,188,186,173,150,127,110,104,99,88,57,55,68,87,93,80,80,78,
        76,78,78,75,76,77,79,80,82,82,83,83,83,83,86,79,27,14,10,11,12,12,14,14,14,14,14,14,14,15,14,13,13,13,13,14,15,14,12,11,15,15,14,15,14,14,14,14,13,11,18,30,38,40,38,35,31,26,11,43,80,55,61,36,40,80,158,200,136,110,106,88,97,86,107,101,108,102,111,131,83,16,26,44,50,126,80,92,86,73,138,90,97,55,28,24,28,32,35,37,41,52,82,131,173,188,193,193,193,194,190,176,145,101,90,102,102,101,99,89,60,56,57,71,88,73,74,74,
        106,119,117,108,112,114,111,117,115,109,117,116,119,122,119,80,21,14,10,12,12,13,14,14,14,14,14,14,15,15,14,13,13,13,13,14,14,14,12,11,15,15,14,15,15,14,15,14,13,13,63,106,111,104,96,96,93,45,10,67,68,102,104,53,71,133,130,138,116,164,183,99,76,69,90,73,74,71,89,121,118,34,19,31,43,125,83,87,86,77,117,85,58,35,29,27,31,34,38,43,44,39,35,43,86,148,181,182,185,184,162,110,67,54,78,103,105,100,99,81,55,56,59,61,72,61,61,66,
        107,128,133,124,125,127,121,131,130,122,129,125,130,141,135,58,16,13,12,12,13,13,13,14,15,14,14,14,14,14,14,14,13,13,14,15,14,13,11,11,15,14,14,15,14,14,15,14,13,24,113,152,155,147,135,130,137,43,11,88,62,126,119,75,88,129,63,72,92,156,173,135,94,109,113,88,77,71,84,102,118,62,13,28,42,121,74,75,79,70,102,90,58,48,34,33,38,37,39,52,47,40,35,32,35,55,113,160,171,143,92,70,65,49,72,103,105,99,93,53,35,40,50,50,46,40,43,45,
        103,126,140,130,128,134,125,132,134,130,131,119,115,126,110,31,15,12,11,12,12,12,13,14,15,15,14,14,13,13,13,13,13,14,14,15,15,14,11,12,14,14,15,15,14,14,15,14,13,51,156,168,172,173,176,169,149,31,18,89,64,111,92,66,117,100,75,93,97,101,94,140,144,103,94,96,96,106,98,73,105,90,20,19,37,110,72,82,95,89,127,102,62,53,43,37,47,61,47,45,46,41,37,38,43,43,54,88,117,97,82,80,77,63,77,101,100,98,82,32,21,24,27,35,31,30,31,29,
        107,130,137,136,138,137,127,130,131,132,144,141,136,141,89,19,15,11,11,11,12,12,12,13,13,14,13,14,14,14,14,15,14,14,14,15,15,14,11,12,14,14,14,15,14,14,14,14,14,108,226,230,231,230,232,232,156,22,33,96,97,126,62,64,139,69,102,150,124,54,47,72,142,138,88,71,85,119,117,93,111,111,51,13,27,69,88,105,120,117,125,108,80,73,77,77,83,113,97,78,80,69,64,72,81,89,99,94,103,102,111,94,92,92,101,104,109,94,63,24,21,23,27,36,33,31,32,30,
        111,115,140,167,159,125,111,108,106,116,149,150,154,141,54,15,14,11,11,12,12,12,12,12,12,12,12,13,14,15,15,15,14,15,15,16,15,14,11,13,15,14,14,14,14,14,13,13,18,100,162,157,164,167,160,164,89,12,45,92,99,126,68,99,128,59,132,193,135,60,55,79,102,151,146,97,91,105,117,107,104,113,87,18,22,48,108,115,108,102,99,101,102,102,105,110,108,118,118,110,111,112,112,116,118,118,121,123,122,119,121,114,108,110,112,110,112,107,93,87,88,89,93,96,98,94,93,93,
        106,122,133,143,141,137,135,131,114,109,136,135,139,109,26,13,12,10,11,11,11,13,13,13,13,12,12,12,12,12,13,14,14,14,15,15,14,13,10,13,14,14,14,14,14,14,13,13,25,102,125,126,135,140,137,136,51,10,67,70,102,140,89,145,128,70,151,209,144,114,118,122,125,112,151,150,123,127,135,129,129,133,126,44,15,33,98,137,134,131,129,131,131,133,133,134,135,135,138,138,137,139,139,139,138,138,138,140,139,139,138,137,136,135,134,133,134,135,134,135,135,134,133,132,133,133,132,133,
        123,137,131,128,127,129,131,130,124,120,128,134,139,89,17,13,14,11,12,12,12,12,13,14,14,14,13,12,12,12,12,14,14,15,15,15,14,13,10,14,15,14,14,14,15,14,14,13,34,127,140,140,141,142,143,132,33,11,86,58,117,121,101,160,127,71,145,205,142,133,138,139,138,134,118,158,158,139,139,140,140,140,142,90,16,24,64,138,140,139,139,137,140,144,144,146,147,147,149,147,146,147,147,146,142,143,143,143,142,142,142,141,139,137,136,138,140,138,140,141,140,138,139,139,139,137,136,135,
        138,136,134,136,136,135,137,135,134,135,136,139,134,53,13,14,15,11,12,12,12,13,13,14,14,14,14,14,13,12,12,13,14,15,16,15,14,13,11,14,15,14,15,15,14,14,14,14,47,129,138,139,141,143,150,116,21,17,88,57,135,93,136,155,140,76,119,192,132,139,143,144,144,143,138,119,159,159,141,144,141,139,146,131,36,17,40,115,142,139,140,136,139,143,140,139,140,142,143,143,143,143,144,145,140,142,141,140,141,141,139,139,139,138,137,138,137,136,137,139,139,138,137,136,135,135,134,132,
        137,137,137,142,138,131,135,135,133,133,133,137,108,24,12,14,14,12,12,12,12,13,13,13,14,14,14,14,13,13,13,14,15,15,15,15,14,12,10,14,15,14,14,14,14,14,14,16,78,147,149,151,152,153,158,94,12,29,94,66,123,84,159,146,142,82,130,207,141,139,141,140,139,139,138,134,115,161,157,139,137,134,134,131,70,14,26,73,134,133,131,128,129,132,129,128,131,130,132,130,129,129,133,133,129,128,129,127,126,125,121,120,124,122,125,127,127,125,125,124,125,121,119,121,120,121,122,120,
        138,139,146,146,145,142,142,146,146,144,143,146,87,15,12,13,14,13,12,12,12,12,12,13,13,14,14,13,13,13,13,14,15,15,15,15,14,12,10,15,16,15,15,14,14,13,14,42,121,140,138,138,141,143,144,66,10,42,93,76,86,115,155,133,130,78,130,205,137,134,134,129,128,129,128,130,117,109,160,142,119,121,121,122,98,23,19,42,108,123,120,121,123,121,116,118,115,113,114,115,113,117,116,115,112,109,110,110,105,103,100,101,100,101,104,105,111,112,116,115,114,112,108,109,111,112,111,112,
        141,140,143,140,142,140,139,142,141,140,139,133,56,12,12,13,14,13,12,12,12,12,12,13,14,14,13,13,14,14,14,15,15,15,15,15,14,12,11,15,16,16,15,15,15,14,16,80,139,134,135,137,140,142,135,40,10,66,78,70,70,155,141,132,132,80,130,208,140,132,131,126,127,127,126,124,121,113,110,159,138,122,118,119,116,52,13,28,79,121,119,119,115,114,112,110,107,104,103,102,100,105,101,98,101,99,100,99,97,96,96,97,98,100,102,102,107,109,115,116,117,116,114,109,112,108,107,106,
        137,136,136,135,135,134,132,133,132,129,130,114,30,11,12,12,12,12,12,12,12,12,12,13,13,13,13,12,13,14,14,14,15,15,15,14,14,11,10,15,16,15,15,15,15,14,26,110,135,131,130,132,133,137,115,25,10,93,62,55,103,162,134,133,132,81,133,209,139,128,128,127,130,129,127,127,125,121,116,114,161,135,116,117,116,85,18,21,46,107,113,110,109,109,112,109,109,105,106,103,102,102,99,102,104,102,103,104,106,107,107,105,104,102,105,106,107,108,113,113,109,108,107,105,105,99,103,106,
        134,133,131,131,132,130,127,128,127,125,125,114,42,12,11,12,12,12,11,12,12,11,12,12,13,13,12,13,13,14,14,15,15,16,15,14,14,11,10,15,15,15,15,15,15,14,41,116,125,129,128,128,129,131,89,15,17,99,42,46,147,143,129,132,130,79,129,208,136,128,128,124,125,121,121,125,126,124,122,116,117,161,135,122,118,114,42,15,30,87,120,117,118,119,120,117,117,113,109,109,112,115,112,112,110,108,110,107,107,108,108,110,112,111,110,107,109,109,113,116,111,110,110,111,107,105,108,110,
        133,130,129,129,127,128,129,128,125,127,126,125,100,38,13,12,12,13,12,12,12,12,12,12,13,13,13,13,13,14,15,15,15,16,16,14,13,11,11,16,15,15,15,15,15,15,65,129,132,135,131,132,138,139,75,11,31,101,29,80,159,130,128,129,128,79,126,208,135,125,128,127,127,123,123,126,124,123,116,114,106,112,161,126,113,119,73,15,22,57,117,120,124,124,125,128,126,125,125,122,124,123,118,119,118,114,120,123,126,125,125,125,122,121,121,119,117,124,123,124,123,121,120,123,119,117,118,121,
        129,130,129,130,131,132,133,137,138,138,135,130,132,108,44,15,12,12,13,12,12,12,12,12,13,13,13,12,13,14,14,15,16,15,15,13,13,10,10,15,15,15,15,14,14,18,85,131,133,134,131,135,140,138,51,9,45,96,33,129,150,128,130,130,128,77,124,206,135,126,129,128,127,126,127,129,128,126,123,127,124,113,123,158,130,123,107,31,15,35,104,129,130,125,125,129,128,132,129,124,125,125,120,125,129,125,125,124,127,127,124,123,121,122,118,116,113,115,115,118,118,114,113,117,112,105,109,114,
        129,133,134,133,129,132,137,138,139,142,142,138,138,134,114,60,22,13,12,12,12,12,11,11,13,13,13,13,13,14,14,15,15,15,14,13,12,9,10,15,16,15,15,15,14,29,121,142,144,143,145,143,143,131,32,10,66,76,58,160,144,142,142,142,140,81,114,202,136,135,139,134,133,136,139,133,135,136,135,137,139,141,125,127,165,144,135,70,14,22,72,136,136,133,132,131,130,132,130,129,125,125,126,126,125,127,128,125,121,120,126,125,124,125,123,125,119,123,117,109,107,101,101,106,109,108,113,114,
        132,132,140,137,135,136,139,146,148,146,144,142,137,138,136,129,99,51,18,12,12,12,12,12,12,13,14,14,14,15,15,16,16,15,14,13,12,9,11,16,16,15,15,15,15,46,136,144,145,150,150,146,146,107,20,12,90,57,104,155,136,133,131,133,133,99,116,171,133,136,137,136,138,134,136,133,137,140,136,134,138,141,138,120,126,165,142,109,27,17,40,119,135,135,138,131,132,137,135,129,132,133,131,126,126,129,131,128,131,126,127,128,130,130,131,132,131,132,127,128,129,126,122,122,123,125,126,128,
        128,134,137,138,137,138,141,144,146,139,136,133,132,133,130,137,138,124,37,13,13,13,13,13,13,14,14,14,14,15,16,17,17,15,14,13,12,9,11,16,16,15,15,14,15,67,133,130,131,135,130,123,122,70,11,18,99,55,142,135,118,115,112,119,128,134,133,139,136,136,140,139,141,137,134,138,136,136,131,134,136,136,136,131,117,129,161,132,58,14,25,82,136,140,141,138,137,140,139,130,129,129,131,132,129,128,129,125,127,125,122,123,126,119,121,119,115,117,115,121,123,124,123,124,119,120,123,128,
        136,138,140,141,141,140,136,135,138,131,132,126,127,133,132,138,142,131,35,16,21,12,12,13,14,14,14,15,15,15,16,16,16,15,14,13,12,9,11,16,15,15,16,14,20,100,146,142,143,145,142,137,131,55,15,42,108,71,153,136,129,138,138,138,136,135,137,138,144,139,140,141,135,134,138,138,133,129,127,132,126,131,132,132,132,113,128,156,95,24,49,115,136,136,137,136,133,134,135,129,128,124,122,121,122,123,121,124,119,118,116,112,114,113,110,112,110,112,116,116,112,110,107,115,118,118,120,122,
        136,133,137,142,138,138,134,134,135,136,131,127,129,132,136,143,143,124,31,17,40,21,10,10,12,14,15,15,15,15,15,16,16,15,14,14,12,9,11,16,15,16,16,15,30,124,141,140,146,144,144,145,124,42,64,154,174,127,146,137,137,138,132,137,139,133,136,140,150,148,151,153,149,150,145,146,141,139,132,130,133,130,124,134,135,128,111,130,163,95,135,193,138,135,133,136,135,134,131,134,136,129,126,120,123,129,124,122,117,120,123,113,114,120,119,117,113,114,112,113,112,110,106,114,122,112,116,114,
        139,138,140,141,135,134,137,133,132,133,133,135,134,138,140,139,136,105,26,15,40,41,13,9,10,11,11,12,14,15,17,17,17,16,14,14,12,9,11,15,15,16,15,15,49,137,141,135,138,141,140,141,111,30,60,165,193,137,135,136,142,143,139,140,141,136,139,136,135,136,141,145,142,136,131,132,140,135,133,136,137,134,139,141,132,130,128,118,181,146,75,74,68,135,135,138,133,125,118,131,137,134,123,119,123,126,123,115,110,117,116,109,111,118,123,118,112,118,119,119,124,122,123,125,128,121,115,118,
        132,135,136,135,137,135,143,140,137,136,139,143,138,135,141,142,146,107,26,14,32,53,24,10,10,10,10,9,9,11,13,15,16,17,15,15,12,9,11,16,16,16,16,16,78,149,137,133,139,147,152,149,96,15,15,109,80,101,135,132,137,144,142,137,130,123,131,129,131,135,128,132,128,124,120,123,133,132,130,127,129,128,131,130,127,133,129,128,144,122,35,15,33,107,133,135,131,120,115,118,118,121,118,119,123,126,124,122,118,122,120,117,119,117,125,122,121,124,124,128,139,136,135,135,130,136,132,132,
        126,132,135,135,140,138,142,143,136,131,135,141,133,135,145,147,149,99,25,16,23,54,43,17,12,11,11,10,9,9,9,10,11,14,15,15,12,9,11,16,17,16,16,21,96,139,133,130,133,141,148,144,77,11,21,112,56,112,137,133,132,135,132,128,126,123,126,129,127,129,132,132,128,130,131,144,143,139,138,132,134,136,135,135,126,135,131,122,127,116,63,13,22,67,117,122,125,124,123,119,111,112,108,116,112,115,115,122,120,118,119,117,119,112,113,114,123,122,120,114,112,111,116,126,117,119,122,119,
        130,128,131,135,135,133,131,135,129,130,133,135,130,131,141,140,139,90,23,17,18,46,55,36,15,11,11,10,9,8,8,9,9,10,10,11,10,8,10,15,16,17,17,32,106,125,126,123,122,126,130,125,45,10,34,107,58,128,138,135,131,125,121,117,125,121,110,114,116,117,117,118,130,134,139,142,135,126,129,126,137,133,131,128,125,126,121,124,126,120,102,27,16,37,101,124,123,129,129,124,124,120,117,122,121,115,115,126,123,113,106,111,107,116,115,112,114,111,107,104,106,99,96,99,93,82,73,76,
        119,124,122,121,115,115,115,116,112,119,125,126,126,133,137,136,135,77,22,20,17,35,55,52,31,16,11,10,10,8,8,9,9,9,9,9,9,8,9,10,10,11,16,71,110,113,113,114,109,112,107,99,25,10,51,101,57,94,90,100,101,103,101,97,109,109,106,118,118,121,125,127,137,141,141,134,131,128,139,140,136,130,135,126,119,116,126,130,125,126,138,68,13,26,75,121,122,125,124,121,123,121,110,115,111,120,121,127,126,119,120,125,122,129,125,129,130,117,109,110,116,111,113,103,107,105,97,96,
        116,121,120,117,119,117,122,132,135,137,135,134,135,138,142,144,142,76,21,21,18,28,50,56,46,31,22,14,11,10,9,9,9,9,10,10,10,9,9,9,10,10,21,99,125,127,132,125,115,117,121,89,17,11,73,80,65,92,87,105,109,117,111,125,122,121,123,129,128,131,133,130,141,137,141,142,141,137,142,141,133,130,137,131,140,136,129,137,144,142,143,107,22,18,45,121,134,136,135,123,130,132,117,113,109,115,125,124,116,111,121,118,113,112,121,123,130,130,128,123,121,117,118,118,112,110,113,114,
        125,131,127,128,129,133,126,136,137,129,133,140,138,141,139,144,143,73,22,21,17,25,45,57,53,48,36,25,16,11,9,9,9,9,10,10,10,10,10,10,10,12,29,93,113,110,102,100,94,98,108,70,11,12,95,62,83,113,127,129,119,114,113,117,110,99,96,99,103,122,128,128,134,123,130,139,141,136,142,139,142,139,137,133,144,142,134,138,144,148,141,127,54,13,27,91,139,140,137,134,132,137,133,121,125,128,123,122,122,123,127,131,128,126,130,134,121,125,133,128,133,127,124,125,117,119,122,117,
        123,127,124,123,122,128,124,125,123,121,130,128,131,126,125,131,135,66,22,22,17,24,43,57,58,56,51,44,35,26,14,9,9,9,10,10,10,10,10,10,13,23,40,102,118,108,105,111,127,122,115,53,9,19,106,54,109,126,125,122,121,119,117,112,110,107,108,112,115,126,142,134,121,120,129,134,132,136,140,142,140,145,147,141,144,141,142,134,138,143,128,128,95,18,19,52,116,120,123,120,127,126,119,108,114,115,115,118,126,128,122,120,122,117,116,115,111,119,114,112,119,119,123,120,117,119,116,113,
        119,121,120,121,127,132,127,126,121,119,128,129,127,126,126,136,138,62,23,25,19,23,43,56,58,58,54,57,58,55,28,10,10,10,10,10,10,12,17,25,35,43,60,115,125,115,120,119,119,118,103,34,9,33,114,52,107,112,116,113,116,131,126,128,132,129,122,130,120,117,124,122,125,128,130,130,133,139,135,131,126,142,137,123,126,130,131,129,128,137,133,136,128,42,14,32,92,130,128,120,121,124,125,123,112,109,122,113,119,129,117,116,117,114,124,118,111,115,112,109,104,113,124,118,110,111,112,110,
        124,121,123,137,139,133,124,125,130,128,128,130,125,131,135,141,132,54,24,24,19,21,39,56,59,54,54,58,61,82,38,9,9,10,10,10,12,21,45,55,57,52,83,130,130,121,122,128,125,124,102,52,38,43,101,67,127,111,108,110,115,130,130,117,123,133,127,129,124,123,118,122,124,127,128,131,125,136,145,144,132,138,129,130,131,140,137,137,128,130,130,120,129,78,16,42,115,133,126,122,117,120,122,121,127,120,113,119,123,126,129,126,120,126,136,121,121,125,113,108,111,109,114,109,105,111,118,110,
        123,129,136,137,138,126,131,136,137,138,132,135,135,142,135,145,128,50,26,27,19,19,35,55,61,49,45,52,65,113,38,15,15,10,10,11,18,28,46,59,59,54,95,139,133,130,125,131,128,125,89,60,50,78,89,77,136,126,122,119,125,137,130,125,120,128,146,140,142,134,127,140,137,134,129,131,133,135,141,140,143,134,138,137,134,143,134,128,129,125,128,121,129,117,72,152,210,165,135,129,129,127,121,120,127,120,112,123,125,123,126,126,119,119,120,125,134,123,120,114,115,116,119,114,132,128,123,111,
        123,125,129,121,123,128,130,122,119,132,123,115,125,126,124,127,112,41,28,34,23,18,30,53,60,50,41,42,84,120,47,25,21,11,17,24,18,26,41,55,59,63,110,140,135,143,132,133,126,123,84,66,151,190,153,105,124,131,125,130,140,138,131,127,121,128,132,133,137,135,127,131,135,132,119,119,125,120,120,122,118,119,130,124,132,135,132,125,133,129,135,124,128,124,97,163,208,179,139,128,124,131,124,122,122,123,119,120,125,128,127,121,129,122,116,119,120,122,117,116,120,115,115,116,118,120,131,124,
        113,114,116,114,109,115,114,110,108,116,118,113,107,105,103,112,114,39,27,38,28,16,25,47,57,49,39,40,102,122,46,26,23,12,28,41,25,21,37,52,56,69,126,133,125,132,124,129,130,124,88,80,206,229,219,147,121,124,128,118,120,125,118,120,124,126,129,126,130,126,125,123,116,119,115,112,112,118,129,120,121,125,133,120,129,127,119,120,136,134,133,129,119,123,101,101,145,162,122,121,122,121,112,112,104,115,120,109,105,105,99,113,122,111,115,104,104,106,108,114,116,108,111,112,113,116,106,100,
        113,116,111,118,111,110,119,105,113,123,133,123,116,121,122,126,126,40,29,39,38,20,23,47,61,50,41,46,123,135,52,31,35,26,25,38,36,23,31,52,59,88,138,140,141,141,135,130,132,132,76,97,218,246,229,146,129,128,130,119,119,122,126,130,135,127,129,131,121,122,134,125,115,126,128,121,127,131,123,116,120,124,143,125,130,128,126,123,132,142,146,131,124,128,126,81,97,193,140,120,116,125,122,117,107,127,121,120,111,111,103,115,119,116,111,108,108,110,113,117,123,111,121,124,126,127,113,104,
        125,128,126,127,136,128,127,120,132,127,135,136,131,126,130,128,112,36,28,39,46,27,20,41,58,47,40,49,128,137,59,34,45,37,28,26,37,29,27,44,64,105,137,132,136,126,120,123,131,126,62,81,184,225,168,148,130,135,129,127,129,134,119,134,143,132,122,132,122,121,124,132,128,129,125,120,125,123,119,115,112,125,128,118,123,132,121,123,121,120,125,125,122,121,124,99,70,167,169,110,108,105,120,119,114,115,104,101,105,102,98,113,111,110,105,106,99,105,102,107,111,103,112,117,129,126,120,123,
        137,141,137,139,140,136,127,126,129,124,130,135,139,133,129,131,119,38,29,37,51,39,19,33,55,45,39,63,120,146,69,36,48,37,28,25,28,31,26,32,60,117,128,121,120,125,119,126,134,110,51,104,207,220,134,132,134,124,126,123,128,139,117,124,135,122,124,127,137,135,130,138,132,119,128,124,125,120,127,132,125,130,130,133,118,110,105,108,115,107,110,116,110,114,116,102,68,116,193,130,122,107,113,121,118,112,112,108,120,109,104,114,117,113,100,101,99,99,106,99,94,100,111,130,122,130,125,120,
        132,142,144,145,133,137,129,124,122,125,127,119,126,124,117,132,126,38,29,32,50,50,26,24,50,43,37,81,124,128,68,39,49,37,24,22,22,24,24,24,52,122,124,122,120,130,127,134,140,100,48,125,224,202,143,136,129,122,128,129,132,141,131,133,132,117,121,126,127,123,114,118,120,121,127,125,130,125,125,136,127,123,128,130,131,124,124,128,128,120,118,121,111,114,128,118,93,77,178,160,122,116,115,120,111,111,105,116,131,115,121,123,126,118,105,108,108,98,111,113,104,114,118,131,116,116,114,108,
        123,123,124,124,124,129,124,115,120,114,120,122,123,123,113,128,126,38,32,27,43,57,37,21,39,39,38,101,134,134,79,41,45,39,27,21,19,18,19,20,40,116,130,125,120,129,137,133,137,86,53,148,233,183,128,123,111,121,138,131,122,136,138,137,142,135,123,128,123,122,120,123,127,121,128,131,129,136,131,125,135,130,133,139,140,136,133,121,125,130,120,117,119,125,131,121,125,76,130,187,131,124,117,116,118,115,99,103,123,120,122,119,109,120,108,115,109,103,114,112,107,122,125,124,114,110,119,123,
        136,125,118,122,126,136,119,121,129,138,132,131,131,132,119,121,116,35,32,27,34,54,49,27,33,39,48,117,137,134,87,49,62,57,51,43,34,26,19,18,31,96,124,123,123,122,109,108,110,63,66,169,235,159,105,107,108,112,114,110,113,115,119,129,133,132,122,127,125,123,107,119,130,121,128,129,128,133,135,126,122,124,121,127,126,127,133,130,123,126,118,122,128,126,121,114,125,88,85,186,146,108,115,118,124,113,115,112,114,124,129,119,111,105,107,110,103,108,110,107,117,129,114,112,118,106,112,115,
        127,111,106,119,135,141,132,125,124,127,121,118,124,127,117,116,99,32,32,27,25,48,54,37,37,48,77,126,133,138,92,41,66,66,66,63,61,58,48,27,22,91,124,118,118,118,115,110,102,52,84,191,232,138,111,119,114,117,122,117,111,113,117,123,133,137,126,123,120,116,115,134,135,129,131,138,141,134,139,130,127,132,124,127,120,119,125,134,132,129,124,118,119,123,118,112,125,104,70,150,181,120,122,122,127,130,133,126,113,114,124,115,104,100,107,109,109,107,109,109,100,104,110,108,107,108,115,117,
        114,115,109,121,125,123,125,123,119,122,131,120,124,117,108,113,95,30,35,31,21,40,56,43,40,51,103,131,145,132,94,29,60,69,64,62,64,65,66,50,22,85,122,118,118,128,129,119,96,47,105,210,223,124,117,134,127,130,133,125,128,129,122,128,137,131,128,118,117,113,109,119,114,120,123,127,137,126,126,120,124,135,131,131,134,133,142,131,134,142,138,119,113,121,119,121,139,118,84,98,193,146,123,126,124,130,133,130,127,127,122,126,110,110,125,123,112,121,117,116,114,125,117,120,122,119,130,120,
        127,118,123,116,122,123,118,127,129,118,129,129,119,116,124,131,90,29,38,34,21,32,55,50,43,69,130,127,135,123,92,24,40,63,63,57,44,33,28,23,23,96,118,120,122,120,132,132,89,49,125,223,206,134,138,133,127,127,127,115,122,131,129,125,124,117,123,124,124,120,123,110,102,111,116,114,124,122,104,110,118,132,135,128,137,131,138,131,131,132,129,128,127,115,116,117,128,119,108,74,160,170,115,119,127,127,126,124,136,136,136,130,123,124,126,111,109,122,111,109,109,129,111,112,123,126,122,109,
        115,110,123,120,120,120,114,123,120,118,117,108,107,113,137,145,84,29,41,40,22,24,47,54,47,69,123,113,108,120,93,23,21,46,59,60,55,54,46,20,27,110,117,116,125,116,128,118,74,57,152,233,184,129,133,120,116,125,129,129,132,133,118,124,130,129,123,122,120,116,121,106,101,116,134,138,127,128,123,121,123,108,116,121,127,123,129,137,126,117,125,122,127,125,125,111,118,119,122,86,110,193,131,112,110,116,122,124,130,119,130,121,113,118,120,108,105,125,111,112,118,113,102,108,105,103,100,105,
        120,115,120,129,132,115,123,130,118,122,117,130,131,127,140,134,73,29,41,44,26,19,34,45,46,60,107,107,110,128,103,24,14,30,51,68,67,67,63,27,40,105,109,122,132,118,119,122,61,69,173,237,166,123,127,122,115,118,123,130,128,129,111,122,129,128,122,125,130,113,118,119,116,115,127,128,139,127,129,132,132,106,110,125,121,122,112,127,126,121,125,118,121,124,118,122,120,119,122,109,76,172,163,118,102,98,98,102,114,106,112,107,104,112,115,112,113,123,114,107,122,118,113,113,103,114,126,123,
        124,135,138,150,151,133,130,134,121,120,128,144,160,146,136,135,72,30,42,46,30,19,30,39,47,54,93,116,120,132,105,24,13,23,43,67,68,65,64,29,54,97,110,119,120,131,120,116,55,81,183,233,142,116,115,115,117,105,97,124,107,116,111,107,117,128,125,124,109,102,124,123,115,115,119,122,131,114,126,126,132,128,118,142,125,114,120,128,121,136,129,112,122,118,105,108,99,119,118,126,71,139,190,148,134,105,82,101,113,109,125,124,114,110,110,103,117,121,130,137,141,128,109,110,106,126,128,122
    };
    return ImageUInt8::fromRaw(img, 128, 128);
}

inline ImageUInt8Ptr getPeppersImage(){
    auto img = new uint8_t[16384]{
        93,103,105,108,110,111,110,108,110,111,110,110,109,110,109,107,107,107,106,105,107,138,162,164,119,86,85,83,99,129,126,152,168,171,175,178,177,177,177,177,176,175,169,164,157,156,132,136,155,158,160,152,146,146,151,154,153,151,148,144,137,129,123,116,118,141,160,164,163,161,163,163,164,166,167,167,169,175,181,179,75,87,142,150,173,167,122,92,97,100,103,101,87,119,108,97,94,90,94,96,112,124,131,132,136,144,150,142,144,142,135,105,99,104,106,87,95,110,115,117,119,127,139,148,150,146,144,136,
        94,105,104,105,104,102,102,100,101,102,104,104,107,107,107,104,104,103,102,98,96,108,143,174,176,128,89,74,97,125,118,151,176,183,188,195,198,195,193,190,188,186,178,165,149,133,131,144,170,150,106,82,85,88,91,92,97,102,107,107,110,110,106,109,147,168,174,182,192,191,188,189,191,192,192,189,192,197,204,167,55,82,158,162,153,162,122,111,105,100,95,92,83,85,90,94,92,101,106,115,133,142,148,156,156,158,161,163,164,164,161,144,109,112,111,91,102,114,122,125,123,130,138,149,152,153,152,143,
        90,104,102,104,101,99,102,101,100,102,105,102,106,105,106,103,103,105,103,100,100,98,103,137,173,181,168,145,131,123,113,148,172,184,189,199,203,205,204,201,197,191,174,147,126,138,157,171,158,93,86,90,92,93,95,96,96,96,99,97,95,96,99,124,169,176,183,192,200,191,192,196,199,199,199,193,194,194,201,113,85,88,175,190,161,149,125,153,151,133,139,99,83,85,84,84,96,111,119,128,138,145,148,153,155,158,161,164,167,166,165,161,137,109,102,86,98,108,110,113,122,132,140,149,151,150,147,142,
        86,97,99,99,98,102,104,105,105,100,98,96,102,104,103,105,105,103,106,106,105,102,101,98,125,161,177,179,163,122,119,146,171,182,185,194,199,208,212,209,202,183,150,130,145,159,173,144,103,92,90,91,92,95,94,97,97,97,98,97,99,101,100,98,108,150,189,194,196,198,201,201,202,203,200,194,189,184,175,90,96,136,190,190,188,193,154,142,151,137,136,152,131,78,64,81,106,122,131,134,141,146,149,151,155,155,157,161,165,168,168,166,154,120,93,80,102,107,104,116,131,138,146,152,152,150,143,139,
        93,119,119,123,126,133,142,146,147,141,116,93,95,100,98,95,97,100,101,100,100,102,100,101,95,98,128,154,149,111,111,152,174,180,177,153,130,121,119,115,135,138,134,153,165,145,91,87,99,94,93,94,95,95,96,98,99,100,99,98,98,100,98,100,100,95,118,183,204,209,205,200,200,200,193,183,172,161,107,105,107,133,177,193,192,195,136,138,161,156,139,155,155,146,56,91,120,129,130,138,142,148,153,156,157,155,157,161,164,168,169,166,159,121,76,73,107,112,116,128,136,145,149,148,149,148,146,138,
        122,146,144,144,145,148,150,153,155,155,152,122,86,88,88,87,83,86,90,88,89,87,87,88,87,89,88,84,78,93,107,147,167,144,91,81,85,91,91,92,92,88,99,119,113,93,92,91,91,91,91,93,95,94,95,96,97,99,99,100,100,100,100,101,101,98,100,101,158,209,208,200,197,190,182,168,152,116,76,103,107,137,157,192,170,156,122,179,172,153,155,165,167,160,151,120,118,126,132,139,143,149,151,154,155,159,159,160,163,168,169,166,153,111,53,76,114,121,116,127,133,139,139,140,138,137,138,136,
        120,146,148,148,150,150,153,154,157,158,158,153,100,77,76,79,93,94,77,77,77,73,73,76,76,77,81,84,88,79,92,139,135,93,87,87,91,93,93,94,92,92,92,89,89,88,88,92,94,93,93,96,96,95,96,96,99,100,99,98,99,101,101,100,101,101,99,100,96,123,191,200,190,178,166,143,135,106,84,69,100,138,122,142,145,152,153,197,152,150,160,179,179,174,163,148,116,124,132,139,141,143,145,147,149,157,158,160,162,165,164,161,141,90,25,76,117,126,117,121,129,133,131,127,128,126,128,131,
        121,151,153,155,157,159,163,165,175,172,165,160,126,74,74,87,129,127,99,99,105,105,91,73,71,72,77,79,103,80,99,134,99,86,88,91,94,93,93,94,93,92,91,90,93,90,91,93,92,96,95,93,95,95,98,98,99,98,100,98,99,100,100,100,99,100,98,99,101,99,101,161,180,160,140,118,102,93,100,61,114,152,128,129,147,191,183,182,177,180,133,189,201,191,177,159,142,117,129,134,133,135,139,143,148,151,154,158,160,161,157,149,130,70,25,98,126,130,121,114,122,125,124,124,125,127,127,132,
        122,154,158,161,164,179,198,200,208,189,167,158,128,71,69,95,127,122,118,124,129,130,128,118,93,74,76,77,94,79,117,108,92,86,88,91,91,93,93,93,92,89,88,89,89,88,89,91,93,94,93,94,94,94,95,96,98,97,97,98,98,99,100,99,99,97,98,98,101,99,99,99,134,130,103,78,122,122,117,89,117,139,126,156,132,156,191,193,201,171,169,192,210,207,185,165,159,124,123,128,133,139,143,150,156,156,156,157,158,161,152,140,116,50,27,113,128,132,127,115,115,122,122,122,127,133,138,137,
        129,156,161,164,167,174,180,184,191,169,157,150,104,69,82,121,121,118,127,130,135,137,138,139,131,93,40,30,29,94,134,90,90,85,87,91,92,93,93,91,89,89,89,86,89,87,86,87,91,94,94,92,95,93,94,95,94,94,95,97,100,99,101,101,101,99,98,97,100,101,99,101,105,100,32,83,141,116,108,87,105,138,112,147,139,126,172,191,171,170,175,196,208,203,185,173,159,142,123,133,136,141,148,155,160,159,160,159,156,156,148,135,105,33,46,121,128,130,130,130,126,121,123,124,132,139,144,143,
        134,160,161,161,161,161,172,184,199,194,161,140,66,56,109,120,120,126,130,135,138,141,142,141,140,133,44,16,29,126,126,88,88,93,88,89,92,91,89,90,88,89,87,88,88,86,88,88,91,90,94,96,93,94,94,94,91,92,94,95,99,99,100,100,99,99,98,96,99,100,103,99,98,102,82,124,129,110,106,88,93,107,116,135,147,141,157,169,179,195,205,206,201,187,176,176,163,155,131,138,143,148,152,157,161,161,159,158,154,149,142,123,81,26,74,123,127,128,129,157,155,127,122,126,130,137,143,143,
        129,157,158,157,155,156,187,204,205,196,163,123,54,91,124,120,124,131,136,138,140,142,143,141,138,125,65,49,85,132,121,89,88,89,89,91,91,92,92,93,91,88,88,88,86,93,118,97,91,91,92,94,94,94,95,99,98,94,88,92,95,98,98,100,100,99,97,99,97,97,96,97,96,99,100,110,111,90,91,91,110,109,132,144,150,163,169,170,188,208,215,211,194,179,173,175,169,158,140,140,142,150,156,161,161,161,161,155,148,142,132,103,46,22,91,123,126,125,127,141,150,127,118,125,129,136,144,145,
        125,157,158,157,156,157,162,182,175,166,144,91,75,113,120,124,130,134,136,138,141,168,177,142,128,102,50,71,126,133,113,90,90,88,87,90,89,93,90,90,87,89,91,91,94,117,138,151,101,94,95,93,96,96,101,105,104,99,92,90,88,93,96,97,97,101,96,96,96,94,94,92,93,94,99,99,102,108,107,100,96,112,130,141,146,155,167,176,190,200,204,196,180,174,171,173,171,161,147,145,144,151,157,161,163,162,159,149,140,131,112,67,16,22,101,124,125,125,128,130,131,129,117,122,133,140,146,148,
        125,154,155,155,152,153,155,156,154,143,103,71,105,118,116,127,134,136,138,138,169,214,213,146,110,56,22,91,130,122,105,91,91,89,88,87,89,90,90,89,87,91,95,93,119,133,185,155,96,93,93,91,91,100,104,109,105,100,92,92,92,94,98,98,96,98,99,98,94,94,93,92,91,93,93,93,94,98,108,131,113,108,126,137,145,154,165,173,182,183,184,176,170,166,169,171,173,165,154,148,150,151,156,160,161,162,158,142,130,113,76,25,13,25,108,126,127,125,124,130,130,128,116,126,133,141,147,147,
        106,124,127,140,147,151,150,152,144,106,69,77,113,117,120,130,137,135,133,153,208,220,188,112,69,19,42,117,130,120,101,93,91,90,89,87,89,91,87,86,90,97,97,104,131,167,178,124,92,93,92,91,89,98,106,103,101,95,95,92,94,93,97,97,95,97,98,99,100,96,95,95,93,95,95,94,94,97,97,99,107,120,125,134,143,153,162,165,165,162,165,168,169,168,171,171,174,171,156,144,154,155,156,159,162,158,150,122,105,68,24,9,11,24,112,126,127,127,122,130,131,134,127,125,134,137,143,147,
        81,95,101,109,119,142,151,141,107,71,55,103,113,117,124,134,135,134,159,197,216,184,108,68,23,22,96,116,101,102,99,92,91,90,88,87,88,88,88,86,93,99,103,122,144,189,148,94,92,91,93,92,89,89,92,102,97,93,91,92,92,93,96,97,100,99,99,100,101,100,100,98,97,94,95,95,97,97,96,96,100,97,111,138,141,149,155,160,156,156,154,157,165,167,168,170,170,170,157,148,155,156,156,159,161,156,138,89,55,21,8,10,10,16,107,128,128,126,122,128,129,135,132,127,134,140,143,144,
        87,111,118,119,121,122,130,127,74,40,78,114,112,118,131,134,131,145,202,211,173,97,55,21,15,67,109,86,87,99,97,93,91,91,89,86,87,86,83,84,99,103,123,128,165,169,116,91,91,93,97,94,95,94,90,94,93,92,91,92,93,93,96,95,98,100,101,101,102,101,102,105,104,98,99,95,96,97,97,98,98,95,95,101,146,145,148,153,149,152,151,154,156,163,165,165,168,168,159,153,154,159,160,159,158,154,123,40,16,9,8,8,9,13,94,125,128,126,122,119,126,132,138,132,136,144,151,149,
        96,115,124,125,128,133,136,134,105,46,107,111,117,126,132,128,131,137,159,144,83,42,19,14,62,109,79,79,107,98,96,94,92,91,90,88,87,87,88,87,98,115,105,149,172,145,91,90,92,95,96,94,95,95,93,92,90,92,92,93,95,93,95,96,97,99,102,102,101,101,101,101,100,103,100,96,96,99,98,96,96,96,97,96,104,149,143,147,150,150,152,151,156,162,163,160,166,166,160,157,154,159,163,163,159,151,89,10,8,9,9,11,11,11,81,122,127,126,125,116,117,131,134,131,134,144,150,152,
        102,119,129,127,132,136,139,142,134,92,107,110,120,126,124,129,147,141,105,74,35,18,15,33,108,84,79,92,114,102,96,93,90,90,89,87,84,84,85,84,95,115,120,168,160,118,90,89,94,94,98,97,98,105,103,95,93,93,94,95,96,95,93,95,94,97,99,101,101,101,101,101,102,100,100,98,98,98,99,99,98,97,97,98,98,109,148,147,149,153,153,150,155,160,160,158,166,165,163,158,155,158,165,164,160,146,55,7,8,9,11,10,12,14,58,115,123,126,126,120,114,126,133,135,135,144,152,155,
        103,119,126,128,132,135,140,139,140,114,104,112,121,123,156,191,157,123,71,34,17,14,18,106,91,85,87,110,118,99,95,94,92,90,89,87,85,84,83,86,115,104,141,162,137,93,90,88,93,93,96,97,99,109,98,96,93,95,96,96,97,96,97,96,96,97,98,100,100,102,101,101,100,100,99,98,96,96,96,99,98,100,99,100,99,99,125,145,151,152,151,149,154,156,160,159,164,165,166,161,157,155,164,164,159,139,34,12,10,13,14,16,18,26,60,110,120,124,122,122,107,112,130,131,132,145,154,152,
        95,114,122,129,129,134,140,142,138,119,93,113,129,185,210,174,99,59,28,15,12,14,94,113,97,89,99,108,107,95,95,93,91,90,90,88,86,84,84,100,102,104,157,156,111,92,90,90,92,94,98,98,99,100,100,99,96,95,97,96,99,99,99,100,98,98,97,98,100,101,103,102,101,101,102,100,99,98,99,99,98,99,98,100,99,102,102,144,144,146,147,147,151,156,157,159,163,166,166,165,160,142,135,127,127,122,71,40,20,24,33,61,92,112,119,121,121,122,124,123,110,103,126,130,134,144,154,154,
        89,110,119,125,126,131,136,139,134,124,101,121,192,209,159,90,54,24,15,13,12,28,140,109,98,93,97,98,91,92,95,93,93,93,90,87,87,86,86,123,112,138,161,133,97,92,91,90,91,91,96,96,98,100,100,99,98,94,97,99,99,99,101,102,101,101,99,100,100,102,102,103,104,102,101,100,99,100,99,99,98,99,99,102,100,102,103,114,136,144,144,145,146,152,158,161,164,165,155,137,119,72,71,98,102,110,129,139,113,81,62,91,107,116,124,127,131,131,134,131,125,117,123,131,132,143,153,153,
        92,109,117,122,126,133,136,136,129,123,102,132,170,136,88,49,20,15,12,10,12,73,129,108,95,90,89,89,86,91,90,91,92,90,89,85,85,84,96,121,129,158,160,115,99,95,94,93,91,92,95,98,99,99,102,100,99,94,95,97,98,98,100,101,102,101,100,100,100,100,103,105,104,104,103,102,99,101,100,99,99,97,100,101,115,105,102,105,107,133,137,141,144,153,159,163,147,124,110,99,93,81,86,96,110,117,128,130,137,141,124,100,113,126,133,130,135,137,138,138,137,129,122,128,132,140,147,152,
        99,113,115,121,125,134,134,130,123,92,72,112,112,85,44,22,14,12,11,10,75,108,117,101,91,88,86,87,85,86,90,89,92,92,90,87,86,82,116,115,140,174,158,102,101,95,94,93,92,92,97,96,97,100,101,104,105,98,93,96,98,100,100,101,102,100,101,102,99,100,101,102,103,100,104,102,101,101,100,100,98,99,98,98,101,101,103,103,107,98,135,136,141,148,148,118,97,97,89,92,94,96,99,109,121,127,131,134,137,140,140,137,119,125,130,135,144,149,149,146,143,137,133,127,126,136,141,143,
        105,113,115,122,124,129,131,127,116,48,19,46,55,35,17,15,14,12,11,69,147,99,107,95,90,88,86,86,85,85,89,90,89,89,88,87,86,91,120,122,148,170,147,97,100,98,94,94,93,92,93,96,98,101,104,119,123,115,93,97,96,99,101,100,100,100,102,102,100,100,99,104,101,102,104,102,102,100,100,100,99,98,100,101,103,102,102,104,102,95,100,132,132,125,90,77,66,73,93,109,98,93,104,121,130,133,137,138,138,149,143,141,139,125,132,140,152,154,156,159,147,141,143,136,134,131,134,138,
        97,107,113,121,123,126,121,115,90,28,17,14,14,12,13,12,14,15,64,153,140,92,95,92,90,88,86,87,88,87,87,88,87,88,87,86,87,109,120,130,166,164,138,97,103,97,94,93,94,93,94,98,101,100,112,127,143,150,101,92,96,97,100,99,99,99,101,101,99,97,98,100,101,102,102,102,105,104,101,100,100,100,104,101,103,102,105,106,100,96,97,111,94,59,47,48,57,73,88,91,88,95,108,119,128,130,136,140,142,148,148,147,146,139,137,150,169,181,179,173,162,151,146,143,139,137,135,134,
        93,113,115,122,130,121,111,98,55,25,20,15,15,15,17,18,22,41,140,157,131,89,88,89,90,91,88,86,89,87,88,87,87,87,87,86,91,120,112,137,178,167,126,101,104,99,95,94,98,96,96,99,101,102,110,118,125,154,127,92,93,95,97,100,97,97,99,101,99,97,97,101,101,101,101,104,106,106,103,100,101,105,109,112,107,106,104,103,100,97,97,95,88,18,27,51,66,74,81,77,87,102,112,118,127,130,136,142,145,148,149,153,148,145,148,163,177,189,197,189,179,160,158,154,147,147,140,134,
        90,113,114,136,184,115,91,62,27,29,27,23,23,27,30,30,30,102,153,144,115,85,86,89,91,89,88,88,87,86,87,88,87,87,87,88,108,120,133,150,180,180,130,100,105,101,97,95,98,100,96,99,103,104,105,103,103,124,148,97,93,95,98,98,99,99,101,102,102,99,97,101,101,100,103,108,108,103,105,102,109,118,120,125,130,109,102,102,108,101,98,97,99,45,22,44,67,85,81,74,90,100,113,117,124,130,134,142,147,147,152,155,155,152,149,167,172,172,173,188,188,172,178,184,168,147,143,140,
        92,113,112,170,183,107,92,70,55,48,33,23,27,33,34,36,62,144,152,135,119,87,87,87,89,90,88,86,87,87,88,89,87,87,86,114,118,115,130,160,186,184,156,98,102,99,97,97,96,99,96,98,102,103,105,105,101,103,116,140,90,94,96,96,99,100,100,101,101,101,100,99,100,101,105,123,168,149,123,106,130,137,151,162,164,120,103,102,147,113,100,96,99,89,19,28,55,73,68,78,90,97,111,119,127,130,136,144,147,150,156,161,163,155,147,172,172,166,154,157,158,176,197,207,203,166,146,144,
        134,157,154,160,159,158,159,156,151,144,128,87,40,31,38,37,115,155,145,127,125,91,89,90,89,87,87,86,84,86,85,87,87,89,97,117,120,129,135,156,187,194,175,106,99,97,96,94,96,98,95,98,100,101,105,106,102,101,102,141,102,92,95,96,98,99,102,102,102,101,100,99,100,103,103,128,190,197,192,138,148,177,182,186,167,128,103,102,166,138,106,97,97,104,43,24,34,48,62,79,91,98,109,116,123,131,138,145,148,161,172,174,163,155,147,171,181,164,145,141,139,161,204,218,217,181,150,148,
        140,164,163,161,161,161,161,161,159,151,136,126,89,49,35,58,143,149,140,124,126,102,83,91,90,87,86,86,85,86,85,87,91,64,100,128,111,137,141,149,186,195,178,118,95,93,93,94,94,98,96,98,100,100,105,110,104,103,101,112,133,89,95,96,95,98,100,101,103,102,101,100,102,106,105,120,200,207,205,188,177,201,207,208,177,125,104,108,172,149,124,103,97,101,77,18,29,40,55,80,98,101,111,118,125,130,140,141,151,169,187,189,167,159,147,165,198,173,144,127,132,163,200,217,214,180,155,151,
        139,165,166,164,165,164,163,164,162,155,137,119,98,73,46,99,151,141,134,124,120,120,76,86,90,88,86,85,83,91,82,61,90,45,98,100,88,136,138,153,175,181,162,143,122,93,90,92,94,96,97,96,99,101,105,109,113,114,104,98,143,98,91,94,95,97,97,101,104,102,101,101,101,107,107,121,195,210,211,208,191,206,214,214,200,129,104,123,170,150,131,109,100,101,101,26,24,36,57,74,95,108,112,118,126,129,139,143,160,184,196,195,178,157,150,160,156,192,140,125,135,158,190,202,196,146,112,109,
        138,167,168,167,167,167,163,162,159,148,128,107,86,70,51,130,149,136,124,120,114,140,90,82,90,87,88,85,109,95,100,99,102,112,105,121,94,106,103,138,167,166,131,164,146,117,90,89,93,97,97,96,98,100,102,110,127,144,111,100,119,102,89,93,93,95,99,100,104,102,100,100,101,105,109,135,185,204,213,202,193,194,207,207,186,121,111,121,129,124,111,105,98,100,103,43,24,36,58,73,95,111,116,122,123,131,135,146,162,189,202,202,191,154,149,154,147,148,172,133,133,144,149,123,81,92,109,128,
        138,170,170,170,168,165,164,159,153,139,120,98,77,61,54,142,145,133,122,116,111,145,129,73,87,86,84,106,158,134,108,129,123,124,119,109,135,123,110,156,170,156,140,171,165,133,125,91,89,95,97,96,98,99,102,108,118,140,122,102,98,117,89,92,94,96,98,100,102,102,101,100,101,104,118,142,181,201,217,203,172,179,201,199,161,110,116,105,101,103,112,104,99,99,103,69,21,35,56,72,87,108,116,119,125,134,135,148,166,197,208,206,195,158,147,146,158,147,153,185,158,125,87,69,99,115,123,133,
        141,172,171,172,171,167,164,161,150,132,108,81,60,38,54,147,143,131,122,111,111,141,150,83,81,87,93,160,153,133,111,95,127,125,114,116,135,131,99,160,157,162,175,156,155,154,158,131,85,93,96,97,98,100,103,112,125,146,138,109,97,111,90,91,93,96,97,100,102,100,101,101,102,104,122,146,179,195,206,191,150,174,196,187,133,110,116,103,98,103,104,104,100,101,108,87,20,36,62,71,90,111,118,122,124,138,140,149,172,203,213,212,200,158,145,143,157,136,121,156,151,83,84,99,117,125,131,133,
        142,172,171,170,171,167,163,159,146,121,88,64,31,19,65,152,144,136,119,105,121,142,150,120,72,84,140,160,138,114,115,120,125,133,133,120,118,135,122,171,176,174,194,189,172,176,154,159,99,89,93,98,98,99,102,113,146,171,150,112,101,105,96,91,92,95,98,99,101,101,100,99,102,105,124,147,173,189,191,152,129,167,167,141,115,109,117,100,97,101,101,102,99,100,113,103,21,38,57,67,89,109,117,120,130,133,143,145,181,207,215,214,202,154,144,146,141,139,115,88,79,82,89,108,127,127,134,140,
        142,168,170,169,170,167,161,154,133,91,60,31,18,16,84,153,143,138,125,124,138,144,145,146,80,101,150,151,122,111,118,120,131,136,137,136,140,136,129,157,149,151,146,148,153,155,152,157,138,87,94,97,99,101,106,116,172,201,190,132,104,99,99,89,92,92,96,97,100,99,98,102,98,105,121,139,154,162,150,126,123,129,124,117,110,108,120,98,96,104,112,115,100,101,119,101,24,34,56,74,89,103,117,119,129,134,140,157,189,209,215,214,197,158,147,139,140,107,79,76,74,79,103,132,122,127,136,143,
        143,169,168,168,168,164,154,135,97,57,27,16,18,21,105,152,144,137,130,137,143,139,136,147,113,123,149,129,119,119,118,126,135,136,141,141,146,152,165,153,149,153,155,153,158,163,157,156,164,91,92,96,99,102,108,119,178,212,211,161,106,99,99,89,91,93,93,97,99,100,98,97,97,104,114,125,131,123,121,118,119,114,110,108,106,115,102,98,98,112,160,183,111,103,128,104,26,36,60,82,86,103,115,120,128,135,141,161,194,210,215,212,191,156,146,146,138,103,75,70,95,142,194,141,118,126,135,140,
        137,162,162,163,165,160,137,99,50,19,17,19,23,30,124,151,141,130,131,141,138,134,136,146,146,125,142,128,124,125,129,131,134,138,144,145,149,156,172,156,155,156,173,179,175,168,165,158,176,97,94,101,101,104,108,120,180,210,216,186,117,100,97,89,89,93,92,95,97,97,99,98,98,101,105,113,118,112,116,120,114,110,106,106,103,108,95,97,96,114,198,219,155,104,144,105,28,33,59,80,89,108,119,124,133,137,141,162,190,207,215,212,191,150,141,146,141,92,78,102,167,202,208,163,112,129,133,142,
        95,110,116,121,123,119,113,103,68,32,20,25,30,41,133,144,137,122,135,144,137,134,139,146,146,124,134,133,128,124,131,136,136,146,148,148,152,160,169,153,152,164,195,211,202,181,169,160,175,95,95,103,102,102,107,130,172,208,217,199,134,101,97,89,87,92,92,94,96,95,97,99,100,100,101,107,110,110,112,113,107,103,101,101,106,98,97,95,98,131,202,220,183,110,150,109,26,29,57,81,88,106,120,121,124,137,143,162,176,198,215,214,191,159,139,146,132,82,118,185,206,205,143,108,121,132,136,144,
        77,95,94,96,97,106,113,115,120,118,90,49,35,55,134,143,141,120,134,141,131,134,139,144,138,114,127,140,131,128,132,139,143,149,150,147,162,167,167,154,154,167,203,215,197,179,167,160,164,92,99,105,103,103,106,134,164,204,217,210,154,105,96,89,85,90,94,93,95,95,95,98,100,102,104,104,107,108,108,105,104,101,101,101,101,94,97,95,103,154,207,213,175,116,135,101,22,31,53,71,88,100,115,119,126,136,146,158,161,180,209,216,195,172,139,152,110,128,174,205,199,116,94,108,123,133,138,145,
        89,106,102,101,99,107,113,122,122,122,113,75,22,50,132,142,141,121,129,125,123,131,136,144,133,109,120,147,134,127,132,137,144,149,147,156,178,178,174,157,156,168,194,201,181,172,160,158,156,85,99,102,101,102,106,121,164,197,213,215,169,109,101,90,87,92,93,94,93,94,95,98,99,100,103,103,106,107,105,103,101,103,103,103,94,98,97,94,109,178,212,206,159,109,117,87,18,32,52,66,83,100,109,112,118,128,142,147,154,168,202,212,202,186,153,124,120,175,198,205,109,93,107,119,123,131,142,147,
        88,107,111,100,97,107,112,117,121,117,107,52,15,36,134,130,133,126,127,126,123,128,131,136,128,108,115,144,143,133,134,137,141,156,147,166,178,178,172,154,151,168,177,171,165,163,157,157,154,86,98,100,99,101,105,119,154,191,208,211,188,117,103,90,85,89,89,92,92,92,94,97,95,99,102,102,106,107,104,104,105,104,104,95,97,100,96,98,130,197,209,196,129,106,105,66,17,35,55,67,74,93,104,107,112,116,133,144,149,167,200,204,199,181,150,115,162,199,207,126,80,97,110,118,121,134,140,147,
        81,109,169,146,100,100,107,112,117,114,89,27,13,29,132,139,123,130,131,120,122,123,123,127,125,114,113,126,146,137,135,139,143,145,151,168,172,174,169,153,149,155,157,158,163,160,156,158,158,98,95,100,101,99,102,113,140,176,193,201,187,122,106,96,86,82,91,93,92,90,91,94,96,97,101,103,105,106,107,107,104,99,96,98,102,98,98,111,161,200,192,152,110,101,102,38,19,33,55,66,72,87,96,101,104,113,122,139,140,161,193,196,188,173,120,144,182,201,171,63,81,98,107,115,120,135,140,147,
        76,82,105,155,143,101,100,106,110,100,54,14,11,31,96,137,123,129,123,117,118,118,114,117,116,121,115,123,142,139,134,141,142,146,147,158,163,167,164,153,147,152,155,159,160,158,159,158,158,115,93,98,100,99,102,109,132,142,148,155,153,121,107,101,86,81,77,90,90,89,87,92,94,96,100,101,102,103,103,101,99,96,97,100,101,100,112,152,190,189,144,114,102,98,94,18,18,30,56,66,67,77,87,95,103,108,110,125,124,141,167,179,168,135,122,170,195,205,112,61,74,84,95,109,121,130,136,144,
        75,74,83,116,140,102,90,94,86,56,19,10,12,43,85,128,114,115,116,115,112,108,107,105,104,125,117,120,132,132,130,134,137,143,148,150,151,154,161,157,152,154,155,160,159,159,157,157,158,119,87,97,98,98,100,106,115,127,122,132,133,121,111,106,96,90,76,81,89,87,93,100,90,96,97,98,98,98,98,99,100,97,99,99,101,105,146,188,201,168,123,107,99,95,77,12,16,23,42,56,61,66,68,72,86,85,89,97,102,103,114,103,84,118,138,186,205,206,173,177,165,115,84,100,115,124,134,143,
        56,73,72,75,77,73,66,66,51,19,10,10,13,36,101,131,112,112,114,108,103,102,105,94,90,126,123,121,126,129,129,132,139,143,147,146,150,151,159,160,157,156,158,163,160,160,159,159,159,121,85,94,97,98,97,102,109,117,118,123,122,120,114,107,98,94,106,119,139,150,175,130,108,85,91,93,94,96,97,100,100,102,102,99,105,120,170,195,193,153,112,100,96,94,50,9,17,22,34,46,49,50,51,58,50,45,44,84,114,84,62,94,149,150,155,190,206,204,193,183,175,189,184,117,111,120,131,141,
        42,55,51,45,38,43,35,28,17,9,9,10,9,31,102,117,106,104,114,110,99,72,50,40,59,128,124,120,128,127,130,129,134,141,147,147,148,151,157,164,173,173,162,161,157,159,157,159,157,114,82,92,96,97,98,99,102,109,110,115,117,119,112,111,113,125,130,121,122,171,189,178,190,76,82,87,94,94,97,98,97,98,101,102,108,145,179,180,157,123,104,100,96,97,29,41,14,18,25,28,31,39,40,35,22,24,120,167,167,126,109,149,165,157,184,192,202,197,192,193,178,183,204,204,123,110,123,133,
        28,33,35,26,23,23,15,12,10,7,9,23,65,80,71,46,40,91,122,99,81,63,60,49,73,129,125,123,130,130,131,128,135,139,142,143,147,149,161,173,181,179,164,159,158,159,158,157,156,107,83,90,95,96,97,99,100,102,103,108,109,114,111,110,107,89,95,94,103,145,184,182,193,107,84,91,97,100,101,99,98,100,99,107,118,154,184,173,132,110,102,97,95,95,108,125,16,16,18,17,22,24,24,18,13,108,159,169,158,155,165,137,162,148,182,192,200,200,189,184,182,201,205,205,186,101,117,126,
        17,16,20,15,15,11,12,10,9,22,94,118,75,51,45,39,65,103,97,87,64,57,58,48,90,132,124,122,128,128,130,131,134,138,140,141,146,149,154,168,181,171,163,158,161,160,158,158,155,99,78,88,93,94,94,97,100,102,101,104,110,111,111,110,107,99,73,75,86,108,172,189,173,154,75,88,96,100,100,99,99,102,107,112,128,138,148,149,129,106,99,96,97,105,142,96,11,16,17,15,16,14,14,14,41,147,158,161,162,169,172,159,163,141,191,182,189,187,186,177,193,204,205,205,209,112,107,119,
        73,61,33,12,7,10,7,8,70,137,103,66,56,52,49,57,100,104,89,67,59,53,54,49,104,135,126,122,128,128,129,132,136,140,142,142,146,150,157,165,179,166,161,160,162,161,160,158,156,91,81,88,91,91,91,94,99,100,101,102,109,114,111,109,108,107,110,78,91,97,149,193,166,189,81,86,91,97,99,99,99,104,105,114,129,133,131,127,111,103,100,99,99,98,148,106,10,13,15,15,14,13,14,12,89,149,156,158,166,176,186,187,180,168,178,184,192,198,201,202,202,203,205,206,208,123,105,117,
        104,116,114,104,67,27,17,113,143,105,66,51,52,51,53,93,107,94,73,54,54,53,50,45,111,132,128,121,127,127,129,130,137,138,142,139,145,151,159,166,164,162,160,160,163,163,163,157,155,82,82,85,88,90,93,96,96,98,98,100,104,110,110,110,111,109,106,91,99,109,129,184,181,196,118,77,86,92,92,96,98,100,102,106,114,112,108,102,101,101,104,101,96,53,141,122,10,10,12,10,12,13,14,13,113,149,153,155,166,180,187,187,186,181,183,189,195,201,206,207,206,205,204,207,209,116,99,106,
        109,126,121,113,105,94,115,134,107,83,63,56,51,48,62,102,93,74,60,55,55,56,51,44,123,134,137,129,126,128,129,132,132,139,139,139,148,154,162,166,164,163,159,160,162,165,163,157,153,76,78,84,86,87,89,92,94,98,100,100,103,105,108,112,108,107,100,91,97,112,113,167,189,172,169,68,79,86,90,93,94,98,98,98,98,99,98,99,99,104,110,103,49,24,77,94,64,8,10,10,10,14,12,14,122,146,149,154,162,177,184,186,184,182,186,191,196,202,209,215,214,209,205,206,210,102,97,106,
        112,128,125,120,100,113,135,104,88,72,59,57,56,58,82,93,79,63,56,54,57,55,50,51,129,134,141,138,125,124,127,127,131,137,140,138,148,155,163,163,164,165,159,160,162,166,160,152,149,68,74,80,84,87,90,95,96,97,99,101,105,110,111,111,109,102,98,94,87,116,108,148,197,157,197,77,75,81,88,89,93,95,95,95,96,95,97,99,100,107,106,71,9,13,38,75,81,43,10,7,7,10,11,14,123,143,146,150,158,170,179,179,181,181,184,189,197,203,208,211,211,209,205,205,210,86,84,98,
        113,129,127,121,103,132,116,93,78,62,54,52,56,62,88,79,65,58,60,56,56,56,54,64,135,137,143,144,133,126,127,127,128,133,135,139,148,152,160,165,163,164,159,160,163,163,159,153,145,67,72,77,81,85,89,92,95,99,101,102,104,104,105,103,98,106,104,100,100,119,105,132,186,165,192,112,70,79,85,85,89,95,100,100,97,99,96,93,91,102,81,15,33,76,113,121,116,101,72,46,20,10,10,14,121,140,142,146,152,164,170,169,172,176,179,187,193,198,204,206,208,206,205,205,204,63,80,96,
        113,127,124,97,126,125,105,90,73,63,59,58,62,73,79,67,58,59,58,59,54,54,48,77,136,147,145,141,138,133,128,126,130,131,134,144,147,151,159,161,162,161,159,161,162,158,158,152,142,60,70,74,75,81,84,92,99,95,96,98,99,99,95,85,101,110,119,127,137,111,119,119,176,180,170,159,65,74,73,82,95,110,113,109,121,139,160,144,109,76,21,33,81,125,132,135,132,131,122,109,86,34,11,10,105,134,140,142,149,160,166,165,170,171,176,183,187,194,202,205,205,205,206,206,189,42,78,93,
        112,121,97,116,129,114,94,75,65,60,60,59,61,65,67,62,60,58,57,56,56,53,46,90,139,150,150,141,137,137,139,137,134,134,135,143,146,151,156,159,162,160,162,165,161,157,160,152,137,57,66,70,74,78,77,83,88,89,89,89,86,76,66,86,112,120,129,138,151,118,126,124,163,191,156,195,84,65,70,85,104,107,93,95,115,135,133,136,149,132,45,39,98,126,130,133,137,138,133,125,114,82,30,10,97,132,136,138,143,153,159,162,164,166,169,175,179,189,199,203,204,204,204,205,155,23,49,73,
        114,100,111,124,120,95,75,63,56,60,60,62,61,67,65,60,59,57,56,56,56,51,42,104,132,149,148,141,137,137,139,135,138,136,138,144,147,153,156,159,159,162,165,163,161,160,160,151,128,48,60,62,65,66,68,69,69,67,63,68,64,60,85,115,117,121,132,143,157,123,118,110,147,190,151,199,126,73,83,90,107,92,85,95,126,131,134,141,145,148,147,99,94,121,134,134,138,137,134,133,123,108,73,15,78,125,129,134,140,146,151,155,159,163,165,170,174,182,194,200,200,200,200,201,100,18,31,57,
        91,118,114,118,100,78,62,56,51,56,56,55,58,61,61,62,58,58,59,56,57,56,43,124,137,148,148,140,138,140,137,135,137,137,142,148,148,156,155,158,161,161,159,158,160,159,158,154,104,38,45,47,48,47,48,42,48,55,68,65,45,76,104,125,130,128,127,146,153,127,117,111,125,187,167,184,167,68,80,95,100,85,83,102,114,125,136,140,140,145,148,149,129,110,130,137,143,140,139,138,134,121,96,33,51,124,129,132,137,140,146,150,153,158,161,164,167,175,187,193,192,192,191,189,48,45,122,121,
        97,116,120,108,87,72,59,58,54,54,50,57,62,65,62,60,57,57,57,57,54,56,55,131,148,151,146,139,137,137,137,137,139,140,147,149,152,157,158,161,160,160,159,161,162,162,157,158,82,26,30,34,37,41,53,71,85,91,99,101,103,130,121,125,137,143,133,142,150,135,118,120,118,164,173,181,197,80,76,87,90,81,89,105,106,126,133,141,151,155,157,156,152,132,128,136,143,148,153,161,148,131,110,53,25,103,121,130,132,135,140,145,146,152,155,156,162,169,178,178,176,175,169,144,104,177,146,87,
        100,121,116,94,78,67,57,54,52,54,53,56,60,63,60,59,56,56,55,56,55,57,75,133,151,151,148,139,136,139,138,138,140,144,148,150,154,159,159,163,161,159,159,162,164,161,157,157,76,55,53,56,54,74,89,102,111,112,106,89,73,93,135,118,124,122,141,158,172,153,109,120,108,154,177,171,195,131,71,112,93,78,88,99,110,116,124,142,148,158,163,161,151,154,135,134,141,157,179,197,187,143,123,77,19,70,102,113,127,127,131,134,138,141,144,144,150,153,159,159,156,154,144,161,166,122,87,79,
        104,120,105,87,77,69,61,57,56,58,57,58,58,62,60,59,57,58,54,55,54,50,91,131,151,147,144,138,137,140,138,140,141,144,148,150,155,161,160,164,161,161,160,163,164,161,153,149,66,65,65,62,94,109,116,119,124,124,140,120,97,89,91,133,59,56,114,141,158,141,103,119,106,133,177,175,198,178,85,140,139,74,91,106,112,117,131,143,156,159,158,152,143,136,141,137,143,162,194,204,202,185,134,102,30,43,88,99,114,122,124,126,130,132,129,127,129,133,134,132,127,137,154,146,98,85,86,79,
        101,113,98,86,76,64,57,58,56,58,57,60,61,60,61,62,61,57,59,56,56,49,103,134,151,150,143,138,139,141,140,143,143,147,148,153,156,162,160,166,162,160,161,164,165,163,153,135,60,62,64,102,117,119,129,129,128,132,136,130,123,104,71,78,52,60,106,94,115,112,121,129,122,134,170,184,184,188,127,139,89,116,105,80,112,124,134,150,138,117,106,106,124,132,124,121,128,160,187,209,211,208,160,116,39,22,67,95,108,113,116,116,120,118,112,108,107,106,94,93,122,135,133,104,89,89,86,80,
        91,102,92,82,71,63,60,62,62,60,59,61,62,65,61,64,65,64,62,60,59,46,104,134,150,147,143,140,141,139,141,144,145,148,150,153,158,162,159,165,161,160,159,164,164,163,148,103,45,57,98,115,120,125,149,148,138,139,141,138,141,146,107,50,48,66,52,39,64,109,121,130,120,132,153,194,191,198,164,128,57,124,129,109,109,121,132,108,82,76,82,92,114,131,128,128,129,126,149,198,210,210,169,125,59,15,47,82,96,103,103,101,104,101,95,95,94,85,82,105,110,99,92,84,84,87,89,82,
        87,88,78,74,70,68,63,60,60,59,60,61,59,64,62,63,66,65,64,60,56,46,108,133,150,147,143,141,142,141,144,144,146,147,148,152,156,159,157,162,162,160,160,164,164,162,139,64,32,76,111,118,124,138,152,147,145,146,147,151,159,174,155,54,72,42,26,20,107,113,128,135,129,136,159,184,196,195,175,155,93,119,149,98,89,112,148,62,64,73,84,100,116,124,131,136,139,138,136,142,199,217,169,129,77,12,26,67,84,92,89,91,90,86,81,83,78,79,91,94,85,78,83,86,90,91,90,91,
        74,81,79,76,73,74,68,63,59,60,60,60,60,65,63,64,62,62,60,56,58,49,109,133,144,146,141,140,142,141,144,147,148,148,149,153,157,160,160,163,161,160,161,164,163,163,118,36,37,101,113,123,127,139,141,146,150,153,155,162,171,185,144,89,115,60,31,58,118,121,123,133,127,137,156,185,198,197,183,165,155,89,122,98,137,111,85,58,71,77,90,109,118,127,136,139,143,145,143,146,145,202,169,131,93,13,17,51,76,79,79,83,75,71,69,62,72,86,85,84,78,82,87,89,93,94,93,91,
        70,82,83,78,76,75,72,69,62,59,58,58,64,68,70,63,63,61,59,60,60,56,113,134,145,144,142,140,140,140,143,149,149,149,151,155,159,161,160,162,159,160,160,163,163,158,78,22,66,111,119,125,131,137,138,142,147,151,162,165,157,137,116,172,149,70,29,51,114,133,136,144,132,137,161,180,199,187,175,176,168,105,138,140,120,101,58,69,77,87,98,111,118,137,137,145,150,150,151,146,143,154,161,131,92,14,17,35,63,69,70,73,68,62,56,59,78,79,78,78,82,87,92,92,93,93,94,91,
        66,82,86,83,78,77,76,73,68,60,60,61,66,73,72,65,60,59,60,60,62,64,113,134,139,142,141,139,140,142,144,148,149,152,151,157,163,163,161,162,158,159,160,164,162,144,40,19,91,115,127,128,131,125,123,127,132,138,140,148,140,67,37,125,165,155,124,55,85,121,155,146,139,148,154,164,188,179,157,172,184,113,135,146,132,141,68,82,91,92,93,104,111,124,131,139,142,148,154,153,146,148,136,124,83,12,18,33,53,61,65,63,60,54,54,73,75,80,78,83,87,87,90,91,91,95,94,93,
        64,81,79,84,80,80,76,77,74,66,63,62,69,77,77,67,60,62,61,63,63,58,118,135,143,144,143,140,142,143,146,150,150,151,154,159,162,161,160,160,160,161,161,164,161,119,22,28,103,125,127,127,128,123,122,120,134,140,144,147,156,162,89,27,93,137,140,147,109,143,144,107,125,148,157,172,191,190,172,148,109,137,152,154,147,106,56,93,98,101,106,109,118,129,138,140,146,149,152,156,153,143,145,120,65,14,14,22,41,51,53,53,46,44,67,69,77,79,86,85,89,88,90,90,93,92,97,91,
        61,79,72,75,84,83,78,78,79,76,66,66,72,81,74,62,60,63,60,60,55,58,119,137,148,145,141,143,145,144,147,150,152,153,157,159,161,160,161,161,160,162,161,162,157,87,16,46,114,131,134,135,134,128,129,130,132,136,142,147,155,161,175,96,22,60,94,114,131,139,148,126,155,175,178,158,148,166,153,88,111,128,173,148,74,12,56,98,100,94,93,113,124,131,142,144,153,160,163,162,156,153,145,120,44,12,13,18,27,38,45,43,34,62,70,77,77,82,85,86,88,86,89,89,89,92,95,94,
        61,76,74,68,75,81,80,79,81,76,67,64,68,79,71,64,65,63,60,57,53,64,121,138,152,142,141,140,143,144,146,150,153,154,156,158,158,160,161,162,161,161,161,159,142,58,17,90,125,139,140,141,141,129,127,127,130,124,123,123,120,120,115,131,134,140,137,125,97,108,155,142,138,137,154,145,168,165,147,113,162,156,95,79,145,39,43,78,94,103,114,124,127,130,139,147,155,162,174,177,169,155,145,132,32,12,13,17,21,26,28,26,48,74,78,77,77,81,86,87,91,90,90,91,91,93,97,107,
        61,69,76,73,71,74,77,79,81,78,68,63,65,72,65,61,59,57,56,47,43,67,122,139,150,142,140,143,143,145,147,148,154,155,157,159,157,160,161,161,160,159,158,152,120,35,32,108,134,139,140,139,131,124,128,129,128,126,117,112,109,110,119,131,133,132,135,127,124,130,130,140,143,127,83,36,57,109,106,106,108,114,127,133,133,139,139,136,135,132,126,130,131,134,136,144,153,170,182,188,172,159,148,142,29,12,14,15,17,18,20,28,81,81,79,79,81,85,86,89,91,91,90,91,91,91,108,119,
        63,67,65,68,67,69,73,73,77,75,67,62,63,69,66,62,61,55,51,46,41,77,121,145,150,144,140,144,140,144,148,151,154,156,158,160,159,164,163,160,159,157,154,139,93,20,50,112,145,148,135,127,121,124,122,128,129,121,123,118,122,125,125,125,130,129,130,125,126,130,133,132,137,143,156,66,12,62,86,92,97,111,126,135,141,134,138,138,137,140,141,135,136,137,139,144,168,160,182,193,182,162,149,144,42,12,11,14,13,14,16,72,81,78,79,83,84,85,89,91,91,93,92,94,93,95,122,149,
        66,65,62,61,65,66,69,72,74,71,68,67,63,63,65,60,57,51,50,43,40,85,121,149,147,144,141,141,141,144,149,151,153,157,159,159,160,163,160,160,158,156,146,126,67,17,71,117,147,145,127,117,123,120,129,127,125,127,123,124,123,122,126,122,129,129,132,135,138,136,136,133,143,144,148,146,108,82,101,102,105,112,123,131,140,138,139,144,142,141,145,146,144,144,140,145,152,166,181,187,180,161,159,142,60,11,12,12,12,12,48,84,76,76,80,84,85,86,87,90,96,95,95,95,97,111,163,182,
        69,64,61,62,69,62,64,69,71,69,68,65,65,62,64,63,57,53,48,42,35,96,122,151,146,140,141,140,142,145,149,150,154,156,159,159,160,163,161,159,156,155,138,108,44,21,85,118,140,134,119,120,120,121,126,124,122,115,114,119,116,121,119,123,129,125,130,134,138,136,134,139,149,158,153,146,114,78,100,104,108,116,116,126,136,141,142,148,151,153,157,164,171,169,167,155,152,164,179,184,181,167,171,142,73,12,11,14,13,25,76,69,71,74,81,85,85,87,90,89,95,97,97,97,108,152,195,203,
        62,67,62,66,65,64,64,67,70,68,65,65,64,63,59,58,54,49,44,41,30,106,125,148,144,140,140,141,143,146,150,151,154,156,158,158,157,161,158,157,154,143,125,90,35,30,93,115,128,119,119,119,121,121,120,123,120,118,120,124,125,123,124,127,127,123,126,134,134,130,136,146,174,195,182,165,142,84,94,103,114,120,122,124,133,143,151,153,177,193,194,200,204,204,199,173,155,156,177,179,176,167,171,139,80,11,12,12,16,69,70,68,69,75,79,86,87,88,89,90,92,95,97,105,117,178,208,211,
        64,68,64,68,65,58,52,50,67,67,65,62,62,58,57,55,51,48,41,41,33,118,127,146,143,140,140,139,143,148,149,152,154,156,159,156,160,160,157,155,149,137,114,74,28,39,97,110,118,114,120,118,119,122,120,123,121,120,122,124,126,126,130,131,130,123,130,135,132,135,141,155,197,213,212,203,175,94,95,102,112,117,112,120,132,140,145,150,179,201,216,216,215,212,208,188,159,159,171,174,168,162,162,138,85,12,13,13,50,70,63,67,67,73,78,87,91,91,90,94,94,96,97,119,144,191,212,210,
        64,66,66,67,65,63,57,44,51,66,65,64,60,54,41,38,43,41,34,31,41,118,127,146,144,140,141,142,143,147,150,151,155,158,159,157,169,168,155,152,142,126,100,58,22,50,94,108,111,117,121,121,121,122,125,124,120,122,124,125,126,127,133,134,130,129,129,128,129,136,145,164,198,214,216,214,209,105,99,108,112,109,109,114,126,139,150,158,171,196,213,213,211,207,197,175,152,154,161,172,161,156,153,137,82,14,17,25,68,62,65,67,69,75,79,87,92,93,90,91,94,95,99,131,174,199,215,202,
        67,68,62,68,68,71,60,49,37,51,60,61,58,43,30,28,48,60,56,30,46,121,129,144,144,141,141,143,144,149,147,151,154,157,159,160,163,159,151,147,133,118,92,50,19,59,97,105,110,120,122,120,123,125,127,120,125,125,120,122,123,127,132,134,130,127,131,124,130,140,164,180,197,209,210,214,211,120,90,106,102,101,107,112,121,130,144,157,171,186,200,202,199,191,175,157,148,152,156,165,155,156,148,140,77,17,21,47,62,62,64,70,73,78,81,88,95,94,94,94,97,98,104,134,188,207,212,182,
        67,69,67,66,70,78,67,53,41,25,42,52,50,52,74,89,82,92,70,25,65,113,129,143,142,138,141,142,146,148,148,151,152,154,154,153,155,152,148,141,127,110,86,49,20,60,94,103,111,121,121,122,127,131,126,121,122,123,121,124,126,128,130,131,127,122,126,129,137,153,178,181,188,196,203,205,204,138,86,96,94,97,103,108,113,121,132,139,151,165,174,181,181,167,152,152,150,145,152,156,154,152,145,141,71,14,17,56,55,63,68,74,78,82,85,89,96,96,94,93,98,101,111,144,188,201,195,143,
        68,69,65,64,67,72,63,52,38,29,37,75,96,72,33,54,80,50,17,15,87,119,132,143,142,141,141,142,146,149,148,152,153,155,153,150,153,152,146,138,126,110,84,46,23,67,95,106,114,123,125,127,131,130,126,122,122,121,121,123,127,129,128,126,119,117,124,134,143,157,166,170,167,166,176,190,189,153,87,90,91,92,98,101,106,117,121,126,130,139,144,156,155,151,151,152,149,144,147,149,148,146,144,143,58,14,25,50,50,57,65,73,79,82,84,89,95,94,95,96,100,105,122,164,186,189,155,103,
        69,70,68,66,65,66,61,51,34,63,75,49,42,60,55,36,68,46,28,25,98,117,134,143,143,142,141,143,148,148,148,152,153,153,151,148,152,151,143,132,120,106,82,43,23,66,94,99,107,120,124,127,132,132,129,126,124,122,121,123,125,127,127,122,113,119,125,132,138,146,142,151,149,154,163,168,167,149,89,86,88,89,96,99,106,115,117,123,122,127,133,137,146,152,148,148,148,143,146,146,142,142,143,139,43,12,27,45,52,57,65,74,80,83,84,91,96,94,94,96,101,107,137,173,183,173,120,97,
        62,67,68,67,65,60,55,41,77,92,45,35,80,106,91,84,107,63,24,25,114,121,136,145,142,142,140,143,148,148,148,152,152,153,149,147,148,148,140,130,120,107,84,41,26,66,90,101,104,110,118,121,125,129,127,125,123,121,120,120,121,119,122,115,107,119,126,130,133,137,135,138,141,143,146,148,148,133,96,83,85,87,97,98,105,112,113,120,119,124,130,137,144,146,147,146,145,138,141,142,139,140,142,140,35,13,37,41,50,60,68,73,78,81,88,95,94,92,95,97,102,115,146,172,171,142,103,95,
        56,63,64,63,65,60,54,56,77,40,30,83,92,57,44,38,53,41,26,36,118,123,140,145,145,142,144,146,148,149,151,151,153,151,148,146,149,145,136,126,120,107,81,32,25,66,89,100,100,104,105,109,116,122,126,123,124,123,118,118,119,117,120,111,106,112,122,128,128,134,133,134,134,135,134,139,136,120,97,86,88,94,99,99,106,107,112,113,119,124,128,137,142,143,143,143,141,134,133,141,139,141,139,133,26,14,39,42,54,60,71,74,79,83,88,94,94,93,92,97,103,118,154,164,153,118,98,93,
        53,60,61,55,54,51,44,80,50,36,38,96,38,113,86,63,68,27,47,38,122,125,143,147,144,142,141,147,148,148,151,150,153,151,145,144,149,142,135,125,118,106,82,27,24,54,87,97,97,102,98,111,100,111,115,115,119,114,114,115,114,114,113,106,107,110,118,125,125,131,130,127,126,126,130,132,130,110,99,87,89,91,96,94,98,106,110,111,116,120,127,134,137,141,140,144,139,134,132,138,140,141,138,126,21,16,37,45,55,61,68,71,75,83,90,91,93,92,94,98,103,121,157,155,153,116,98,90,
        56,62,59,55,52,50,80,74,42,29,80,80,44,103,108,88,66,34,33,40,124,126,145,148,144,143,144,148,149,149,149,152,152,153,145,142,146,141,130,123,116,106,76,25,22,42,81,96,97,97,96,92,95,98,103,109,106,104,103,107,106,102,102,105,108,107,110,121,125,129,132,127,127,124,123,126,122,110,98,93,88,94,98,95,100,106,108,111,115,117,123,127,129,133,138,137,137,131,132,139,135,138,130,110,19,20,37,48,56,64,70,73,77,85,90,90,91,93,94,98,101,116,149,156,148,106,95,88,
        68,79,67,60,61,56,48,37,25,28,99,91,68,94,106,95,82,72,34,60,123,127,145,145,143,143,143,147,149,149,151,152,150,146,144,149,149,140,131,127,119,110,80,31,22,36,76,94,97,96,91,86,88,90,97,104,97,98,97,99,98,101,98,102,102,109,111,118,126,129,131,126,123,125,127,126,121,109,101,97,95,106,101,96,99,106,107,109,115,118,124,127,127,128,133,137,136,130,133,137,136,136,134,86,17,20,34,48,59,64,66,73,77,82,88,90,92,90,94,100,103,114,141,145,121,98,91,87,
        64,70,67,64,66,63,62,62,28,42,77,71,55,82,103,99,93,89,42,72,123,128,145,142,142,140,143,149,149,150,152,152,152,147,144,151,145,141,130,126,121,109,73,29,21,28,68,91,101,99,94,90,88,91,95,98,92,93,94,95,99,99,101,98,102,103,108,119,125,126,127,123,118,120,125,125,113,107,102,102,97,104,102,100,101,107,107,111,116,114,121,124,123,126,129,133,130,128,130,132,133,136,133,66,16,20,34,54,61,66,69,75,79,84,88,89,90,90,91,99,100,109,132,122,104,92,88,84,
        52,59,62,61,60,55,47,50,39,28,66,47,75,89,99,100,109,86,47,81,124,126,146,143,141,141,144,149,150,150,151,151,149,142,140,143,145,139,130,127,120,112,69,28,21,24,58,85,98,99,96,90,87,87,96,98,95,92,88,91,98,101,102,100,102,103,110,115,123,129,126,124,126,124,123,120,111,103,97,97,97,104,103,102,103,108,104,106,116,117,120,123,125,126,134,131,130,131,130,133,134,130,135,55,16,22,29,51,62,67,69,75,78,82,85,87,88,90,93,96,100,106,114,111,101,95,86,77,
        46,50,57,58,58,54,44,40,49,45,49,40,65,90,94,109,90,62,95,103,122,125,146,144,143,141,145,149,149,150,151,151,147,140,140,142,142,139,131,124,119,110,60,24,20,20,39,74,91,96,100,97,95,91,99,107,100,93,89,93,98,106,105,100,105,102,106,114,115,125,127,125,127,127,124,119,105,96,92,96,100,106,104,100,107,108,106,109,115,118,123,124,120,118,121,128,131,125,129,129,131,132,138,44,16,27,25,48,64,73,72,75,79,86,87,86,88,88,91,95,97,99,103,103,100,91,82,70,
        28,38,48,48,44,45,34,27,35,32,28,53,54,68,64,75,67,44,57,107,124,125,144,143,143,143,145,150,150,151,153,152,146,138,138,142,143,140,130,127,121,111,54,25,21,21,29,68,81,90,99,98,95,92,99,105,101,90,90,92,101,110,110,104,101,99,102,108,115,124,122,129,126,130,125,117,102,92,87,92,94,102,102,102,109,108,108,110,113,117,117,121,120,120,118,125,127,125,129,131,133,137,137,32,16,25,18,45,63,70,73,72,77,85,87,88,88,87,90,92,96,96,99,101,99,91,83,72,
        21,27,20,25,26,29,32,34,53,47,41,59,53,62,68,62,39,25,18,109,125,128,144,144,143,143,146,150,149,151,152,152,148,141,138,142,139,138,131,128,121,103,46,23,20,22,23,54,73,77,94,100,104,99,99,107,104,91,87,92,104,110,110,105,100,94,98,101,110,119,118,123,123,127,123,117,98,89,90,91,95,103,105,106,112,111,109,110,113,114,114,117,117,118,117,120,125,126,132,131,133,141,129,22,13,18,21,38,64,70,75,75,77,82,86,88,88,89,91,92,93,92,93,96,93,86,79,69,
        23,21,16,16,11,26,42,51,58,62,55,35,48,58,53,42,46,31,28,123,124,127,143,143,141,144,145,149,149,151,153,152,148,138,137,143,143,139,133,129,123,102,42,20,22,20,22,35,69,71,80,93,101,101,98,104,107,97,84,90,101,106,109,105,99,94,92,100,106,124,122,124,127,125,121,113,100,86,85,89,93,95,103,101,104,106,106,107,114,112,110,112,121,119,118,122,127,128,132,134,139,138,127,22,13,17,25,20,60,66,73,76,77,78,82,84,85,89,88,92,92,89,91,92,88,82,77,70,
        36,28,21,18,13,15,37,44,51,53,54,55,82,78,41,28,43,38,45,127,128,126,142,147,142,145,147,148,149,151,151,153,149,138,140,142,146,140,133,128,124,93,37,19,21,20,23,27,58,69,74,84,97,104,101,103,108,104,94,89,102,106,108,106,103,93,96,98,102,111,121,122,125,124,120,110,97,86,88,93,90,94,96,100,102,106,103,108,112,112,114,115,118,118,118,126,127,129,134,136,143,141,127,17,13,15,26,14,42,63,76,78,81,80,82,84,86,87,90,89,90,87,89,89,85,82,74,69,
        46,47,47,38,22,14,21,37,41,41,50,47,74,45,33,60,67,52,55,131,127,126,141,146,142,145,148,148,150,151,152,152,146,137,140,145,148,144,135,130,124,83,30,20,20,21,21,20,42,70,73,76,87,101,97,92,102,104,95,86,93,106,110,108,110,101,97,96,103,107,115,122,122,118,117,105,90,83,85,85,85,92,93,90,91,99,98,105,109,110,114,116,121,119,122,128,132,134,135,135,141,142,127,21,17,19,31,24,16,55,73,77,80,84,84,84,85,88,88,87,90,88,86,85,82,76,70,64,
        56,63,67,66,56,36,17,25,33,35,39,39,31,22,37,70,76,47,68,129,127,127,140,146,143,146,147,148,150,152,152,150,146,140,141,148,151,144,134,129,125,69,23,19,21,22,24,20,36,62,70,76,85,96,96,90,95,104,103,85,91,102,109,108,110,109,100,91,95,103,105,116,119,121,120,109,92,86,88,89,89,94,100,98,91,98,98,104,108,113,112,114,125,121,122,128,132,133,133,137,139,139,130,26,16,18,27,31,14,20,61,77,81,83,83,82,83,83,81,84,84,81,82,79,73,71,67,63,
        50,66,73,78,74,64,49,24,20,29,32,34,29,22,28,35,40,29,87,128,127,127,138,143,141,143,148,148,151,152,152,150,141,142,149,150,149,146,138,132,123,58,18,21,21,21,22,21,28,53,68,78,82,94,99,89,94,98,107,94,83,93,105,111,110,110,101,94,97,99,107,112,117,124,126,118,98,90,86,85,90,99,101,100,96,101,98,98,101,107,106,115,124,121,123,130,133,134,133,139,137,138,128,31,16,22,32,44,24,19,20,58,76,85,85,84,82,85,82,82,81,79,73,71,69,70,68,66,
        52,70,72,88,87,76,67,62,38,20,25,26,23,25,38,41,40,39,107,131,126,128,136,139,138,147,148,148,152,153,155,154,146,147,156,153,150,145,138,135,117,44,18,21,19,20,21,21,26,45,66,77,84,95,101,93,90,99,106,103,87,87,98,106,109,112,104,99,91,93,104,111,117,120,127,119,103,92,85,82,90,99,102,103,108,110,109,104,105,107,111,113,115,117,123,127,134,133,133,132,133,135,127,25,17,21,41,58,45,28,18,16,39,69,82,85,82,82,81,78,75,69,66,70,70,76,70,67,
        48,58,68,87,94,86,68,70,67,49,22,18,18,23,37,37,35,33,118,134,127,130,136,136,139,145,148,148,152,155,166,174,157,154,161,156,150,145,137,135,107,31,21,21,19,20,21,22,23,40,61,67,84,86,93,99,96,100,105,100,91,80,90,101,108,111,104,99,91,94,100,110,120,122,125,123,102,90,87,84,88,98,100,101,111,115,114,111,112,110,117,117,119,122,127,125,133,135,135,139,135,137,125,20,16,27,50,74,60,45,20,14,14,14,30,45,56,64,67,65,64,65,66,68,71,75,72,65,
        68,76,68,66,59,61,62,59,64,59,41,16,15,19,25,29,29,37,127,136,129,131,136,140,140,145,148,148,152,150,152,148,142,152,156,153,151,145,138,135,86,20,18,21,20,20,22,22,23,34,58,69,84,91,94,93,98,100,105,104,88,79,85,97,105,103,107,95,88,91,93,105,114,116,118,118,95,84,90,88,88,95,97,102,110,114,112,116,118,116,115,117,120,119,123,124,133,139,137,129,134,135,121,12,15,20,44,74,75,73,28,13,12,12,18,27,40,57,62,65,69,70,68,69,68,68,71,72,
        82,93,86,83,81,78,69,57,50,47,39,26,12,14,21,22,24,46,133,135,125,132,138,141,143,144,148,148,150,151,151,146,141,146,153,154,153,148,140,133,60,17,19,22,22,20,23,23,23,28,57,67,78,88,93,92,97,99,103,104,89,79,84,94,103,106,108,99,95,87,90,100,113,114,113,106,91,81,85,91,88,96,94,102,106,111,112,115,119,119,120,124,126,122,127,126,136,142,138,134,136,133,110,12,13,16,30,59,77,108,64,14,14,15,29,44,58,66,65,70,72,77,70,69,70,69,74,68,
        76,87,82,84,82,79,79,75,73,71,67,68,65,66,69,69,59,82,134,137,122,133,141,139,143,145,146,147,149,150,152,149,143,148,154,154,150,147,140,120,36,17,20,21,22,25,23,26,28,32,59,71,77,87,95,91,94,104,105,98,86,79,80,91,98,107,107,101,93,91,88,99,111,112,107,102,89,80,81,89,92,95,98,104,112,112,115,120,122,123,123,126,126,123,124,125,137,141,133,131,134,133,93,11,12,13,18,38,55,80,119,13,14,23,45,52,62,68,72,71,74,71,70,69,70,66,68,65,
        95,110,110,104,101,96,92,87,86,82,87,85,81,79,77,75,67,102,137,139,118,132,137,139,140,144,145,147,149,150,151,147,145,151,154,153,150,148,141,93,22,19,18,21,20,22,25,24,24,28,51,65,75,80,85,88,86,97,103,93,78,82,81,88,95,107,108,102,93,91,95,101,106,106,107,104,80,80,80,82,91,94,99,105,108,110,113,118,118,123,119,124,124,125,126,126,136,137,134,132,128,130,71,9,11,13,11,19,29,27,77,16,14,27,45,57,64,69,76,79,77,77,73,72,70,68,68,68,
        111,131,129,126,118,110,102,98,92,89,83,85,80,80,78,75,68,115,137,133,117,132,135,138,139,144,145,149,150,150,150,147,147,152,155,153,150,146,135,65,16,18,20,19,21,24,25,27,30,30,42,62,71,79,85,84,86,94,104,95,78,85,88,91,102,107,105,97,92,88,91,99,104,111,107,99,85,76,80,82,90,96,94,106,108,112,112,117,115,116,113,119,122,123,123,124,132,133,132,130,128,131,41,12,11,10,12,12,16,17,20,15,13,35,53,63,70,76,80,80,79,75,73,72,66,67,67,67,
        120,144,141,136,132,125,119,114,104,100,93,90,85,84,83,81,76,124,135,126,121,132,134,139,142,146,147,149,150,151,151,146,148,153,155,153,152,147,127,38,17,20,20,22,23,25,29,31,33,35,34,59,63,77,79,82,87,95,102,103,88,87,95,97,104,108,108,103,98,93,96,102,109,110,105,91,85,81,86,80,83,87,91,100,103,107,107,112,115,121,118,119,123,119,119,124,134,132,134,132,127,121,15,13,12,13,13,13,13,13,16,14,15,36,57,64,75,83,83,79,76,75,74,71,69,67,71,72,
        125,150,148,147,141,135,128,124,115,109,102,93,88,83,80,78,82,124,131,115,127,135,132,135,143,147,149,150,150,152,151,149,151,157,156,154,154,145,106,19,17,21,22,23,25,26,29,35,33,33,33,54,62,79,84,82,84,94,101,99,90,83,91,95,103,109,110,101,97,96,98,104,110,110,104,92,88,83,87,80,85,89,91,92,94,99,105,111,113,117,116,117,120,115,117,124,136,135,133,129,122,106,11,13,13,12,12,13,13,14,13,14,14,35,58,66,76,87,83,81,75,73,71,67,65,67,72,75,
        130,155,153,151,146,139,135,129,125,120,116,100,89,80,75,70,94,120,125,111,128,131,132,135,141,147,149,150,151,151,148,151,155,157,156,155,152,144,74,17,19,23,25,27,27,33,30,33,33,39,36,45,60,71,83,83,83,92,101,100,89,84,91,99,102,109,114,107,102,101,104,106,112,110,98,88,73,72,80,72,75,72,80,78,85,90,98,106,113,116,115,116,119,118,121,131,138,134,131,128,110,44,11,11,13,11,10,10,10,10,12,12,13,31,56,66,74,78,79,82,77,72,70,70,71,73,77,77,
        135,156,154,154,153,144,139,135,127,123,117,101,94,80,77,63,106,121,124,114,128,129,130,135,142,147,149,150,151,152,150,153,154,157,157,154,150,142,42,22,25,27,28,31,31,33,33,34,36,38,39,40,55,66,78,81,84,90,97,104,90,81,91,96,104,109,111,107,104,100,98,103,108,111,104,98,74,69,75,73,68,69,75,77,76,85,96,105,109,110,111,117,117,116,121,133,132,130,127,124,100,16,18,29,33,27,15,7,8,10,10,11,12,20,50,60,69,73,74,75,72,72,73,72,72,76,79,77,
        136,160,158,154,154,152,145,140,133,130,120,110,100,89,77,71,113,122,115,117,131,128,131,134,141,148,150,152,152,150,150,153,157,157,157,154,153,126,25,24,27,30,31,32,31,35,36,35,38,38,37,37,47,60,70,73,79,91,97,104,93,79,85,93,106,110,117,114,106,102,103,101,106,108,105,99,77,64,67,69,67,71,70,75,81,86,98,97,104,108,109,112,117,115,123,129,130,130,126,107,34,56,92,100,97,101,94,76,48,15,7,9,9,10,36,54,64,70,71,73,73,72,74,74,77,83,82,82,
        140,164,162,160,158,159,155,142,137,132,123,113,99,88,72,78,113,118,112,122,132,132,133,136,140,147,151,152,151,149,150,153,156,157,156,155,152,86,20,25,26,30,32,32,32,34,36,34,39,34,39,36,40,58,66,68,72,89,96,102,96,81,83,92,98,108,118,118,110,102,103,107,107,107,106,95,80,65,66,67,64,68,71,81,92,87,90,97,106,107,110,109,116,118,121,127,127,126,113,46,54,86,96,100,102,105,104,99,100,92,62,27,9,9,16,39,57,64,66,70,74,73,76,76,80,84,85,82,
        147,173,177,185,180,180,169,148,139,132,123,115,100,85,71,87,116,116,107,125,129,129,131,133,142,146,150,153,154,152,150,154,155,156,156,157,151,46,24,28,29,29,32,34,32,36,33,36,36,34,36,35,33,47,63,66,74,84,92,98,95,82,79,88,93,108,118,117,108,105,102,107,107,104,103,89,76,68,66,65,68,69,73,81,85,84,92,103,110,115,114,117,118,117,123,127,126,116,54,18,65,86,99,115,120,122,123,118,110,103,98,97,77,40,12,18,36,50,62,69,71,76,77,81,83,85,86,83,
        169,202,206,206,205,199,174,153,139,131,120,107,92,78,60,94,112,112,107,127,125,128,132,135,146,148,150,152,153,153,151,154,157,161,157,155,129,23,25,30,30,31,32,35,36,35,38,34,36,39,38,37,33,43,60,64,72,81,91,96,95,85,76,79,88,103,106,113,106,102,99,103,101,107,96,85,67,63,57,62,65,68,71,75,79,85,94,103,106,112,117,115,118,118,121,124,117,69,12,21,40,90,102,115,130,138,143,138,127,117,108,104,101,103,93,61,37,29,46,58,68,72,77,83,85,89,86,83,
        169,207,212,211,212,203,174,154,140,128,116,97,83,68,55,92,104,106,108,123,120,124,128,130,143,148,151,153,154,154,154,156,159,159,158,154,78,22,24,29,30,35,33,36,36,36,35,32,31,34,32,35,32,36,49,61,67,75,91,101,99,87,77,77,83,89,102,104,102,96,101,101,103,100,82,78,68,57,51,51,56,60,62,66,73,75,80,97,101,108,120,120,116,119,124,116,77,18,14,75,69,85,97,118,138,149,159,167,158,139,121,113,112,106,104,102,102,85,56,47,62,70,77,82,84,86,87,80,
        164,200,204,204,204,199,171,147,136,124,102,85,71,55,38,74,93,92,98,119,114,118,124,129,142,148,150,152,151,153,154,157,158,157,156,142,31,26,26,30,34,33,34,35,35,33,35,34,34,33,34,34,34,33,37,53,67,74,91,99,104,86,75,70,79,84,92,90,83,86,95,99,94,94,89,77,67,55,49,55,54,54,57,61,65,67,70,83,92,105,109,111,113,115,107,72,15,15,68,95,66,84,99,128,144,147,165,189,204,205,170,147,130,122,121,123,115,111,103,97,76,63,70,77,81,83,83,79,
        154,183,183,182,178,172,159,144,132,111,89,74,56,35,25,61,73,74,78,105,103,109,115,126,135,145,149,149,149,150,154,157,157,157,151,94,21,28,28,29,33,32,31,33,34,35,33,32,32,34,31,30,35,30,34,44,56,66,87,100,104,92,79,76,74,78,80,83,83,83,95,93,93,92,88,69,50,38,43,47,53,52,48,47,57,63,56,67,73,81,89,80,82,96,115,86,99,129,110,87,89,89,104,120,131,142,167,196,209,215,216,201,197,183,175,178,170,184,144,136,108,96,79,72,76,80,81,78,
        142,171,171,168,162,156,151,137,118,99,91,90,92,91,91,111,124,127,118,126,122,116,115,117,124,135,141,140,142,147,152,156,154,152,135,34,22,25,26,28,27,27,28,28,30,29,31,30,30,31,31,32,32,35,29,35,50,59,74,94,99,95,77,71,68,70,70,74,79,81,83,91,91,93,78,39,41,27,35,39,45,50,47,46,43,45,42,44,45,35,24,73,157,194,195,189,183,149,114,80,85,89,102,116,129,142,156,191,207,217,215,217,206,209,214,212,214,213,208,204,137,127,104,89,72,71,74,72,
        140,168,166,163,159,152,145,140,146,156,161,161,161,157,159,158,159,162,162,161,163,164,163,161,156,154,150,138,136,138,143,148,147,138,70,19,20,20,22,22,23,24,25,23,25,26,24,26,27,28,29,31,29,31,33,26,37,46,60,80,93,93,73,65,69,67,72,72,77,77,79,83,79,68,53,22,17,20,28,31,38,39,40,40,36,32,24,23,20,18,20,81,165,188,181,178,175,146,117,85,88,89,101,105,118,134,156,174,196,207,210,214,209,209,209,210,217,211,217,192,132,157,138,115,96,84,69,64,
        138,164,162,157,154,155,158,158,160,160,163,161,161,159,160,160,161,160,162,162,163,162,162,161,162,165,168,167,166,161,159,156,151,140,97,90,93,88,77,65,52,37,24,19,18,21,19,21,23,24,27,29,25,32,35,33,29,31,45,59,72,74,58,49,56,59,65,71,73,77,82,76,68,43,22,19,17,19,22,29,30,37,34,33,32,32,26,24,19,16,12,55,153,167,153,155,110,54,80,85,87,82,102,108,119,125,135,153,166,179,184,190,196,195,199,210,214,217,219,199,130,125,119,131,107,100,95,69,
        133,157,155,151,157,164,162,162,164,164,164,163,164,162,161,161,162,161,162,160,160,161,163,162,163,165,165,163,163,167,166,166,162,161,159,157,160,161,161,162,163,162,147,124,89,54,24,17,20,20,22,23,25,26,26,27,31,27,29,40,51,48,43,38,47,53,62,69,74,75,74,67,48,27,16,17,16,17,15,23,22,26,29,29,28,23,23,23,19,15,13,11,18,11,12,35,41,32,25,43,47,60,94,113,122,124,132,137,146,154,157,161,163,168,162,168,192,200,205,203,168,121,115,99,102,106,100,91,
        128,147,139,152,165,163,163,167,166,165,164,164,163,163,161,160,162,161,161,163,164,163,165,165,164,165,165,164,163,163,164,164,164,163,164,164,165,165,164,166,166,167,166,165,165,164,151,114,71,46,24,21,21,25,27,27,26,24,21,22,32,35,33,27,30,41,52,65,71,71,62,50,34,21,18,18,18,16,15,15,17,21,20,20,19,17,18,16,15,13,10,10,10,9,9,10,10,12,23,50,62,77,97,97,98,100,99,99,106,118,125,134,144,147,152,161,175,182,181,189,161,133,109,95,94,96,100,94,
        122,138,126,159,163,163,164,166,167,166,168,165,166,160,161,161,161,161,160,163,164,162,166,165,164,165,168,165,165,164,165,165,165,167,166,169,168,169,169,169,169,171,173,169,169,166,165,163,164,161,148,116,65,27,20,23,24,23,22,22,22,24,27,23,22,31,46,61,66,62,47,31,22,22,23,21,18,18,18,15,16,14,15,15,15,16,14,14,11,10,10,8,7,7,8,26,65,89,99,98,96,96,94,95,96,96,98,94,90,91,88,85,79,81,83,83,89,93,99,103,108,109,104,92,92,92,92,87,
        103,108,105,164,163,167,169,168,169,170,169,167,166,163,162,162,163,161,161,164,165,166,166,167,167,167,169,168,168,166,165,167,168,170,172,172,172,171,174,174,175,178,178,178,175,173,170,171,173,170,169,166,167,150,100,47,24,23,26,27,23,20,23,27,30,35,45,51,47,40,37,39,34,30,27,27,22,22,21,22,20,17,18,17,16,13,14,13,11,10,10,9,14,39,68,95,94,92,91,89,90,88,88,88,90,92,93,92,96,93,94,93,91,90,90,89,89,92,91,93,96,97,99,104,126,117,98,85,
        76,65,105,162,166,170,172,172,173,176,174,172,170,166,166,164,163,164,165,166,164,168,168,169,173,166,167,167,167,168,169,171,177,176,176,173,171,173,179,186,190,194,189,185,182,183,184,182,179,180,180,176,175,175,172,167,140,79,32,35,33,35,33,31,43,44,55,63,66,67,68,69,73,78,86,80,71,61,54,43,36,28,25,21,20,18,17,16,13,11,14,40,70,80,82,86,88,90,89,87,89,87,85,85,86,87,86,84,86,86,88,89,90,91,95,96,94,102,97,95,98,101,106,131,153,150,137,136,
        59,69,142,153,166,170,172,171,174,174,177,171,172,168,168,166,165,165,167,166,166,168,169,172,185,168,170,170,169,172,174,178,189,188,190,186,181,186,194,198,201,202,199,201,202,203,200,198,200,195,192,188,183,184,181,184,183,177,106,46,55,64,67,73,74,70,70,76,82,82,85,92,90,94,93,91,95,95,97,95,93,94,91,85,73,62,63,64,65,78,93,94,84,79,83,88,90,92,90,89,84,85,84,83,86,88,62,81,81,80,83,83,84,87,88,89,90,92,95,99,104,108,116,158,179,191,169,158,
        49,99,147,138,162,165,171,169,172,173,174,170,170,169,167,166,164,168,167,166,166,168,169,177,190,174,167,170,171,180,188,187,196,192,196,202,204,206,209,210,206,199,195,193,199,202,205,203,203,203,206,202,195,193,197,198,203,199,150,64,69,69,72,74,73,66,74,79,81,83,86,90,87,87,90,89,90,92,92,90,91,90,90,87,88,84,85,87,95,109,124,129,143,125,96,84,91,94,93,91,89,86,81,83,87,86,91,94,81,82,84,86,86,88,92,93,91,94,95,100,107,117,138,179,164,140,176,154,
        96,124,130,120,162,165,168,167,169,170,170,167,168,168,168,164,162,163,163,165,166,169,175,178,192,180,167,176,176,192,201,189,191,197,199,202,198,183,168,156,150,153,158,163,169,172,171,172,172,180,188,192,194,190,190,186,171,160,144,120,116,101,65,31,33,67,70,75,82,87,90,92,86,91,90,92,92,91,89,91,89,88,84,83,83,86,90,95,104,117,127,142,164,177,140,95,76,84,87,87,89,90,91,92,92,84,81,153,91,89,89,90,92,94,96,95,93,98,97,102,108,146,145,205,179,157,175,185,
        94,104,139,132,161,164,168,165,166,167,171,168,168,168,165,164,163,162,163,164,166,174,177,179,188,182,170,177,180,186,197,185,178,179,173,154,147,152,157,163,166,168,171,174,181,183,186,187,187,185,177,169,165,162,161,166,169,172,174,176,178,181,183,168,128,84,67,70,78,88,96,100,97,93,92,93,95,94,91,90,87,84,84,84,85,91,97,103,110,122,142,146,128,169,141,137,82,77,78,84,82,83,88,88,83,77,116,144,95,90,91,90,89,92,95,96,97,98,100,109,117,129,181,179,158,194,180,188,
        89,93,142,143,160,164,166,162,163,166,173,168,170,170,169,168,170,166,164,167,175,181,178,181,185,181,174,172,167,163,165,161,162,155,148,153,158,162,165,168,169,171,172,179,182,184,185,189,192,193,192,191,187,185,187,188,188,188,186,186,186,186,188,186,186,186,169,122,73,74,81,90,97,101,102,98,94,96,94,93,91,89,89,90,94,101,110,112,123,139,173,184,147,147,152,144,102,74,79,81,83,85,84,78,55,98,148,111,87,89,88,85,85,90,91,94,96,97,105,109,117,140,174,201,172,192,190,182        
    };
    return ImageUInt8::fromRaw(img, 128, 128);
}

inline ImageUInt8Ptr getCharImage(){
    auto img = new uint8_t[17920]{
        118,118,118,118,118,118,118,118,118,118,118,112,112,112,109,107,109,116,117,117,119,123,132,134,134,138,153,160,162,160,153,138,127,120,101,99,92,80,82,84,90,88,80,71,67,67,56,56,42,33,33,33,33,33,33,39,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,39,33,33,33,33,33,33,33,33,33,37,44,44,41,35,33,33,33,33,33,33,39,50,50,50,50,50,50,65,67,65,50,44,44,44,44,42,44,37,33,33,33,33,33,33,33,33,33,33,27,16,16,
        118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,134,136,138,134,134,134,134,134,134,162,168,168,168,168,164,151,145,130,101,101,101,101,101,101,101,101,97,84,69,67,67,67,67,44,33,33,33,33,33,42,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,39,33,33,33,33,33,33,33,33,35,50,50,50,50,50,37,33,33,33,33,33,39,50,50,50,50,50,56,67,67,67,56,50,50,50,50,50,50,50,37,39,39,39,39,37,33,33,33,33,27,16,16,
        122,118,118,118,118,118,118,118,118,118,118,118,118,118,118,126,140,151,151,145,134,134,134,134,145,168,168,168,174,174,170,151,151,145,105,101,101,101,101,101,101,101,101,84,82,71,67,67,67,56,37,33,33,33,39,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,48,33,33,33,33,33,33,33,33,33,42,50,50,50,50,50,44,33,33,33,33,33,39,50,50,50,50,50,50,65,67,65,50,50,50,50,50,50,50,50,50,50,50,50,50,50,42,33,33,33,27,16,16,
        134,127,118,118,118,118,122,129,129,129,129,129,129,129,130,134,151,151,151,151,147,145,153,177,185,198,221,228,230,228,221,208,198,179,159,155,146,136,132,125,119,112,112,107,103,95,90,90,81,71,67,58,56,58,61,61,61,54,50,50,50,50,50,50,50,50,50,50,48,41,37,33,33,37,39,33,33,41,44,44,36,33,33,33,42,50,50,50,50,50,50,48,44,44,44,44,44,46,50,50,50,50,50,50,50,56,50,50,50,50,50,50,47,50,48,50,50,50,50,50,50,50,33,33,33,27,16,16,
        134,134,118,118,118,123,134,134,134,134,134,134,134,134,134,134,145,151,151,151,151,155,194,202,206,245,253,253,253,253,249,236,236,225,187,185,181,164,151,147,134,129,118,118,118,112,101,101,101,86,84,82,67,67,67,67,67,67,52,50,50,50,50,50,50,50,50,48,35,33,33,33,33,33,33,33,41,50,50,50,50,35,33,33,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,48,36,33,33,37,50,50,50,50,50,50,50,33,33,33,27,16,16,
        134,134,134,129,129,134,134,134,134,134,134,134,134,134,134,134,148,151,151,151,153,166,202,204,217,253,253,253,253,253,253,242,236,236,196,185,185,168,162,151,134,134,129,118,118,118,107,101,101,90,84,84,75,67,67,67,67,67,56,50,50,50,50,50,50,50,50,41,33,33,33,33,33,33,33,33,44,50,50,50,50,37,33,33,42,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,37,33,33,33,33,42,50,50,50,50,50,42,33,33,33,27,16,16,
        132,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,151,151,151,157,176,179,210,236,242,253,253,253,253,253,253,253,247,247,236,230,219,204,198,183,168,164,149,140,138,123,118,112,104,93,84,84,82,67,67,67,67,67,63,61,61,61,61,54,50,50,50,39,33,33,33,33,33,33,33,33,42,50,50,50,42,33,33,33,33,37,41,48,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,46,39,39,39,39,39,35,33,33,33,33,33,33,37,39,39,39,37,33,33,33,33,25,16,16,
        125,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,151,151,151,174,185,185,238,253,253,253,253,253,253,253,253,253,253,253,253,253,249,232,219,213,185,185,179,151,151,147,120,118,118,103,86,84,84,69,67,67,67,67,67,67,67,67,67,67,52,50,50,42,33,33,33,33,33,33,33,33,35,46,48,42,33,33,33,33,33,33,33,33,46,50,50,50,50,50,50,50,50,50,50,50,50,50,50,44,31,18,16,16,18,27,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,24,16,16,
        123,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,151,151,151,179,185,187,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,236,230,219,191,185,185,159,151,151,129,118,118,107,99,86,84,82,73,73,73,73,71,67,67,67,67,67,56,50,50,50,39,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,39,50,50,50,50,50,50,50,50,50,50,50,50,50,50,44,22,16,16,16,16,18,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,31,16,16,
        123,134,134,140,140,138,134,134,134,134,134,134,134,134,134,134,151,151,151,183,200,211,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,247,247,247,227,221,210,198,187,176,168,160,143,136,132,118,109,107,105,101,101,88,84,80,78,78,78,71,63,61,61,61,61,50,35,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,39,50,50,50,50,50,50,50,50,50,50,50,50,50,50,44,22,16,16,16,16,16,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,33,29,22,
        118,134,149,151,151,151,145,134,134,134,134,134,134,134,134,134,151,151,151,200,219,219,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,219,219,208,185,185,182,153,151,149,120,118,118,118,118,116,86,84,84,84,84,84,69,67,67,67,67,67,44,33,33,33,33,33,42,50,46,36,33,33,33,33,33,33,33,33,39,50,50,50,50,50,50,50,50,50,50,50,50,50,50,44,24,16,16,16,16,20,33,33,33,33,33,33,33,33,33,33,33,33,41,42,35,33,33,33,
        114,140,151,151,151,151,151,138,134,134,134,134,134,134,134,134,151,151,159,208,219,219,249,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,219,219,219,185,185,185,166,151,151,133,118,118,118,118,118,95,84,84,84,84,84,73,67,67,67,67,67,56,39,33,33,33,44,50,50,50,47,33,33,33,33,33,33,33,33,44,50,50,50,50,50,50,50,50,50,50,50,50,50,50,41,33,24,22,22,24,31,33,33,33,35,39,39,35,33,33,33,33,41,50,50,46,33,33,33,
        112,134,149,151,151,151,151,140,134,134,134,134,134,134,134,140,154,172,179,211,227,230,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,240,232,219,219,202,191,183,166,147,140,140,140,140,136,127,123,95,84,84,84,73,67,67,67,67,67,63,61,58,56,52,50,50,50,50,50,47,41,39,39,39,39,39,41,50,50,50,50,50,50,50,50,50,50,50,50,50,50,42,33,33,33,33,33,33,33,33,39,44,50,50,50,50,44,44,44,46,50,50,50,50,44,44,39,
        116,134,138,151,151,151,149,134,134,134,134,134,134,134,134,145,179,185,185,225,236,236,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,249,236,236,236,206,202,200,168,151,151,151,151,151,151,151,149,90,84,84,73,67,67,67,67,67,67,67,67,67,67,58,56,54,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,39,33,33,33,33,33,33,33,35,48,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,
        120,134,134,149,151,151,143,134,134,134,134,134,134,134,145,151,185,185,185,230,236,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,215,202,202,179,162,151,151,151,151,151,151,151,106,84,84,73,67,67,67,67,67,67,73,69,67,67,67,67,67,63,53,50,50,50,50,50,50,50,50,50,50,50,50,50,50,52,63,65,56,50,50,46,33,33,33,33,33,33,33,33,44,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,
        125,134,134,140,149,147,136,134,134,134,134,134,134,143,159,162,185,200,211,236,249,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,247,247,247,242,236,236,225,217,193,174,174,166,162,162,155,127,112,110,103,101,101,101,97,82,82,84,84,80,75,67,67,67,67,64,61,61,61,61,61,61,61,61,61,61,61,61,61,61,63,67,67,65,59,52,44,33,33,33,33,33,33,37,46,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,
        134,134,134,134,134,134,134,134,134,134,134,134,134,160,168,168,211,221,236,245,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,247,196,185,185,170,168,168,142,134,134,129,118,118,118,118,116,86,84,84,84,84,82,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,65,50,33,33,33,33,33,37,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,
        134,134,134,134,134,134,134,134,134,134,134,134,134,168,168,168,219,230,236,245,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,215,187,185,174,168,168,145,134,134,134,132,125,118,118,118,95,84,84,84,84,84,73,67,67,67,67,67,67,67,67,67,67,67,67,69,76,75,67,67,67,67,67,67,67,67,59,42,33,33,33,35,48,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,
        129,134,134,134,134,134,134,134,134,134,134,134,138,170,177,179,230,236,236,238,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,234,223,215,213,187,160,157,149,145,143,140,136,134,129,118,112,92,84,84,84,80,69,67,67,67,67,67,67,67,67,67,67,67,80,84,84,80,73,67,67,67,67,67,67,65,61,59,56,56,58,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,
        118,134,134,134,134,134,134,134,134,134,134,134,142,181,185,185,236,236,236,236,236,244,247,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,238,236,236,181,168,168,153,151,151,151,151,149,134,134,132,88,84,84,84,78,67,67,67,67,67,67,67,67,67,67,67,84,84,84,84,84,82,67,67,67,67,67,67,67,67,67,67,67,58,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,42,
        118,134,134,134,134,134,134,134,134,134,134,140,151,185,185,185,236,236,236,236,236,236,236,236,244,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,247,236,236,200,168,168,157,151,151,151,151,151,140,134,134,101,84,84,84,84,80,73,67,67,67,67,67,67,67,67,67,84,84,84,84,84,84,80,73,73,67,67,67,67,67,67,67,67,61,50,50,50,50,50,50,50,50,50,52,54,50,50,50,50,50,46,33,33,33,33,
        118,134,134,134,134,134,134,134,134,134,145,155,162,189,204,208,236,236,236,236,236,236,236,236,236,249,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,249,240,217,206,183,164,162,155,151,151,151,140,134,134,108,95,95,95,95,88,84,73,67,67,67,67,67,67,67,71,84,84,86,93,95,95,95,95,95,93,82,67,67,67,67,67,67,61,50,50,50,50,50,50,54,61,61,67,67,59,50,50,50,48,35,33,33,33,33,
        118,134,134,134,134,134,134,134,134,138,160,168,174,211,219,219,236,236,236,236,236,236,236,236,236,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,227,219,219,179,168,168,153,151,151,140,134,134,116,101,101,101,101,101,87,84,69,67,67,67,67,67,67,73,84,84,97,101,101,101,101,101,101,101,101,84,67,67,67,67,67,65,50,50,50,50,50,56,67,67,67,67,67,67,50,50,50,35,33,33,33,33,33,
        123,134,134,134,134,134,134,134,134,140,168,168,174,219,219,219,236,236,236,236,236,247,253,247,242,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,236,219,219,194,168,168,160,151,151,140,134,134,123,101,101,101,101,101,93,84,84,70,67,67,67,67,67,73,84,93,101,101,101,101,101,101,101,101,101,95,70,67,67,67,67,67,59,50,50,50,50,67,67,67,67,67,67,67,53,50,50,33,33,33,33,33,33,
        132,134,134,134,134,134,134,134,134,142,168,172,185,219,223,230,242,242,245,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,251,247,247,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,230,225,206,191,179,174,164,157,147,140,134,127,121,114,112,112,107,101,93,90,86,78,75,73,73,76,82,95,103,107,107,118,118,118,114,112,112,105,97,82,71,67,67,67,67,67,59,56,58,64,67,67,67,67,67,67,67,65,54,50,33,33,33,33,33,33,
        136,147,149,140,134,134,134,134,134,145,177,185,202,234,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,247,236,236,236,236,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,251,236,236,227,200,185,183,168,160,149,136,134,134,134,134,132,118,118,112,101,101,101,101,101,101,101,107,118,118,147,151,151,151,151,138,134,134,108,88,84,71,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,59,33,33,33,33,33,33,
        143,151,151,149,134,134,134,134,134,151,185,191,208,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,240,236,236,236,236,236,238,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,236,236,211,196,185,174,168,160,145,134,134,134,134,134,126,118,118,109,101,101,101,101,101,103,118,118,129,151,151,151,151,151,145,134,134,123,99,86,80,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,37,33,33,33,33,33,
        142,151,151,149,145,140,140,140,149,162,194,210,217,253,253,253,253,253,253,253,253,253,242,238,236,236,236,236,236,236,236,242,234,230,236,236,236,236,236,236,240,247,247,247,247,247,247,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,245,242,242,227,219,215,202,193,185,177,164,155,151,151,151,146,138,135,131,123,123,123,123,123,127,135,148,159,181,196,196,189,181,176,168,155,142,123,114,99,88,84,75,73,69,67,67,67,67,67,67,67,67,67,67,67,67,67,50,41,39,41,44,44,
        134,142,145,151,151,151,151,160,168,177,221,236,242,253,253,253,253,253,253,253,253,253,213,202,202,196,185,185,185,187,196,219,219,219,236,236,236,236,236,236,236,236,236,236,236,236,236,236,239,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,244,219,219,211,194,185,185,185,185,183,174,168,168,168,168,168,170,208,219,221,249,253,253,253,253,242,236,234,194,185,181,142,120,118,116,88,84,71,67,67,67,67,67,67,67,67,67,67,67,67,59,50,54,67,67,67,
        134,134,140,151,151,151,151,168,168,181,236,236,247,253,253,253,253,253,253,253,253,253,208,202,202,185,185,185,185,185,187,215,219,219,228,236,236,236,236,236,230,221,227,236,236,236,236,236,236,244,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,230,219,219,206,191,185,185,185,185,183,170,168,168,168,168,189,219,219,236,253,253,253,253,253,251,238,236,217,185,185,166,129,118,118,101,84,78,67,67,67,67,67,67,67,67,67,67,67,67,58,50,61,67,67,67,
        134,134,134,151,151,151,153,168,179,196,238,247,253,253,253,253,253,253,253,253,253,253,208,202,202,179,179,179,179,179,179,185,200,202,211,221,225,234,236,234,221,219,219,223,230,230,236,236,236,236,245,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,245,232,230,223,219,213,211,202,202,202,198,185,185,185,193,210,225,236,245,253,253,253,253,253,253,251,240,223,194,187,170,140,132,118,88,84,78,67,67,67,67,61,61,61,61,61,61,61,52,44,50,54,67,67,67,
        134,134,138,151,151,151,157,187,202,208,249,253,253,253,253,253,253,253,253,253,253,253,208,202,202,168,162,151,151,155,166,168,168,168,185,191,202,219,225,219,219,219,219,219,219,219,236,236,236,236,236,242,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,249,238,236,236,236,236,236,236,251,253,253,253,253,253,253,253,253,253,253,234,219,215,179,151,151,125,84,84,78,67,67,67,56,50,50,50,50,50,48,35,33,33,35,48,67,67,67,
        134,140,149,151,151,151,157,202,202,208,253,253,253,253,253,253,253,253,253,253,253,253,210,202,202,164,151,151,151,151,155,168,168,168,181,185,193,215,219,219,219,219,219,219,219,219,236,236,236,236,236,236,245,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,245,236,236,236,236,236,242,253,253,253,253,253,253,253,253,253,253,253,242,219,219,196,151,151,129,84,84,78,67,67,67,50,50,50,50,50,50,37,33,33,33,33,37,67,67,67,
        136,151,151,151,151,151,157,202,202,210,253,253,253,253,253,253,253,253,253,253,253,253,225,213,202,160,151,151,151,151,151,155,162,168,170,177,179,194,206,208,217,219,219,219,219,219,236,236,236,236,236,236,236,245,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,249,242,242,242,242,242,245,253,253,253,253,253,253,253,253,253,253,253,242,219,219,202,153,151,129,84,84,78,67,67,67,50,50,50,50,50,50,33,33,33,33,33,33,67,67,67,
        140,151,151,151,151,151,157,202,202,213,253,253,253,253,253,253,253,253,253,253,253,253,236,236,221,157,151,151,151,151,151,151,151,155,168,168,168,185,185,185,208,219,219,219,219,219,236,236,236,236,236,236,236,236,242,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,219,219,202,164,151,129,84,84,78,67,67,67,50,50,50,50,50,50,35,33,33,33,33,35,67,67,67,
        140,151,151,151,151,151,164,202,202,219,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,157,151,151,151,151,151,151,151,151,164,168,168,181,185,185,198,210,219,219,219,219,236,236,236,236,236,236,236,236,236,245,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,219,219,202,168,151,129,84,84,78,67,67,67,50,50,50,50,50,50,44,33,33,33,33,44,67,67,67,
        140,151,151,151,153,160,176,204,217,228,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,157,151,151,145,145,145,145,145,145,153,160,162,170,177,179,187,194,211,219,219,221,236,236,236,236,236,236,236,236,236,236,244,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,219,219,202,168,151,129,84,84,78,67,67,67,50,50,50,50,50,50,50,46,41,44,52,56,67,67,67,
        140,151,151,151,164,168,179,225,236,236,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,157,151,151,134,132,129,129,134,134,151,151,151,168,168,168,185,185,191,219,219,225,236,236,236,236,236,236,236,236,236,236,236,238,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,219,219,196,166,151,129,84,84,78,67,67,67,50,50,50,50,50,50,56,65,67,67,67,67,67,67,67,
        136,151,151,151,168,168,179,236,236,236,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,157,151,151,134,120,118,118,123,134,143,151,151,164,168,168,181,185,185,211,219,221,236,236,236,236,236,236,236,236,236,236,236,236,244,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,219,219,196,168,153,131,84,84,78,67,67,67,54,50,50,50,50,52,67,67,67,67,67,67,67,67,67,
        134,140,149,151,166,168,179,236,236,238,253,253,253,253,253,253,253,253,253,253,253,253,242,236,236,162,155,151,134,118,118,118,118,130,134,142,145,153,160,162,164,172,174,183,198,202,211,219,219,225,230,230,234,236,236,236,236,236,236,244,247,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,219,219,198,168,166,138,95,88,80,73,70,67,65,52,50,50,50,61,67,67,67,67,71,73,73,80,84,
        134,134,136,149,157,166,179,236,236,242,253,253,253,253,253,253,253,253,253,253,253,253,253,251,242,174,168,160,134,120,118,118,118,120,130,134,134,138,145,149,151,151,151,153,162,168,185,185,185,202,219,219,221,228,230,230,232,236,236,236,236,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,219,219,202,168,168,149,118,118,93,84,84,73,67,65,52,52,65,67,67,67,67,76,84,84,103,118,118,
        134,134,134,138,151,155,179,228,236,242,253,253,253,253,253,253,253,253,253,253,253,253,253,253,251,174,168,168,134,132,122,118,118,118,118,125,134,134,134,134,136,145,151,151,151,162,177,185,185,194,219,219,219,219,219,219,219,230,236,236,236,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,219,219,202,168,168,155,118,118,110,86,84,84,67,67,67,67,67,67,67,67,67,84,84,84,118,118,118,
        129,129,129,129,134,145,166,206,221,232,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,189,179,168,136,134,134,127,118,118,118,118,132,134,134,134,134,136,151,151,151,151,159,168,174,176,191,202,204,208,208,213,219,219,227,230,236,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,245,230,223,208,185,172,159,136,127,120,110,95,84,73,71,67,67,67,67,67,67,67,84,84,84,118,118,118,
        118,118,118,118,118,123,145,202,202,211,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,219,219,196,140,134,134,134,118,118,118,118,120,127,134,134,134,134,138,145,145,145,145,151,151,151,151,162,168,172,183,196,211,219,219,219,225,242,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,227,219,217,177,160,151,143,134,134,104,84,84,76,67,67,67,67,67,67,84,84,84,118,118,118,
        116,118,118,118,118,118,136,194,202,208,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,225,219,219,143,134,134,134,118,118,118,118,118,118,123,134,134,134,134,134,134,134,134,137,151,151,151,151,164,168,170,193,202,211,219,219,219,232,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,219,219,200,168,157,151,134,134,134,90,84,84,69,67,67,67,67,69,84,84,86,118,118,118,
        112,118,118,118,118,118,127,172,191,204,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,234,227,219,162,145,134,134,122,118,118,118,118,118,118,134,134,134,134,134,134,134,134,134,145,151,151,151,153,160,162,170,187,191,204,213,213,221,240,247,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,245,230,223,206,174,170,155,140,134,134,107,97,95,93,84,84,78,78,88,90,90,101,118,118,118,
        114,118,118,118,118,118,123,168,168,187,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,238,174,168,153,134,123,118,118,118,118,118,118,134,134,134,134,134,134,134,134,134,134,142,149,151,151,151,151,162,168,168,185,202,202,213,221,234,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,221,189,185,166,140,134,134,118,118,118,118,118,118,116,101,101,101,101,110,118,118,118,
        118,118,118,118,118,118,123,168,168,179,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,174,168,168,134,123,118,118,118,118,118,120,134,134,134,134,134,134,134,134,134,134,134,134,147,151,151,151,159,168,168,176,200,202,213,219,221,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,236,200,187,174,149,137,134,118,118,118,118,118,118,118,107,101,101,101,109,118,118,118,
        118,118,118,118,118,118,123,168,168,179,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,245,174,168,168,134,122,118,118,118,118,118,129,134,134,134,134,134,134,134,134,134,134,134,134,136,147,151,151,151,159,162,166,177,189,208,219,219,245,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,240,213,204,177,157,154,137,123,122,118,118,118,118,118,103,101,101,101,101,109,112,112,
        118,118,118,118,118,118,123,168,168,179,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,251,244,236,174,168,168,129,118,118,118,118,118,122,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,145,151,151,151,162,168,168,193,217,219,242,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,247,236,236,202,168,168,154,134,134,127,118,118,118,107,101,101,101,101,101,101,101,101,
        118,118,118,118,118,118,123,168,168,179,249,253,253,253,253,253,253,253,253,253,253,253,253,253,253,240,236,236,174,168,168,123,118,118,118,118,118,120,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,147,151,151,159,168,168,183,204,219,234,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,249,236,236,219,170,168,168,136,134,134,118,118,118,101,101,101,101,101,101,101,101,101,
        110,118,118,118,123,125,133,168,168,179,238,245,249,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,174,168,168,123,118,118,118,118,118,112,116,123,123,123,123,123,123,123,132,134,134,134,134,129,129,129,129,131,138,140,140,148,151,157,177,189,202,223,230,240,247,247,247,247,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,247,242,230,204,185,174,159,145,134,123,122,118,101,101,101,101,101,101,101,101,101,
        93,118,118,132,140,151,151,166,168,179,236,236,242,253,253,253,253,253,253,253,253,253,253,253,253,238,236,236,174,168,168,123,118,118,118,118,118,101,101,101,101,101,101,101,101,101,123,134,134,132,120,118,118,118,118,118,118,118,118,118,118,134,151,151,162,176,185,213,236,236,236,236,236,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,211,185,185,164,134,134,127,101,101,101,101,101,101,101,101,101,
        90,118,123,134,149,151,151,159,168,179,228,236,240,253,253,253,253,253,253,253,253,253,253,253,253,247,238,236,181,168,168,125,118,118,118,118,118,101,101,101,101,101,101,101,101,101,114,118,123,118,118,118,118,118,118,118,118,118,118,118,118,134,151,151,159,168,174,202,236,236,236,236,236,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,189,185,185,140,134,134,103,101,101,101,101,101,101,101,101,
        90,118,123,134,151,151,151,151,168,177,206,221,228,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,196,183,170,142,129,118,118,118,118,101,101,95,95,92,90,90,92,95,101,110,112,112,112,112,116,118,118,118,118,118,118,118,118,134,151,151,151,159,166,172,198,213,225,230,236,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,211,204,194,168,153,140,127,114,101,101,101,101,101,101,101,
        88,118,125,134,151,151,151,157,168,172,202,202,211,253,253,253,253,253,253,253,253,253,253,253,253,253,253,251,208,202,191,157,151,136,118,118,118,101,90,84,78,67,67,67,67,78,84,84,90,95,101,101,112,118,118,118,118,118,118,118,118,134,151,151,151,151,151,162,168,168,202,219,225,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,238,213,202,179,151,151,134,103,101,101,101,101,97,
        80,123,134,134,151,151,153,168,168,168,194,202,208,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,208,202,202,159,151,151,120,118,118,95,84,84,67,67,67,67,67,67,84,84,84,84,86,101,112,118,118,118,118,118,118,118,118,134,151,151,151,151,151,151,157,168,202,219,219,242,253,253,253,253,245,242,242,242,242,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,219,210,202,153,151,151,110,101,101,101,92,84,
        78,132,134,136,151,155,166,168,168,168,172,191,202,242,251,253,253,253,253,253,253,253,253,253,253,253,253,253,223,210,202,168,159,151,132,122,118,95,84,84,67,67,67,67,67,67,84,84,84,84,84,90,109,118,118,118,118,118,118,118,118,136,151,151,151,151,151,151,151,162,202,219,219,240,253,253,253,249,236,236,236,236,236,236,244,247,247,247,247,247,247,247,247,247,247,247,247,247,247,249,253,253,253,253,253,253,253,253,247,247,247,250,253,253,253,253,228,223,211,170,155,151,118,114,103,101,84,84,
        73,134,134,140,160,168,168,168,168,168,168,168,185,236,238,250,253,253,253,253,253,253,253,253,253,253,253,253,236,236,221,174,168,160,134,134,127,95,84,84,67,67,67,67,67,67,76,84,84,84,84,84,101,118,118,118,118,118,118,118,118,140,151,151,151,151,151,151,151,157,202,219,219,232,249,253,249,238,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,247,253,253,253,253,253,253,242,236,236,236,236,250,253,253,253,253,240,228,185,172,160,120,118,114,101,88,84,
        67,134,134,145,168,168,168,168,168,168,168,168,179,236,236,236,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,174,168,168,138,134,134,101,90,84,67,67,67,67,67,67,67,70,73,73,82,84,93,118,118,118,118,118,118,118,118,140,151,151,151,151,151,151,151,159,202,219,221,234,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,238,253,253,253,253,253,245,236,236,236,236,236,236,253,253,253,253,247,236,185,179,168,129,118,118,101,99,92,
        67,134,145,151,168,168,168,168,168,168,168,168,179,236,236,236,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,174,168,168,149,136,134,101,101,90,75,67,67,67,67,67,67,67,67,67,69,76,78,90,101,101,101,101,110,118,118,140,151,151,151,151,151,151,155,166,202,223,234,236,236,236,236,236,236,236,236,236,236,236,230,230,230,230,230,230,230,230,230,234,236,236,236,236,236,236,236,240,247,247,247,244,236,236,236,236,236,236,236,253,253,253,253,240,236,185,172,168,134,130,120,107,107,107,
        67,138,151,151,168,168,168,168,168,168,168,168,179,236,236,236,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,174,168,168,151,143,134,107,101,99,78,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,95,118,118,140,151,151,157,166,164,166,168,168,204,236,236,236,236,236,236,236,236,236,236,236,236,225,219,219,219,219,219,219,219,219,219,221,234,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,245,253,253,240,236,236,174,168,168,134,134,127,118,118,118,
        59,129,151,151,164,168,168,168,168,168,168,168,179,236,236,236,253,253,253,253,253,253,253,253,253,253,253,253,238,236,236,174,168,168,151,151,140,107,101,101,78,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,88,118,118,140,151,151,166,168,168,168,168,168,208,236,236,238,249,253,253,253,245,236,236,236,230,219,219,219,219,219,219,219,219,219,219,219,225,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,240,238,236,236,225,168,168,164,134,134,125,118,118,118,
        56,78,112,129,131,138,140,136,134,144,168,168,179,236,236,236,253,253,253,253,253,253,253,253,253,253,253,253,251,240,236,185,176,168,153,151,149,107,101,101,78,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,78,95,109,136,151,151,166,168,168,168,172,185,219,238,242,249,253,253,253,253,253,247,242,238,230,219,219,219,219,219,219,219,219,219,219,219,225,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,219,213,191,162,160,145,129,125,118,118,118,118,
        56,67,67,73,84,84,84,84,84,95,162,168,179,236,236,236,253,253,253,253,253,253,253,253,253,253,253,253,253,253,245,217,202,187,157,151,151,107,101,101,78,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,78,84,84,127,151,151,162,168,168,176,217,219,234,253,253,253,253,253,253,253,253,253,253,253,234,219,219,219,219,219,219,219,219,219,219,219,225,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,185,185,185,151,142,134,118,118,118,118,118,118,
        52,67,67,67,71,78,84,84,84,95,151,168,179,232,236,236,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,219,213,202,157,151,151,109,101,101,84,73,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,78,84,84,123,151,151,162,168,168,196,219,221,245,253,253,253,253,253,253,253,253,253,253,253,242,219,219,219,219,219,219,219,219,219,219,219,225,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,228,185,185,177,143,134,132,118,118,118,118,118,118,
        50,67,67,67,67,67,75,84,84,95,151,160,177,221,232,236,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,230,225,211,168,159,151,121,108,101,86,84,76,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,78,84,84,123,151,151,162,168,170,200,223,234,247,253,253,253,253,253,253,253,253,253,253,253,238,208,208,208,208,208,208,208,208,210,213,213,215,234,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,225,221,206,179,174,155,138,134,123,118,118,112,112,112,112,
        50,67,67,67,67,67,73,84,84,95,151,151,168,219,219,228,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,234,191,185,170,140,134,119,93,84,84,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,78,84,84,123,151,151,162,168,181,210,236,236,247,253,253,253,253,253,253,253,253,253,253,253,228,185,185,185,185,185,185,185,185,185,185,196,208,219,230,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,202,202,202,162,151,151,132,120,118,116,103,101,101,101,101,
        50,67,67,67,67,67,73,84,84,92,144,151,162,213,219,219,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,251,191,185,185,140,134,134,95,84,84,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,78,84,84,125,151,151,162,179,185,213,236,236,247,253,253,253,253,253,253,253,253,253,253,251,217,185,185,185,174,168,172,181,185,185,185,185,200,219,219,232,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,202,202,198,151,151,151,122,118,118,105,101,101,101,101,101,
        50,67,67,67,67,67,73,84,84,90,112,146,162,196,216,219,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,191,185,185,148,134,134,95,84,84,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,78,84,90,129,151,151,168,185,185,213,236,236,247,253,253,253,253,253,253,253,253,253,249,233,208,185,185,182,168,168,168,168,179,185,185,185,188,205,219,222,233,236,236,236,236,236,236,236,236,236,236,236,236,236,236,202,199,188,151,151,151,118,118,118,101,101,101,101,101,101,
        39,67,67,67,67,67,73,84,90,101,101,101,126,185,185,198,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,239,211,174,168,151,90,84,84,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,78,98,101,129,151,151,174,185,185,225,250,253,253,253,253,253,253,253,253,253,253,245,219,219,208,185,171,168,168,168,168,168,168,168,168,168,168,179,185,196,219,219,225,236,236,236,236,236,236,236,236,236,236,236,236,191,185,185,151,151,151,118,118,118,101,101,101,101,101,101,
        39,67,67,67,67,67,73,84,95,101,101,101,112,185,185,188,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,245,230,174,168,164,90,84,84,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,82,101,101,129,151,151,174,185,191,228,253,253,253,253,253,253,253,253,253,253,253,236,219,219,204,181,168,168,168,168,168,168,168,168,168,168,168,168,176,189,217,219,223,236,236,236,236,236,236,236,236,236,236,236,233,185,185,183,151,151,147,118,118,116,101,101,101,95,86,84,
        39,67,67,67,67,67,73,84,88,101,101,101,112,185,185,185,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,245,236,174,168,168,92,84,84,69,67,67,67,67,67,67,67,67,67,67,67,67,67,67,90,101,103,138,151,157,179,193,202,230,253,253,253,253,253,253,253,253,253,253,253,236,219,211,189,172,168,168,168,168,168,168,168,168,168,168,168,168,168,177,200,217,219,228,236,236,236,236,236,236,236,236,236,233,222,185,183,174,151,147,129,118,116,107,101,101,101,84,84,84,
        39,67,67,67,67,67,67,84,84,90,101,101,112,185,185,185,250,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,238,236,174,168,168,95,84,84,73,67,67,67,67,67,67,67,67,73,82,84,84,93,101,103,123,134,147,174,185,194,225,236,247,253,253,253,253,253,253,253,253,253,253,253,219,185,185,179,157,151,151,151,151,145,134,134,134,134,134,134,134,134,145,168,168,174,185,185,196,219,225,236,236,236,228,219,219,208,168,168,160,118,109,101,101,101,101,101,101,101,86,84,84,
        39,67,67,67,67,67,67,84,84,88,101,101,112,183,185,185,239,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,247,236,236,174,168,168,105,86,84,73,67,67,67,67,67,67,67,67,82,84,84,88,101,101,114,134,134,159,185,185,210,236,236,247,253,253,253,253,253,253,253,253,253,253,253,208,185,185,168,151,151,151,151,151,143,134,134,134,134,134,134,134,134,140,168,168,172,185,185,193,219,219,225,234,232,221,219,205,196,168,159,147,109,101,101,101,101,101,101,101,101,95,86,84,
        39,67,67,67,67,67,67,84,84,84,101,101,112,174,183,185,236,242,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,247,236,236,174,168,168,107,99,86,78,67,67,67,67,67,67,67,67,84,84,84,90,101,101,118,134,134,162,185,185,213,236,236,247,253,253,253,253,253,253,253,253,253,253,223,185,185,174,153,151,151,151,151,147,136,134,134,132,129,129,130,134,134,134,149,166,168,177,185,185,204,219,219,219,219,217,208,196,174,160,147,129,101,101,101,101,101,101,101,101,101,101,101,101,
        65,67,67,67,67,67,67,84,84,84,101,101,112,168,168,168,236,236,236,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,244,236,236,174,168,168,122,116,103,78,67,67,67,67,67,67,67,67,80,84,84,90,101,101,125,157,168,183,221,236,238,249,253,253,253,253,253,253,253,242,236,236,200,185,185,165,151,151,145,123,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,129,151,151,151,151,151,151,123,118,118,101,101,101,101,101,101,101,101,101,101,101,101,
        67,67,67,67,67,67,67,84,84,86,101,101,112,168,168,168,233,236,236,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,174,168,168,123,118,116,82,69,67,67,67,67,67,67,67,78,84,84,92,101,101,136,168,168,202,236,236,245,253,253,253,253,253,253,253,251,236,234,223,194,183,170,157,151,151,132,118,118,118,118,118,118,118,118,118,118,118,118,118,110,109,112,112,112,107,114,118,118,118,125,151,151,151,151,151,146,118,118,118,101,101,101,101,101,101,101,101,101,101,101,101,
        67,67,67,67,67,67,78,84,84,90,101,103,121,164,168,168,222,232,236,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,174,168,168,132,120,118,84,82,69,67,67,67,67,67,67,78,84,84,95,101,101,140,168,170,211,236,236,247,253,253,253,253,253,253,253,240,236,225,211,191,172,162,151,148,140,129,118,118,118,118,118,118,118,118,118,118,118,118,114,101,101,101,101,101,101,103,116,118,118,118,136,151,151,151,146,124,118,118,118,101,101,101,101,101,101,101,101,101,101,101,101,
        61,67,67,67,82,99,101,101,101,101,107,118,123,153,166,168,219,219,219,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,174,168,168,134,134,123,101,99,88,67,67,67,67,67,67,78,84,84,99,136,151,164,187,200,225,245,253,253,253,253,253,253,253,253,242,236,213,202,202,177,155,151,143,134,134,129,118,118,118,118,118,118,118,118,118,118,118,118,114,101,101,101,101,101,101,101,103,118,118,118,118,118,118,118,118,118,118,118,118,101,101,101,101,97,90,90,95,101,101,101,101,
        50,67,67,71,99,101,101,101,101,101,116,118,123,151,157,168,219,219,219,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,236,236,236,174,168,168,134,134,132,101,101,99,73,67,67,67,67,67,78,84,84,119,151,151,174,202,202,228,253,253,253,253,253,253,253,253,245,236,236,213,202,202,168,151,151,140,134,134,127,118,116,112,112,112,112,107,114,118,118,118,118,116,101,101,101,101,101,101,101,101,105,112,112,107,114,118,118,118,118,118,118,118,109,101,101,97,84,84,84,84,95,101,101,101,
        50,67,67,73,101,101,101,101,101,112,118,118,123,151,155,168,213,219,219,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,244,236,236,174,168,168,134,134,134,101,101,101,80,70,67,67,67,71,84,84,95,129,151,157,185,202,202,230,253,253,253,253,253,253,253,253,242,236,236,213,202,194,162,151,143,134,134,130,120,114,103,101,101,101,101,101,103,114,118,118,116,107,101,101,101,101,101,101,101,101,101,101,101,101,103,118,118,118,118,118,118,118,118,109,101,90,84,84,84,84,90,101,101,101,
        50,67,67,73,101,101,101,110,123,134,147,151,151,151,157,168,202,210,219,238,253,253,253,253,253,253,253,253,253,253,253,253,253,253,247,236,236,179,168,168,134,134,130,101,101,101,84,84,84,84,84,84,84,99,101,134,162,168,191,202,202,230,253,253,253,253,253,253,253,253,242,236,236,208,187,185,160,136,134,126,118,118,101,67,67,67,67,67,69,80,84,84,84,84,90,101,101,101,101,101,101,101,101,101,101,101,101,101,101,118,118,118,118,118,118,118,118,118,110,90,84,84,84,84,90,101,101,101,
        54,67,67,73,101,101,105,118,132,143,151,151,151,151,157,168,202,202,215,240,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,238,236,179,168,168,134,132,120,101,101,101,84,84,84,84,84,84,90,101,101,138,168,168,191,202,204,236,253,253,253,253,253,253,253,253,240,234,223,194,185,185,149,134,134,123,118,118,90,67,67,67,67,67,67,67,67,67,71,82,88,101,101,101,101,101,101,101,101,101,101,101,101,101,101,118,118,118,118,118,118,118,118,118,118,95,84,84,84,84,95,101,101,101,
        65,67,67,73,101,101,107,118,130,142,151,151,151,151,157,168,202,202,213,244,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,242,179,168,168,134,125,118,101,101,101,84,84,84,84,84,84,95,101,101,140,168,168,191,206,217,242,253,253,253,253,253,253,253,245,234,225,211,191,185,176,140,134,127,118,115,99,71,67,67,67,67,67,67,67,67,67,67,75,84,93,101,101,101,101,101,101,101,101,101,101,101,101,101,118,118,118,118,118,118,118,118,118,118,101,95,90,90,95,101,101,101,101,
        67,67,67,80,101,101,107,118,120,132,151,151,151,151,151,165,202,202,219,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,251,179,168,168,134,123,118,101,101,101,101,101,101,101,101,101,108,138,151,166,194,204,217,242,253,253,253,253,251,238,236,236,227,219,208,202,202,179,159,151,129,107,101,93,67,67,67,67,67,67,67,67,67,67,67,67,67,67,82,84,84,84,84,84,90,101,101,101,101,101,101,101,101,114,118,118,118,118,118,118,118,118,116,101,103,114,118,118,118,118,118,118,
        67,67,67,82,99,101,105,118,118,122,149,151,151,151,151,151,199,202,219,245,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,183,170,168,134,132,120,101,101,101,101,101,101,101,101,103,131,151,151,181,202,213,236,253,253,253,253,253,242,236,236,228,219,219,206,202,202,168,151,151,120,101,101,84,67,67,67,67,67,67,67,67,67,67,67,67,67,67,76,84,84,84,84,84,88,101,101,101,101,101,101,101,101,107,118,118,118,118,118,118,118,116,107,101,110,118,118,118,118,118,118,118,
        67,67,67,73,88,99,101,110,118,118,133,149,151,151,151,151,188,198,211,238,249,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,191,183,170,140,134,132,107,101,101,101,101,101,101,105,116,140,151,155,193,215,219,242,253,253,253,253,253,242,236,236,225,219,211,202,202,185,153,151,143,112,101,101,84,67,67,67,67,67,67,67,67,67,67,67,67,67,67,69,80,84,84,84,84,84,93,101,101,101,101,101,101,101,107,118,118,118,118,118,118,118,103,101,101,112,118,118,118,118,118,118,118,
        61,54,50,50,50,52,65,84,84,84,84,86,105,151,151,151,185,185,185,230,236,236,236,238,251,253,253,253,253,253,253,253,253,253,253,253,253,208,200,187,151,151,151,151,151,151,151,151,151,151,151,151,159,179,185,213,242,253,253,253,253,253,253,253,242,236,236,208,185,185,174,162,153,140,134,134,108,101,101,84,67,67,67,67,67,56,50,50,50,37,33,33,33,35,46,50,50,58,76,84,84,84,84,90,101,101,101,101,101,107,118,118,118,118,118,118,118,105,101,101,114,118,118,118,118,118,118,118,
        50,50,50,50,50,50,56,84,84,84,84,84,95,151,151,151,182,185,185,223,236,236,236,236,238,253,253,253,253,253,253,253,253,253,253,253,253,208,202,200,153,151,151,151,151,151,151,151,151,151,151,153,177,185,191,227,253,253,253,253,253,253,253,253,242,236,236,206,185,185,168,151,151,140,132,122,107,101,101,78,67,65,61,59,52,50,50,50,44,33,33,33,33,33,35,50,50,56,67,69,76,84,84,90,101,101,101,101,101,105,118,118,118,118,118,118,118,114,112,114,118,118,118,118,118,118,118,118,
        50,50,50,50,50,50,56,84,84,84,84,84,95,151,151,151,171,181,185,215,232,236,236,236,236,245,253,253,253,253,253,253,253,253,253,253,253,208,202,202,166,153,151,151,151,151,151,151,151,151,155,166,179,193,202,230,253,253,253,253,253,253,253,253,242,236,230,202,185,179,157,151,151,140,125,118,107,101,86,67,61,52,50,50,50,50,50,50,44,33,33,33,33,33,33,50,50,56,67,67,67,71,84,90,101,101,101,101,101,101,110,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,118,
        50,50,50,50,50,50,56,84,84,84,84,84,95,151,151,151,168,168,168,213,219,219,236,236,236,236,240,251,253,253,253,253,253,253,253,253,253,253,245,215,174,168,157,151,151,151,151,151,151,153,168,168,189,215,228,242,253,253,253,253,253,253,253,253,242,225,219,196,172,168,157,151,151,140,125,118,90,67,67,59,36,33,33,33,35,46,50,50,41,33,33,33,33,33,33,50,50,56,67,67,67,67,84,86,101,101,101,90,84,84,84,84,84,84,84,95,118,118,118,118,118,118,118,118,116,99,88,84,
        50,50,50,50,50,50,56,82,84,84,84,84,95,151,151,151,165,168,168,204,219,219,236,236,236,236,236,240,253,253,253,253,253,253,253,253,253,253,253,245,176,168,168,153,151,151,151,151,151,157,168,168,200,225,236,242,253,253,253,253,253,253,253,253,232,219,219,191,168,168,157,151,151,140,132,122,84,67,67,50,33,33,33,33,33,35,42,41,33,33,33,33,33,33,35,50,50,56,67,67,67,67,84,84,88,95,88,84,84,84,84,84,84,84,84,84,118,118,118,118,118,114,112,110,103,88,84,84,
        50,50,50,50,50,50,56,73,82,84,84,84,95,151,151,151,154,164,168,189,213,219,236,236,236,236,236,236,253,253,253,253,253,253,253,253,253,253,253,253,191,174,168,159,151,151,151,151,151,159,168,168,204,236,236,242,253,253,253,253,253,253,253,245,225,219,219,187,168,168,157,151,151,140,134,134,86,67,67,50,33,33,33,33,33,33,33,33,33,33,33,33,33,35,46,50,50,56,67,67,67,67,84,84,84,84,84,84,84,84,84,84,84,84,84,84,118,118,118,118,114,101,101,101,101,84,84,84,
        50,50,50,50,50,50,56,67,67,67,67,69,86,143,151,151,151,151,151,179,185,196,234,236,236,236,236,236,253,253,253,253,253,253,253,253,253,253,253,253,213,200,183,162,151,151,151,151,151,166,168,168,208,236,236,238,253,253,253,244,236,236,236,236,225,217,206,185,168,168,157,151,151,147,134,134,89,67,67,50,25,16,16,16,20,31,33,33,33,33,42,56,65,67,67,67,67,67,67,67,67,69,84,84,84,84,84,84,84,84,84,84,84,84,84,84,118,118,118,118,112,101,101,101,101,84,84,84,
        50,50,50,50,50,50,56,67,67,67,67,67,78,132,142,151,151,151,151,179,185,185,230,236,236,236,236,238,253,253,253,253,253,253,253,253,253,253,253,253,219,206,200,166,153,151,151,151,160,168,168,168,210,236,236,236,240,247,240,236,236,236,236,236,225,204,202,181,168,168,155,151,151,149,134,134,84,67,67,44,16,16,16,16,16,24,33,33,33,37,50,65,67,67,67,67,67,71,71,69,75,82,84,84,88,95,88,84,84,84,84,84,84,84,84,84,118,118,118,118,112,101,101,101,101,86,84,84,
        61,54,50,50,50,52,63,67,67,67,67,67,78,120,132,143,151,151,151,179,185,185,230,236,236,236,238,251,253,253,253,253,253,253,253,253,253,253,253,253,219,217,208,168,166,157,157,160,168,168,168,179,213,236,240,238,236,236,236,236,236,236,236,236,217,202,191,170,168,160,151,151,149,140,134,112,71,67,67,42,16,16,16,16,16,22,33,33,33,35,50,63,67,67,67,67,76,84,84,84,84,84,84,88,101,101,101,90,84,84,84,84,84,84,84,95,118,118,118,118,112,101,101,101,101,99,86,84,
        67,67,52,50,50,63,67,67,67,67,67,67,73,103,112,118,118,118,129,174,185,185,230,236,236,242,253,253,253,253,253,253,253,253,253,253,253,253,253,253,219,219,217,174,168,168,168,168,168,174,219,236,238,249,253,253,240,236,236,236,236,236,236,230,196,174,168,157,145,134,101,84,84,84,84,76,67,67,67,42,16,16,16,16,16,22,33,33,33,33,37,54,67,67,67,69,84,84,103,118,118,118,118,118,118,114,103,101,101,101,101,101,101,101,101,109,118,118,118,118,114,101,101,110,118,118,118,118,
        67,67,54,50,50,65,67,67,67,67,67,67,69,101,101,112,118,118,118,172,185,185,230,236,236,251,253,253,253,253,253,253,253,253,253,253,253,253,253,253,219,219,219,174,168,168,168,168,168,200,236,236,245,253,253,253,249,236,236,236,236,234,223,219,185,168,168,151,134,134,95,84,75,67,67,67,67,67,67,47,16,16,16,16,16,24,33,33,33,33,33,46,67,67,69,82,84,92,118,118,118,118,118,118,118,118,110,101,101,101,101,101,101,101,101,110,118,118,118,118,116,103,114,118,118,118,118,118,
        67,59,50,50,50,56,65,67,67,67,67,67,67,101,101,101,118,118,118,164,181,185,230,236,242,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,219,219,219,183,170,168,168,168,177,213,236,236,247,253,253,253,253,242,236,236,232,221,219,200,170,168,157,136,134,112,84,76,67,67,67,67,67,67,67,50,25,18,16,16,20,31,33,33,33,33,33,44,67,69,78,84,84,95,118,118,118,118,118,118,118,118,112,101,101,101,101,101,101,101,101,103,114,118,118,116,107,114,118,118,118,118,118,118,
        44,37,33,33,33,33,33,33,33,41,44,50,56,101,101,101,118,118,118,162,168,168,230,236,236,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,238,225,219,191,168,168,140,118,118,105,86,84,73,67,67,67,67,67,67,67,67,50,33,33,33,33,33,33,33,33,33,33,33,44,76,84,84,93,101,107,118,118,118,118,118,118,118,118,118,118,118,118,118,115,104,101,101,101,101,101,101,101,101,116,118,118,118,118,118,118,
        33,33,33,33,33,33,33,33,33,33,33,39,56,97,101,101,118,118,118,155,168,168,228,236,236,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,236,219,219,189,168,168,135,118,118,92,84,84,73,67,65,61,67,67,67,67,67,50,33,33,33,33,33,33,33,33,33,33,33,50,84,84,88,101,101,105,118,118,118,118,118,118,118,118,118,118,118,118,118,118,115,101,101,101,101,101,101,101,101,109,118,118,118,118,118,118,
        33,33,33,33,33,33,33,33,33,33,33,33,56,78,97,101,118,118,118,147,164,168,221,232,236,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,247,228,219,211,181,168,160,129,116,107,90,84,84,69,67,52,50,58,67,67,67,67,50,33,33,33,33,33,33,33,33,33,33,37,59,84,86,99,101,101,101,110,118,118,118,118,118,118,118,118,118,118,118,118,118,118,101,101,101,101,101,101,101,101,101,118,118,118,118,118,118,
        33,33,33,33,33,33,33,33,33,33,33,33,56,67,67,78,118,118,118,145,151,151,213,219,219,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,244,236,213,200,189,168,151,151,123,101,101,90,84,73,67,67,50,50,56,67,67,67,67,50,33,33,33,33,33,33,42,59,67,67,67,73,93,101,101,101,101,101,101,101,101,101,103,116,118,118,118,118,118,118,118,118,118,101,101,101,101,101,101,101,101,101,118,118,123,132,134,134,
        33,33,33,33,33,33,33,33,33,33,33,41,56,67,67,67,115,118,118,142,151,151,210,219,219,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,251,236,236,213,187,185,166,151,151,123,101,101,90,73,67,67,67,50,50,54,67,67,67,67,47,33,33,33,33,33,37,59,67,67,67,67,76,101,101,101,101,101,101,101,101,101,101,101,103,118,118,118,118,118,118,118,118,118,109,101,101,101,101,101,101,101,101,118,118,132,134,134,134,
        33,33,33,33,33,33,33,33,33,41,46,50,56,67,67,67,104,116,118,125,145,151,193,213,219,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,249,236,236,208,185,185,162,151,151,123,101,101,90,67,67,67,65,50,50,50,59,67,67,59,42,33,33,33,33,33,39,67,67,67,67,67,78,101,101,101,101,101,101,101,101,101,101,101,101,118,118,118,118,118,118,118,118,118,118,109,101,101,101,101,101,101,101,118,118,134,134,134,134,
        33,33,33,33,33,33,33,44,63,67,67,67,67,67,67,67,101,101,112,118,118,118,179,185,185,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,236,236,208,185,185,162,151,151,123,101,101,90,67,67,67,56,50,50,50,50,50,50,50,42,33,33,33,33,33,39,67,67,67,67,67,78,101,101,101,101,101,101,101,101,101,101,101,101,118,118,118,118,118,118,118,118,118,118,118,116,101,101,101,101,101,101,118,120,134,134,134,134,
        33,33,33,33,33,33,35,65,67,67,67,67,67,67,67,67,101,101,101,118,118,118,179,185,185,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,236,236,204,185,185,160,151,151,118,101,101,86,67,67,59,50,50,50,50,50,50,50,50,39,33,33,33,33,33,48,67,67,67,67,67,78,101,101,101,101,101,101,101,101,101,101,101,101,118,118,118,118,118,118,118,118,118,118,118,118,107,101,101,101,101,109,120,129,134,134,134,134,
        54,41,33,33,33,35,44,67,67,67,67,67,67,67,67,67,101,101,101,112,118,118,172,185,185,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,249,236,221,196,185,176,151,151,133,103,101,92,73,67,65,52,50,50,50,50,50,50,50,42,33,33,33,33,35,44,56,67,67,67,67,67,78,101,101,101,101,101,101,101,101,101,101,101,103,118,118,118,118,118,118,118,118,118,118,118,116,101,107,101,101,109,118,130,134,134,134,134,134,
        67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,101,101,101,112,118,118,164,183,185,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,225,202,202,179,162,151,127,103,101,90,73,67,67,61,52,50,50,50,39,33,33,33,33,33,33,44,50,50,50,50,56,67,67,67,67,67,78,101,101,101,101,101,101,101,101,101,101,101,107,118,118,118,118,118,118,116,107,101,101,107,101,116,118,118,118,118,118,134,134,134,134,134,134,
        67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,101,101,101,118,118,118,160,170,183,247,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,217,202,202,172,151,151,120,101,101,84,67,67,63,50,50,50,50,50,37,33,33,33,33,33,39,50,50,50,50,50,63,67,67,67,67,67,82,101,101,101,101,101,101,101,101,101,101,101,107,118,118,118,118,118,118,103,101,101,101,101,107,118,118,118,118,118,125,134,134,134,134,134,134,
        67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,101,101,112,118,118,118,153,164,176,234,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,249,247,247,247,228,204,202,187,162,151,136,112,101,99,73,67,67,56,50,50,50,50,42,33,33,33,33,33,33,42,50,50,50,52,61,67,67,67,67,67,75,90,101,101,101,101,101,101,101,101,101,101,101,107,118,118,118,118,118,112,101,101,101,101,101,101,116,118,118,118,125,134,134,134,134,134,134,134,
        67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,101,109,118,118,118,125,151,151,151,230,238,251,253,253,253,253,253,253,253,253,245,236,236,236,236,236,236,236,236,240,247,247,245,238,236,236,221,204,202,179,168,168,140,118,118,107,95,84,73,67,67,56,50,50,39,33,33,33,33,33,33,33,33,42,50,50,56,67,67,67,95,118,118,110,101,101,101,107,116,118,114,103,101,101,101,101,101,107,118,118,118,118,118,112,101,101,101,101,101,101,101,109,118,127,134,134,134,134,134,134,134,134,
        67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,101,110,118,118,118,127,143,151,151,225,236,238,253,253,253,253,253,253,253,238,236,236,236,236,236,236,236,236,236,236,236,236,236,236,234,223,208,202,202,177,168,168,133,118,118,99,84,84,71,67,67,56,50,50,37,33,33,33,33,33,33,33,33,42,50,50,65,67,67,78,118,118,118,118,105,101,101,116,118,118,118,110,101,101,101,101,101,107,118,118,118,118,118,112,101,101,101,101,101,101,101,101,109,134,134,134,134,134,134,134,134,134,
        67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,84,99,114,118,118,120,130,143,151,198,228,236,242,251,253,253,253,253,247,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,227,219,208,202,194,170,168,160,129,116,107,90,84,78,67,67,67,56,50,42,33,33,33,33,33,33,33,33,33,42,50,50,67,67,67,84,118,118,118,118,103,101,101,114,118,118,118,112,101,101,101,101,101,107,118,118,118,118,118,105,101,101,101,101,101,101,101,101,101,134,134,134,134,134,134,134,134,134,
        39,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,95,101,101,101,101,110,177,185,185,230,236,247,253,253,253,247,236,236,236,236,236,236,236,236,236,236,236,236,236,236,230,219,219,196,185,174,160,151,151,123,101,101,88,71,67,67,67,67,44,33,33,33,33,33,33,33,33,33,33,33,42,50,50,67,67,73,92,118,118,118,107,101,101,101,103,112,118,118,105,101,101,101,101,101,107,118,118,118,114,105,101,101,101,101,101,101,101,101,101,101,132,134,134,134,134,134,134,134,134,
        39,67,67,67,67,67,67,67,64,53,50,56,67,67,67,67,67,67,67,95,101,101,101,101,101,162,182,185,230,236,236,253,253,253,247,236,236,236,236,236,236,236,236,236,236,236,236,236,223,219,219,219,196,177,168,157,151,151,123,101,101,81,67,67,67,67,67,44,33,33,33,33,33,33,33,33,33,33,33,42,50,53,67,67,81,90,107,118,107,101,101,101,101,101,101,107,104,101,101,101,101,101,101,104,118,118,118,104,101,101,101,101,101,101,101,101,101,101,107,129,134,134,134,134,134,134,134,134,
        39,59,67,67,67,67,67,67,54,50,50,50,59,67,67,67,67,67,67,80,101,101,101,101,101,155,170,185,208,236,236,245,253,253,240,236,236,236,236,236,236,236,236,236,236,236,236,225,219,219,219,219,196,168,168,157,151,149,112,101,99,73,67,67,67,67,67,44,33,33,33,33,33,33,33,33,33,33,33,42,56,65,67,67,84,90,101,101,101,101,101,101,101,101,101,101,101,101,101,101,101,101,101,110,118,118,112,101,101,101,101,101,101,101,101,101,101,103,116,129,134,134,134,134,134,134,134,134,
        39,50,54,56,67,67,67,67,50,50,50,50,50,58,61,59,61,65,67,67,73,86,101,101,101,147,155,170,185,194,208,228,238,238,236,236,240,247,245,238,236,236,236,236,236,236,236,219,219,219,219,219,193,168,168,149,140,127,107,92,84,73,67,67,67,67,67,44,33,33,33,33,33,33,33,33,33,41,44,50,67,67,67,67,84,90,101,101,101,101,101,101,101,101,101,101,101,101,105,112,112,112,114,118,118,118,112,101,101,101,101,101,101,101,101,101,107,118,118,129,134,134,134,134,134,134,134,134,
        37,50,50,50,67,67,67,67,58,50,50,50,50,50,50,50,50,56,67,67,67,69,101,101,101,145,151,153,185,185,185,219,221,234,236,238,253,253,253,249,236,236,236,236,236,236,236,230,219,219,219,219,191,168,168,145,123,118,101,84,84,73,67,67,67,67,67,44,33,33,33,33,33,33,33,33,41,50,50,59,67,67,67,78,84,90,101,107,101,101,101,101,101,101,101,101,101,103,118,118,118,118,118,118,118,118,112,101,101,101,101,101,101,101,101,101,116,118,118,129,134,134,134,134,134,134,134,134,
        39,50,50,50,67,67,67,67,67,63,61,63,67,59,50,50,50,54,67,67,67,67,101,101,101,131,151,151,170,185,185,204,219,225,236,240,253,253,253,253,242,236,236,236,236,236,236,236,233,219,219,217,183,168,166,136,118,116,88,84,82,67,67,67,67,67,67,44,33,33,33,33,33,33,33,33,44,50,52,64,67,67,75,84,84,92,116,118,116,103,101,101,101,101,101,101,101,107,118,118,118,118,118,118,118,118,112,101,101,101,101,101,101,101,101,101,118,118,123,134,134,134,134,134,134,134,134,134,
        46,50,50,50,67,67,67,67,67,67,67,67,67,67,54,50,50,58,67,67,67,67,101,101,101,118,123,129,151,162,174,185,193,198,236,236,247,253,253,253,253,238,236,236,236,236,236,236,236,217,208,196,176,159,151,123,105,88,84,75,67,67,67,67,67,67,67,52,37,33,33,33,37,44,48,56,59,65,76,78,84,92,97,105,107,110,118,118,118,116,107,101,101,101,101,101,101,107,118,118,118,118,118,118,118,118,112,101,101,101,101,101,101,101,101,101,118,122,134,134,134,134,134,134,134,134,134,134,
        50,50,50,50,67,67,67,67,67,67,67,67,67,67,56,50,59,65,67,67,67,67,101,101,101,118,118,118,151,151,157,183,185,185,230,236,236,251,253,253,253,240,236,236,236,236,236,236,236,213,196,185,168,151,151,116,86,84,75,67,67,67,67,67,67,67,67,56,50,35,33,35,50,52,67,67,69,84,84,87,101,101,114,118,118,118,118,118,118,118,116,101,101,101,101,101,101,107,118,118,118,118,118,118,118,118,112,101,101,101,101,101,101,101,101,101,118,129,134,134,134,134,134,134,134,134,129,120,
        50,50,50,50,67,67,67,67,67,67,67,67,67,65,50,50,67,67,67,67,67,67,86,101,101,110,118,118,143,151,151,168,185,185,208,236,236,242,251,253,245,236,236,236,236,236,236,236,230,200,185,183,155,151,147,99,84,82,67,67,67,67,67,67,67,67,67,56,50,39,33,39,50,56,67,67,73,84,84,93,101,101,116,118,118,118,118,118,118,118,114,101,101,101,101,101,101,107,118,118,118,118,118,118,118,118,112,101,101,101,101,101,101,101,101,101,110,122,134,134,134,134,134,134,134,132,120,118,
        59,61,61,61,67,67,67,67,67,67,56,52,44,39,46,50,67,67,67,67,67,67,67,78,90,90,93,101,130,138,140,140,145,151,159,172,179,198,202,202,221,225,225,225,225,225,206,202,191,170,162,149,134,129,110,92,76,69,67,67,67,67,67,67,67,67,67,56,50,39,33,39,50,56,67,67,73,84,84,93,101,101,109,118,118,118,118,118,118,110,103,101,101,101,101,101,101,107,118,118,118,118,118,118,118,118,112,101,101,101,101,101,101,101,101,101,101,109,118,123,122,123,123,123,123,118,118,118,
        67,67,67,67,67,67,67,67,67,65,44,33,33,33,33,42,67,67,67,67,67,67,67,67,73,84,84,84,127,134,134,134,134,134,145,151,151,177,185,185,204,217,219,219,219,219,189,185,185,162,145,134,123,107,101,82,67,67,67,67,67,67,67,67,67,67,67,56,50,48,39,48,50,58,67,67,73,84,84,93,101,101,107,118,118,118,118,118,114,101,101,101,101,101,101,101,101,107,118,118,118,118,118,118,118,118,116,101,101,101,101,101,101,101,101,101,101,101,103,114,118,118,118,118,118,118,118,118,
        67,67,67,67,67,67,67,67,67,61,35,33,33,33,33,33,67,67,67,67,67,67,67,67,67,78,84,84,105,130,134,134,134,134,138,151,151,162,183,185,194,204,217,219,219,206,185,185,182,155,134,132,112,101,97,69,67,67,67,67,67,67,67,67,67,67,67,56,50,50,50,50,50,61,67,67,78,84,84,93,101,101,107,118,118,118,118,118,112,101,101,101,101,101,101,101,101,110,118,118,118,118,118,118,118,118,118,112,101,101,101,101,101,101,101,101,101,101,101,101,101,101,103,116,118,118,118,118,
        65,67,67,67,67,67,67,67,67,61,33,33,33,33,33,33,67,67,67,67,67,67,67,67,67,67,71,73,73,80,84,95,95,106,112,114,118,118,129,140,162,166,177,185,181,174,174,174,168,140,118,105,93,80,67,67,67,67,67,67,67,67,67,67,67,67,67,56,50,50,50,50,50,61,69,76,82,92,95,98,101,101,107,118,118,118,118,118,112,101,101,101,101,101,101,108,125,129,129,129,129,129,129,129,129,129,129,129,125,114,112,112,104,101,101,101,101,101,101,101,101,101,101,112,118,118,118,118,
        54,67,67,67,67,67,67,67,67,61,37,33,33,33,33,33,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,84,84,84,101,101,101,101,101,112,147,151,153,168,168,168,168,168,168,129,101,101,80,67,67,67,67,67,67,67,67,67,67,67,67,67,67,56,50,50,50,50,50,63,82,84,92,101,101,101,101,101,107,118,118,118,118,118,112,101,101,101,101,101,105,134,134,134,134,134,134,134,134,134,134,134,134,134,134,132,118,118,118,105,101,101,101,101,101,103,107,107,109,118,118,118,118,118,
        50,63,67,67,67,67,67,67,67,67,48,35,33,33,35,44,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,76,84,84,93,101,101,101,101,103,131,151,151,160,168,168,168,168,160,121,101,101,75,67,67,67,67,67,67,67,67,67,67,67,67,67,67,56,50,50,50,50,50,71,84,84,95,101,101,101,101,101,107,118,118,120,130,132,121,101,101,101,101,101,116,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,129,118,118,116,103,101,101,101,107,118,118,118,118,118,118,118,118,105,
        50,52,63,67,67,67,67,67,67,67,61,61,56,56,61,61,67,67,67,67,67,67,67,67,67,63,56,56,56,56,63,67,71,73,84,88,90,90,92,101,118,123,136,140,143,145,153,151,143,114,99,88,73,67,67,67,67,67,67,67,67,67,67,67,67,67,67,63,61,61,61,61,67,80,84,90,99,101,101,101,101,101,107,118,120,130,134,134,132,127,123,123,123,125,130,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,129,129,129,123,119,114,118,118,118,118,118,118,118,118,112,101,
        50,50,50,61,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,52,50,50,50,50,52,65,67,67,84,84,84,84,84,90,118,118,118,122,132,134,134,134,134,110,86,84,73,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,69,84,84,84,99,101,101,101,101,101,101,107,118,127,134,134,134,134,134,134,134,134,134,134,134,134,134,134,129,118,118,118,118,120,132,134,134,134,134,134,134,134,134,134,134,134,125,118,118,118,118,118,118,118,118,114,101,
        50,50,50,52,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,59,50,50,50,50,50,50,61,67,67,82,84,84,84,84,84,110,118,118,118,120,134,134,134,134,105,84,82,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,67,73,84,84,84,97,101,101,101,101,101,101,101,116,125,134,134,134,134,134,134,134,134,134,134,134,134,134,134,118,118,118,118,118,118,120,134,134,134,134,134,134,134,134,134,134,134,134,130,125,122,118,118,118,118,118,118,110,
        50,50,50,50,67,67,67,67,67,67,67,67,67,67,67,67,67,63,56,56,56,56,56,54,50,50,50,50,50,50,50,61,67,67,73,82,84,84,84,84,101,105,107,107,114,122,130,134,126,93,73,67,67,58,56,56,54,46,44,44,44,44,44,44,56,67,67,67,67,67,67,67,82,84,84,86,97,101,101,101,93,90,90,90,97,119,134,134,134,142,157,157,157,157,157,157,149,134,134,118,118,118,118,118,118,118,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,127,118,118,118,118,118,118,
        50,50,50,50,67,67,67,67,67,67,67,67,67,67,67,67,59,50,50,50,50,50,50,50,50,50,50,50,50,50,50,61,67,67,67,67,67,67,75,84,99,101,101,101,101,112,118,118,118,88,67,67,58,50,50,48,35,33,33,33,33,33,33,33,44,61,67,67,67,67,67,67,67,78,84,84,86,97,99,93,84,84,84,84,84,103,134,134,138,168,168,170,177,176,168,168,168,149,134,125,118,118,118,118,118,123,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,118,118,118,118,118,118,
        50,50,50,50,59,67,67,67,67,67,67,67,67,67,67,64,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,54,67,67,67,67,67,67,67,78,90,101,101,101,101,103,114,118,107,78,67,67,54,50,50,41,33,33,33,33,33,33,33,33,44,50,65,67,67,67,67,67,67,67,78,84,84,84,84,84,84,84,84,84,84,101,134,134,145,168,170,183,185,185,176,168,168,157,134,134,130,129,129,129,129,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,127,118,118,118,118,118,118,
        50,50,50,50,50,54,56,56,56,56,56,56,56,56,54,50,50,50,50,50,50,50,50,50,50,50,50,50,50,48,41,39,48,56,56,56,56,56,56,59,71,75,82,90,90,90,90,90,84,75,67,59,50,50,50,39,33,33,33,33,33,33,33,33,44,50,61,67,67,67,67,67,67,67,69,82,84,84,84,84,84,84,84,84,84,101,140,155,160,183,215,219,219,219,217,213,198,164,145,142,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,132,122,118,118,118,118,107,107,105,
        50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,48,35,33,33,33,44,50,50,50,50,50,50,63,67,67,67,69,78,84,84,84,71,59,50,50,50,50,39,33,33,33,33,33,33,33,33,44,52,65,67,67,67,67,67,67,67,67,71,84,84,84,84,84,84,84,84,84,103,164,168,174,232,236,236,236,236,236,236,236,198,151,151,142,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,120,118,118,118,118,118,101,99,92,
        50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,41,33,33,33,33,33,48,50,50,50,50,50,61,67,67,67,67,69,80,84,76,67,52,50,50,50,48,33,33,33,33,33,33,33,33,33,46,65,67,67,67,67,67,67,67,67,67,67,84,84,84,84,84,84,84,84,84,112,168,168,185,234,236,236,236,236,236,236,236,213,162,151,145,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,127,118,118,118,118,118,118,101,88,84,
        31,50,50,50,44,44,46,50,50,50,50,50,50,44,44,44,50,50,50,50,50,50,50,50,50,50,50,50,39,33,33,33,33,33,22,27,27,27,27,27,61,67,67,67,67,67,67,67,67,58,41,39,39,39,33,33,33,33,33,33,33,33,33,33,58,67,67,67,67,67,67,67,67,67,67,67,80,84,84,84,84,84,84,84,88,133,168,168,193,225,234,236,236,236,236,236,236,217,168,155,151,134,134,134,134,134,130,123,123,123,123,123,123,123,123,123,121,110,107,107,107,116,118,118,118,101,84,84};
    return ImageUInt8::fromRaw(img, 140, 128);
        
}


inline ImageUInt8Ptr getLenaCropImage(){
    auto img = new uint8_t[5508]{
    184,	188,	186,	185,	191,	190,	192,	195,	195,	196,	194,	198,	201,	200,	200,	202,	202,	205,	205,	205,	205,	208,	211,	211,	213,	213,	215,	216,	219,	218,	217,	214,	205,	182,	109,	60,	50,	50,	60,	57,	60,	69,	68,	56,	57,	61,	78,	65,	50,	51,	69,	104,	159,	160,	143,	140,	131,	142,	144,	144,	140,	150,	147,	149,	147,	142,	148,	149,	144,	147,	144,	146,	141,	145,	143,	144,	147,	141,	145,	142,	139,
    181,	187,	185,	186,	187,	188,	189,	192,	193,	192,	198,	197,	198,	197,	200,	201,	201,	206,	205,	206,	201,	206,	210,	212,	211,	210,	215,	216,	219,	222,	217,	215,	213,	200,	153,	71,	49,	49,	53,	57,	66,	113,	67,	52,	57,	64,	75,	63,	50,	52,	67,	112,	163,	155,	143,	135,	133,	144,	137,	146,	151,	148,	153,	145,	146,	146,	146,	145,	146,	147,	146,	147,	142,	147,	144,	151,	145,	141,	147,	143,	140,
    185,	188,	193,	187,	185,	189,	188,	192,	193,	192,	198,	199,	199,	201,	200,	200,	203,	205,	204,	205,	204,	209,	211,	207,	211,	211,	212,	216,	217,	221,	219,	215,	215,	206,	188,	112,	51,	55,	51,	53,	59,	59,	66,	53,	57,	65,	71,	54,	52,	52,	67,	113,	160,	150,	150,	134,	129,	141,	143,	151,	150,	153,	147,	145,	145,	147,	149,	143,	146,	145,	145,	143,	144,	142,	139,	145,	148,	140,	142,	138,	143,
    182,	183,	186,	185,	184,	188,	192,	191,	189,	193,	197,	197,	198,	202,	204,	203,	205,	206,	204,	206,	205,	207,	210,	210,	213,	211,	213,	217,	215,	218,	218,	216,	215,	213,	205,	159,	76,	70,	49,	56,	56,	62,	60,	59,	53,	63,	60,	51,	46,	53,	70,	131,	160,	151,	153,	136,	131,	142,	142,	152,	152,	145,	148,	147,	149,	148,	147,	147,	145,	146,	145,	141,	142,	143,	141,	147,	145,	140,	143,	140,	143,
    180,	185,	186,	187,	188,	190,	189,	193,	196,	192,	194,	198,	197,	199,	200,	203,	206,	204,	205,	209,	206,	204,	209,	209,	211,	211,	214,	215,	215,	215,	218,	215,	216,	215,	211,	190,	109,	54,	54,	53,	58,	59,	58,	54,	54,	58,	56,	48,	45,	57,	78,	141,	154,	152,	160,	128,	139,	145,	139,	149,	147,	150,	150,	143,	148,	144,	147,	144,	147,	144,	146,	142,	141,	145,	143,	147,	147,	147,	143,	143,	142,
    181,	184,	182,	185,	184,	185,	187,	194,	191,	192,	194,	200,	196,	199,	202,	202,	202,	206,	207,	206,	206,	207,	211,	210,	211,	212,	214,	215,	216,	218,	216,	218,	217,	217,	213,	205,	159,	66,	47,	57,	50,	59,	61,	51,	50,	55,	54,	57,	53,	55,	82,	146,	147,	153,	168,	133,	140,	140,	143,	151,	148,	151,	148,	144,	149,	144,	146,	143,	143,	149,	145,	145,	144,	141,	144,	145,	143,	146,	141,	140,	145,
    180,	183,	185,	186,	184,	185,	187,	193,	192,	192,	193,	196,	200,	202,	202,	201,	204,	206,	207,	208,	205,	210,	210,	208,	211,	210,	212,	213,	215,	218,	218,	217,	218,	217,	216,	210,	188,	104,	52,	52,	53,	58,	61,	50,	51,	52,	57,	45,	44,	55,	86,	154,	145,	151,	158,	135,	132,	140,	150,	150,	147,	148,	149,	144,	147,	147,	146,	150,	145,	149,	142,	149,	144,	146,	146,	148,	147,	140,	145,	142,	140,
    179,	187,	185,	183,	186,	184,	190,	195,	193,	196,	195,	194,	200,	200,	202,	202,	206,	205,	205,	206,	206,	209,	210,	209,	212,	209,	212,	216,	216,	217,	216,	219,	218,	218,	220,	212,	205,	145,	67,	52,	53,	56,	51,	50,	48,	51,	54,	43,	44,	55,	99,	156,	148,	153,	155,	140,	130,	144,	148,	150,	150,	151,	146,	149,	148,	153,	150,	143,	145,	143,	151,	142,	145,	145,	148,	147,	142,	145,	146,	143,	143,
    184,	182,	183,	186,	182,	185,	188,	190,	190,	192,	196,	199,	200,	201,	204,	200,	206,	206,	208,	205,	206,	208,	212,	208,	210,	210,	212,	213,	214,	215,	216,	219,	219,	218,	220,	214,	209,	181,	91,	51,	48,	52,	53,	45,	45,	53,	50,	45,	45,	57,	104,	148,	147,	154,	152,	146,	131,	149,	148,	148,	151,	150,	147,	149,	148,	153,	146,	141,	147,	149,	142,	148,	148,	145,	141,	145,	144,	144,	144,	145,	147,
    183,	180,	182,	183,	186,	185,	185,	189,	191,	191,	196,	194,	199,	201,	201,	202,	204,	204,	205,	205,	206,	209,	211,	209,	210,	207,	212,	212,	213,	217,	218,	219,	218,	218,	219,	219,	212,	200,	141,	61,	43,	48,	52,	44,	48,	49,	52,	46,	49,	60,	114,	146,	145,	160,	148,	143,	136,	143,	146,	148,	147,	149,	149,	150,	149,	150,	146,	147,	144,	145,	145,	149,	146,	143,	150,	145,	144,	147,	141,	142,	146,
    185,	183,	183,	186,	183,	186,	186,	189,	190,	192,	196,	195,	200,	199,	202,	200,	203,	204,	206,	204,	203,	208,	208,	210,	210,	212,	211,	212,	211,	214,	217,	217,	218,	218,	218,	219,	213,	206,	174,	78,	42,	50,	50,	49,	44,	47,	52,	45,	50,	61,	124,	144,	143,	153,	146,	142,	144,	141,	151,	148,	143,	142,	149,	147,	146,	152,	154,	145,	148,	148,	145,	143,	143,	146,	144,	141,	142,	142,	144,	142,	142,
    183,	182,	192,	184,	186,	186,	189,	190,	191,	193,	194,	197,	198,	198,	200,	201,	202,	202,	206,	205,	207,	209,	209,	210,	210,	209,	211,	211,	212,	212,	215,	213,	218,	219,	219,	219,	214,	210,	197,	114,	50,	58,	60,	45,	46,	48,	52,	46,	45,	66,	135,	145,	143,	147,	142,	136,	153,	142,	152,	151,	151,	149,	146,	144,	144,	148,	145,	143,	145,	144,	145,	149,	144,	144,	145,	139,	145,	145,	141,	147,	141,
    180,	185,	185,	183,	186,	184,	189,	189,	192,	194,	192,	195,	199,	199,	201,	203,	206,	203,	206,	204,	207,	208,	208,	208,	208,	209,	210,	210,	212,	212,	213,	213,	217,	218,	219,	218,	216,	212,	204,	160,	69,	54,	53,	50,	46,	46,	50,	45,	47,	71,	147,	146,	143,	142,	147,	138,	155,	146,	145,	149,	146,	149,	149,	149,	147,	151,	147,	148,	146,	146,	148,	149,	145,	146,	143,	143,	146,	141,	143,	139,	141,
    178,	182,	184,	184,	186,	186,	185,	190,	196,	190,	195,	196,	199,	203,	200,	200,	203,	205,	207,	206,	208,	207,	208,	209,	207,	211,	213,	213,	213,	219,	215,	213,	215,	217,	220,	217,	218,	215,	211,	183,	93,	44,	42,	41,	46,	48,	47,	43,	51,	81,	142,	145,	138,	144,	147,	137,	151,	149,	141,	150,	147,	149,	144,	144,	147,	144,	148,	144,	147,	141,	144,	144,	142,	146,	140,	141,	146,	143,	144,	136,	138,
    179,	183,	183,	185,	184,	182,	189,	189,	193,	193,	191,	195,	197,	199,	202,	200,	202,	205,	205,	206,	210,	208,	207,	206,	209,	212,	212,	210,	214,	213,	213,	214,	213,	216,	216,	220,	219,	215,	211,	198,	123,	49,	42,	37,	40,	44,	43,	41,	52,	92,	141,	142,	138,	143,	146,	135,	149,	156,	147,	150,	150,	148,	144,	146,	147,	146,	144,	146,	143,	143,	139,	150,	143,	147,	147,	144,	141,	146,	144,	139,	139,
    181,	180,	185,	183,	185,	184,	186,	190,	189,	193,	190,	196,	198,	199,	203,	202,	202,	203,	206,	205,	207,	205,	208,	207,	212,	210,	211,	211,	214,	213,	214,	213,	213,	217,	220,	218,	219,	217,	213,	204,	153,	64,	38,	39,	39,	45,	44,	40,	56,	104,	144,	139,	143,	140,	154,	141,	148,	159,	151,	148,	146,	151,	150,	149,	144,	143,	149,	143,	149,	145,	147,	145,	150,	141,	146,	140,	143,	138,	144,	140,	141,
    179,	181,	181,	185,	187,	185,	187,	187,	189,	189,	190,	193,	194,	198,	201,	200,	201,	207,	205,	205,	203,	210,	205,	208,	207,	210,	212,	209,	214,	213,	213,	213,	215,	215,	217,	218,	218,	220,	214,	210,	184,	87,	45,	48,	43,	43,	42,	45,	57,	105,	142,	138,	140,	139,	147,	142,	145,	158,	149,	144,	142,	152,	150,	143,	147,	144,	146,	139,	143,	143,	146,	145,	146,	146,	145,	144,	141,	147,	143,	139,	144,
    180,	180,	181,	182,	184,	187,	186,	189,	191,	193,	193,	195,	196,	199,	198,	198,	199,	205,	206,	207,	201,	205,	206,	207,	207,	209,	210,	209,	213,	213,	212,	213,	216,	215,	216,	218,	218,	217,	215,	209,	196,	121,	45,	42,	44,	41,	47,	43,	62,	113,	139,	138,	139,	140,	149,	141,	145,	157,	156,	150,	142,	149,	148,	145,	146,	144,	146,	144,	145,	143,	142,	147,	139,	142,	146,	140,	140,	142,	139,	138,	140,
    179,	180,	181,	183,	183,	185,	188,	188,	189,	194,	194,	195,	197,	199,	197,	201,	200,	204,	205,	207,	204,	204,	205,	206,	208,	206,	211,	208,	210,	210,	212,	213,	215,	213,	214,	217,	217,	218,	215,	210,	205,	156,	55,	40,	42,	45,	42,	48,	63,	125,	142,	135,	138,	137,	151,	146,	142,	156,	156,	146,	143,	142,	146,	147,	142,	144,	145,	143,	145,	144,	145,	145,	140,	145,	143,	136,	141,	140,	143,	140,	145,
    176,	179,	179,	186,	186,	188,	188,	186,	188,	193,	191,	194,	197,	196,	198,	200,	203,	207,	202,	205,	203,	206,	207,	206,	208,	209,	210,	208,	212,	211,	210,	212,	212,	212,	216,	215,	217,	216,	216,	211,	206,	176,	77,	37,	40,	42,	44,	51,	71,	131,	140,	131,	140,	139,	146,	145,	143,	150,	155,	147,	147,	147,	142,	146,	145,	148,	145,	145,	144,	144,	147,	148,	142,	143,	140,	140,	144,	138,	144,	137,	140,
    178,	179,	182,	185,	183,	187,	189,	187,	190,	191,	188,	194,	196,	195,	196,	198,	202,	204,	205,	206,	201,	204,	206,	206,	207,	209,	209,	211,	213,	210,	210,	212,	212,	212,	214,	216,	217,	215,	216,	215,	209,	196,	112,	39,	40,	42,	44,	44,	71,	129,	137,	132,	143,	135,	148,	148,	147,	146,	153,	153,	145,	146,	145,	147,	144,	147,	143,	143,	144,	144,	142,	142,	142,	137,	141,	147,	140,	141,	139,	138,	138,
    179,	177,	186,	183,	186,	187,	193,	187,	186,	190,	192,	194,	196,	196,	199,	197,	202,	203,	204,	203,	203,	205,	204,	206,	207,	208,	210,	209,	211,	210,	209,	211,	215,	213,	212,	215,	217,	217,	215,	218,	212,	203,	138,	45,	42,	42,	42,	45,	79,	134,	138,	138,	138,	139,	148,	146,	145,	149,	153,	158,	142,	145,	140,	147,	140,	146,	143,	147,	142,	143,	138,	146,	138,	143,	138,	144,	139,	137,	144,	139,	140,
    179,	179,	182,	183,	182,	186,	188,	187,	187,	190,	189,	195,	194,	197,	198,	198,	199,	203,	204,	202,	204,	205,	206,	208,	208,	207,	208,	208,	210,	210,	210,	212,	215,	214,	212,	214,	215,	215,	217,	217,	213,	203,	168,	72,	37,	40,	44,	45,	86,	129,	137,	142,	139,	139,	148,	144,	146,	151,	152,	155,	146,	145,	142,	141,	141,	145,	142,	142,	145,	141,	144,	143,	144,	137,	139,	149,	136,	143,	136,	139,	139,
    176,	181,	182,	181,	181,	186,	187,	185,	188,	190,	190,	195,	195,	199,	198,	199,	202,	205,	201,	203,	204,	204,	204,	206,	207,	208,	208,	213,	209,	210,	211,	212,	214,	212,	214,	215,	215,	216,	216,	215,	212,	208,	184,	94,	39,	39,	46,	46,	99,	129,	133,	136,	139,	140,	151,	140,	146,	151,	145,	156,	152,	145,	143,	145,	142,	144,	141,	143,	145,	134,	141,	141,	140,	145,	140,	141,	139,	139,	138,	140,	142,
    177,	180,	180,	179,	182,	186,	187,	186,	186,	191,	189,	193,	194,	195,	199,	198,	200,	200,	200,	204,	204,	202,	205,	205,	205,	207,	207,	209,	209,	212,	209,	210,	213,	212,	214,	214,	215,	216,	214,	214,	213,	207,	196,	126,	46,	46,	44,	48,	102,	132,	130,	142,	136,	140,	152,	142,	145,	149,	147,	151,	160,	146,	146,	144,	142,	141,	142,	143,	145,	141,	146,	141,	148,	139,	144,	139,	139,	142,	137,	142,	143,
    173,	178,	179,	184,	181,	184,	182,	186,	188,	186,	192,	195,	194,	194,	195,	197,	196,	200,	203,	202,	203,	201,	204,	205,	205,	207,	206,	207,	209,	209,	212,	212,	213,	212,	212,	212,	213,	215,	214,	215,	213,	208,	204,	153,	52,	43,	47,	50,	116,	130,	135,	143,	141,	139,	149,	140,	147,	146,	142,	142,	158,	146,	146,	149,	145,	143,	145,	144,	141,	145,	143,	143,	143,	138,	140,	143,	140,	139,	137,	139,	141,
    174,	179,	175,	177,	181,	183,	183,	185,	190,	188,	192,	191,	192,	195,	195,	195,	200,	199,	203,	203,	202,	205,	204,	205,	208,	207,	205,	208,	209,	210,	209,	209,	211,	212,	212,	213,	212,	214,	213,	214,	214,	211,	206,	179,	81,	41,	47,	57,	125,	131,	134,	142,	140,	142,	151,	141,	145,	145,	146,	144,	152,	152,	141,	144,	141,	145,	140,	143,	146,	140,	147,	145,	140,	136,	139,	141,	139,	139,	142,	137,	143,
    173,	174,	176,	178,	176,	183,	186,	186,	191,	190,	192,	189,	193,	195,	195,	197,	197,	201,	200,	204,	204,	205,	207,	206,	204,	205,	207,	208,	209,	210,	210,	212,	213,	212,	212,	213,	212,	215,	213,	213,	212,	210,	206,	190,	102,	40,	42,	63,	121,	132,	133,	141,	139,	137,	147,	141,	145,	147,	147,	146,	149,	159,	143,	144,	142,	147,	144,	144,	145,	145,	143,	140,	141,	143,	137,	139,	138,	140,	138,	139,	138,
    174,	173,	173,	179,	180,	180,	185,	185,	188,	189,	188,	187,	191,	195,	193,	194,	197,	199,	201,	204,	204,	206,	205,	207,	205,	205,	209,	207,	209,	207,	210,	211,	212,	211,	210,	212,	213,	212,	212,	213,	213,	212,	209,	200,	135,	46,	44,	62,	128,	131,	128,	144,	139,	139,	150,	143,	151,	149,	148,	148,	145,	157,	146,	143,	144,	143,	145,	142,	150,	143,	148,	140,	142,	141,	139,	138,	142,	138,	141,	139,	137,
    169,	172,	174,	180,	180,	182,	183,	182,	189,	187,	188,	187,	197,	196,	194,	196,	198,	199,	198,	202,	201,	206,	208,	204,	206,	206,	205,	205,	207,	208,	211,	212,	208,	212,	210,	213,	212,	212,	212,	213,	217,	213,	208,	203,	160,	53,	44,	70,	131,	122,	131,	146,	142,	141,	152,	143,	148,	148,	148,	148,	145,	148,	145,	144,	141,	148,	143,	144,	140,	143,	146,	148,	143,	139,	136,	140,	141,	143,	141,	142,	138,
    171,	173,	174,	177,	178,	180,	181,	181,	183,	186,	190,	188,	194,	192,	194,	197,	197,	197,	199,	199,	198,	202,	207,	205,	204,	205,	205,	204,	208,	209,	208,	209,	211,	213,	212,	212,	212,	212,	212,	212,	215,	210,	211,	205,	180,	80,	45,	72,	128,	124,	136,	144,	144,	146,	151,	144,	149,	155,	148,	149,	146,	144,	143,	148,	140,	143,	145,	142,	145,	141,	148,	148,	147,	140,	139,	145,	141,	143,	143,	141,	139,
    170,	175,	174,	178,	181,	180,	182,	181,	184,	188,	191,	191,	193,	191,	197,	198,	195,	200,	199,	199,	200,	202,	205,	202,	203,	204,	206,	206,	205,	209,	207,	209,	214,	209,	212,	215,	212,	212,	214,	213,	213,	212,	210,	205,	192,	112,	47,	78,	124,	126,	138,	141,	143,	146,	149,	136,	156,	146,	153,	147,	147,	147,	148,	149,	142,	143,	141,	144,	145,	143,	143,	142,	147,	141,	143,	142,	142,	143,	144,	139,	145,
    171,	172,	175,	178,	179,	177,	183,	180,	185,	188,	189,	191,	192,	191,	193,	197,	194,	199,	198,	197,	199,	201,	203,	203,	204,	204,	205,	202,	205,	212,	207,	209,	210,	211,	212,	213,	210,	213,	211,	215,	212,	211,	211,	206,	202,	152,	57,	87,	123,	123,	142,	145,	140,	141,	149,	140,	147,	149,	146,	150,	144,	143,	147,	142,	141,	142,	146,	142,	143,	142,	143,	139,	146,	141,	145,	145,	145,	145,	141,	141,	144,
    169,	171,	168,	174,	179,	176,	182,	182,	183,	184,	186,	190,	192,	190,	191,	198,	196,	199,	197,	197,	201,	200,	201,	203,	206,	206,	206,	204,	207,	208,	206,	208,	209,	209,	211,	210,	210,	210,	212,	210,	209,	212,	213,	206,	204,	177,	82,	92,	124,	131,	142,	141,	141,	144,	141,	132,	140,	150,	147,	146,	149,	145,	142,	144,	146,	145,	146,	144,	149,	145,	147,	141,	146,	147,	148,	146,	144,	144,	141,	145,	140,
    170,	170,	169,	172,	177,	175,	181,	182,	183,	182,	183,	190,	190,	190,	191,	193,	198,	195,	199,	197,	199,	201,	203,	203,	206,	207,	204,	203,	205,	208,	208,	209,	210,	210,	211,	211,	211,	210,	211,	212,	212,	211,	211,	209,	206,	196,	112,	99,	122,	133,	142,	143,	138,	143,	134,	127,	133,	140,	140,	141,	145,	145,	144,	142,	145,	148,	142,	145,	146,	148,	143,	145,	148,	144,	148,	146,	144,	143,	143,	145,	141,
    171,	169,	172,	172,	175,	173,	179,	182,	183,	182,	185,	187,	193,	189,	189,	195,	194,	194,	197,	199,	197,	198,	204,	204,	204,	206,	205,	206,	208,	210,	208,	208,	212,	209,	210,	212,	209,	209,	211,	212,	213,	211,	211,	210,	204,	200,	143,	108,	123,	136,	146,	144,	140,	136,	130,	118,	125,	133,	134,	135,	134,	135,	137,	137,	146,	144,	141,	142,	149,	144,	149,	148,	146,	146,	144,	146,	148,	143,	145,	143,	143,
    170,	170,	172,	176,	172,	176,	181,	180,	180,	182,	186,	187,	190,	191,	192,	193,	196,	194,	199,	198,	200,	199,	201,	203,	204,	205,	205,	206,	207,	208,	209,	212,	212,	210,	212,	209,	210,	209,	210,	211,	212,	213,	213,	211,	209,	202,	175,	115,	122,	135,	144,	140,	143,	137,	120,	116,	120,	123,	127,	132,	129,	132,	132,	133,	142,	134,	139,	143,	140,	141,	145,	147,	144,	148,	144,	147,	148,	143,	148,	144,	141,
    171,	172,	174,	172,	172,	179,	177,	179,	181,	178,	185,	188,	189,	192,	193,	193,	193,	196,	194,	195,	199,	201,	200,	201,	204,	207,	207,	208,	208,	210,	209,	211,	211,	209,	209,	210,	213,	212,	208,	209,	213,	212,	210,	211,	212,	207,	191,	131,	124,	143,	143,	141,	129,	131,	109,	111,	111,	118,	125,	116,	123,	122,	127,	126,	128,	133,	133,	127,	136,	137,	146,	142,	145,	143,	146,	147,	147,	150,	144,	142,	144,
    170,	170,	171,	171,	173,	174,	175,	179,	178,	183,	176,	185,	186,	190,	191,	189,	195,	193,	194,	196,	200,	199,	200,	202,	202,	205,	207,	208,	208,	209,	207,	209,	210,	210,	209,	211,	211,	212,	210,	211,	213,	213,	210,	212,	212,	211,	201,	156,	124,	141,	146,	135,	123,	119,	101,	103,	107,	111,	118,	108,	116,	114,	121,	119,	121,	129,	132,	129,	132,	136,	135,	140,	141,	144,	144,	144,	146,	146,	144,	146,	148,
    167,	169,	171,	172,	174,	176,	176,	178,	179,	182,	180,	186,	189,	189,	191,	189,	196,	195,	194,	194,	198,	199,	199,	203,	203,	203,	205,	209,	207,	210,	208,	209,	209,	211,	207,	210,	213,	213,	211,	213,	212,	212,	210,	211,	212,	213,	207,	174,	129,	137,	142,	131,	121,	112,	98,	94,	94,	100,	115,	97,	107,	110,	113,	110,	116,	120,	122,	119,	126,	129,	131,	135,	130,	135,	142,	136,	146,	141,	140,	146,	150,
    166,	171,	170,	170,	172,	178,	173,	174,	179,	179,	182,	187,	186,	187,	188,	192,	193,	193,	195,	198,	198,	200,	201,	201,	204,	202,	206,	206,	205,	207,	211,	209,	209,	212,	209,	211,	209,	213,	211,	210,	210,	212,	210,	212,	214,	212,	207,	191,	142,	135,	135,	115,	109,	109,	91,	87,	90,	101,	108,	95,	97,	96,	102,	105,	110,	107,	114,	112,	123,	122,	127,	128,	130,	129,	131,	131,	136,	141,	143,	138,	141,
    169,	170,	166,	169,	172,	172,	169,	176,	181,	182,	183,	186,	185,	186,	192,	188,	190,	190,	196,	195,	197,	200,	201,	201,	201,	199,	208,	205,	208,	212,	210,	208,	207,	211,	211,	209,	210,	212,	210,	211,	212,	210,	213,	213,	214,	213,	211,	200,	155,	132,	129,	107,	96,	96,	85,	81,	86,	92,	108,	88,	86,	92,	97,	99,	99,	103,	103,	112,	115,	115,	123,	119,	122,	122,	129,	129,	130,	137,	137,	132,	135,
    164,	168,	169,	170,	168,	169,	170,	176,	178,	179,	178,	184,	185,	186,	188,	189,	189,	194,	193,	194,	192,	197,	201,	201,	204,	203,	206,	206,	207,	210,	207,	208,	209,	213,	214,	212,	212,	217,	211,	213,	215,	214,	214,	213,	214,	215,	212,	208,	173,	128,	120,	92,	89,	86,	85,	78,	86,	85,	101,	82,	83,	86,	94,	91,	92,	100,	100,	103,	105,	107,	107,	110,	112,	116,	116,	117,	126,	122,	123,	124,	128,
    165,	169,	171,	168,	169,	169,	169,	172,	173,	178,	177,	183,	186,	187,	185,	189,	191,	190,	190,	192,	194,	198,	202,	203,	204,	204,	207,	207,	206,	208,	206,	210,	210,	211,	210,	210,	210,	215,	211,	212,	213,	214,	213,	214,	216,	216,	212,	212,	184,	128,	104,	89,	89,	87,	84,	69,	72,	80,	84,	76,	75,	86,	86,	86,	89,	96,	94,	99,	96,	91,	102,	102,	106,	104,	108,	114,	114,	119,	121,	122,	116,
    166,	169,	165,	168,	168,	172,	169,	172,	173,	180,	178,	184,	186,	184,	188,	189,	188,	191,	192,	192,	195,	197,	200,	202,	201,	203,	205,	207,	205,	209,	206,	209,	210,	211,	211,	209,	210,	212,	211,	211,	214,	214,	214,	214,	215,	216,	214,	210,	198,	133,	96,	82,	82,	84,	86,	84,	82,	93,	83,	75,	75,	78,	83,	84,	82,	86,	86,	98,	86,	91,	95,	95,	106,	101,	99,	104,	102,	112,	109,	110,	107,
    164,	168,	168,	166,	168,	168,	172,	174,	175,	175,	177,	182,	188,	185,	180,	186,	188,	189,	191,	194,	194,	196,	201,	201,	203,	205,	203,	207,	205,	206,	205,	208,	207,	209,	213,	210,	211,	209,	213,	211,	212,	214,	214,	214,	215,	216,	215,	214,	203,	147,	94,	77,	86,	90,	91,	76,	97,	102,	85,	79,	83,	78,	78,	76,	76,	80,	82,	88,	85,	86,	87,	90,	94,	98,	91,	95,	97,	99,	99,	102,	103,
    170,	170,	166,	168,	173,	171,	173,	171,	171,	176,	178,	182,	183,	184,	183,	187,	190,	189,	191,	195,	195,	198,	198,	201,	199,	203,	204,	205,	207,	207,	208,	208,	211,	209,	210,	211,	211,	212,	211,	213,	212,	213,	215,	214,	216,	216,	215,	213,	210,	167,	101,	81,	89,	90,	94,	79,	94,	101,	91,	81,	89,	88,	89,	86,	84,	87,	80,	87,	83,	81,	84,	88,	88,	91,	89,	90,	89,	88,	85,	93,	91,
    166,	167,	166,	169,	169,	169,	170,	170,	172,	178,	178,	180,	183,	181,	186,	186,	188,	187,	195,	194,	197,	199,	200,	199,	199,	202,	204,	206,	205,	205,	207,	207,	208,	209,	209,	210,	212,	213,	212,	212,	213,	213,	215,	215,	217,	214,	213,	214,	210,	185,	103,	78,	90,	89,	97,	91,	94,	101,	90,	90,	90,	91,	89,	85,	89,	90,	82,	87,	92,	83,	83,	81,	82,	77,	83,	79,	79,	85,	82,	85,	86,
    163,	168,	167,	167,	169,	166,	170,	167,	171,	176,	176,	177,	184,	180,	183,	187,	189,	189,	192,	193,	196,	198,	199,	200,	201,	200,	203,	205,	205,	207,	206,	208,	209,	209,	212,	210,	210,	210,	211,	213,	212,	213,	215,	214,	217,	213,	215,	213,	209,	197,	125,	87,	82,	91,	93,	87,	104,	100,	92,	96,	102,	95,	90,	94,	89,	89,	86,	93,	83,	86,	76,	79,	74,	80,	79,	83,	73,	73,	77,	78,	81,
    160,	165,	165,	164,	167,	167,	167,	169,	172,	174,	174,	175,	182,	182,	183,	185,	188,	190,	192,	193,	196,	195,	196,	199,	198,	201,	203,	204,	207,	205,	205,	206,	208,	205,	208,	208,	209,	211,	209,	212,	211,	211,	213,	213,	217,	214,	214,	215,	210,	204,	148,	83,	80,	86,	95,	89,	105,	98,	101,	101,	104,	98,	98,	91,	90,	95,	88,	101,	89,	88,	85,	85,	79,	83,	75,	69,	71,	75,	76,	72,	77,
    162,	165,	164,	167,	168,	165,	171,	169,	175,	175,	176,	177,	180,	180,	189,	183,	187,	184,	190,	192,	191,	197,	199,	198,	196,	199,	201,	207,	205,	207,	203,	208,	208,	208,	206,	211,	208,	212,	212,	211,	212,	212,	213,	214,	214,	217,	214,	215,	213,	207,	175,	96,	78,	86,	96,	96,	105,	98,	103,	102,	101,	101,	94,	98,	100,	101,	93,	93,	88,	93,	84,	91,	85,	85,	77,	82,	77,	73,	74,	73,	69,
    163,	162,	165,	167,	168,	168,	168,	169,	177,	176,	176,	176,	178,	180,	185,	185,	183,	186,	187,	189,	195,	197,	197,	199,	200,	198,	203,	201,	205,	203,	202,	207,	206,	207,	209,	208,	209,	210,	211,	211,	211,	213,	213,	216,	215,	216,	216,	214,	214,	210,	188,	113,	84,	88,	96,	96,	101,	98,	111,	104,	105,	104,	100,	101,	102,	101,	101,	98,	98,	96,	90,	95,	94,	86,	82,	81,	81,	74,	74,	72,	63,
    160,	163,	162,	162,	163,	167,	168,	169,	171,	176,	174,	173,	176,	182,	181,	182,	182,	185,	186,	188,	193,	192,	196,	194,	197,	197,	201,	201,	203,	203,	204,	204,	207,	207,	206,	208,	208,	211,	210,	211,	211,	211,	212,	214,	215,	214,	215,	212,	212,	209,	201,	140,	86,	88,	92,	99,	101,	100,	106,	108,	106,	106,	106,	110,	100,	98,	102,	98,	96,	98,	95,	96,	91,	88,	86,	84,	79,	77,	73,	73,	69,
    159,	162,	164,	168,	162,	166,	167,	167,	174,	178,	173,	174,	176,	176,	177,	180,	180,	182,	186,	191,	192,	195,	195,	193,	196,	198,	203,	202,	203,	203,	205,	204,	206,	208,	207,	205,	208,	211,	212,	211,	212,	212,	212,	214,	216,	214,	213,	213,	212,	210,	207,	168,	96,	86,	92,	97,	103,	100,	107,	114,	111,	111,	107,	108,	104,	107,	107,	103,	99,	100,	95,	99,	94,	87,	87,	85,	81,	82,	81,	82,	73,
    159,	162,	163,	163,	162,	167,	166,	170,	174,	177,	172,	175,	176,	180,	182,	182,	182,	186,	188,	191,	191,	191,	195,	194,	195,	196,	197,	201,	200,	203,	204,	205,	206,	208,	208,	205,	209,	210,	210,	213,	211,	211,	212,	213,	214,	214,	213,	214,	211,	211,	206,	185,	109,	87,	92,	98,	100,	104,	105,	111,	112,	106,	108,	105,	107,	109,	108,	107,	108,	101,	102,	97,	96,	94,	93,	88,	89,	88,	79,	83,	77,
    157,	157,	161,	162,	166,	166,	168,	170,	173,	177,	175,	171,	175,	178,	181,	181,	185,	186,	188,	191,	190,	193,	194,	198,	196,	194,	199,	199,	204,	203,	204,	207,	204,	207,	206,	206,	208,	209,	207,	210,	209,	212,	211,	212,	214,	214,	212,	215,	211,	212,	208,	199,	133,	91,	93,	96,	99,	102,	106,	111,	111,	108,	112,	115,	108,	108,	108,	108,	101,	107,	104,	96,	103,	100,	95,	94,	92,	90,	82,	88,	82,
    157,	158,	163,	162,	164,	164,	166,	168,	173,	169,	170,	171,	177,	175,	183,	176,	181,	182,	185,	190,	192,	194,	195,	193,	195,	194,	201,	198,	201,	200,	205,	206,	204,	206,	208,	209,	209,	208,	206,	211,	210,	211,	212,	213,	213,	214,	212,	214,	214,	214,	210,	203,	154,	95,	93,	93,	98,	100,	106,	105,	108,	115,	111,	110,	112,	112,	110,	115,	106,	105,	105,	102,	105,	104,	97,	92,	94,	93,	88,	87,	87,
    158,	161,	161,	165,	160,	163,	169,	169,	167,	168,	171,	170,	173,	178,	180,	175,	178,	183,	186,	189,	190,	190,	194,	196,	197,	197,	198,	196,	199,	201,	204,	202,	202,	205,	206,	208,	209,	210,	206,	206,	210,	210,	211,	212,	214,	212,	213,	214,	214,	213,	213,	207,	177,	107,	88,	90,	101,	99,	107,	103,	106,	112,	111,	113,	110,	115,	110,	113,	106,	109,	107,	103,	101,	101,	104,	97,	96,	100,	93,	94,	84,
    161,	160,	161,	160,	159,	166,	171,	170,	167,	167,	169,	170,	172,	176,	177,	177,	181,	186,	187,	186,	189,	190,	192,	193,	196,	196,	199,	197,	200,	201,	204,	204,	203,	207,	204,	205,	206,	208,	206,	208,	209,	211,	211,	211,	211,	214,	212,	214,	214,	214,	212,	211,	195,	121,	87,	91,	93,	94,	104,	109,	109,	113,	113,	107,	111,	112,	112,	111,	106,	108,	108,	108,	99,	103,	100,	91,	96,	98,	92,	92,	86,
    160,	162,	159,	159,	164,	164,	167,	167,	168,	166,	168,	168,	174,	173,	172,	175,	179,	184,	187,	190,	187,	195,	190,	195,	199,	195,	194,	199,	201,	201,	203,	203,	207,	205,	205,	204,	206,	208,	208,	210,	210,	210,	211,	213,	211,	211,	213,	213,	212,	213,	210,	207,	198,	142,	85,	90,	92,	94,	106,	103,	109,	111,	113,	109,	113,	113,	112,	112,	109,	109,	104,	106,	99,	111,	99,	93,	96,	99,	98,	93,	95,
    158,	162,	162,	161,	159,	164,	163,	161,	162,	166,	166,	168,	169,	175,	174,	176,	183,	181,	179,	187,	185,	189,	192,	193,	191,	195,	194,	197,	200,	200,	202,	202,	205,	206,	206,	209,	206,	207,	208,	208,	211,	212,	211,	214,	211,	210,	208,	211,	213,	212,	213,	210,	204,	162,	105,	82,	87,	95,	100,	100,	108,	110,	108,	115,	110,	111,	113,	113,	115,	110,	111,	108,	103,	104,	97,	106,	97,	101,	95,	93,	86,
    161,	160,	160,	160,	163,	165,	163,	159,	163,	164,	166,	171,	174,	177,	171,	175,	183,	181,	183,	183,	183,	188,	191,	189,	190,	193,	195,	197,	201,	200,	203,	203,	204,	205,	206,	208,	205,	208,	207,	208,	207,	212,	210,	211,	212,	208,	205,	210,	212,	215,	214,	211,	206,	183,	111,	85,	88,	94,	101,	101,	109,	112,	114,	109,	113,	114,	111,	114,	109,	106,	108,	102,	112,	103,	101,	103,	101,	99,	100,	95,	89,
    153,	161,	163,	158,	168,	164,	167,	163,	162,	165,	166,	168,	172,	179,	172,	174,	176,	176,	180,	180,	184,	190,	189,	190,	195,	191,	195,	197,	198,	200,	201,	203,	205,	206,	206,	205,	205,	208,	209,	209,	211,	212,	210,	209,	212,	210,	206,	210,	213,	212,	212,	214,	205,	197,	141,	86,	87,	91,	95,	95,	98,	105,	108,	109,	114,	117,	116,	112,	113,	110,	108,	106,	107,	102,	101,	103,	101,	99,	101,	94,	93,
    158,	157,	161,	165,	160,	165,	167,	159,	162,	164,	165,	169,	170,	175,	172,	171,	175,	177,	176,	181,	183,	186,	189,	190,	190,	191,	194,	194,	195,	199,	200,	200,	201,	204,	204,	204,	206,	207,	207,	210,	211,	209,	209,	212,	212,	210,	208,	211,	213,	211,	213,	212,	209,	202,	159,	103,	88,	91,	90,	97,	103,	97,	108,	106,	109,	112,	113,	114,	109,	108,	110,	107,	112,	99,	98,	102,	106,	99,	104,	100,	97,
    154,	158,	163,	161,	163,	162,	164,	162,	165,	168,	165,	171,	168,	169,	167,	170,	172,	173,	176,	182,	181,	182,	187,	188,	191,	191,	194,	193,	194,	199,	199,	199,	202,	202,	204,	205,	206,	208,	207,	209,	207,	211,	208,	211,	210,	210,	208,	211,	213,	211,	216,	211,	211,	206,	179,	111,	84,	90,	93,	99,	98,	100,	108,	103,	107,	112,	110,	111,	107,	109,	114,	108,	105,	100,	104,	97,	107,	105,	103,	98,	98,
    152,	159,	160,	160,	160,	163,	160,	165,	165,	164,	169,	168,	165,	166,	170,	169,	173,	175,	178,	178,	182,	182,	188,	188,	190,	188,	190,	194,	196,	196,	196,	200,	200,	202,	204,	205,	207,	205,	208,	209,	208,	209,	209,	211,	210,	210,	209,	209,	214,	215,	211,	213,	212,	207,	194,	128,	86,	85,	94,	95,	103,	108,	101,	105,	103,	110,	109,	111,	107,	107,	107,	114,	104,	109,	102,	98,	98,	104,	110,	103,	102,
    154,	164,	164,	159,	159,	162,	161,	162,	165,	165,	171,	167,	166,	169,	167,	170,	174,	173,	177,	179,	181,	183,	183,	187,	188,	189,	191,	194,	196,	196,	198,	200,	200,	200,	204,	203,	202,	203,	207,	209,	210,	210,	210,	210,	212,	210,	212,	211,	212,	217,	214,	211,	212,	207,	200,	146,	88,	91,	86,	97,	96,	100,	99,	107,	108,	103,	113,	111,	106,	106,	107,	111,	108,	104,	104,	105,	103,	107,	106,	100,	106,
    153,	155,	158,	160,	160,	159,	161,	163,	163,	166,	167,	166,	166,	170,	170,	169,	178,	174,	171,	178,	182,	182,	183,	183,	190,	188,	187,	191,	191,	194,	195,	197,	202,	199,	204,	203,	204,	203,	207,	209,	210,	210,	210,	210,	213,	210,	209,	211,	211,	216,	211,	213,	212,	206,	202,	164,	98,	84,	88,	92,	103,	101,	103,	106,	106,	104,	106,	107,	103,	108,	108,	115,	114,	102,	110,	101,	100,	102,	104,	100,	105};
    return ImageUInt8::fromRaw(img, 68, 81);
    
}


/**
 * @brief Retorna uma imagem de teste simples 17x15.
 * @note Mantém apenas a segunda definição (a primeira está comentada).
 * @return Ponteiro para imagem 17x15.
 */
inline ImageUInt8Ptr getSimpleImage(){
    /*
    auto img=new uint8_t[255]{
        122, 127, 166, 201, 152,  96,  54,  44,  40,  41,  42,  43,  44,
        44,  37, 133, 143, 213, 246, 236, 196, 137,  85,  55,  43,  44,
        45,  35,  40,  42, 133, 168, 231, 242, 246, 246, 228, 172, 111,
        74,  76,  80,  54,  52,  41, 147, 215, 222, 199, 220, 235, 244,
       237, 205, 172, 181, 186, 106,  57,  47, 164, 235, 224, 149, 168,
       208, 231, 244, 248, 246, 246, 230, 133,  58,  62, 140, 224, 237,
       161, 128, 149, 180, 227, 245, 248, 247, 243, 189, 103,  94, 134,
       211, 240, 181, 109, 105, 120, 168, 223, 240, 241, 246, 237, 176,
       110, 117, 188, 244, 210, 111,  74,  86, 144, 215, 230, 219, 227,
       232, 212, 133,  66, 159, 242, 238, 149,  75,  78, 163, 238, 212,
       172, 198, 219, 175, 111,  75, 144, 231, 244, 171,  81, 113, 212,
       222, 149, 108, 115, 137, 118,  99,  78, 139, 222, 245, 185, 115,
       176, 229, 176,  85,  62,  79,  95,  98, 107,  48, 102, 199, 241,
       220, 171, 220, 208, 125,  47,  45,  73,  90,  98, 104,  41,  72,
       171, 240, 242, 233, 226, 149,  65,  39,  60,  97, 104, 106, 112,
        54,  68, 140, 228, 238, 236, 194, 100,  44,  48,  85, 100, 104,
       107, 122,  54,  54,  94, 181, 222, 214, 141,  67,  40,  72,  99,
       105, 106, 109, 123,  54,  48,  59,  95, 145, 158,  84,  52,  60,
        96, 110, 115, 116, 110, 113,  49,  45,  44,  48,  71,  89,  49,
        47,  71,  95, 162, 156, 119, 122, 111};
    */

        auto img=new uint8_t[255]{
            10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
            10, 10,215,255,255,150,150, 10, 10, 10, 10, 10, 10, 10, 10,
            10,150,255,255,255,255,255,150, 10, 10, 10, 10, 10, 10, 10,
            10,215,225,150,225,255,255,255,255,150,150,150, 10, 10, 10,
            10,225,225,150,150,215,255,255,255,255,255,255,150, 10, 10,
            10,225,255,150, 10, 10,150,255,255,255,255,255,150, 10, 10,
            10,215,255,150, 10, 10, 10, 150,225,255,255,255,255,150,10,
            10,150,255,255, 10, 10, 10, 150,225,255,255,255,255,215,10,
            10,150,255,255,150, 10, 10, 150,225,150,150,150,255,150,10,
            10,150,255,255,150, 10, 10, 215,215, 10, 10, 10,150, 10,10,
            10,150,255,255,150, 10,150, 215,150, 10, 10, 10, 10, 10,10,
            10, 10,150,255,225,150,225, 215,150, 10, 10, 10, 10, 10,10,
            10, 10,150,255,255,255,225, 150, 10, 10, 10, 10, 10, 10,10,
            10, 10,150,255,255,255,150,  10, 10, 10, 10, 10, 10, 10,10,
            10, 10, 10,150,225,255,150,  10, 10, 10, 10, 10, 10, 10,10,
            10, 10, 10, 10,150,150, 10,  10, 10, 10, 10, 10, 10, 10,10,
            10, 10, 10, 10, 10, 10, 10,  10, 10, 10, 10, 10, 10, 10,10};
    return ImageUInt8::fromRaw(img, 17, 15);
}

inline ImageUInt8Ptr getICIP14Image(){
    auto img=new uint8_t[49]{
        0, 0, 0, 0, 0, 0, 0,
        0, 4, 4, 4, 7, 7, 7,
        0, 7, 7, 4, 7, 4, 7,
        0, 7, 4, 4, 7, 4, 7,
        0, 4, 4, 4, 7, 4, 7,
        0, 7, 7, 4, 7, 7, 7,
        0, 0, 0, 0, 0, 0, 0
    };
    return ImageUInt8::fromRaw(img, 7, 7);
}

