/*
 * Copyright (c) 2024 MPI-M, Clara Bayley
 *
 * ----- CLEO -----
 * File: createsupers.hpp
 * Project: runcleo
 * Created Date: Tuesday 17th October 2023
 * Author: Clara Bayley (CB)
 * Additional Contributors:
 * -----
 * License: BSD 3-Clause "New" or "Revised" License
 * https://opensource.org/licenses/BSD-3-Clause
 * -----
 * File Description:
 * classes and templated functions required by RunCLEO to create a view of superdroplets
 * (on device) using some initial conditions
 */

#ifndef LIBS_RUNCLEO_CREATESUPERS_HPP_
#define LIBS_RUNCLEO_CREATESUPERS_HPP_

#include <Kokkos_Core.hpp>
#include <Kokkos_Profiling_ScopedRegion.hpp>
#include <iostream>
#include <stdexcept>
#include <string>

#include "../kokkosaliases.hpp"
#include "gridboxes/supersindomain.hpp"
#include "runcleo/gensuperdrop.hpp"

/**
 * @brief Return an initialised view of superdrops in device memory.
 *
 * This function initialises a view of superdrops in device memory by creating
 * a view on the device and copying a host mirror view that is initialised using
 * the `SuperdropInitConds` instance.
 *
 * @tparam SuperdropInitConds The type of the super-droplets' initial conditions data.
 * @param sdic The instance of the super-droplets' initial conditions data.
 * @return A view of superdrops in device memory.
 */
template <typename SuperdropInitConds>
viewd_supers initialise_supers(const SuperdropInitConds &sdic);

/**
 * @brief Return a mirror view of superdrops on host memory.
 *
 * This function initialises a mirror view of superdrops on host memory, using the super-droplet
 * generator instance `GenSuperdrop` to generate the kk'th super-droplet with their initial Gridbox
 * index, spatial coordinates, and attributes. Kokkos::parallel_for is used to perform parallel
 * initialisation of the mirror view, where each superdrop is generated by the provided `gen`
 * function object.
 *
 * The equivalent serial version of the Kokkos::parallel_for([...]) is:
 * @code
 * for (size_t kk(0); kk < ntotsupers; ++kk)
 * {
 *  h_totsupers(kk) = gen(kk);
 * }
 * @endcode
 *
 * @param sdic The instance of the super-droplets' initial conditions data.
 * @param totsupers The view of superdrops on device memory.
 * @return A mirror view of superdrops on host memory.
 */
template <typename SuperdropInitConds>
viewd_supers::HostMirror initialise_supers_on_host(const SuperdropInitConds &sdic,
                                                   const viewd_supers totsupers);

/**
 * @brief Check if superdroplets initialisation is complete.
 *
 * This function checks if the initialisation of supers view is complete by checking if the
 * superdroplets are sorted by ascending gridbox indexes. If the initialisation is incomplete
 * (the superdroplets are not sorted), it throws an exception with an appropriate error message.
 *
 * @param allsupers Struct to handle all the super-droplets in device memory.
 *
 * @throws std::invalid_argument If the initialisation is incomplete i.e. the super-droplets
 * are not ordered correctly.
 */
void is_sdsinit_complete(const SupersInDomain allsupers);

/**
 * @brief Print statement about initialised super-droplets.
 *
 * This function prints information about each superdroplet, including its ID, Gridbox index,
 * spatial coordinates, and attributes.
 *
 * @param totsupers The view of super-droplets in device memory.
 */
void print_supers(const viewd_constsupers totsupers);

/**
 * @brief Create a view of super-droplets in (device) memory.
 *
 * This function creates an ordered view of superdrops in device memory, where the number
 * of superdrops is specified by the parameter `ntotsupers`. The superdrops are
 * ordered by the gridbox indexes and generated using a generator which uses
 * the initial conditions provided by the `SuperdropInitConds` type.
 *
 * Kokkos::Profiling are null pointers unless a Kokkos profiler library has been
 * exported to "KOKKOS_TOOLS_LIBS" prior to runtime so the lib gets dynamically loaded.
 *
 * @tparam SuperdropInitConds The type of the super-droplets' initial conditions data.
 * @param sdic The instance of the super-droplets' initial conditions data.
 * @param gbxindex_max max value for superdroplet gridbox indexes (0 <= sdgbxindex < gbxindex)
 * @return Struct for handling super-droplets in device memory.
 */
template <typename SuperdropInitConds>
SupersInDomain create_supers(const SuperdropInitConds &sdic, const unsigned int gbxindex_max) {
  Kokkos::Profiling::ScopedRegion region("init_supers");

  // Log message and create superdrops using the initial conditions
  std::cout << "\n--- create superdrops ---\ninitialising\n";
  auto totsupers = initialise_supers(sdic);

  // Log message and sort the view of superdrops
  std::cout << "sorting and finding superdrops in domain\n";
  auto allsupers = SupersInDomain(totsupers, gbxindex_max);

  // Log message and perform checks on the initialisation of superdrops
  std::cout << "checking initialisation\n";
  is_sdsinit_complete(allsupers);

  // // Print information about the created superdrops
  // print_supers(totsupers);

  // Log message indicating the successful creation of superdrops
  std::cout << "--- create superdrops: success ---\n";

  return allsupers;
}

/**
 * @brief Return an initialised view of superdrops in device memory.
 *
 * This function initialises a view of superdrops in device memory by creating
 * a view on the device and copying a host mirror view that is initialised using
 * the `SuperdropInitConds` instance.
 *
 * @tparam SuperdropInitConds The type of the super-droplets' initial conditions data.
 * @param sdic The instance of the super-droplets' initial conditions data.
 * @return A view of superdrops in device memory.
 */
template <typename SuperdropInitConds>
viewd_supers initialise_supers(const SuperdropInitConds &sdic) {
  GenSuperdrop gen(sdic);

  // create superdrops view on device
  auto totsupers = viewd_supers("totsupers", gen.get_maxnsupers());

  // initialise a mirror of superdrops view on host
  auto h_totsupers = initialise_supers_on_host(sdic, totsupers);

  // Copy host view to device (h_totsupers to totsupers)
  Kokkos::deep_copy(totsupers, h_totsupers);

  return totsupers;
}

/**
 * @brief Return a mirror view of superdrops on host memory.
 *
 * This function initialises a mirror view of superdrops on host memory, using the super-droplet
 * generator instance `GenSuperdrop` to generate the kk'th super-droplet with their initial Gridbox
 * index, spatial coordinates, and attributes. Kokkos::parallel_for is used to perform parallel
 * initialisation of the mirror view, where each superdrop is generated by the provided `gen`
 * function object.
 *
 * The equivalent serial version of the Kokkos::parallel_for([...]) is:
 * @code
 * for (size_t kk(0); kk < ntotsupers; ++kk)
 * {
 *  h_totsupers(kk) = gen(kk);
 * }
 * @endcode
 *
 * @param sdic The instance of the super-droplets' initial conditions data.
 * @param totsupers The view of superdrops on device memory.
 * @return A mirror view of superdrops on host memory.
 */
template <typename SuperdropInitConds>
viewd_supers::HostMirror initialise_supers_on_host(const SuperdropInitConds &sdic,
                                                   const viewd_supers totsupers) {
  // Create a mirror view of supers in case the original view is on device memory
  auto h_totsupers = Kokkos::create_mirror_view(totsupers);

  // Parallel initialisation of the mirror view
  const auto ntotsupers = totsupers.extent(0);
  const GenSuperdrop gen(sdic);
  Kokkos::parallel_for("initialise_supers_on_host", Kokkos::RangePolicy<HostSpace>(0, ntotsupers),
                       [=](const size_t kk) { h_totsupers(kk) = gen(kk); });

  return h_totsupers;
}

#endif  // LIBS_RUNCLEO_CREATESUPERS_HPP_
